#include "sharing.h"

#include "chargable.h"
#include "notifications_tags.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/library/json/field.h>
#include <rtline/library/json/proto/adapter.h>

namespace {
    const TString GeocodedLocationMacro = "_GeocodedLocation_";
    const TString ModelNameMacro = "_Model_";
    const TString OwnerMacro = "_Owner_";
}

NDrive::TScheme TSessionSharingTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    {
        NDrive::TScheme& localizationScheme = result.Add<TFSArray>("localizations", "Ключи для локализации", 100000).SetElement<NDrive::TScheme>();
        localizationScheme.Add<TFSString>("key", "Ключ текста");
        localizationScheme.Add<TFSString>("value", "Текст (параметризуемый для локализации)");
    }
    result.Add<TFSVariants>("offer_builder_id", "Offer constructor to use").SetVariants(IOfferBuilderAction::GetNames(server));
    result.Add<TFSVariants>("booking_tag", "Booking tag to be added for the owner").SetReference("user_tags");
    result.Add<TFSVariants>("interruption_tag", "Interruption tag to be added for the user").SetReference("user_tags");
    result.Add<TFSVariants>("invitation_tag", "Invitation tag to be added for the user").SetReference("user_tags");
    result.Add<TFSVariants>("rejection_tag", "Rejection tag to be added for the owner").SetReference("user_tags");
    result.Add<TFSVariants>("revocation_tag", "Revocation tag to be added for the user").SetReference("user_tags");
    result.Add<TFSVariants>("visibility_tag", "Tag to be added while booking").SetReference("device_tags");
    return result;
}

NJson::TJsonValue TSessionSharingTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TSessionSharingTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

TDBTag TSessionSharingTag::Get(const TString& sessionId, TStringBuf userId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(sessionId, tx);
    R_ENSURE(optionalSession, {}, "cannot GetSession", tx);
    auto session = *optionalSession;
    R_ENSURE(session, HTTP_NOT_FOUND, "cannot find session " << sessionId, tx);
    R_ENSURE(!session->GetClosed(), HTTP_BAD_REQUEST, "session " << sessionId << " is already closed", tx);
    return Get(session, userId, server, tx);
}

TDBTag TSessionSharingTag::Get(ISession::TConstPtr session, TStringBuf userId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    const auto& tagsManager = Yensured(server.GetDriveAPI())->GetTagsManager();
    auto optionalExistingTags = tagsManager.GetUserTags().RestoreTags(TVector<TString>{ Yensured(session)->GetUserId() }, { Type() }, tx);
    R_ENSURE(optionalExistingTags, {}, "cannot restore existing sharing tags", tx);
    for (auto&& tag : *optionalExistingTags) {
        auto impl = tag.GetTagAs<TSessionSharingTag>();
        R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast tag " << tag.GetTagId() << " to SessionSharingTag", tx);
        if (impl->GetSessionId() == session->GetSessionId() && impl->GetUserId() == userId) {
            return std::move(tag);
        }
    }
    return {};
}

TDBTag TSessionSharingTag::Book(TDBTag&& sessionSharingTag, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    auto impl = sessionSharingTag.MutableTagAs<TSessionSharingTag>();
    R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast " << sessionSharingTag.GetTagId() << " to SessionSharingTag", tx);
    R_ENSURE(impl->GetUserId() == permissions.GetUserId(), HTTP_FORBIDDEN, "user_id mismatch " << impl->GetUserId() << ' ' << permissions.GetUserId(), tx);
    R_ENSURE(impl->GetPerformer() == permissions.GetUserId(), HTTP_FORBIDDEN, "performer mismatch " << impl->GetPerformer() << ' ' << permissions.GetUserId(), tx);

    auto description = impl->GetDescriptionAs<TDescription>(server, tx);
    if (!description) {
        return {};
    }

    const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
    auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(impl->GetSessionId(), tx);
    R_ENSURE(optionalSession, {}, "cannot GetSession", tx);
    auto session = *optionalSession;
    R_ENSURE(session, HTTP_NOT_FOUND, "cannot find session " << impl->GetSessionId(), tx);
    R_ENSURE(!session->GetClosed(), HTTP_FORBIDDEN, "session " << impl->GetSessionId() << " is closed", tx);

    IOffer::TPtr offer;
    {
        auto action = server.GetDriveDatabase().GetUserPermissionManager().GetAction(description->GetOfferBuilderId());
        R_ENSURE(action, HTTP_INTERNAL_SERVER_ERROR, "cannot find action " << description->GetOfferBuilderId(), tx);
        auto builder = action->GetAs<IOfferBuilderAction>();
        R_ENSURE(builder, HTTP_INTERNAL_SERVER_ERROR, "cannot cast action " << description->GetOfferBuilderId() << " to OfferBuilder", tx);

        TUserOfferContext uoc{&server, permissions.Self()};
        uoc.SetFilterAccounts(false);
        TOffersBuildingContext context(std::move(uoc));
        context.InitializeCar(session->GetObjectId());
        context.SetVisibilityRequired(false);

        auto offers = builder->BuildOffers(context, tx, /*useCorrectors=*/false);
        R_ENSURE(offers.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "invalid offers count: " << offers.size(), tx);
        R_ENSURE(offers.front(), HTTP_INTERNAL_SERVER_ERROR, "null offer created", tx);
        offer = offers.front()->GetOfferPtrAs<IOffer>();
    }
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "cannot create offer", tx);
    offer->SetSharedSessionId(impl->GetSessionId());
    {
        auto to = tagsManager.GetTagsMeta().CreateTag(TChargableTag::Sharing);
        R_ENSURE(to, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << TChargableTag::Sharing, tx);
        auto sharing = std::dynamic_pointer_cast<TChargableTag>(to);
        R_ENSURE(sharing, HTTP_INTERNAL_SERVER_ERROR, "cannot cast tag " << to->GetName() << " to ChargableTag", tx);
        sharing->SetSharedSessionId(offer->GetOfferId());

        auto sessionTag = tagsManager.GetDeviceTags().RestoreTag(session->GetInstanceId(), tx);
        if (!sessionTag) {
            tx.Check();
        }
        bool evolvedToSharing = TChargableTag::DirectEvolve(*sessionTag, sharing, permissions, server, tx, nullptr);
        if (!evolvedToSharing) {
            tx.Check();
        }
    }
    {
        auto reservation = tagsManager.GetTagsMeta().CreateTag(TChargableTag::Reservation);
        R_ENSURE(reservation, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << TChargableTag::Reservation, tx);
        auto added = tagsManager.GetDeviceTags().AddTag(reservation, permissions.GetUserId(), session->GetObjectId(), &server, tx);
        if (!added) {
            tx.Check();
        }
    }
    auto visibilityTag = TDBTag();
    {
        auto visibility = tagsManager.GetTagsMeta().CreateTag(description->GetVisibilityTag());
        R_ENSURE(visibility, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << description->GetVisibilityTag(), tx);
        visibility->SetPerformer(permissions.GetUserId());
        auto added = tagsManager.GetDeviceTags().AddTag(visibility, permissions.GetUserId(), session->GetObjectId(), &server, tx, EUniquePolicy::NoUnique);
        R_ENSURE(added, {}, "cannot add VisibilityTag " << visibility->GetName(), tx);
        R_ENSURE(added->size() == 1, HTTP_INTERNAL_SERVER_ERROR, "unexpected number of VisibilityTags added: " << added->size(), tx);
        visibilityTag = std::move(added->front());
    }

    TChargableTag::TBookOptions bookOptions;
    bookOptions.MultiRent = true;
    auto booked = TChargableTag::Book(offer, permissions, server, tx, bookOptions);
    if (!booked) {
        tx.Check();
    }
    if (visibilityTag) {
        const bool removed = tagsManager.GetDeviceTags().RemoveTag(visibilityTag, permissions.GetUserId(), &server, tx, /*force=*/true);
        R_ENSURE(removed, {}, "cannot remove VisibilityTag", tx);
    }
    if (impl->GetSharingType() == ESharingType::OneTime) {
        impl->SkipNotification = true;
        const bool removed = tagsManager.GetUserTags().RemoveTag(sessionSharingTag, permissions.GetUserId(), &server, tx, /*force=*/true);
        R_ENSURE(removed, {}, "cannot remove SessionSharingTag", tx);
        const auto& bookingTagName = description->GetBookingTag();
        if (bookingTagName) {
            auto bookingTag = tagsManager.GetTagsMeta().CreateTag(bookingTagName);
            R_ENSURE(bookingTag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << bookingTagName, tx);

            auto genericUserState = std::dynamic_pointer_cast<TGenericUserStateTag>(bookingTag);
            if (genericUserState) {
                genericUserState->SetSessionId(impl->GetSessionId());
                genericUserState->SetUserId(impl->GetUserId());
            }

            auto addedTags = tagsManager.GetUserTags().AddTag(bookingTag, permissions.GetUserId(), session->GetUserId(), &server, tx);
            R_ENSURE(addedTags, {}, "cannot add BookingTag", tx);
        }
    }
    tx.Committed().Subscribe([
        offerId = offer->GetOfferId(),
        sessionSharingTag = std::move(sessionSharingTag)
    ](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            NDrive::TEventLog::Log("BookSharedSession", NJson::TMapBuilder
                ("offer_id", offerId)
                ("session_sharing_tag", NJson::ToJson(sessionSharingTag))
            );
        }
    });
    return booked;
}

TDBTag TSessionSharingTag::Invite(ISession::TConstPtr session, const TString& userId, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    auto existing = Get(session, userId, server, tx);
    if (existing) {
        return existing;
    }

    const auto& tagsManager = Yensured(server.GetDriveAPI())->GetTagsManager();
    auto tag = tagsManager.GetTagsMeta().CreateTag(Type());
    R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << Type(), tx);
    auto impl = std::dynamic_pointer_cast<TSessionSharingTag>(tag);
    R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast tag " << tag->GetName() << " to SessionSharingTag", tx);
    impl->SetPerformer(userId);
    impl->SetSessionId(Yensured(session)->GetSessionId());
    impl->SetUserId(userId);

    {
        auto owner = server.GetDriveDatabase().GetUserManager().Fetch(session->GetUserId(), tx);
        R_ENSURE(owner, {}, "cannot fetch owner info", tx);
        impl->MutableMacros().emplace(OwnerMacro, owner->GetShortName());
    }
    {
        auto deviceFetchInfo = server.GetDriveDatabase().GetCarManager().FetchInfo(session->GetObjectId(), tx);
        auto device = deviceFetchInfo.GetResultPtr(session->GetObjectId());
        R_ENSURE(device, {}, "cannot fetch device " << session->GetObjectId() << " info", tx);
        auto model = server.GetDriveDatabase().GetModelsDB().Fetch(device->GetModel(), tx);
        R_ENSURE(model, {}, "cannot fetch model " << device->GetModel() << " info", tx);
        impl->MutableMacros().emplace(ModelNameMacro, model->GetName());
    }
    {
        auto snapshot = server.GetSnapshotsManager().GetSnapshot(session->GetObjectId());
        auto geocodedLocation = snapshot.GetGeocoded();
        impl->MutableMacros().emplace(GeocodedLocationMacro, geocodedLocation ? geocodedLocation->Content : "–");
    }

    auto added = tagsManager.GetUserTags().AddTag(tag, permissions.GetUserId(), session->GetUserId(), &server, tx);
    R_ENSURE(added, {}, "cannot AddTag " << tag->GetName(), tx);
    R_ENSURE(added->size() == 1, HTTP_INTERNAL_SERVER_ERROR, "incorrect number of tags added: " << added->size(), tx);
    return std::move(added->front());
}

bool TSessionSharingTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(self);
    if (SkipNotification) {
        return true;
    }

    if (!server) {
        tx.SetErrorInfo("SessionSharingTag::OnAfterAdd", "no Server");
        return false;
    }
    auto description = GetDescriptionAs<TDescription>(*server, tx);
    if (!description) {
        return false;
    }

    const auto& invitationTagName = description->GetInvitationTag();
    const auto& tagsManager = server->GetDriveDatabase().GetTagsManager();
    if (invitationTagName) {
        auto invitationTag = tagsManager.GetTagsMeta().CreateTag(invitationTagName);
        if (!invitationTag) {
            tx.SetErrorInfo("SessionSharingTag::OnAfterAdd", "cannot create tag " + invitationTagName);
            return false;
        }

        auto userMessageTag = std::dynamic_pointer_cast<TUserMessageTagBase>(invitationTag);
        if (userMessageTag) {
            for (auto&& [name, value] : Macros) {
                userMessageTag->AddMacro(name, value);
            }
        }
        auto addedTags = tagsManager.GetUserTags().AddTag(invitationTag, userId, UserId, server, tx);
        if (!addedTags) {
            return false;
        }
    }
    return true;
}

bool TSessionSharingTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(self);
    if (SkipNotification) {
        return true;
    }
    if (!server) {
        tx.SetErrorInfo("SessionSharingTag::OnAfterRemove", "no Server");
        return false;
    }

    const auto description = GetDescriptionAs<TDescription>(*server, tx);
    if (!description) {
        return false;
    }

    const auto& rejectionTagName = description->GetRejectionTag();
    const auto& revocationTagName = description->GetRevocationTag();
    const auto& tagsManager = server->GetDriveDatabase().GetTagsManager();
    if (userId == UserId && rejectionTagName) {
        auto rejectionTag = tagsManager.GetTagsMeta().CreateTag(rejectionTagName);
        if (!rejectionTag) {
            tx.SetErrorInfo("SessionSharingTag::OnAfterRemove", "cannot create tag " + rejectionTagName);
            return false;
        }

        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetSession(SessionId, tx);
        if (!optionalSession) {
            return false;
        }
        auto session = *optionalSession;
        if (!session) {
            tx.SetErrorInfo("SessionSharingTag::OnAfterRemove", "cannot find session " + SessionId);
            return false;
        }

        auto genericUserState = std::dynamic_pointer_cast<TGenericUserStateTag>(rejectionTag);
        if (genericUserState) {
            genericUserState->SetSessionId(SessionId);
            genericUserState->SetUserId(UserId);
        }

        auto addedTags = tagsManager.GetUserTags().AddTag(rejectionTag, userId, session->GetUserId(), server, tx);
        if (!addedTags) {
            return false;
        }
    }
    if (userId != UserId && revocationTagName) {
        auto revocationTag = tagsManager.GetTagsMeta().CreateTag(revocationTagName);
        if (!revocationTag) {
            tx.SetErrorInfo("SessionSharingTag::OnAfterRemove", "cannot create tag " + revocationTagName);
            return false;
        }

        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetSession(SessionId, tx);
        if (!optionalSession) {
            return false;
        }
        auto session = *optionalSession;
        if (!session) {
            tx.SetErrorInfo("SessionSharingTag::OnAfterRemove", "cannot find session " + SessionId);
            return false;
        }

        auto genericUserState = std::dynamic_pointer_cast<TGenericUserStateTag>(revocationTag);
        if (genericUserState) {
            genericUserState->SetCarId(session->GetObjectId());
            genericUserState->SetUserId(session->GetUserId());
        }

        auto addedTags = tagsManager.GetUserTags().AddTag(revocationTag, userId, UserId, server, tx);
        if (!addedTags) {
            return false;
        }
    }
    return true;
}

NJson::TJsonValue TSessionSharingTag::GetStateReport(ELocalization locale, const TConstDBTag& self, const TUserPermissions& permissions, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    auto sharingTypeString = ToString(SharingType);
    NJson::TJsonValue result;
    result["sharing_type"] = sharingTypeString;
    result["session_id"] = SessionId;
    result["tag_id"] = self.GetTagId();
    if (permissions.GetUserId() == UserId) {
        result["name"] = "shared_session";
    } else {
        auto user = server.GetDriveDatabase().GetUsersData()->GetCachedObject(UserId);
        result["name"] = "sharing_session";
        result["user"] = user->GetPublicReport();
    }
    auto getLocalString = [&](const TString& key) {
        auto value = localization ? localization->GetLocalString(locale, key) : key;
        for (auto&& [macro, substitute] : Macros) {
            SubstGlobal(value, macro, substitute);
        }
        return value;
    };
    result["info_subtitle"] = getLocalString(TStringBuilder() << "sharing." << sharingTypeString << ".info_subtitle");
    {
        NJson::TJsonValue item;
        item["key"] = getLocalString(TStringBuilder() << "sharing." << sharingTypeString << ".items.place.title");
        item["value"] = getLocalString(GeocodedLocationMacro);
        result["info_items"].AppendValue(std::move(item));
    }
    {
        NJson::TJsonValue item;
        item["key"] = getLocalString(TStringBuilder() << "sharing." << sharingTypeString << ".items.owner.title");
        item["value"] = getLocalString(OwnerMacro);
        result["info_items"].AppendValue(std::move(item));
    }

    auto description = GetDescriptionAs<TDescription>(server);
    R_ENSURE(description, {}, "cannot get description for " << GetName());
    for (auto&& [name, key] : description->GetLocalizations()) {
        result["localizations"].InsertValue(name, getLocalString(key));
    }
    return result;
}

bool TSessionSharingTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return NJson::TryFieldsFromJson(value, GetFields(), errors);
}

void TSessionSharingTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

TSharingSession::TSharingSession(ISession::TConstPtr real, ISession::TConstPtr shared)
    : ISession(*Yensured(real))
    , Real(std::move(real))
    , Shared(std::move(shared))
{
}

NJson::TJsonValue TSharingSession::DoGetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const {
    if (!Shared) {
        return NJson::JSON_NULL;
    }
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(Real);
    auto billingCustomization = std::dynamic_pointer_cast<TBillingReportCustomization>(customization);
    auto offer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
    auto standardOffer = std::dynamic_pointer_cast<TStandartOffer>(offer);
    if (billingCustomization) {
        auto cloned = MakeAtomicShared<TBillingReportCustomization>(*billingCustomization);
        cloned->ClearPaymentsData();
        if (standardOffer) {
            cloned->SetAgreement(standardOffer->GetAgreement());
        }
        customization = cloned;
    }
    return Shared->GetReportImpl(locale, server, customization);
}

template <>
NJson::TJsonValue NJson::ToJson(const TSessionSharingTag::ESharingType& object) {
    return ToString(object);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TSessionSharingTag::ESharingType& result) {
    return TryFromJson(value, Stringify(result));
}

TSessionSharingTag::TFactory::TRegistrator<TSessionSharingTag> TSessionSharingTag::Registrator(TSessionSharingTag::Type());
TTagDescription::TFactory::TRegistrator<TSessionSharingTag::TDescription> SessionSharingTagDescriptionRegistrator(TSessionSharingTag::Type());
