#pragma once

#include "device_tags.h"
#include "state.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/history/session.h>

#include <rtline/util/types/field.h>

class TSessionSharingTag
    : public IJsonSerializableTag
    , public NDrive::IUserState
{
private:
    using TBase = IJsonSerializableTag;
    using TStringDict = TMap<TString, TString>;

public:
    enum class ESharingType {
        OneTime     /* "one_time" */,
        Permanent   /* "permanent" */,
    };

    class TDescription: public TTagDescription {
    public:
        using TTagDescription::TTagDescription;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        R_FIELD(TStringDict, Localizations);
        R_FIELD(TString, OfferBuilderId);
        R_FIELD(TString, BookingTag);
        R_FIELD(TString, InterruptionTag);
        R_FIELD(TString, InvitationTag);
        R_FIELD(TString, RejectionTag);
        R_FIELD(TString, RevocationTag);
        R_FIELD(TString, VisibilityTag, TDeviceTagRecord::TypeName);

        DECLARE_FIELDS(
            Field(NJson::KeyValue(Localizations), "localizations"),
            Field(OfferBuilderId, "offer_builder_id"),
            Field(BookingTag, "booking_tag"),
            Field(InterruptionTag, "interruption_tag"),
            Field(InvitationTag, "invitation_tag"),
            Field(RejectionTag, "rejection_tag"),
            Field(RevocationTag, "revocation_tag"),
            Field(VisibilityTag, "visibility_tag")
        );
    };

public:
    R_FIELD(ESharingType, SharingType, ESharingType::OneTime);
    R_FIELD(TStringDict, Macros);
    R_FIELD(TString, SessionId);
    R_FIELD(TString, UserId);

    DECLARE_FIELDS(
        Field(Macros, "macros"),
        Field(SharingType, "sharing_type"),
        Field(SessionId, "session_id"),
        Field(UserId, "user_id")
    );

public:
    static TString Type() {
        return "session_sharing_tag";
    }

    static TDBTag Get(const TString& sessionId, TStringBuf userId, const NDrive::IServer& server, NDrive::TEntitySession& tx);
    static TDBTag Get(ISession::TConstPtr session, TStringBuf userId, const NDrive::IServer& server, NDrive::TEntitySession& tx);
    static TDBTag Book(TDBTag&& sessionSharingTag, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);
    static TDBTag Invite(ISession::TConstPtr session, const TString& userId, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);

public:
    using TBase::TBase;

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    NJson::TJsonValue GetStateReport(ELocalization locale, const TConstDBTag& self, const TUserPermissions& permissions, const NDrive::IServer& server) const override;

    bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;
    bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    bool SkipNotification = false;

private:
    static TFactory::TRegistrator<TSessionSharingTag> Registrator;
};

class TSharingSession: public ISession {
public:
    TSharingSession(ISession::TConstPtr real, ISession::TConstPtr shared);

    ISession::TConstPtr GetReal() const {
        return Real;
    }

    const TString& GetSessionId() const override {
        return Real->GetSessionId();
    }

protected:
    bool DoCompile() const override {
        return Real->Compile() && Shared && Shared->Compile();
    }
    NJson::TJsonValue DoGetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const override;

private:
    ISession::TConstPtr Real;
    ISession::TConstPtr Shared;
};
