#include "support_tags.h"

#include "container_tag.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/chat/engine.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/support_center/ifaces.h>
#include <drive/backend/support_center/yandex/client.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/instant_model.h>

#include <util/string/split.h>

bool IDeferrableTag::ExecDeferChatActions(const TString& userId, const TString& actorId, const TDeferInfo& deferInfo, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    if (deferInfo.GetNextNode().empty() && deferInfo.GetMessages().empty()) {
        return true;
    }
    if (server->GetDriveAPI()->GetDatabaseName() == server->GetChatEngine()->GetDatabaseName()) {
        return ExecDeferChatActions(userId, actorId, deferInfo, session, session, server);
    } else {
        auto chatSession = server->GetChatEngine()->BuildSession();
        if (!ExecDeferChatActions(userId, actorId, deferInfo, session, chatSession, server)) {
            return false;
        }
        if (!chatSession.Commit()) {
            session.AddErrorMessage("IDeferrableTag", "can't commit chat session");
            session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
        return true;
    }
}

bool IDeferrableTag::ExecDeferChatActions(const TString& userId, const TString& actorId, const TDeferInfo& deferInfo, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession, const NDrive::IServer* server) const {
    TString chatId, topic;
    IChatRobot::ParseTopicLink(deferInfo.GetTopicLink(), chatId, topic);
    auto chatRobot = server->GetChatRobot(chatId);
    if (!chatRobot) {
        session.AddErrorMessage("IDeferrableTag", "chat robot " + chatId + " not configured");
        return false;
    }
    if (!chatRobot->AcceptMessages(userId, topic, deferInfo.MutableMessages(), actorId, chatSession)) {
        session.AddErrorMessage("IDeferrableTag", "could not send arbitrary message");
        return false;
    }
    if (deferInfo.GetNextNode() && !chatRobot->MoveToStep(userId, topic, deferInfo.GetNextNode(), &chatSession, true)) {
        session.AddErrorMessage("IDeferrableTag", "can't not move to step " + deferInfo.GetNextNode());
        return false;
    }
    return true;
}

bool IDeferrableTag::ExecDeferTagActions(TDBTag& self, const TDeferInfo& deferInfo, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    if (deferInfo.GetDropPerformer()) {
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().DropPerformer({}, {self.GetTagId()}, permissions, server, session, true)) {
            session.AddErrorMessage("IDeferrableTag", "can't drop performer");
            return false;
        }
        self->SetPerformer("");
    }
    if (deferInfo.GetRemoveTag()) {
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(self, permissions.GetUserId(), server, session, true)) {
            session.AddErrorMessage("IDeferrableTag", "can't remove tags");
            return false;
        }
    } else if (deferInfo.GetEvolveTag() && deferInfo.GetNextTag()) {
        ITag::TPtr targetEvolveTag;
        if (deferInfo.GetCopyTag()) {
            auto tagDestination = self.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            auto destTagImpl = tagDestination.MutableTagAs<ITopicLinkOwner>();
            if (deferInfo.GetOriginalSupportLine() && destTagImpl) {
                destTagImpl->SetOriginalSupportLine(deferInfo.GetOriginalSupportLine());
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tagDestination, permissions.GetUserId(), session)) {
                    session.AddErrorMessage("IDeferrableTag", "can't update tag data");
                    return false;
                }
            }
            tagDestination->SetName(deferInfo.GetNextTag());
            NJson::TJsonValue json;
            tagDestination->SerializeSpecialDataToJson(json);
            json["tag_name"] = deferInfo.GetNextTag();
            targetEvolveTag = IJsonSerializableTag::BuildFromJson(server->GetDriveAPI()->GetTagsManager(), json);
        } else {
            targetEvolveTag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(deferInfo.GetNextTag());
        }
        if (!deferInfo.GetDropPerformer()) {
            targetEvolveTag->SetPerformer(self->GetPerformer());
        }
        if (!targetEvolveTag) {
            session.AddErrorMessage("IDeferrableTag", "can't convert to evolve target tag");
            return false;
        }
        if (deferInfo.GetUntilSLA()) {
            targetEvolveTag->SetSLAInstant(deferInfo.GetUntilSLA());
        }
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(self, targetEvolveTag, permissions, server, session)) {
            session.AddErrorMessage("IDeferrableTag", "can't direct evolve");
            return false;
        }
    }
    return true;
}

bool IDeferrableTag::ExecDeferActions(TDBTag& self, TDeferInfo& deferInfo, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    deferInfo.SetMessages(messages);
    return (ExecDeferChatActions(self.GetObjectId(), permissions.GetUserId(), deferInfo, session, server) &&
            ExecDeferTagActions(self, deferInfo, permissions, session, server));
}

bool IDeferrableTag::OnDeferUntil(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TInstant untilSLA, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    TDeferInfo deferInfo = GetDeferUntilInfo(untilSLA, server);
    return ExecDeferActions(self, deferInfo, messages, permissions, session, server);
}

bool IDeferrableTag::OnDefer(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    TDeferInfo deferInfo = GetDeferInfo(server);
    return ExecDeferActions(self, deferInfo, messages, permissions, session, server);
}

bool IDeferrableTag::OnClose(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const {
    TDeferInfo deferInfo = GetDeferCloseInfo(self, server);
    return ExecDeferActions(self, deferInfo, messages, permissions, session, server);
}

const TString TSupportChatTag::TypeName = "user_support_chat_tag";
const TString TSupportChatTag::DeferredName = "support_chat_deferred";
const TString TSupportChatTag::FeedbackName = "support_chat_waiting_feedback";
const TString TSupportChatTag::DeferredContainerName = "support_chat_container";
ITag::TFactory::TRegistrator<TSupportChatTag> TSupportChatTag::Registrator(TSupportChatTag::TypeName);
TSupportChatTag::TDescription::TFactory::TRegistrator<TSupportChatTag::TDescription> TSupportChatTag::TDescription::Registrator(TSupportChatTag::TypeName);

NDrive::TScheme TSupportChatTag::TBehaviour::GetScheme(const NDrive::IServer* server, const TString& evolveToTagTypeName) {
    NDrive::TScheme result;
    auto tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({evolveToTagTypeName});
    result.Add<TFSVariants>("evolve_to_tag", "Эволюционировать в тег").SetVariants(tags);
    result.Add<TFSString>("move_to_node", "Переводить в вершину");
    result.Add<TFSBoolean>("remove_tag", "Снимать тег");
    result.Add<TFSBoolean>("drop_performer", "Снимать исполнителя");
    return result;
}

NJson::TJsonValue TSupportChatTag::TBehaviour::SerializeToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    NJson::InsertField(jsonMeta, "evolve_to_tag", EvolveToTag);
    NJson::InsertField(jsonMeta, "move_to_node", MoveToNode);
    NJson::InsertField(jsonMeta, "remove_tag", RemoveTag);
    NJson::InsertField(jsonMeta, "drop_performer", DropPerformer);
    return jsonMeta;
}

bool TSupportChatTag::TBehaviour::DeserializeFromJson(const NJson::TJsonValue& jsonMeta) {
    return
        NJson::ParseField(jsonMeta, "evolve_to_tag", EvolveToTag, false) &&
        NJson::ParseField(jsonMeta, "move_to_node", MoveToNode, false) &&
        NJson::ParseField(jsonMeta, "remove_tag", RemoveTag, false) &&
        NJson::ParseField(jsonMeta, "drop_performer", DropPerformer, false);
}

NJson::TJsonValue TSupportChatTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TDescription::TBase::DoSerializeMetaToJson();
    NJson::InsertField(jsonMeta, "operator_entrance_message", OperatorEntranceMessage);
    NJson::InsertField(jsonMeta, "operator_name", OperatorName);
    NJson::InsertField(jsonMeta, "redirect_chat_node", RedirectChatNode);
    NJson::InsertField(jsonMeta, "need_notification", NeedNotification);
    NJson::InsertField(jsonMeta, "on_defer_behaviour", OnDeferBehaviour.SerializeToJson());
    NJson::InsertField(jsonMeta, "on_defer_until_behaviour", OnDeferUntilBehaviour.SerializeToJson());
    NJson::InsertField(jsonMeta, "on_close_behaviour", OnCloseBehaviour.SerializeToJson());
    TJsonProcessor::WriteContainerArrayStrings(jsonMeta, "included_filter_message_types", IncludedFilterMessageTypes);
    return jsonMeta;
}

bool TSupportChatTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    if (jsonMeta.Has("on_defer_behaviour") && !OnDeferBehaviour.DeserializeFromJson(jsonMeta["on_defer_behaviour"])) {
        return false;
    }
    if (jsonMeta.Has("on_defer_until_behaviour") && !OnDeferUntilBehaviour.DeserializeFromJson(jsonMeta["on_defer_until_behaviour"])) {
        return false;
    }
    if (jsonMeta.Has("on_close_behaviour") && !OnCloseBehaviour.DeserializeFromJson(jsonMeta["on_close_behaviour"])) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonMeta, "included_filter_message_types", IncludedFilterMessageTypes)) {
        return false;
    }
    return
        TDescription::TBase::DoDeserializeMetaFromJson(jsonMeta) &&
        NJson::ParseField(jsonMeta, "operator_entrance_message", OperatorEntranceMessage) &&
        NJson::ParseField(jsonMeta, "operator_name", OperatorName) &&
        NJson::ParseField(jsonMeta, "redirect_chat_node", RedirectChatNode) &&
        NJson::ParseField(jsonMeta, "need_notification", NeedNotification, false);
}

NDrive::TScheme TSupportChatTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TDescription::TBase::GetScheme(server);
    result.Add<TFSString>("operator_entrance_message", "Какое сообщение выводить по входу оператора");
    result.Add<TFSString>("operator_name", "Подпись оператора в чате");
    result.Add<TFSString>("redirect_chat_node", "В какую ноду переводить чат после эволюции");
    result.Add<TFSBoolean>("need_notification", "Высылать пуши для непрочитанных сообщений");
    result.Add<TFSStructure>("on_defer_behaviour", "Поведение при откладывании").SetStructure(TBehaviour::GetScheme(server, TSupportChatTag::TypeName));
    result.Add<TFSStructure>("on_defer_until_behaviour", "Поведение при откладывании до даты").SetStructure(TBehaviour::GetScheme(server, TUserContainerTag::TypeName));
    result.Add<TFSStructure>("on_close_behaviour", "Поведение при закрытии").SetStructure(TBehaviour::GetScheme(server, TSupportChatTag::TypeName));
    result.Add<TFSVariants>("included_filter_message_types", "Сообщения, которые учитываются при фильтрации").InitVariants<EInsignificantMessageType>().SetMultiSelect(true);
    return result;
}

void TSupportChatTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json["feedback"] = Feedback.SerializeToJson();
    json["muted"] = Muted;
    json["initial_node"] = InitialNode;
    json["chat_title"] = ChatTitle;
    json["external_chat"] = ExternalChatInfo.SerializeToJson();
    if (Meta.IsDefined()) {
        json["chat_meta"] = Meta;
    }
}

bool TSupportChatTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (json.Has("feedback")) {
        Feedback.DeserializeFromJson(json["feedback"]);
    }
    JREAD_BOOL_OPT(json, "muted", Muted);
    JREAD_STRING_OPT(json, "initial_node", InitialNode);
    JREAD_STRING_OPT(json, "chat_title", ChatTitle);
    if (json.Has("external_chat") && !ExternalChatInfo.DeserializeFromJson(json["external_chat"])) {
        return false;
    }
    Meta = json["chat_meta"];
    return TBase::DoSpecialDataFromJson(json, errors);
}

TSupportChatTag::TProto TSupportChatTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    Feedback.SerializeToProto(proto.MutableFeedback());
    ExternalChatInfo.SerializeToProto(proto.MutableExternalChatInfo());
    proto.SetMuted(Muted);
    proto.SetInitialNode(InitialNode);
    proto.SetChatTitle(ChatTitle);
    if (Meta.IsDefined()) {
        proto.SetMeta(Meta.GetStringRobust());
    }
    return proto;
}

bool TSupportChatTag::DoDeserializeSpecialDataFromProto(const TSupportChatTag::TProto& proto) {
    Feedback.DeserializeFromProto(proto.GetFeedback());
    ExternalChatInfo.DeserializeFromProto(proto.GetExternalChatInfo());
    Muted = proto.GetMuted();
    InitialNode = proto.GetInitialNode();
    ChatTitle = proto.GetChatTitle();
    Meta = NJson::ToJson(NJson::JsonString(proto.GetMeta()));
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

TString TSupportChatTag::BuildChatLink(const TString& objectId) const {
    return "<a href=\"" + TCarsharingUrl().ChatPage(objectId, TopicLink) + "\">" + GetName() + "</a>";
}

NDrive::TScheme TSupportChatTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = ISerializableTag<TProto>::GetScheme(server);
    result.Add<TFSString>("topic_link", "Ссылка на коммуникацию с клиентом").SetDefault("support." + ToString(ModelingNow().Seconds()));
    result.Add<TFSString>("initial_node", "Сразу поместить чат в ноду");
    result.Add<TFSString>("chat_title", "Заголовок чата");
    result.Add<TFSStructure>("external_chat", "Данные о чатах от внешней системы").SetStructure(TExternalChatInfo::GetScheme());
    return result;
}

IDeferrableTag::TDeferInfo TSupportChatTag::GetDeferInfo(const NDrive::IServer* server) const {
    IDeferrableTag::TDeferInfo deferInfo;
    auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TSupportChatTag::TDescription>();
    deferInfo.SetNextTag(description->GetOnDeferBehaviour().GetEvolveToTag());
    deferInfo.SetNextNode(description->GetOnDeferBehaviour().GetMoveToNode());
    deferInfo.SetRemoveTag(description->GetOnDeferBehaviour().GetRemoveTag());
    deferInfo.SetDropPerformer(description->GetOnDeferBehaviour().GetDropPerformer());
    deferInfo.SetEvolveTag(true);
    deferInfo.SetCopyTag(true);
    deferInfo.SetTopicLink(GetTopicLink());
    if (GetOriginalSupportLine() != GetName() && GetName() != TSupportChatTag::DeferredName && GetName() != deferInfo.GetNextTag()) {
        deferInfo.SetOriginalSupportLine(GetName());
    }
    return deferInfo;
}

IDeferrableTag::TDeferInfo TSupportChatTag::GetDeferUntilInfo(TInstant deferUntil, const NDrive::IServer* server) const {
    IDeferrableTag::TDeferInfo deferInfo;
    auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TSupportChatTag::TDescription>();
    deferInfo.SetNextTag(description->GetOnDeferUntilBehaviour().GetEvolveToTag());
    deferInfo.SetNextNode(description->GetOnDeferUntilBehaviour().GetMoveToNode());
    deferInfo.SetRemoveTag(description->GetOnDeferUntilBehaviour().GetRemoveTag());
    deferInfo.SetDropPerformer(description->GetOnDeferUntilBehaviour().GetDropPerformer());
    deferInfo.SetEvolveTag(true);
    deferInfo.SetTopicLink(GetTopicLink());
    deferInfo.SetUntilSLA(deferUntil);
    if (GetOriginalSupportLine() != GetName() && GetName() != TSupportChatTag::DeferredContainerName && GetName() != deferInfo.GetNextTag()) {
        deferInfo.SetOriginalSupportLine(GetName());
    }
    return deferInfo;
}

IDeferrableTag::TDeferInfo TSupportChatTag::GetDeferCloseInfo(const TDBTag& /*self*/, const NDrive::IServer* server) const {
    IDeferrableTag::TDeferInfo deferInfo;
    auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TSupportChatTag::TDescription>();
    deferInfo.SetNextTag(description->GetOnCloseBehaviour().GetEvolveToTag());
    deferInfo.SetNextNode(description->GetOnCloseBehaviour().GetMoveToNode());
    deferInfo.SetRemoveTag(description->GetOnCloseBehaviour().GetRemoveTag());
    deferInfo.SetDropPerformer(description->GetOnCloseBehaviour().GetDropPerformer());
    deferInfo.SetEvolveTag(true);
    deferInfo.SetCopyTag(true);
    deferInfo.SetTopicLink(GetTopicLink());
    if (GetOriginalSupportLine() != GetName() && GetName() != TSupportChatTag::DeferredName && GetName() != TSupportChatTag::DeferredContainerName && GetName() != deferInfo.GetNextTag()) {
        deferInfo.SetOriginalSupportLine(GetName());
    }
    return deferInfo;
}

NJson::TJsonValue TSupportChatTag::TExternalChatInfo::SerializeToJson() const {
    NJson::TJsonValue json(NJson::JSON_MAP);
    TJsonProcessor::Write<TString>(json, "external_chat_id", Id, "");
    TJsonProcessor::Write<TString>(json, "external_chat_provider", Provider, "");
    return json;
}

bool TSupportChatTag::TExternalChatInfo::DeserializeFromJson(const NJson::TJsonValue& json) {
    return TJsonProcessor::Read(json, "external_chat_id", Id)
        && TJsonProcessor::Read(json, "external_chat_provider", Provider);
}

TString TSupportChatTag::TExternalChatInfo::MakeChatTopic() const {
    return MakeChatTopic(Id, Provider);
}

TString TSupportChatTag::TExternalChatInfo::MakeChatTopic(const TString& id, const TString& provider) {
    if (provider) {
        return provider + "-" + id;
    }
    return id;
}

void TSupportChatTag::TExternalChatInfo::SerializeToProto(NDrive::NProto::TExternalChatInfoData* proto) const {
    proto->SetId(Id);
    proto->SetProvider(Provider);
}

void TSupportChatTag::TExternalChatInfo::DeserializeFromProto(const NDrive::NProto::TExternalChatInfoData& proto) {
    Id = proto.GetId();
    Provider = proto.GetProvider();
}

NDrive::TScheme TSupportChatTag::TExternalChatInfo::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSString>("external_chat_id", "Внешний id чата").SetReadOnly(true);
    result.Add<TFSString>("external_chat_provider", "Внешняя чатовая система").SetReadOnly(true);
    return result;
}

const TString TSupportPhoneCallTag::TypeName = "user_support_call_tag";
ITag::TFactory::TRegistrator<TSupportPhoneCallTag> TSupportPhoneCallTag::Registrator(TSupportPhoneCallTag::TypeName);
TSupportPhoneCallTag::TDescription::TFactory::TRegistrator<TSupportPhoneCallTag::TDescription> TSupportPhoneCallTag::TDescription::Registrator(TSupportPhoneCallTag::TypeName);

NDrive::TScheme TSupportPhoneCallTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("call_queue", "Название очереди звонков на стороне КЦ");
    result.Add<TFSString>("url_template", "Шаблон ссылки на звонок с '[call_id]'");
    return result;
}

NJson::TJsonValue TSupportPhoneCallTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE_DEF(jsonMeta, "call_queue", CallQueue, "");
    JWRITE_DEF(jsonMeta, "url_template", UrlTemplate, "");
    return jsonMeta;
}

bool TSupportPhoneCallTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_STRING_OPT(jsonMeta, "call_queue", CallQueue);
    JREAD_STRING_OPT(jsonMeta, "url_template", UrlTemplate);
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}

void TSupportPhoneCallTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "meta", Meta.GetStringRobust());
    JWRITE(json, "status", ToString(Status));
    JWRITE(json, "call_id", InternalCallId);
}

bool TSupportPhoneCallTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    {
        TString metaStr;
        JREAD_STRING_OPT(json, "meta", metaStr);
        if (metaStr && !NJson::ReadJsonFastTree(metaStr, &Meta)) {
            return false;
        }
    }
    {
        TString statusStr;
        JREAD_STRING_OPT(json, "status", statusStr);
        if (!TryFromString(statusStr, Status)) {
            return false;
        }
    }
    JREAD_STRING_OPT(json, "call_id", InternalCallId);
    return TBase::DoSpecialDataFromJson(json, errors);
}

TSupportPhoneCallTag::TProto TSupportPhoneCallTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetTopicLink(TopicLink);
    proto.SetMeta(Meta.GetStringRobust());
    proto.SetStatus(ToString(Status));
    proto.SetInternalCallId(InternalCallId);
    return proto;
}

bool TSupportPhoneCallTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    TopicLink = proto.GetTopicLink();
    {
        TString metaStr = proto.GetMeta();
        if (metaStr && !NJson::ReadJsonFastTree(metaStr, &Meta)) {
            return false;
        }
    }
    {
        TString statusStr = proto.GetStatus();
        if (!TryFromString(statusStr, Status)) {
            return false;
        }
    }
    InternalCallId = proto.GetInternalCallId();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TSupportPhoneCallTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("status", "Статус звонка").InitVariants<ECallStatus>();
    result.Add<TFSString>("meta", "Метаинформация, json");
    result.Add<TFSString>("call_id", "ID звонка");
    return result;
}

const TString TSupportEmailTag::TypeName = "user_support_mail_tag";
ITag::TFactory::TRegistrator<TSupportEmailTag> TSupportEmailTag::Registrator(TSupportEmailTag::TypeName);
TSupportEmailTag::TDescription::TFactory::TRegistrator<TSupportEmailTag::TDescription> TSupportEmailTag::TDescription::Registrator(TSupportEmailTag::TypeName);

bool TSupportChatTag::SendChatSeparator(const TString& userId, const TString& performerId, const TString& topicLink, const TString& tagName, const NDrive::IServer* server, NDrive::TEntitySession& chatSession) const {
    TString chatId;
    TString topic;
    IChatRobot::ParseTopicLink(topicLink, chatId, topic);

    auto chatRobot = server->GetChatRobot(chatId);
    if (!chatRobot) {
        chatSession.SetErrorInfo("chat_robot", "doesn't exist", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName)->GetAs<TSupportChatTag::TDescription>();
    auto operatorName = tagDescription->GetOperatorName();
    if (operatorName) {
        if (!chatRobot->SetChatOperatorName(userId, topic, operatorName, chatSession)) {
            return false;
        }
    }

    auto separatorText = tagDescription->GetOperatorEntranceMessage();
    if (!separatorText) {
        return true;
    }

    NDrive::NChat::TMessage separatorMessage;
    separatorMessage.SetText(separatorText);
    separatorMessage.SetType(NDrive::NChat::TMessage::EMessageType::Separator);
    if (!chatRobot->SendArbitraryMessage(userId, topic, performerId, separatorMessage, chatSession)) {
        return false;
    }
    return true;
}

template <class T>
TString GetChatId(const NDrive::IServer& server, const TString& tagName) {
    TString result;
    if (auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName)) {
        if (auto supportBaseDescription = tagDescription->GetAs<T>()) {
            result = supportBaseDescription->GetChatId();
        }
    }
    return result;
}

template <class T>
bool AddTopicLink(const TDBTag& self, const TString& topicLink, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto selfCopy = self.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    if (!selfCopy) {
        session.SetErrorInfo("tag", "incorrect deep copy", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto selfCopyImpl = selfCopy.MutableTagAs<T>();
    if (!selfCopyImpl) {
        ERROR_LOG << "Can't cast to target type" << Endl;
        session.SetErrorInfo("user_tags", "can't cast tag to target type", EDriveSessionResult::InternalError);
        return false;
    }
    selfCopyImpl->SetTopicLink(topicLink);
    return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(selfCopy, userId, session);
}

bool TSupportChatTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto tagImpl = self.GetTagAs<TSupportChatTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag_impl", "null", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    if (!tagImpl->GetInitialNode()) {
        return true;
    }

    TString chatId;
    TString topic;
    if (tagImpl->GetTopicLink().empty()) {
        auto descrChatId = GetChatId<TSupportChatTag::TDescription>(*server, GetName());
        chatId = descrChatId.empty() ? "support" : descrChatId;
        topic = self.GetTagId();
        if (!AddTopicLink<TSupportChatTag>(self, chatId + "." + topic, userId, server, session)) {
            return false;
        }
    } else {
        IChatRobot::ParseTopicLink(tagImpl->GetTopicLink(), chatId, topic);
    }

    auto chatRobot = server->GetChatRobot(chatId);
    if (!chatRobot) {
        session.SetErrorInfo("chat_robot", "does not exist: " + chatId, EDriveSessionResult::InconsistencySystem);
        return false;
    }

    auto execute = [&](NDrive::TEntitySession& tx) {
        if (!chatRobot->MoveToStep(self.GetObjectId(), topic, tagImpl->GetInitialNode(), &tx, true, tagImpl->GetChatTitle())) {
            session.SetErrorInfo("chat_robot", "could not move to step: " + tagImpl->GetInitialNode(), EDriveSessionResult::InternalError);
            return false;
        }

        if (self->GetComment()) {
            NDrive::NChat::TMessage supportComment;
            supportComment.SetText(self->GetComment());
            supportComment.SetType(NDrive::NChat::TMessage::EMessageType::Plaintext);
            supportComment.SetTraits((ui32)NDrive::NChat::TMessage::EMessageTraits::StaffOnly);
            if (!chatRobot->SendArbitraryMessage(self.GetObjectId(), topic, userId, supportComment, tx)) {
                session.SetErrorInfo("chat_robot", "could not send support comment", EDriveSessionResult::InternalError);
                return false;
            }
        }
        return true;
    };

    if (server->GetDriveAPI()->GetDatabaseName() != server->GetChatEngine()->GetDatabaseName()) {
        auto chatSession = chatRobot->BuildChatEngineSession();
        if (!execute(chatSession)) {
            return false;
        }
        if (!chatSession.Commit()) {
            session.SetErrorInfo("chat_robot", "could not commit chat session", EDriveSessionResult::InternalError);
            session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
    } else {
        if (!execute(session)) {
            return false;
        }
    }

    return true;
}

bool TSupportChatTag::OnAfterPerform(TDBTag& self, const TUserPermissions& /*permissions*/, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto tagImpl = self.GetTagAs<TSupportChatTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag_impl", "null", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto userId = self.GetObjectId();
    auto performerId = self->GetPerformer();

    TString chatId;
    TString topic;
    IChatRobot::ParseTopicLink(tagImpl->GetTopicLink(), chatId, topic);

    auto chatRobot = server->GetChatRobot(chatId);
    if (!chatRobot) {
        session.SetErrorInfo("chat_robot", "doesn't exist", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    auto execute = [&](NDrive::TEntitySession& tx) {
        auto count = chatRobot->GetMessagesCount(userId, topic, NDrive::NChat::TMessage::EMessageType::Separator, tx);
        if (!count) {
            return false;
        } else if (*count > 0) {
            // Already sent entrance message
            return true;
        }
        return SendChatSeparator(userId, performerId, tagImpl->GetTopicLink(), self->GetName(), server, tx);
    };

    if (server->GetChatEngine()->GetDatabaseName() == server->GetDriveAPI()->GetDatabaseName()) {
        if (!execute(session)) {
            return false;
        }
    } else {
        auto chatSession = chatRobot->BuildChatEngineSession();
        if (!execute(chatSession)) {
            session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
        if (!chatSession.Commit()) {
            session.SetErrorInfo("SupportChatTag::SendChatSeparator", "cannot commit ChatSession", EDriveSessionResult::TransactionProblem);
            session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
    }
    return true;
}

bool TSupportChatTag::OnAfterDropPerform(const TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto impl = self.GetTagAs<TSupportChatTag>();
    if (!impl || !impl->IsMuted()) {
        return true;
    }
    return ChangeMutedStatus(self, false, permissions.GetUserId(), server, session);
}

bool TSupportChatTag::ProvideDataOnEvolve(const TDBTag& fromTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto fromTagImpl = fromTag.GetTagAs<TSupportChatTag>();
    if (fromTagImpl) {
        SetMeta(fromTagImpl->GetMeta());
        SetExternalChatInfo(fromTagImpl->GetExternalChatInfo());
    }
    return TBase::ProvideDataOnEvolve(fromTag, permissions, server, session);
}

bool TSupportChatTag::OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* /*eContext*/) const {
    auto tagImpl = fromTag.GetTagAs<TSupportChatTag>();

    auto destinationTagName = toTag->GetName();
    {
        auto newNodeId = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(destinationTagName)->GetAs<TSupportChatTag::TDescription>()->GetRedirectChatNode();
        if (newNodeId) {
            TString topicLink = tagImpl->GetTopicLink();
            TString chatId;
            TString topic;
            IChatRobot::ParseTopicLink(topicLink, chatId, topic);

            auto chatRobot = server->GetChatRobot(chatId);
            if (!chatRobot) {
                session.SetErrorInfo("chat_robot", "doesn't exist", EDriveSessionResult::InconsistencySystem);
                ERROR_LOG << "there is no such chat robot: " << chatId << Endl;
                return false;
            }

            auto execute = [&](NDrive::TEntitySession& tx) {
                if (!chatRobot->MoveToStep(fromTag.GetObjectId(), topic, newNodeId, &tx)) {
                    ERROR_LOG << "can't move to new step" << Endl;
                    return false;
                }
                return true;
            };
            if (server->GetChatEngine()->GetDatabaseName() != server->GetDriveAPI()->GetDatabaseName()) {
                auto chatSession = chatRobot->BuildChatEngineSession();
                if (!execute(chatSession)) {
                    session.SetErrorInfo("SupportChatTag::OnAfterEvolve", "cannot move to the new step");
                    session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
                    return false;
                }
                if (!chatSession.Commit()) {
                    session.SetErrorInfo("SupportChatTag::OnAfterEvolve", "cannot commit ChatSession", EDriveSessionResult::TransactionProblem);
                    session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
                    return false;
                }
            } else {
                if (!execute(session)) {
                    return false;
                }
            }
        }
    }

    auto toTagImpl = dynamic_cast<const TSupportChatTag*>(toTag.Get());
    if (toTagImpl && toTagImpl->IsMuted()) {
        TDBTag evolvedTag;
        evolvedTag.SetData(toTag);
        evolvedTag.SetObjectId(fromTag.GetObjectId());
        evolvedTag.SetTagId(fromTag.GetTagId());
        return ChangeMutedStatus(evolvedTag, false, permissions.GetUserId(), server, session);
    }

    return true;
}

bool TSupportChatTag::ChangeMutedStatus(const TDBTag& tagContainer, const bool targetStatus, const TString& operatorId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    TDBTag tagCopy = tagContainer.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    if (!tagCopy) {
        session.SetErrorInfo("tag", "incorrect deep copy", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto tagImpl = tagCopy.MutableTagAs<TSupportChatTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag", "incorrect_type", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    if (tagImpl->IsMuted() == targetStatus) {
        return true;
    }
    tagImpl->SetMuted(targetStatus);
    return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tagCopy, operatorId, session);
}

const TString TSupportOutgoingCommunicationTag::TypeName = "user_outgoing_communication";
const TString TSupportOutgoingCommunicationTag::DeferredName = "user_outgoing_communication_deferred";
TSupportOutgoingCommunicationTag::TDescription::TFactory::TRegistrator<TSupportOutgoingCommunicationTag::TDescription> TSupportOutgoingCommunicationTag::TDescription::Registrator(TSupportOutgoingCommunicationTag::TypeName);
ITag::TFactory::TRegistrator<TSupportOutgoingCommunicationTag> TSupportOutgoingCommunicationTag::Registrator(TSupportOutgoingCommunicationTag::TypeName);

IDeferrableTag::TDeferInfo TSupportOutgoingCommunicationTag::GetDeferInfo(const NDrive::IServer* /*server*/) const {
    return TDeferInfo();
}

IDeferrableTag::TDeferInfo TSupportOutgoingCommunicationTag::GetDeferUntilInfo(TInstant deferUntil, const NDrive::IServer* server) const {
    IDeferrableTag::TDeferInfo deferInfo;
    deferInfo.SetTopicLink(GetTopicLink());
    auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TSupportOutgoingCommunicationTag::TDescription>();
    deferInfo.SetNextTag(description->GetDeferContainerTagName());
    deferInfo.SetEvolveTag(true);
    deferInfo.SetUntilSLA(deferUntil);
    return deferInfo;
}

IDeferrableTag::TDeferInfo TSupportOutgoingCommunicationTag::GetDeferCloseInfo(const TDBTag& self, const NDrive::IServer* /*server*/) const {
    IDeferrableTag::TDeferInfo deferInfo;
    deferInfo.SetTopicLink("outgoing_communication." + self.GetTagId());
    deferInfo.SetRemoveTag(true);
    return deferInfo;
}

NDrive::TScheme TSupportOutgoingCommunicationTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TDescriptionBase::GetScheme(server);
    auto tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TUserContainerTag::TypeName});
    result.Add<TFSVariants>("defer_container_name", "Имя контейнер тега для откладывания").SetVariants(tags);
    result.Add<TFSVariants>("linked_tag_name", "Эволюционировать связанный тег в").SetVariants(
        NContainer::Keys(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags())
    );
    return result;
}

NJson::TJsonValue TSupportOutgoingCommunicationTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TDescriptionBase::DoSerializeMetaToJson();
    NJson::InsertField(result, "defer_container_name", DeferContainerTagName);
    NJson::InsertField(result, "linked_tag_name", LinkedTagName);
    return result;
}

bool TSupportOutgoingCommunicationTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TDescriptionBase::DoDeserializeMetaFromJson(value) &&
        NJson::ParseField(value["defer_container_name"], DeferContainerTagName) &&
        NJson::ParseField(value["linked_tag_name"], LinkedTagName);
}

TSupportOutgoingCommunicationTag::TProto TSupportOutgoingCommunicationTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    if (LinkedTagId) {
        proto.SetLinkedTagId(LinkedTagId);
    }
    auto& userProblemData = *proto.MutableUserProblemData();
    userProblemData = UserProblemData.DoSerializeSpecialDataToProto();
    return proto;
}

bool TSupportOutgoingCommunicationTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (proto.HasUserProblemData() && !UserProblemData.DoDeserializeSpecialDataFromProto(proto.GetUserProblemData())) {
        return false;
    }
    LinkedTagId = proto.GetLinkedTagId();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

void TSupportOutgoingCommunicationTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::TJsonValue userJson;
    UserProblemData.SerializeSpecialDataToJson(userJson);
    json = NJson::MergeJson(json, userJson);
    JWRITE(json, "linked_tag_id", LinkedTagId);
}

bool TSupportOutgoingCommunicationTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!UserProblemData.DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    JREAD_STRING_OPT(json, "linked_tag_id", LinkedTagId);
    return TBase::DoSpecialDataFromJson(json, errors);
}

NDrive::TScheme TSupportOutgoingCommunicationTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = ISerializableTag<TProto>::GetScheme(server);
    result.Add<TFSString>("topic_link", "Ссылка на коммуникацию с клиентом").SetReadOnly(true);
    auto userScheme  = UserProblemData.GetScheme(server);
    result.ExtendSafe(userScheme);
    return result;
}

bool TSupportOutgoingCommunicationTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TopicLink) {
        auto descrChatId = GetChatId<TSupportOutgoingCommunicationTag::TDescription>(*server, GetName());
        auto chatId = descrChatId.empty() ? "outgoing_communication" : descrChatId;
        auto chatRobot = server->GetChatRobot(chatId);
        if (!chatRobot) {
            ERROR_LOG << "Chat robot for outgoing communications is not configured" << Endl;
            session.SetErrorInfo("user_tags", "chat logic not configured", EDriveSessionResult::InternalError);
            return false;
        }

        if (!AddTopicLink<TSupportOutgoingCommunicationTag>(self, chatId + "." + self.GetTagId(), userId, server, session)) {
            return false;
        }

        auto execute = [&](NDrive::TEntitySession& tx) {
            if (!chatRobot->EnsureChat(self.GetObjectId(), self.GetTagId(), tx, false)) {
                session.SetErrorInfo("chat_messages", "ensure failed", EDriveSessionResult::InconsistencySystem);
                return false;
            }
            if (GetComment()) {
                NDrive::NChat::TMessage commentMessage;
                commentMessage.SetText(GetComment());
                commentMessage.SetType(NDrive::NChat::TMessage::EMessageType::Plaintext);
                commentMessage.SetTraits((ui32)NDrive::NChat::TMessage::EMessageTraits::StaffOnly);
                chatRobot->SendArbitraryMessage(self.GetObjectId(), self.GetTagId(), userId, commentMessage, tx);
            }
            return true;
        };

        if (server->GetDriveAPI()->GetDatabaseName() == server->GetChatEngine()->GetDatabaseName()) {
            if (!execute(session)) {
                return false;
            }
        } else {
            auto chatSession = server->GetChatEngine()->BuildSession();
            if (!execute(chatSession)) {
                return false;
            }
            if (!chatSession.Commit()) {
                session.SetErrorInfo("chat_messages", "send failed", EDriveSessionResult::InconsistencySystem);
                session.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
                return false;
            }
        }
    }
    return true;
}

bool TSupportOutgoingCommunicationTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnAfterEvolve(from, to, permissions, server, session, eContext)) {
        return false;
    }
    if (LinkedTagId) {
        const auto& database = Yensured(server)->GetDriveDatabase();
        const auto& tagManager = database.GetTagsManager().GetUserTags();

        auto expectedLinkedFrom = tagManager.RestoreTag(LinkedTagId, session);
        if (!expectedLinkedFrom) {
            session.AddErrorMessage("SupportOutgoingCommunicationTag::OnAfterEvolve", "cannot RestoreTag " + LinkedTagId);
            return false;
        }
        const auto& linkedFrom = *expectedLinkedFrom;
        if (!linkedFrom) {
            session.SetErrorInfo("SupportOutgoingCommunicationTag::OnAfterEvolve", "tag " + LinkedTagId + " is missing");
            return false;
        }

        auto linkedTo = LinkedTag;
        if (!linkedTo) {
            linkedTo = MakeLinkedTag(*server, session);
        }
        if (!linkedTo) {
            return false;
        }
        if (!linkedTo->CopyOnEvolve(*linkedFrom, nullptr, *server)) {
            session.SetErrorInfo("SupportOutgoingCommunicationTag::OnAfterEvolve", "cannot CopyOnEvolve");
            return false;
        }
        if (!tagManager.EvolveTag(linkedFrom, linkedTo, permissions, server, session, eContext)) {
            return false;
        }
    }
    return true;
}

bool TSupportOutgoingCommunicationTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnBeforeEvolve(from, to, permissions, server, session, eContext)) {
        return false;
    }
    auto target = std::dynamic_pointer_cast<TSupportOutgoingCommunicationTag>(to);
    if (target && target->GetLinkedTagId().empty()) {
        target->SetLinkedTagId(LinkedTagId);
    }
    return true;
}

ITag::TPtr TSupportOutgoingCommunicationTag::MakeLinkedTag(const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    const auto& tagMeta = server.GetDriveDatabase().GetTagsManager().GetTagsMeta();
    auto description = GetDescriptionAs<TDescription>(server, session);
    if (!description) {
        return nullptr;
    }
    const auto& linkedName = description->GetLinkedTagName();
    auto result = tagMeta.CreateTag(linkedName);
    if (!result) {
        session.SetErrorInfo("SupportOutgoingCommunicationTag::MakeLinkedTag", "cannot create tag " + linkedName);
        return nullptr;
    }
    if (auto jsonTag = std::dynamic_pointer_cast<IJsonSerializableTag>(result); jsonTag && RawData.IsDefined()) {
        TMessagesCollector errors;
        if (!jsonTag->SpecialDataFromJson(RawData, &errors)) {
            session.SetErrorInfo("SupportOutgoingCommunicationTag::MakeLinkedTag", "cannot deserialize tag " + linkedName);
            session.MergeErrorMessages(errors, "SpecialDataFromJson");
            return nullptr;
        }
    }
    return result;
}

ITag::TPtr TSupportOutgoingCommunicationTag::MutableLinkedTag(const NDrive::IServer& server, NDrive::TEntitySession& session) {
    if (!LinkedTag) {
        LinkedTag = MakeLinkedTag(server, session);
    }
    return LinkedTag;
}

const TString TSupportOutgoingCallTag::TypeName = "user_outgoing_call";
TSupportOutgoingCallTag::TDescription::TFactory::TRegistrator<TSupportOutgoingCallTag::TDescription> TSupportOutgoingCallTag::TDescription::Registrator(TSupportOutgoingCallTag::TypeName);
ITag::TFactory::TRegistrator<TSupportOutgoingCallTag> TSupportOutgoingCallTag::Registrator(TSupportOutgoingCallTag::TypeName);

NDrive::TScheme TSupportOutgoingCallTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = ISerializableTag<TProto>::GetScheme(server);
    result.Add<TFSString>("call_id", "ID звонка").SetReadOnly(true);
    result.Add<TFSString>("call_task_tag", "Ссылка на исходящую коммуникацию").SetReadOnly(true);
    return result;
}

TSupportOutgoingCallTag::TProto TSupportOutgoingCallTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    if (CallId) {
        proto.SetCallId(CallId);
    }
    if (CallTaskTag) {
        proto.SetCallTaskTag(CallTaskTag);
    }
    return proto;
}

bool TSupportOutgoingCallTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (proto.HasCallId()) {
        CallId = proto.GetCallId();
    }
    if (proto.HasCallTaskTag()) {
        CallTaskTag = proto.GetCallTaskTag();
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

void TSupportOutgoingCallTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "call_id", CallId);
    NJson::InsertField(json, "call_task_tag", CallTaskTag);
}

bool TSupportOutgoingCallTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return NJson::ParseField(json, "call_id", CallId, /* required = */ true)
        && NJson::ParseField(json, "call_task_tag", CallTaskTag, /* required = */ false)
        && TBase::DoSpecialDataFromJson(json, errors);
}

NDrive::TScheme TSupportOutgoingCallTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("url_template", "Шаблон ссылки на звонок с '[call_id]'");
    return result;
}

NJson::TJsonValue TSupportOutgoingCallTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::InsertField(result, "url_template", UrlTemplate);
    return result;
}

bool TSupportOutgoingCallTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeMetaFromJson(value) &&
        NJson::ParseField(value["url_template"], UrlTemplate);
}


ITag::TFactory::TRegistrator<TSupportUserRoutingTag> TSupportUserRoutingTag::Registrator(TSupportUserRoutingTag::GetTypeName());
TSupportUserRoutingTagDescription::TFactory::TRegistrator<TSupportUserRoutingTagDescription> TSupportUserRoutingTagDescription::Registrator(TSupportUserRoutingTag::GetTypeName());

NDrive::TScheme TSupportUserRoutingTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("action_type", "Тип роутинга").InitVariants<ESupportUserRoutingAction>();
    return result;
}

NJson::TJsonValue TSupportUserRoutingTagDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    NJson::InsertField(jsonMeta, "action_type", NJson::Stringify(ActionType));
    return jsonMeta;
}

bool TSupportUserRoutingTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    return (NJson::ParseField(jsonMeta["action_type"], NJson::Stringify(ActionType)) &&
            TBase::DoDeserializeMetaFromJson(jsonMeta));
}

TString TSupportUserRoutingTag::GetTypeName() {
    return "support_user_routing_tag";
}

EUniquePolicy TSupportUserRoutingTag::GetUniquePolicy() const {
    return EUniquePolicy::NoUnique;
}

TSet<NEntityTagsManager::EEntityType> TSupportUserRoutingTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

NDrive::TScheme TSupportUserRoutingTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme = TBase::GetScheme(server);
    scheme.Add<TFSNumeric>("ActionPriority", "Приоритет действия").SetMin(0).SetMax(15).SetDefault(0);
    scheme.Add<TFSString>("ActionData", "Спецификация действия (номер телефона/очередь)");
    return scheme;
}

bool TSupportUserRoutingTag::OnAfterAdd(const TDBTag& self, const TString& /* userId */, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!server->GetSupportCenterManager() || !server->GetSupportCenterManager()->GetCallCenterYandexClient()) {
        session.SetErrorInfo("support_user_routing_tag::add", "Yandex callcenter client is not configured", EDriveSessionResult::InternalError);
        return false;
    }

    {
        const auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
        TVector<TString> registeredUserRoutingTags = MakeVector(tagsManager.GetTagsMeta().GetRegisteredTagNames({GetTypeName()}));

        const auto& entityTagsManager = tagsManager.GetUserTags();
        auto optionalTags = entityTagsManager.RestoreEntityTags(self.GetObjectId(), registeredUserRoutingTags, session);
        if (!optionalTags) {
            return false;
        }

        for (auto&& tag: *optionalTags) {
            if (auto userRoutingTag = tag.GetTagAs<TSupportUserRoutingTag>()) {
                if (tag.GetTagId() != self.GetTagId() && userRoutingTag->GetActionPriority() == GetActionPriority()) {
                    session.SetErrorInfo("support_user_routing_tag::add", "Another routing tag with same priopity exists", EDriveSessionResult::IncorrectRequest);
                    session.SetCode(HTTP_BAD_REQUEST);
                    return false;
                }
            }
        }
    }

    TString phoneNumber;
    if (!GetUserPhone(server, self.GetObjectId(), phoneNumber, session)) {
        return false;
    }

    auto tagDescriptionPtr = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TTagDescription>();
    if (!tagDescriptionPtr) {
        session.SetErrorInfo("support_user_routing_tag::add", "Invalid tag description", EDriveSessionResult::InternalError);
        return false;
    }

    switch (tagDescriptionPtr->GetActionType()) {
    case ESupportUserRoutingAction::DropCall:
        if (!EnableDropCall(server, phoneNumber, session)) {
            return false;
        }
        break;
    default:
        session.SetErrorInfo("support_user_routing_tag::add", "Unsupported action type: " + ::ToString(tagDescriptionPtr->GetActionType()), EDriveSessionResult::IncorrectRequest);
        session.SetCode(HTTP_BAD_REQUEST);
        return false;
    }

    return true;
}

bool TSupportUserRoutingTag::OnAfterRemove(const TDBTag& self, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!server->GetSupportCenterManager() || !server->GetSupportCenterManager()->GetCallCenterYandexClient()) {
        session.SetErrorInfo("support_user_routing_tag::remove", "Yandex callcenter client is not configured", EDriveSessionResult::InternalError);
        return false;
    }

    TString phoneNumber;
    if (!GetUserPhone(server, self.GetObjectId(), phoneNumber, session)) {
        return false;
    }

    auto tagDescriptionPtr = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TTagDescription>();
    if (!tagDescriptionPtr) {
        session.SetErrorInfo("support_user_routing_tag::remove", "Invalid tag description", EDriveSessionResult::InternalError);
        return false;
    }

    switch (tagDescriptionPtr->GetActionType()) {
    case ESupportUserRoutingAction::DropCall:
        if (!DisableDropCall(server, phoneNumber, session)) {
            return false;
        }
        break;
    default:
        session.SetErrorInfo("support_user_routing_tag::remove", "Unsupported action type: " + ::ToString(tagDescriptionPtr->GetActionType()), EDriveSessionResult::IncorrectRequest);
        session.SetCode(HTTP_BAD_REQUEST);
        return false;
    }

    return true;
}

bool TSupportUserRoutingTag::GetUserPhone(const NDrive::IServer* server, const TString& phoneUserId, TString& phoneNumber, NDrive::TEntitySession& session) const {
    auto optionalUser = server->GetDriveAPI()->GetUsersData()->RestoreUser(phoneUserId, session);
    if (!optionalUser) {
        session.AddErrorMessage("support_user_routing_tag", "Cannot restore user info");
        return false;
    }

    if (!optionalUser->GetPhone()) {
        session.SetErrorInfo("support_user_routing_tag", "User has no phone assigned", EDriveSessionResult::InternalError);
        return false;
    }

    phoneNumber = optionalUser->GetPhone();
    return true;
}

bool TSupportUserRoutingTag::GetTargetQueues(TSet<TString>& queues, TMessagesCollector& errors) const {
    auto queueConfig = NCallCenterYandex::TCallCenterYandexQueueConfig::Construct();
    if (!queueConfig) {
        errors.AddMessage(__LOCATION__, "Error constructing support queue config");
        return false;
    }

    TSet<TString> targetQueues = StringSplitter(GetActionData()).Split(',').SkipEmpty();
    if (targetQueues) {
        if (!queueConfig->AreQueuesValid(targetQueues, errors)) {
            return false;
        }
    } else {
        targetQueues = queueConfig->GetAllQueues();
    }

    queues = std::move(targetQueues);
    return true;
}

bool TSupportUserRoutingTag::EnableDropCall(const NDrive::IServer* server, const TString& phoneNumber, NDrive::TEntitySession& session) const {
    auto internalCCClientPtr = server->GetSupportCenterManager()->GetCallCenterYandexClient();
    TMessagesCollector errors;

    TSet<TString> targetQueues;
    if (!GetTargetQueues(targetQueues, errors)) {
        session.SetErrorInfo("support_user_routing_tag::add", errors.GetStringReport(), EDriveSessionResult::InternalError);
        return false;
    }

    TSet<TString> successfullyProcessedQueues;
    for (const auto& queue: targetQueues) {
        if (internalCCClientPtr->EnableCallerDrop(queue, phoneNumber, errors)) {
            successfullyProcessedQueues.emplace(queue);
        }
    }

    // there are 2 problems:
    // - once new queue added, routing has to be updated (to be solved routing users dynamically)
    // - the best way in case of error in one of multiple requests is to try assign the tag again and check out telephony permissions

    const bool success = (successfullyProcessedQueues.size() == targetQueues.size());
    if (!success) {
        for (const auto& queue: successfullyProcessedQueues) {
            internalCCClientPtr->DisableCallerRouting(queue, phoneNumber, errors);
        }
        session.SetErrorInfo("support_user_routing_tag::add", "Error performing telephony requests: " + errors.GetStringReport(), EDriveSessionResult::InternalError);
        return false;
    }

    return true;
}

bool TSupportUserRoutingTag::DisableDropCall(const NDrive::IServer* server, const TString& phoneNumber, NDrive::TEntitySession& session) const {
    auto internalCCClientPtr = server->GetSupportCenterManager()->GetCallCenterYandexClient();
    TMessagesCollector errors;

    TSet<TString> targetQueues;
    if (!GetTargetQueues(targetQueues, errors)) {
        session.SetErrorInfo("support_user_routing_tag::remove", errors.GetStringReport(), EDriveSessionResult::InternalError);
        return false;
    }

    TSet<TString> successfullyProcessedQueues;
    for (const auto& queue: targetQueues) {
        if (internalCCClientPtr->DisableCallerRouting(queue, phoneNumber, errors)) {
            successfullyProcessedQueues.emplace(queue);
        }
    }

    const bool success = (successfullyProcessedQueues.size() == targetQueues.size());
    if (!success) {
        session.SetErrorInfo("support_user_routing_tag::remove", "Error performing telephony requests: " + errors.GetStringReport(), EDriveSessionResult::InternalError);
        return false;
    }

    return true;
}

TSupportUserRoutingTag::TProto TSupportUserRoutingTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    if (ActionPriority) {
        proto.SetActionPriority(ActionPriority);
    }
    if (ActionData) {
        proto.SetActionData(ActionData);
    }
    return proto;
}

bool TSupportUserRoutingTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (proto.HasActionPriority()) {
        ActionPriority = proto.GetActionPriority();
    }
    if (proto.HasActionData()) {
        ActionData = proto.GetActionData();
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}
