#pragma once

#include "user_tags.h"

#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/chat/message.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/support_center/feedback/chat.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/string/vector.h>

class ITopicLinkOwner {
protected:
    TString OriginalSupportLine;
    TString TopicLink;

public:
    void SetTopicLink(const TString& newTopicLink) {
        TopicLink = newTopicLink;
    }

    void SetOriginalSupportLine(const TString& originalSupportLine) {
        OriginalSupportLine = originalSupportLine;
    }

    TString GetTopicLink() const {
        return TopicLink;
    }

    TString GetOriginalSupportLine() const {
        return OriginalSupportLine;
    }
};

template <class TProto>
class TSupportUserTagBase
    : public ISerializableTag<TProto>
    , public ITopicLinkOwner
{
protected:
    using TBase = ISerializableTag<TProto>;

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;
        static TFactory::TRegistrator<TDescription> Registrator;

    private:
        R_FIELD(TString, ChatId);

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme scheme = TBase::GetScheme(server);
            scheme.Add<TFSString>("chat_id", "id чата для автоматической генерации topic link");
            return scheme;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
            NJson::InsertField(result, "chat_id", ChatId);
            return result;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            return
                TBase::DoDeserializeMetaFromJson(jsonMeta) &&
                NJson::ParseField(jsonMeta["chat_id"], ChatId);
        }
    };

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        NJson::InsertField(json, "topic_link", TopicLink);
        NJson::InsertField(json, "original_support_line", OriginalSupportLine);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        RawData = json;
        if (!NJson::ParseField(json, "original_support_line", OriginalSupportLine, false)
            || !NJson::ParseField(json, "topic_link", TopicLink, false)) {
            return false;
        }
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetTopicLink(TopicLink);
        proto.SetOriginalSupportLine(OriginalSupportLine);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        TopicLink = proto.GetTopicLink();
        OriginalSupportLine = proto.GetOriginalSupportLine();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSString>("topic_link", "Ссылка на коммуникацию с клиентом");
        return result;
    }

protected:
    NJson::TJsonValue RawData;
};

class IDeferrableTag {
public:
    class TDeferInfo {
        R_FIELD(TString, NextTag);
        R_FIELD(TString, NextNode);
        R_FIELD(TString, TopicLink);
        R_FIELD(TString, OriginalSupportLine);
        R_FIELD(TInstant, UntilSLA);
        R_FIELD(TVector<NDrive::NChat::TMessage>, Messages, {}, mutable);
        R_FIELD(bool, RemoveTag, false);
        R_FIELD(bool, EvolveTag, false);
        R_FIELD(bool, CopyTag, false);
        R_FIELD(bool, DropPerformer, true);
    };

    virtual bool OnDeferUntil(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TInstant untilSLA, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    virtual bool OnDefer(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    virtual bool OnClose(TDBTag& self, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const;

protected:
    virtual TDeferInfo GetDeferInfo(const NDrive::IServer* server) const = 0;
    virtual TDeferInfo GetDeferUntilInfo(TInstant deferUntil, const NDrive::IServer* server) const = 0;
    virtual TDeferInfo GetDeferCloseInfo(const TDBTag& self, const NDrive::IServer* server) const = 0;

private:
    bool ExecDeferActions(TDBTag& self, TDeferInfo& deferInfo, const TVector<NDrive::NChat::TMessage>& messages, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    bool ExecDeferChatActions(const TString& userId, const TString& actorId, const TDeferInfo& deferInfo, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
    bool ExecDeferChatActions(const TString& userId, const TString& actorId, const TDeferInfo& deferInfo, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession, const NDrive::IServer* server) const;
    bool ExecDeferTagActions(TDBTag& self, const IDeferrableTag::TDeferInfo& deferInfo, const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer* server) const;
};

class TSupportChatTag
    : public TSupportUserTagBase<NDrive::NProto::TSupportChatTagData>
    , public IDeferrableTag
{
private:
    using TBase = TSupportUserTagBase<NDrive::NProto::TSupportChatTagData>;

public:
    class TExternalChatInfo {
        R_FIELD(TString, Id);
        R_FIELD(TString, Provider);
    public:
        TExternalChatInfo() = default;

        TExternalChatInfo(const TString& id, const TString& provider)
            : Id(id)
            , Provider(provider)
        {
        }

        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& json);
        void SerializeToProto(NDrive::NProto::TExternalChatInfoData* proto) const;
        void DeserializeFromProto(const NDrive::NProto::TExternalChatInfoData& proto);
        TString MakeChatTopic() const;
        static TString MakeChatTopic(const TString& id, const TString& provider);
        static NDrive::TScheme GetScheme();
    };

private:
    R_FIELD(TSupportChatFeedback, Feedback);
    R_FIELD(bool, Muted, false);
    R_FIELD(TString, InitialNode);
    R_FIELD(TString, ChatTitle);
    R_FIELD(TExternalChatInfo, ExternalChatInfo);
    R_FIELD(NJson::TJsonValue, Meta, NJson::JSON_NULL);

public:
    static const TString TypeName;
    static const TString DeferredName;
    static const TString FeedbackName;
    static const TString DeferredContainerName;
    static TFactory::TRegistrator<TSupportChatTag> Registrator;

public:
    using TBase::TBase;

    class TBehaviour {
        R_FIELD(TString, EvolveToTag);
        R_FIELD(TString, MoveToNode);
        R_FIELD(bool, RemoveTag, false);
        R_FIELD(bool, DropPerformer, true);

    public:
        TBehaviour(const TString evolveToTagDefault, const TString& moveToNode, const bool removeTag = false)
            : EvolveToTag(evolveToTagDefault)
            , MoveToNode(moveToNode)
            , RemoveTag(removeTag)
        {
        }

        static NDrive::TScheme GetScheme(const NDrive::IServer* server, const TString& evolveToTagTypeName);
        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& jsonMeta);
    };

    enum EInsignificantMessageType {
        Robot /* "from_robot" */,
        Separator /* "separator" */
    };

    class TDescription: public TBase::TDescription {
    private:
        using TBase = TSupportUserTagBase<NDrive::NProto::TSupportChatTagData>::TDescription;

    private:
        R_FIELD(TString, OperatorEntranceMessage);
        R_FIELD(TString, OperatorName);
        R_FIELD(TString, RedirectChatNode);
        R_FIELD(bool, NeedNotification, false);
        R_FIELD(TSet<EInsignificantMessageType>, IncludedFilterMessageTypes);
        R_FIELD(TBehaviour, OnCloseBehaviour, TBehaviour(FeedbackName, ""));
        R_FIELD(TBehaviour, OnDeferBehaviour, TBehaviour(DeferredName, ""));
        R_FIELD(TBehaviour, OnDeferUntilBehaviour, TBehaviour(DeferredContainerName, ""));

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool ProvideDataOnEvolve(const TDBTag& fromTag, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) override;
    virtual bool OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& /*permissions*/, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;
    virtual bool OnAfterDropPerform(const TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    static bool ChangeMutedStatus(const TDBTag& container, const bool targetStatus, const TString& operatorId, const NDrive::IServer* server, NDrive::TEntitySession& session);

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    TString BuildChatLink(const TString& objectId) const;

protected:
    virtual IDeferrableTag::TDeferInfo GetDeferInfo(const NDrive::IServer* server) const override;
    virtual IDeferrableTag::TDeferInfo GetDeferUntilInfo(TInstant deferUntil, const NDrive::IServer* server) const override;
    virtual IDeferrableTag::TDeferInfo GetDeferCloseInfo(const TDBTag& self, const NDrive::IServer* server) const override;

private:
    bool SendChatSeparator(const TString& userId, const TString& performerId, const TString& topicLink, const TString& tagName, const NDrive::IServer* server, NDrive::TEntitySession& chatSession) const;
};

class TSupportPhoneCallTag: public TSupportUserTagBase<NDrive::NProto::TSupportPhoneCallTagData> {
public:
    enum ECallStatus {
        EnterQueue /* "enter_queue" */,
        RingNoAnswer /* "ring_no_answer" */,
        Connect /* "connect" */,
        CompleteAgent /* "complete_agent" */,
        CompleteCaller /* "complete_caller" */,
        Abandon /* "abandon" */,
        ExitEmpty /* "exit_empty" */,
        ExitWithTimeout /* "exit_with_timeout" */,
        AttendedTransfer /* "attended_transfer" */,
        BlindTransfer /* "blind_transfer" */,
    };

private:
    using TBase = TSupportUserTagBase<NDrive::NProto::TSupportPhoneCallTagData>;

private:
    R_FIELD(NJson::TJsonValue, Meta, NJson::JSON_NULL);
    R_FIELD(ECallStatus, Status, ECallStatus::EnterQueue);
    R_FIELD(TString, InternalCallId);

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TSupportPhoneCallTag> Registrator;

public:
    using TBase::TBase;

public:
    class TDescription: public TBase::TDescription {
    private:
        using TBase = TSupportUserTagBase<NDrive::NProto::TSupportPhoneCallTagData>::TDescription;

    public:
        R_FIELD(TString, UrlTemplate);
        R_FIELD(TString, CallQueue);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
};

class TSupportEmailTag: public TSupportUserTagBase<NDrive::NProto::TSupportEmailTagData> {
private:
    using TBase = TSupportUserTagBase<NDrive::NProto::TSupportEmailTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TSupportEmailTag> Registrator;

public:
    using TBase::TBase;

    class TDescription: public TBase::TDescription {
        static TFactory::TRegistrator<TDescription> Registrator;
    };
};

class TSupportOutgoingCommunicationTag
    : public TSupportUserTagBase<NDrive::NProto::TSupportOutgoingCommunicationTagData>
    , public IDeferrableTag
{
private:
    using TBase = TSupportUserTagBase<NDrive::NProto::TSupportOutgoingCommunicationTagData>;
    using TDescriptionBase = TBase::TDescription;

    TUserProblemTag UserProblemData;

public:
    static const TString TypeName;
    static const TString DeferredName;
    static TFactory::TRegistrator<TSupportOutgoingCommunicationTag> Registrator;

public:
    using TBase::TBase;

public:
    class TDescription: public TDescriptionBase {
    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        R_FIELD(TString, DeferContainerTagName, "container_tag");
        R_FIELD(TString, LinkedTagName);

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

public:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;
    virtual bool OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

    virtual IDeferrableTag::TDeferInfo GetDeferInfo(const NDrive::IServer* server) const override;
    virtual IDeferrableTag::TDeferInfo GetDeferUntilInfo(TInstant deferUntil, const NDrive::IServer* server) const override;
    virtual IDeferrableTag::TDeferInfo GetDeferCloseInfo(const TDBTag& self, const NDrive::IServer* server) const override;

    ITag::TPtr MutableLinkedTag(const NDrive::IServer& server, NDrive::TEntitySession& session);

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    ITag::TPtr MakeLinkedTag(const NDrive::IServer& server, NDrive::TEntitySession& session) const;

private:
    R_FIELD(TString, LinkedTagId);

private:
    ITag::TPtr LinkedTag;
};

class TSupportOutgoingCallTag
    : public ISerializableTag<NDrive::NProto::TSupportOutgoingCallTagData>
{
    R_FIELD(TString, CallId);
    R_FIELD(TString, CallTaskTag);

protected:
    using TBase = ISerializableTag<TProto>;

public:
    using TBase::TBase;
    static const TString TypeName;
    static TFactory::TRegistrator<TSupportOutgoingCallTag> Registrator;


    class TDescription: public TTagDescription {
        R_FIELD(TString, UrlTemplate);
        static TFactory::TRegistrator<TDescription> Registrator;
        using TBase = TTagDescription;

    public:
        using TBase::TBase;
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
};

enum class ESupportUserRoutingAction : ui32 {
    Undefined = 0 /* "undefined" */,
    RedirectToQueue = 1 /* "redirect_to_queue" */,
    RedirectToNumber = 2 /* "redirect_to_number" */,
    DropCall = 3 /* "drop_call" */,
};

class TSupportUserRoutingTagDescription: public TTagDescription {
    using TBase = TTagDescription;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

protected:
    virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;

private:
    static TFactory::TRegistrator<TSupportUserRoutingTagDescription> Registrator;

    R_FIELD(ESupportUserRoutingAction, ActionType, ESupportUserRoutingAction::Undefined);
};

class TSupportUserRoutingTag: public INativeSerializationTag<NDrive::NProto::TSupportUserRoutingTagData> {
    using TBase = INativeSerializationTag<NDrive::NProto::TSupportUserRoutingTagData>;

public:
    using TTagDescription = TSupportUserRoutingTagDescription;

    using TBase::TBase;

    static TString GetTypeName();

    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

private:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    bool GetUserPhone(const NDrive::IServer* server, const TString& phoneUserId, TString& phoneNumber, NDrive::TEntitySession& session) const;

    bool GetTargetQueues(TSet<TString>& queues, TMessagesCollector& errors) const;

    bool EnableDropCall(const NDrive::IServer* server, const TString& phoneNumber, NDrive::TEntitySession& session) const;
    bool DisableDropCall(const NDrive::IServer* server, const TString& phoneNumber, NDrive::TEntitySession& session) const;

private:
    static TFactory::TRegistrator<TSupportUserRoutingTag> Registrator;

    R_FIELD(ui32, ActionPriority, 0);
    R_FIELD(TString, ActionData);
};
