#include "switch.h"

#include "chargable.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>

#include <rtline/util/algorithm/ptr.h>

NDrive::TScheme TSwitchOfferTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("target_tag_name", "Target state").SetVariants(TChargableTag::GetTagNames(server->GetDriveAPI()->GetTagsManager().GetTagsMeta()));
    return result;
}

bool TSwitchOfferTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto driveApi = Checked(server)->GetDriveAPI();
    auto removed = Checked(driveApi)->GetTagsManager().GetDeviceTags().RemoveTag(self, userId, server, session);
    return removed;
}

bool TSwitchOfferTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(userId);
    auto driveApi = Checked(server)->GetDriveAPI();
    auto sessionBuilder = Checked(driveApi)->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    auto lastSession = sessionBuilder->GetLastObjectSession(objectId);
    if (!lastSession) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "cannot find last session for " << objectId,
            EDriveSessionResult::InternalError
        );
        return false;
    }
    Y_ASSERT(lastSession->GetObjectId() == objectId);
    if (lastSession->GetClosed()) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "last session " << lastSession->GetSessionId() << " is finished",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    auto currentSession = lastSession;
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(currentSession);
    if (!billingSession) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "cannot cast session " << currentSession->GetSessionId() << " to BillingSession",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    auto currentOffer = billingSession->GetCurrentOffer();
    if (!currentOffer) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "cannot get current offer from session " << currentSession->GetSessionId(),
            EDriveSessionResult::InternalError
        );
        return false;
    }
    const auto& currentUserId = currentSession->GetUserId();
    const auto& nextOfferId = currentOffer->GetNextOfferId();
    if (!nextOfferId) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "no next offer in session " << currentSession->GetSessionId(),
            EDriveSessionResult::InternalError
        );
        return false;
    }
    auto offersStorage = Checked(server)->GetOffersStorage();
    auto asyncNextOffer = Checked(offersStorage)->RestoreOffer(nextOfferId, currentUserId, session);
    if (!asyncNextOffer.Initialized()) {
        session.AddErrorMessage("SwitchOfferTag::OnBeforeAdd", TStringBuilder() << "cannot restore offer " << nextOfferId);
        return false;
    }

    auto permissions = driveApi->GetUserPermissions(currentUserId, TUserPermissionsFeatures{});
    if (!permissions) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "cannot build permission for " << currentUserId,
            EDriveSessionResult::InternalError
        );
        return false;
    }

    asyncNextOffer.Wait();
    if (!asyncNextOffer.HasValue()) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "cannot restore offer " << nextOfferId << ": " << NThreading::GetExceptionMessage(asyncNextOffer),
            EDriveSessionResult::InternalError
        );
        return false;
    }

    TChargableTag::TSwitchOptions switchOptions;
    switchOptions.IgnoreNonSwitchable = true;
    switchOptions.TargetTagName = TargetTagName;
    auto nextOffer = std::dynamic_pointer_cast<IOffer>(asyncNextOffer.GetValue());
    if (!nextOffer) {
        session.SetErrorInfo(
            "SwitchOfferTag::OnBeforeAdd",
            TStringBuilder() << "non-vehicle offer: " << nextOfferId,
            EDriveSessionResult::InternalError
        );
        return false;
    }
    return TChargableTag::Switch(currentSession, nextOffer, permissions, *server, session, switchOptions);
}

bool TSwitchOfferTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return
        NJson::ParseField(value["target_tag_name"], TargetTagName);
}

void TSwitchOfferTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::InsertNonNull(value, "target_tag_name", TargetTagName);
}

ITag::TFactory::TRegistrator<TSwitchOfferTag> TSwitchOfferTag::Registrator(TSwitchOfferTag::Type());
