#include "tag_with_subtags.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/field.h>

bool TTagWithSubtags::RemoveSubtags(const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto api = Yensured(server)->GetDriveAPI();
    const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
    const auto& userTagManager = Yensured(api)->GetTagsManager().GetUserTags();

    auto deviceTags = deviceTagManager.RestoreTags(CarTagIds, session);
    if (!deviceTags) {
        return false;
    }
    for (auto&& tag : *deviceTags) {
        if (tag->GetPerformer()) {
            continue;
        }
        if (!deviceTagManager.RemoveTag(tag, userId, server, session)) {
            return false;
        }
    }

    auto userTags = userTagManager.RestoreTags(UserTagIds, session);
    if (!userTags) {
        return false;
    }
    for (auto&& tag : *userTags) {
        if (tag->GetPerformer()) {
            continue;
        }
        if (!userTagManager.RemoveTag(tag, userId, server, session)) {
            return false;
        }
    }

    return true;
}

NDrive::TScheme TTagWithSubtags::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    if (server) {
        const auto& tagMeta = server->GetDriveDatabase().GetTagsManager().GetTagsMeta();
        auto carTagDescriptions = tagMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car);
        auto carTagNames = NContainer::Keys(carTagDescriptions);
        result.Add<TFSVariants>("car_tags", "Car tags to add on evolution").SetVariants(carTagNames).SetMultiSelect(true);
        result.Add<TFSVariants>("user_tags", "User tags to add on evolution").SetReference("user_tags").SetMultiSelect(true);
    }
    return result;
}

NJson::TJsonValue TTagWithSubtags::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TTagWithSubtags::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}
