#include "taxi.h"

#include "billing_tags.h"
#include "chargable.h"
#include "switch.h"
#include "transformation.h"

#include <drive/backend/device_snapshot/snapshots/snapshot.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/taxi/routehistory/client.h>

#include <rtline/util/algorithm/ptr.h>

bool TTaxiValidationTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* evolutionContext) const {
    Y_UNUSED(evolutionContext);
    if (!server) {
        session.SetErrorInfo("TaxiValidationTag::OnBeforeEvolve", "no Server", EDriveSessionResult::InternalError);
        return false;
    }

    const auto& name = to->GetName();
    if (name == Ok || name == CarFailure || name == UserFailure) {
        return server->GetDriveDatabase().GetTagsManager().GetDeviceTags().RemoveTag(from, permissions.GetUserId(), server, session);
    }
    return true;
}

bool TTaxiValidationTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* evolutionContext) const {
    if (!server) {
        session.SetErrorInfo("TaxiValidationTag::OnBeforeEvolve", "no Server", EDriveSessionResult::InternalError);
        return false;
    }
    const auto& database = server->GetDriveDatabase();

    const auto& name = to->GetName();
    const auto& objectId = from.GetObjectId();
    if (name == Ok) {
        auto switchOfferTag = MakeAtomicShared<TSwitchOfferTag>(TSwitchOfferTag::Type());
        switchOfferTag->SetTargetTagName(TChargableTag::Parking);
        auto optionalAddedTags = database.GetTagsManager().GetDeviceTags().AddTag(switchOfferTag, permissions.GetUserId(), objectId, server, session);
        return optionalAddedTags.Defined();
    }
    auto sessionBuilder = database.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    auto objectSession = sessionBuilder ? sessionBuilder->GetLastObjectSession(objectId) : nullptr;
    if (name == CarFailure) {
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(objectSession);
        auto currentOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
        auto previousSessionId = currentOffer ? currentOffer->GetParentId() : TString();
        if (previousSessionId) {
            auto deferredRefundTag = MakeAtomicShared<TDeferredSessionRefundTag>();
            if (!database.GetTagsManager().GetTraceTags().AddTag(deferredRefundTag, permissions.GetUserId(), previousSessionId, server, session)) {
                return false;
            }
        }
        return TChargableTag::DirectEvolve(objectSession, TChargableTag::Reservation, permissions, *server, session, evolutionContext);
    }
    if (name == UserFailure) {
        return TChargableTag::DirectEvolve(objectSession, TChargableTag::Reservation, permissions, *server, session, evolutionContext);
    }
    return true;
}

bool TTaxiRouteHistoryPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    auto client = server ? server->GetTaxiRouteHistoryClient() : nullptr;
    if (!client) {
        return true;
    }

    if (Yensured(tag)->GetName() == TTransformationTag::TypeName) {
        return true;
    }
    if (Yensured(tag)->GetName() != TChargableTag::Reservation) {
        NDrive::TEventLog::Log("TaxiRouteHistoryPolicyError", NJson::TMapBuilder
            ("type", "BadTagName")
            ("tag", tag->GetName())
        );
        return false;
    }

    auto sessionBuilder = Yensured(server)->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
    auto session = sessionBuilder ? sessionBuilder->GetLastObjectSession(objectId) : nullptr;
    auto billingSession = std::dynamic_pointer_cast<TBillingSession>(session);
    if (!billingSession) {
        NDrive::TEventLog::Log("TaxiRouteHistoryPolicyError", NJson::TMapBuilder
            ("type", "BadLastObjectSession")
        );
        return false;
    }

    TSnapshotsDiffCompilation snapshotCompilation;
    if (!billingSession->FillCompilation(snapshotCompilation)) {
        NDrive::TEventLog::Log("TaxiRouteHistoryPolicyError", NJson::TMapBuilder
            ("type", "BadSnapshots")
        );
        return false;
    }

    NDrive::TTaxiRouteHistoryOrder order;
    order.Id = session->GetSessionId();
    order.Created = session->GetStartTS();
    order.YandexUid = permissions.GetUid();

    bool accepted = false;
    bool finished = false;
    for (auto&& [tag, snapshot] : snapshotCompilation.GetSnapshots()) {
        auto deviceSnapshot = std::dynamic_pointer_cast<THistoryDeviceSnapshot>(snapshot);
        if (!deviceSnapshot) {
            continue;
        }
        auto location = deviceSnapshot->GetHistoryLocation();
        if (!location) {
            continue;
        }
        NDrive::TTaxiRouteHistoryPoint point;
        point.Position = location->GetCoord();
        if (tag == TChargableTag::Acceptance) {
            accepted = true;
            point.Type = NDrive::TTaxiRouteHistoryPoint::EType::Source;
            order.Points.push_back(point);
        }
        if (tag == TChargableTag::Parking) {
            point.Type = NDrive::TTaxiRouteHistoryPoint::EType::Waiting;
            order.Points.push_back(point);
        }
        if (tag == TChargableTag::Reservation && accepted && !finished) {
            finished = true;
            point.Type = NDrive::TTaxiRouteHistoryPoint::EType::DestinationActual;
            order.Points.push_back(point);
        }
    }
    if (!accepted) {
        NDrive::TEventLog::Log("TaxiRouteHistorySkip", NJson::TMapBuilder
            ("accepted", accepted)
        );
        return true;
    }

    auto offer = billingSession->GetCurrentOffer();
    auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer);
    if (fixPointOffer) {
        NDrive::TTaxiRouteHistoryPoint point;
        point.Position = fixPointOffer->GetFinish();
        point.Type = NDrive::TTaxiRouteHistoryPoint::EType::DestinationFix;
        order.Points.push_back(point);
    }

    NDrive::TEventLog::Log("TaxiRouteHistoryAdd", NJson::ToJson(order));
    auto start = Now();
    auto result = client->Add(order);
    auto eventLogState = NDrive::TEventLog::CaptureState();
    result.Subscribe([orderId = order.Id, start, eventLogState = std::move(eventLogState)](const NThreading::TFuture<void>& w) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        if (w.HasValue()) {
            auto finish = Now();
            auto duration = finish - start;
            NDrive::TEventLog::Log("TaxiRouteHistoryAddSuccess", NJson::TMapBuilder
                ("order_id", orderId)
                ("duration", NJson::ToJson(duration))
            );
        } else {
            NDrive::TEventLog::Log("TaxiRouteHistoryAddError", NJson::TMapBuilder
                ("order_id", orderId)
                ("result", NThreading::GetExceptionInfo(w))
            );
        }
    });

    return true;
}

bool TTaxiRouteHistoryPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(toTag);
    Y_UNUSED(server);
    Y_UNUSED(permissions);
    Y_UNUSED(eContext);
    Y_UNUSED(session);
    return true;
}

bool TTaxiRouteHistoryPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TTaxisharingPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    Y_UNUSED(objectId);
    Y_UNUSED(tag);
    Y_UNUSED(server);
    Y_UNUSED(permissions);
    return true;
}

bool TTaxisharingPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    Y_UNUSED(toTag);
    Y_UNUSED(permissions);
    if (eContext) {
        if (eContext->GetMode() != EEvolutionMode::Default) {
            return true;
        }
        if (eContext->IsSwitching()) {
            return true;
        }
    }
    const auto& objectId = fromTag.GetObjectId();
    auto snapshot = Yensured(server)->GetSnapshotsManager().GetSnapshot(objectId);
    auto fuelLevel = snapshot.GetFuelLevel();
    if (!fuelLevel) {
        session.SetErrorInfo("TaxisharingPolicy::ExecuteBeforeEvolution", "cannot get fuel_level for " + objectId);
        return false;
    }
    if (*fuelLevel < FuelLevelThreshold) {
        session.SetCode(HTTP_BAD_REQUEST);
        session.SetErrorInfo(
            "TaxisharingPolicy::ExecuteBeforeEvolution",
            TStringBuilder() << "fuel level " << fuelLevel << " is low",
            NDrive::MakeError("low_fuel_level")
        );
        return false;
    }
    return true;
}

bool TTaxisharingPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

ITag::TFactory::TRegistrator<TTaxiValidationTag> TTaxiValidationTag::Registrator(TTaxiValidationTag::Type());
