#include "tech_dispatch.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/field.h>

DECLARE_FIELDS_JSON_SERIALIZER(TOrderTechDispatchTag::TDescription::TRepairType);
DECLARE_FIELDS_JSON_SERIALIZER(TOrderTechDispatchTag::TDescription::TRepairGroupInfo);
DECLARE_FIELDS_JSON_SERIALIZER(TAssignmentTechDispatchTag::TDescription::TRepairGroup);

bool ITechDispatchTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(objectId);
    Y_UNUSED(userId);
    Y_UNUSED(server);
    Y_UNUSED(session);
    // TODO: check types
    return true;
}

bool ITechDispatchTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    Y_UNUSED(session);
    Y_UNUSED(eContext);
    auto targetTag = std::dynamic_pointer_cast<ITechDispatchTag>(toTag);
    if (targetTag) {
        targetTag->SetOperations(Operations);
        targetTag->SetSpecials(Specials);
    }
    return true;
}

bool TAssignmentTechDispatchTag::OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    Y_UNUSED(eContext);
    Y_UNUSED(toTag);
    const auto& manager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    auto orderTag = fromTag.GetTagAs<TOrderTechDispatchTag>();
    auto assignmentDescription = Yensured(GetDescriptionAs<TAssignmentTechDispatchTag::TDescription>(*server));

    if (orderTag) {
        auto orderDescription = Yensured(Yensured(manager.GetTagsMeta().GetDescriptionByName(TOrderTechDispatchTag::TypeName))->GetAs<TOrderTechDispatchTag::TDescription>());
        TVector<ITag::TPtr> additionalTags;
        for (const auto& group : orderDescription->GetGroupInfos()) {
            auto groupPtr = orderTag->GetOperations().FindPtr(group.GetName());
            if (!groupPtr) {
                continue;
            }
            for (const auto& subType : *groupPtr) {
                for (const auto& commonSubType : group.GetSubTypes()) {
                    if (commonSubType.GetName() == subType) {
                        for (const auto& tagName : commonSubType.GetAdditionalTags()) {
                            auto addTag = manager.GetTagsMeta().CreateTag(tagName, assignmentDescription->GetAdditionalComment());
                            if (addTag) {
                                additionalTags.emplace_back(std::move(addTag));
                            }
                        }
                        break;
                    }
                }
            }
        }
        if (additionalTags) {
            auto dbTags = manager.GetDeviceTags().AddTags(additionalTags, permissions.GetUserId(), fromTag.GetObjectId(), server, session);
            if (!dbTags) {
                return false;
            }
        }
    }
    return true;
}

bool ITechDispatchTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!NJson::TryFieldsFromJson(json, GetFields())) {
        return false;
    }
    for (auto&& [key, value] : json.GetMap()) {
        TVector<TString> parts(SplitString(key, "__", 2));
        if (parts.size() < 2 || !value.IsBoolean() || !value.GetBoolean()) {
            continue;
        }
        Operations[parts[0]].emplace_back(parts[1]);
        Specials[parts[0]].emplace(parts[1]);
    }
    return
        TBase::DoSpecialDataFromJson(json, errors);
}

void ITechDispatchTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    auto& operationsJson = value.InsertValue("operations", NJson::JSON_ARRAY);
    for (auto&& [type, subTypes] : Operations) {
        NJson::TJsonValue item;
        NJson::InsertField(item, "type", type);
        auto specialPtr = Specials.FindPtr(type);
        if (!specialPtr) {
            NJson::InsertField(item, "sub_types", subTypes);
            operationsJson.AppendValue(item);
            continue;
        }
        for (auto&& name : subTypes) {
            if (specialPtr->contains(name)) {
                value[type + "__" + name] = true;
            } else {
                item["sub_types"].AppendValue(name);
            }
        }
        if (item.Has("sub_types")) {
            operationsJson.AppendValue(item);
        }
    }
}

TExpected<TMap<TString, ui32>, TString> ITechDispatchTag::GetRates(const NDrive::IServer* server) const {
    return TOrderTechDispatchTag::TDescription::GetRates(Operations, server);
}

NDrive::TScheme ITechDispatchTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    auto description = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(TOrderTechDispatchTag::TypeName);
    auto orderDescription = Yensured(std::dynamic_pointer_cast<const TOrderTechDispatchTag::TDescription>(description));

    NDrive::TScheme simpleElementScheme;
    TOperations defaultOperations;
    auto& typeSchemeVariable = simpleElementScheme.Add<TFSVariable>("type", "Вид работ");
    for (const auto& group : orderDescription->GetGroupInfos()) {
        const auto& type = group.GetName();
        const auto& subTypes = group.GetSubTypes();
        TVector<TFSVariants::TCompoundVariant> subTypeNames;
        for (const auto& subType : subTypes) {
            if (subType.IsCommonOption()) {
                result.Add<TFSBoolean>(type + "__" + subType.GetName(), subType.GetHrName()).SetDefault(subType.IsDefault());
                continue;
            }
            subTypeNames.emplace_back(subType.GetName(), subType.GetHrName());
            if (subType.IsDefault()) {
                defaultOperations[type].emplace_back(subType.GetName());
            }
        }
        if (subTypeNames) {
            NDrive::TScheme typeScheme;
            typeScheme.Add<TFSVariants>("sub_types").SetCompoundVariants(std::move(subTypeNames)).SetMultiSelect(true);
            TFSVariants::TCompoundVariant variantName{ type, group.GetHrName() };
            typeSchemeVariable.AddVariant(variantName, std::move(typeScheme));
        }
    }

    auto& operations = result.Add<TFSArray>("operations", "Необходимые работы");
    operations.SetElement(simpleElementScheme);
    operations.SetRequired(true);
    operations.SetDefaultValueView(NJson::ToJson(NJson::KeyValue(defaultOperations, "type", "sub_types")));
    return result;
}

NDrive::TScheme TOrderTechDispatchTag::TDescription::TRepairType::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSString>("name", "Код работы").SetRequired(true);
    result.Add<TFSString>("hr_name", "Название работы");
    result.Add<TFSNumeric>("rate", "Норма работы (x10)").SetMin(0).SetDefault(0).SetRequired(true);
    result.Add<TFSVariants>("additional_tags", "При назначении добавить теги").SetReference("device_tags").SetMultiSelect(true).SetEditable(true);
    result.Add<TFSBoolean>("is_default", "Добавить по умолчанию(для отображения)");
    result.Add<TFSBoolean>("is_common_option", "Основная настройка(для отображения)");
    return result;
}

NDrive::TScheme TOrderTechDispatchTag::TDescription::TRepairGroupInfo::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSString>("name", "Код группы работ").SetRequired(true);
    result.Add<TFSString>("hr_name", "Название группы работ");
    result.Add<TFSArray>("sub_types", "Детализация работ").SetElement(TRepairType::GetScheme()).SetRequired(true);
    return result;
}

NDrive::TScheme TOrderTechDispatchTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSArray>("repair_groups", "Описание работ").SetElement(TRepairGroupInfo::GetScheme()).SetRequired(true);
    return result;
}

NJson::TJsonValue TOrderTechDispatchTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TOrderTechDispatchTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

TExpected<TMap<TString, ui32>, TString> TOrderTechDispatchTag::TDescription::GetRates(const TMap<TString, TVector<TString>>& types, const NDrive::IServer* server) {
    auto description = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(TOrderTechDispatchTag::TypeName);
    auto orderDescription = Yensured(std::dynamic_pointer_cast<const TOrderTechDispatchTag::TDescription>(description));
    TMap<TString, ui32> result;
    for (const auto& [type, subTypes] : types) {
        bool found = false;
        ui32 rate = 0;
        for (const auto& group : orderDescription->GetGroupInfos()) {
            if (type == group.GetName()) {
                found = true;
                for (const auto& subType : subTypes) {
                    bool subFound = false;
                    for (const auto& commonSubType : group.GetSubTypes()) {
                        if (commonSubType.GetName() == subType) {
                            subFound = true;
                            rate += commonSubType.GetRate();
                            break;
                        }
                    }
                    if (!subFound) {
                        return MakeUnexpected<TString>("unknown sub type " + subType);
                    }
                }
                break;
            }
        }
        if (!found) {
            return MakeUnexpected<TString>("unknown type " + type);
        }
        result.emplace(type, rate);
    }
    return result;
}

NDrive::TScheme TAssignmentTechDispatchTag::TDescription::TRepairGroup::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSNumeric>("max_resource", "Максимальный ресурс").SetRequired(true);
    return result;
}

NDrive::TScheme TAssignmentTechDispatchTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("service_name", "Название сервиса").SetRequired(true);
    result.Add<TFSNumeric>("service_priority", "Приоритет назначения(0 - самый высокий)").SetMin(0).SetDefault(0).SetRequired(true);
    result.Add<TFSString>("additional_comment", "Комментарий тега назначения(адрес и т.д.)");

    NDrive::TScheme simpleElementScheme;
    auto& name = simpleElementScheme.Add<TFSVariants>("type", "Тип работ").SetMultiSelect(false).SetEditable(true);
    auto description = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(TOrderTechDispatchTag::TypeName);
    auto orderDescription = std::dynamic_pointer_cast<const TOrderTechDispatchTag::TDescription>(description);
    if (orderDescription) {
        TVector<TFSVariants::TCompoundVariant> typeNames;
        for (const auto& group: orderDescription->GetGroupInfos()) {
            typeNames.emplace_back(group.GetName(), group.GetHrName());
        }
        name.SetCompoundVariants(std::move(typeNames));
    }
    simpleElementScheme.Add<TFSStructure>("info", "Детали").SetStructure(TRepairGroup::GetScheme()).SetRequired(true);

    result.Add<TFSArray>("repair_groups", "Описание работ").SetElement(simpleElementScheme).SetRequired(true);
    return result;
}

NJson::TJsonValue TAssignmentTechDispatchTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TAssignmentTechDispatchTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

TMaybe<TVector<TAssignmentTechDispatchTag::TStationInfo>> TAssignmentTechDispatchTag::GetStationsInfo(NDrive::TEntitySession& tx, const NDrive::IServer* server) {
    const auto& manager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    TMap<TString, TStationInfo> stations;
    {
        auto assignmentTagDescriptions = manager.GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car, {TAssignmentTechDispatchTag::TypeName});
        ForEach(assignmentTagDescriptions.begin(), assignmentTagDescriptions.end(), [&stations](const auto& desc) {
            if (auto assigmentPtr = std::dynamic_pointer_cast<const TAssignmentTechDispatchTag::TDescription>(desc.second)) {
                stations.emplace(desc.first, assigmentPtr);
            }
        });
    }

    auto taggedCars = manager.GetDeviceTags().RestoreTagsRobust({}, MakeVector(NContainer::Keys(stations)), tx);
    if (!taggedCars) {
        return {};
    }

    for (const auto& tag : *taggedCars) {
        auto stationPtr = stations.FindPtr(tag->GetName());
        if (stationPtr) {
            auto result = stationPtr->AddRate(tag, server);
            if (result == TStationInfo::ERateResult::IncorrectType || result == TStationInfo::ERateResult::IncorrectRate) {
                tx.SetErrorInfo("IncorrectRate", tag.GetTagId());
                return {};
            }
        } else {
            tx.SetErrorInfo("UnknownTag", tag->GetName());
            return {};
        }
    }
    auto priorityStations = MakeVector(NContainer::Values(stations));
    Sort(priorityStations.begin(), priorityStations.end(), [](const auto& a, const auto& b) {
        return a.GetDescription().GetServicePriority() < b.GetDescription().GetServicePriority();
    });
    return priorityStations;
}

TAssignmentTechDispatchTag::TStationInfo::ERateResult TAssignmentTechDispatchTag::TStationInfo::AddRate(const TDBTag& tag, const NDrive::IServer* server) {
    auto dispatchTag = tag.GetTagAs<ITechDispatchTag>();
    if (!dispatchTag) {
        return ERateResult::IncorrectType;
    }
    auto tagRates = dispatchTag->GetRates(server);
    if (!tagRates) {
        return ERateResult::IncorrectRate;
    }
    for (const auto& [name, rate] : *tagRates) {
        auto itRate = Rates.find(name);
        if (itRate == Rates.end()) {
            itRate = Rates.emplace(name, 0).first;
        }
        itRate->second += rate;
    }

    TString error;
    if (!GetDescription().Check(Rates, error)) {
        Errors.AddMessage(tag.GetTagId(), error);
        return ERateResult::LimitError;
    }

    return ERateResult::Success;
}

bool TAssignmentTechDispatchTag::TDescription::Check(const TMap<TString, ui32>& rates, TString& error) const {
    for (const auto& [name, rate] : rates) {
        auto group = RepairGroups.FindPtr(name);
        if (!group) {
            error = "no " + name;
            return false;
        }
        if (group->GetMaxResource() && group->GetMaxResource() < rate) {
            error = "limit resource for " + name + " " + ToString(rate) + "/" + ToString(group->GetMaxResource());
            return false;
        }
    }
    return true;
}

const TString TOrderTechDispatchTag::TypeName = "order_tech_dispatch_tag";
ITag::TFactory::TRegistrator<TOrderTechDispatchTag> TOrderTechDispatchTag::Registrator(TOrderTechDispatchTag::TypeName);
TTagDescription::TFactory::TRegistrator<TOrderTechDispatchTag::TDescription> OrderTechDispatchTagDescriptionRegistrator(TOrderTechDispatchTag::TypeName);

const TString TAssignmentTechDispatchTag::TypeName = "assignment_tech_dispatch_tag";
ITag::TFactory::TRegistrator<TAssignmentTechDispatchTag> TAssignmentTechDispatchTag::Registrator(TAssignmentTechDispatchTag::TypeName);
TTagDescription::TFactory::TRegistrator<TAssignmentTechDispatchTag::TDescription> AssignmentTechDispatchTagDescriptionRegistrator(TAssignmentTechDispatchTag::TypeName);
