#pragma once

#include <drive/backend/data/common/serializable.h>

#include <rtline/util/types/field.h>

class ITechDispatchTag: public IJsonSerializableTag {
private:
    using TBase = IJsonSerializableTag;

public:
    using TBase::TBase;

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;
    bool OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

    TExpected<TMap<TString, ui32>, TString> GetRates(const NDrive::IServer* server) const;

protected:
    bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    using TOperations = TMap<TString, TVector<TString>>;
    using TSpecials = TMap<TString, TSet<TString>>;
    R_FIELD(TOperations, Operations);
    R_FIELD(TSpecials, Specials);

public:
    DECLARE_FIELDS(
        Field(NJson::KeyValue(Operations, "type", "sub_types"), "operations", true),
        Field(Specials, "specials")
    );
};

class TOrderTechDispatchTag: public ITechDispatchTag {
private:
    using TBase = ITechDispatchTag;

public:
    class TDescription: public TTagDescription {
    public:
        class TRepairType {
            R_FIELD(TString, Name);
            R_FIELD(TString, HrName);
            R_FIELD(ui32, Rate, 0);
            R_FIELD(TSet<TString>, AdditionalTags);
            R_FIELD(bool, Default, false);
            R_FIELD(bool, CommonOption, false);

        public:
            DECLARE_FIELDS(
                Field(Name, "name", true),
                Field(HrName, "hr_name"),
                Field(Rate, "rate"),
                Field(AdditionalTags, "additional_tags"),
                Field(Default, "is_default"),
                Field(CommonOption, "is_common_option")
            );

        public:
            static NDrive::TScheme GetScheme();
        };

        class TRepairGroupInfo {
            R_FIELD(TString, Name);
            R_FIELD(TString, HrName);
            R_FIELD(TVector<TRepairType>, SubTypes);

        public:
            DECLARE_FIELDS(
                Field(Name, "name", true),
                Field(HrName, "hr_name"),
                Field(SubTypes, "sub_types", true)
            );

            bool operator<(const TRepairGroupInfo& a) const {
                return Name < a.Name;
            }

        public:
            static NDrive::TScheme GetScheme();
        };

    public:
        using TTagDescription::TTagDescription;

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        static TExpected<TMap<TString, ui32>, TString> GetRates(const TMap<TString, TVector<TString>>& types, const NDrive::IServer* server);

    protected:
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        R_FIELD(TSet<TRepairGroupInfo>, GroupInfos);

    public:
        DECLARE_FIELDS(
            Field(GroupInfos, "repair_groups", true)
        );
    };

public:
    using TBase::TBase;
    static const TString TypeName;

public:

private:
    static TFactory::TRegistrator<TOrderTechDispatchTag> Registrator;
};

class TAssignmentTechDispatchTag: public ITechDispatchTag {
private:
    using TBase = ITechDispatchTag;

public:
    class TDescription: public TTagDescription {
    public:
       class TRepairGroup {
            R_FIELD(ui32, MaxResource, 0);

        public:
            DECLARE_FIELDS(
                Field(MaxResource, "max_resource", true)
            );

        public:
            static NDrive::TScheme GetScheme();
        };
    public:
        using TTagDescription::TTagDescription;

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        bool Check(const TMap<TString, ui32>& rates, TString& error) const;

    protected:
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        using TRepairGroups = TMap<TString, TRepairGroup>;

        R_FIELD(ui32, ServicePriority, 0);
        R_FIELD(TString, ServiceName);
        R_FIELD(TString, AdditionalComment);
        R_FIELD(TRepairGroups, RepairGroups);

    public:
        DECLARE_FIELDS(
            Field(ServiceName, "service_name"),
            Field(ServicePriority, "service_priority"),
            Field(AdditionalComment, "additional_comment"),
            Field(NJson::KeyValue(RepairGroups, "type", "info"), "repair_groups")
        );
    };

    class TStationInfo {
        using TRates = TMap<TString, ui32>;
        TAtomicSharedPtr<const TAssignmentTechDispatchTag::TDescription> Description;
        R_FIELD(TRates, Rates);
        R_READONLY(TMessagesCollector, Errors);

    public:
        enum class ERateResult {
            Success,
            IncorrectType,
            IncorrectRate,
            LimitError
        };

    public:
        TStationInfo(TAtomicSharedPtr<const TAssignmentTechDispatchTag::TDescription> desc)
            : Description(desc)
        {}

        const TAssignmentTechDispatchTag::TDescription& GetDescription() const {
            return *Yensured(Description);
        }

        ERateResult AddRate(const TDBTag& tag, const NDrive::IServer* server);
    };

public:
    bool OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

public:
    static const TString TypeName;
    static TMaybe<TVector<TStationInfo>> GetStationsInfo(NDrive::TEntitySession& tx, const NDrive::IServer* server);

private:
    using TBase::TBase;
    static TFactory::TRegistrator<TAssignmentTechDispatchTag> Registrator;
};
