#include "telematics.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/searchserver/http_status_config.h>
#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/protocol/errors.h>
#include <drive/telematics/protocol/mio.h>
#include <drive/telematics/protocol/nonce.h>
#include <drive/telematics/protocol/settings.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>
#include <rtline/util/types/cast.h>

#include <util/digest/fnv.h>
#include <util/generic/serialized_enum.h>

namespace {
    template <class T>
    IEntityTagsManager::TOptionalTag GetTelematicsTag(const TString& carId, const NDrive::IServer* server, NDrive::TEntitySession& session, ui32 attempt = 0) {
        if (attempt >= 2) {
            session.SetErrorInfo("GetTelematicsTag", "cannot acquire tag after " + ToString(attempt) + " attempts");
            return {};
        }

        CHECK_WITH_LOG(server);
        auto api = server->GetDriveAPI();
        const auto& tagManager = api->GetTagsManager().GetDeviceTags();

        auto tags = tagManager.RestoreEntityTags(carId, { T::Type() }, session);
        if (!tags) {
            return {};
        }
        if (!tags->empty()) {
            return std::move(tags->at(0));
        }

        auto tag = MakeAtomicShared<T>();
        auto addedTags = tagManager.AddTag(tag, "frontend", carId, server, session);
        if (!addedTags) {
            return {};
        }
        if (!addedTags->empty()) {
            return std::move(addedTags->front());
        }

        return GetTelematicsTag<T>(carId, server, session, attempt + 1);
    }

    TTelematicsConfigurationTag::TPinInfo GetPinByIMEIImpl(const TString& objectId, const NDrive::IServer* server, ui16 serverId, TInstant deadline);

    TTelematicsConfigurationTag::TPinInfo GetPinImpl(const TString& objectId, const NDrive::IServer* server, ui16 serverId, TInstant deadline) {
        auto api = Yensured(server)->GetDriveAPI();
        auto imei = Yensured(api)->GetIMEI(objectId);
        Y_ENSURE(imei, "cannot get IMEI for " << objectId);
        auto configurationTag = Yensured(api)->GetTagsManager().GetDeviceTags().GetTagFromCache(objectId, TTelematicsConfigurationTag::Type(), TInstant::Zero());
        auto configuration = configurationTag ? configurationTag->GetTagAs<TTelematicsConfigurationTag>() : nullptr;
        if (configuration && configuration->GetConfigurationHash()) {
            const auto& pins = configuration->GetPins();
            auto pin = pins.find(serverId);
            if (pin != pins.end()) {
                return {
                    imei,
                    pin->second.Value.Get(),
                    TTelematicsConfigurationTag::Type()
                };
            }
        }

        return GetPinByIMEIImpl(imei, server, serverId, deadline);
    }

    TTelematicsConfigurationTag::TPinInfo GetPinByIMEIImpl(const TString& imei, const NDrive::IServer* server, ui16 serverId, TInstant deadline) {
        auto sensorApi = Yensured(server)->GetSensorApi();
        if (sensorApi) {
            auto timeout = deadline - Now();
            auto sensor = sensorApi->GetSensor(imei, { VEGA_SETTING_SERVER_PIN, serverId }).ExtractValue(timeout);
            if (sensor) {
                auto value = sensor->ConvertTo<TBuffer>();
                return {
                    imei,
                    LoadFrom<NDrive::NVega::THardPasswordParameter>(value).Value.Get(),
                    "sensor_api"
                };
            }
        }

        const auto& client = Yensured(server)->GetTelematicsClient();
        {
            auto command = NDrive::NVega::TCommand::GetParameter({ VEGA_SETTING_SERVER_PIN, serverId });
            auto timeout = deadline - Now();
            auto handler = client.Command(imei, command, timeout);
            Y_ENSURE(handler.GetFuture().Wait(deadline), "unsuccesful GET_PARAM VEGA_SETTING_SERVER_PIN: wait timeout");
            auto status = handler.GetStatus();
            Y_ENSURE(status == NDrive::TTelematicsClient::EStatus::Success, "unsuccesful GET_PARAM VEGA_SETTING_SERVER_PIN: " << status << ' ' << handler.GetMessage());
            auto response = handler.GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            Y_ENSURE(response, "unsuccesful GET_PARAM VEGA_SETTING_SERVER_PIN: bad cast");
            return {
                imei,
                LoadFrom<NDrive::NVega::THardPasswordParameter>(response->Sensor.ConvertTo<TBuffer>()).Value.Get(),
                "telematics_client"
            };
        }
    }

    TSet<std::pair<NDrive::NVega::ECommandCode, NDrive::NVega::TCommandRequest::TCommandCode>> ExternalCommandCodeMatches = {
        { NDrive::NVega::ECommandCode::SCENARIO_STOP_WARMING_AND_OPEN_DOORS, NDrive::NVega::ECommandCode::OPEN_DOORS },
    };

    bool MatchExternalCommandCode(NDrive::NVega::TCommandRequest::TCommandCode executed, NDrive::NVega::ECommandCode expected) {
        if (executed == expected) {
            return true;
        }
        if (ExternalCommandCodeMatches.contains(std::make_pair(expected, executed))) {
            return true;
        }
        return false;
    }

    NDrive::TScheme KeyValueScheme(const NDrive::TScheme& value) {
        NDrive::TScheme result;
        result.Add<TFSNumeric>("key", "id");
        result.Add<TFSStructure>("value").SetStructure(value);
        return result;
    };

    NDrive::TScheme GetLogicScriptScheme() {
        NDrive::TScheme result;

        result
            .Add<TFSNumeric>("command_id", "Command id")
            .SetMin(0)
            .SetMax(0xFFFF)
            .SetRequired(true);

        result.Add<TFSString>("name", "Name").SetRequired(true);

        {
            NDrive::TScheme scheme;
            scheme
                .Add<TFSVariants>("type", "Type")
                .InitVariants<NDrive::NVega::TLogicStep::EType>()
                .SetRequired(true);
            scheme.Add<TFSNumeric>("action_id", "Action id").SetRequired(true);
            result.Add<TFSArray>("steps", "Steps").SetElement(KeyValueScheme(scheme));
        }

        return result;
    }

    NDrive::TScheme GetLogicCommandScheme() {
        NDrive::TScheme result;

        result
            .Add<TFSVariants>("type", "Type")
            .InitVariants<NDrive::NVega::TLogicCommand::EType>()
            .SetRequired(true);

        NDrive::TScheme argument;
        {
            {
                NDrive::TScheme scheme;
                scheme.Add<TFSNumeric>("sensor_id", "Sensor Id").SetDefault(0);
                scheme.Add<TFSNumeric>("sensor_sub_id", "Sensor Sub Id").SetDefault(0);
                scheme.Add<TFSNumeric>("value", "Value").SetDefault(0);

                argument.Add<TFSStructure>("set_value", "Set value").SetStructure(scheme).SetRequired(false);
            }

            argument.Add<TFSNumeric>("variable", "Variable").SetRequired(false);
        }

        result.Add<TFSStructure>("argument", "Argument").SetStructure(std::move(argument)).SetRequired(true);

        return std::move(result);
    }

    NDrive::TScheme GetLogicCheckScheme() {
        NDrive::TScheme result;

        result
            .Add<TFSVariants>("type", "Type")
            .InitVariants<NDrive::NVega::TLogicCheck::EType>()
            .SetRequired(true);

        result
            .Add<TFSVariants>("operation", "Operation")
            .InitVariants<NDrive::NVega::TLogicCheck::EOperation>()
            .SetRequired(true);

        result.Add<TFSNumeric>("sensor_id", "Sensor Id").SetRequired(true);
        result.Add<TFSNumeric>("sensor_sub_id", "Sensor sub id").SetDefault(0);

        {
            NDrive::TScheme scheme;
            scheme
                .Add<TFSVariants>("type", "Type")
                .InitVariants<NDrive::NVega::ECommonValueType>()
                .SetRequired(true);
            scheme.Add<TFSNumeric>("value").SetDefault(0).SetRequired(true);
            result.Add<TFSStructure>("value", "Value").SetStructure(std::move(scheme));
        }

        result.Add<TFSNumeric>("argument", "Argument").SetDefault(0);

        return result;
    }

    NDrive::TScheme GetLogicActiveIdScheme() {
        NDrive::TScheme result;

        result.Add<TFSBoolean>("active", "Active");
        result.Add<TFSNumeric>("id", "Id").SetMin(0).SetMax(0xFFFF).SetDefault(0);

        return result;
    }

    NDrive::TScheme GetLogicTriggerScheme() {
        NDrive::TScheme result;

        result
            .Add<TFSVariants>("type", "Type")
            .InitVariants<NDrive::NVega::TLogicTrigger::EType>()
            .SetRequired(true);

        result
            .Add<TFSArray>("checks", "Checks")
            .SetElement(KeyValueScheme(GetLogicActiveIdScheme()));

        result.Add<TFSStructure>("triggered_script_id").SetStructure(GetLogicActiveIdScheme());
        result.Add<TFSStructure>("untriggered_script_id").SetStructure(GetLogicActiveIdScheme());
        result.Add<TFSNumeric>("timeout").SetMin(0).SetMax(0xFFFF).SetDefault(0);

        return result;
    }

    NDrive::TScheme GetLogicNotifyScheme() {
        NDrive::TScheme result;
        result.Add<TFSString>("message", "Message").SetRequired(true);
        return result;
    }

    NUnistat::TIntervals TelematicsIntervals = {
        0,
        100,
        200,
        500,
        1000,
        2000,
        3000,
        4000,
        5000,
        10000,
        20000,
        100000,
    };

    TEnumSignal<NDrive::TTelematicsClient::EStatus> TelematicsResults {{ "telematics-results"}, false};
    TEnumSignal<NDrive::NVega::ECommandCode> TelematicsTimes {{ "telematics-times" }, TelematicsIntervals};
    TUnistatSignal<> TelematicsOptimisticFailed {{ "telematics-optimistic-failed" }, false};
    TUnistatSignal<> TelematicsTerminationProcessed {{ "telematics-termination-processed" }, false};
    TUnistatSignal<> TelematicsTerminationRecovered {{ "telematics-termination-recovered" }, false};
}

bool NDrive::ContainsActionInTag(
    const TString& userId,
    const TDriveAPI* driveApi,
    const THttpStatusManagerConfig& configHttpStatus,
    const TString& carId,
    NDrive::TEntitySession& session,
    TStringBuf actionName
) {
    auto tagsByObject = TMap<TString, TVector<TDBTag>>();

    R_ENSURE(
        driveApi->GetTagsManager().GetDeviceTags().GetPerformObjects(userId, tagsByObject, session),
        configHttpStatus.PermissionDeniedStatus,
        "cannot get performed tags: " << session.GetStringReport()
    );

    auto tags = tagsByObject.find(carId);
    R_ENSURE(tags != tagsByObject.end(), configHttpStatus.PermissionDeniedStatus, "no performed tags for " << carId);

    for (auto&& tag : tags->second) {
        auto actions = driveApi->GetActions(tag);
        if (actions.contains(actionName)) {
            return true;
        }
    }

    return false;
}

template <>
NJson::TJsonValue NJson::ToJson(const TTelematicsCommandTagTraits::TRuntimeContext& object) {
    NJson::TJsonValue result;
    if (object.Callback2) {
        result["callback"] = true;
    }
    if (object.Handler2) {
        result["handler"] = object.Handler2.GetId();
    }
    if (object.CallbackTimeout) {
        result["callback_timeout"] = NJson::ToJson(object.CallbackTimeout);
    }
    if (object.Timeout) {
        result["timeout"] = NJson::ToJson(object.Timeout);
    }
    if (object.WaitConnectionTimeout) {
        result["wait_connection_timeout"] = NJson::ToJson(object.WaitConnectionTimeout);
    }
    if (object.DataLifetime) {
        result["data_lifetime"] = NJson::ToJson(object.DataLifetime);
    }
    if (object.Prelinger) {
        result["prelinger"] = NJson::ToJson(object.Prelinger);
    }
    if (object.Postlinger) {
        result["postlinger"] = NJson::ToJson(object.Postlinger);
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const TTelematicsConfigurationTraits::TCustom& object) {
    return object.ToJson();
}

template <>
NJson::TJsonValue NJson::ToJson(const TTelematicsConfigurationTag::TSensorMeta& object) {
    NJson::TJsonValue result;
    result["display_name"] = object.DisplayName;
    result["min_value"] = NJson::ToJson(NJson::Nullable(object.MinValue));
    result["max_value"] = NJson::ToJson(NJson::Nullable(object.MaxValue));
    result["critical_values"] = NJson::ToJson(object.CriticalValues);
    result["icon"] = NJson::ToJson(NJson::Nullable(object.Icon));
    result["dim"] = NJson::ToJson(NJson::Nullable(object.Dim));
    result["format_pattern"] = NJson::ToJson(NJson::Nullable(object.FormatPattern));
    result["decoding"] = NJson::ToJson(NJson::KeyValue(object.Decoding, "code", "decode"));
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsConfigurationTraits::TCustom& object) {
    TString tagsFilter;
    bool result =
        NJson::ParseField(value["model"], object.MutableModel()) &&
        NJson::ParseField(value["tags_filter"], tagsFilter) &&
        NJson::ParseField(value["sensors"], NJson::KeyValue(object.MutableSensors())) &&
        NJson::ParseField(value["can_sensors"], NJson::KeyValue(object.MutableCanParameters())) &&
        NJson::ParseField(value["server_settings"], NJson::KeyValue(object.MutableSensors())) &&
        NJson::ParseField(value["apns"], NJson::KeyValue(object.MutableAPNs())) &&
        NJson::ParseField(value["logic_scripts"], NJson::KeyValue(object.MutableLogicScripts())) &&
        NJson::ParseField(value["logic_commands"], NJson::KeyValue(object.MutableLogicCommands())) &&
        NJson::ParseField(value["logic_checks"], NJson::KeyValue(object.MutableLogicChecks())) &&
        NJson::ParseField(value["logic_triggers"], NJson::KeyValue(object.MutableLogicTriggers())) &&
        NJson::ParseField(value["logic_notifies"], NJson::KeyValue(object.MutableLogicNotifies()));

    if (result) {
        object.MutableTagsFilter() = TTagsFilter::BuildFromString(tagsFilter);
    }

    return result;
}

TTelematicsCommandTagTraits::TTelematicsCommandTagTraits(TCommand command, TDuration timeout, NDrive::TTelematicsClient::THandlerCallback&& callback)
    : Command(std::move(command))
{
    GetContext().Callback2 = std::move(callback);
    GetContext().CallbackTimeout = timeout;
    GetContext().Timeout = timeout;
}

void TTelematicsCommandTagTraits::SetPrelinger(TDuration value) {
    auto& context = GetContext();
    context.Prelinger = value;
}

void TTelematicsCommandTagTraits::SetPostlinger(TDuration value) {
    auto& context = GetContext();
    context.Postlinger = value;
}

void TTelematicsCommandTagTraits::SetEvolveTag(bool value) {
    GetContext().EvolveTag = value;
}

void TTelematicsCommandTagTraits::SetExternalCommandId(const TString& value) {
    GetContext().ExternalCommandId = value;
}

void TTelematicsCommandTagTraits::SetWaitConnectionTimeout(TDuration value) {
    GetContext().WaitConnectionTimeout = value;
}

bool TTelematicsCommandTagTraits::FromJson(const NJson::TJsonValue& value) {
    NDrive::NVega::ECommandCode command;
    if (!TryFromString(value["command"].GetStringRobust(), command)) {
        Command = NDrive::NVega::ECommandCode::UNKNOWN;
    } else {
        Command = command;
    }

    if (!value["handler"].GetString(&HandlerId)) {
        return false;
    }

    return true;
}

void TTelematicsCommandTagTraits::ToJson(NJson::TJsonValue& result) const {
    result["context"] = NJson::ToJson(Context);
    result["command"] = ToString(GetCommand());
    result["handler"] = HandlerId;
    result["timeout"] = 0;
    result["cb_timeout"] = 0;
}

const TTelematicsCommandTagTraits::TRuntimeContext& TTelematicsCommandTagTraits::GetContext() const {
    if (Context) {
        return *Context;
    } else {
        return Default<TRuntimeContext>();
    }
}

TTelematicsCommandTagTraits::TRuntimeContext& TTelematicsCommandTagTraits::GetContext() {
    if (!Context) {
        Context = MakeHolder<TRuntimeContext>();
    }
    return *Context;
}

void TTelematicsCommandTagTraits::SetHandler(NDrive::TTelematicsClient::THandler&& handler) {
    HandlerId = handler.GetId();
    GetContext().Handler2 = std::move(handler);
}

bool TTelematicsCommandTagTraits::Invoke(const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    CHECK_WITH_LOG(server);
    auto api = server->GetDriveAPI();
    CHECK_WITH_LOG(api);

    auto imei = api->GetIMEI(objectId);
    if (!imei) {
        session.SetErrorInfo("telematics_command", "cannot get IMEI for " + objectId, EDriveSessionResult::InternalError);
        return false;
    }

    return Invoke2(objectId, imei, server, session);
}

bool TTelematicsCommandTagTraits::Invoke2(const TString& objectId, const TString& imei, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(session);
    const NDrive::TTelematicsClient& client = server->GetTelematicsClient();
    auto& context = GetContext();
    auto start = Now();
    auto handler = client.Command(imei, Command, context.Timeout, context.WaitConnectionTimeout, context.ExternalCommandId);
    {
        NDrive::TEventLog::Log("InvokeTelematicsCommand", NJson::TMapBuilder
            ("object_id", objectId)
            ("id", handler.GetId())
            ("command", NJson::ToJson(Command))
            ("context", NJson::ToJson(context))
        );
    }
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "invoked_telematics_command")
            ("id", handler.GetId())
        );
    }
    if (context.Callback2) {
        handler.Subscribe(std::move(context.Callback2), start + context.CallbackTimeout);
    }
    auto eventLogState = NDrive::TEventLog::CaptureState();
    handler.Subscribe([command = Command, eventLogState = std::move(eventLogState), objectId, start](const NDrive::TTelematicsClient::THandler& handler) {
        auto elsg = NDrive::TEventLog::Guard(eventLogState);
        auto finish = Now();
        auto duration = finish - start;
        bool optimistic = handler.GetOptimisticRequestSucceeded();
        auto status = handler.GetStatus();
        bool terminationProcessed = handler.GetTerminationProcessed();
        TelematicsResults.Signal(status, 1);
        TelematicsTimes.Signal(command.Code, duration.MilliSeconds());
        if (!optimistic) {
            TelematicsOptimisticFailed.Signal(1);
        }
        if (terminationProcessed) {
            TelematicsTerminationProcessed.Signal(1);
            if (status != NDrive::TTelematicsClient::EStatus::Terminated) {
                TelematicsTerminationRecovered.Signal(1);
            }
        }
        NDrive::TEventLog::Log("FinishTelematicsCommand", NJson::TMapBuilder
            ("command", NJson::ToJson(command))
            ("duration", NJson::ToJson(duration))
            ("object_id", objectId)
            ("optimistic", optimistic)
            ("id", handler.GetId())
            ("events", NJson::ToJson(handler.GetEvents()))
            ("shard", handler.GetShard())
            ("status", ToString(status))
            ("termination_processed", terminationProcessed)
        );
    });
    SetHandler(std::move(handler));
    return true;
}

NDrive::TScheme TTelematicsCommandTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("notifier", "Notifier name").SetVariants(server ? server->GetNotifierNames() : TSet<TString>{});
    return result;
}

NJson::TJsonValue TTelematicsCommandTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    if (Notifier) {
        result["notifier"] = Notifier;
    }
    return result;
}

bool TTelematicsCommandTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    if (!TTagDescription::DoDeserializeMetaFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["notifier"], Notifier);
}

bool TTelematicsCommandTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    Y_UNUSED(errors);
    return TTelematicsCommandTagTraits::FromJson(value);
}

void TTelematicsCommandTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TTelematicsCommandTagTraits::ToJson(value);
}

TTelematicsCommandTag::TProto TTelematicsCommandTag::DoSerializeSpecialDataToProto() const {
    TTelematicsCommandTag::TProto result = TBase::DoSerializeSpecialDataToProto();
    result.SetCommand(GetCommand());
    if (const TString& handler = GetHandlerId()) {
        result.SetHandler(handler);
    }
    return result;
}

bool TTelematicsCommandTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }

    auto command = static_cast<NDrive::NVega::ECommandCode>(proto.GetCommand());
    const auto& commands = GetEnumNames<NDrive::NVega::ECommandCode>();
    if (commands.find(command) == commands.end()) {
        command = NDrive::NVega::ECommandCode::UNKNOWN;
    }

    SetCommand(command);
    SetHandlerId(proto.GetHandler());
    return true;
}

TAtomicSharedPtr<TTelematicsCommandTag> TTelematicsCommandTag::Command(const TString& carId, TAtomicSharedPtr<TTelematicsCommandTag> command, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto tag = GetTelematicsTag<TTelematicsCommandTag>(carId, server, session);
    if (!tag) {
        return nullptr;
    }
    if (!command) {
        session.SetErrorInfo("TelematicsCommandTag::Command", "command is null", EDriveSessionResult::InternalError);
        return nullptr;
    }

    auto hash =
        FnvHash<ui32>(carId) ^
        FnvHash<ui32>(permissions.GetUserId());

    CHECK_WITH_LOG(server);
    auto api = server->GetDriveAPI();
    auto commandString = ToString(command->GetCommand());
    const ISettings& settings = server->GetSettings();
    const TDeviceTagsManager& tagsManager = api->GetTagsManager().GetDeviceTags();

    auto description = std::dynamic_pointer_cast<const TDescription>(api->GetTagsManager().GetTagsMeta().GetDescriptionByName(command->GetName()));
    if (!description) {
        description = MakeAtomicShared<TDescription>();
    }

    auto countInvocations = [command](const auto& events, TStringBuf userId = {}) {
        ui64 result = 0;
        for (auto&& ev : events) {
            if (!ev) {
                continue;
            }
            if (userId && userId != ev.GetHistoryUserId()) {
                continue;
            }
            auto tag = ev.template GetTagAs<TTelematicsCommandTag>();
            if (!tag) {
                continue;
            }
            if (tag->GetCommand() == command->GetCommand()) {
                result += 1;
            }
        }
        return result;
    };

    auto cooldown = permissions.GetSetting<TDuration>(settings, "telematics.command." + commandString + ".cooldown");
    bool control = permissions.CheckAdministrativeActions(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Car, {}, commandString);
    if (cooldown && !control) {
        auto since = TInstant::Now() - *cooldown;
        auto optionalEvents = tagsManager.GetEventsByObject(carId, session, 0, since);
        if (!optionalEvents) {
            return nullptr;
        }

        ui64 invocations = countInvocations(*optionalEvents, permissions.GetUserId());
        if (invocations > 0) {
            session.SetErrorInfo("TelematicsCommandTag::Command", commandString + " cooldown", EDriveSessionResult::CarCooldown);
            session.SetLocalizedMessageKey(permissions.GetSetting<TString>(settings, "telematics.command." + commandString + ".cooldown_localization_message_key").GetOrElse({}));
            session.SetLocalizedTitleKey(permissions.GetSetting<TString>(settings, "telematics.command." + commandString + ".cooldown_localization_title_key").GetOrElse({}));
            return nullptr;
        }
    }

    TMaybe<double> limit;
    if (!limit) {
        limit = permissions.GetAdministrativeLimit(commandString);
    }
    if (!limit) {
        limit = permissions.GetSetting<double>(settings, "telematics.command." + commandString + ".limit");
    }
    if (limit) {
        auto since = TInstant::Now() - TDuration::Days(1);
        const auto& userId = session.GetOriginatorId() ? session.GetOriginatorId() : permissions.GetUserId();

        auto optionalEventsByUser = tagsManager.GetEvents(since, session, TTagEventsManager::TQueryOptions()
            .SetTags({ Type() })
            .SetUserIds({ userId })
        );
        if (!optionalEventsByUser) {
            return nullptr;
        }

        auto optionalEventsByOriginator = tagsManager.GetEvents(since, session, TTagEventsManager::TQueryOptions()
            .SetTags({ Type() })
            .SetOriginatorIds({ userId })
        );
        if (!optionalEventsByOriginator) {
            return nullptr;
        }

        ui64 invocationsByUser = countInvocations(*optionalEventsByUser);
        ui64 invocationsByOriginator = countInvocations(*optionalEventsByOriginator);
        ui64 invocations = invocationsByUser + invocationsByOriginator;
        if (invocations >= *limit) {
            session.SetErrorInfo(
                "TelematicsCommandTag::Command",
                "command invocations limit exceeded: " + ToString(invocationsByUser) + "+" + ToString(invocationsByOriginator) + "/" + ToString(*limit),
                EDriveSessionResult::NoUserPermissions
            );
            return nullptr;
        }
    }

    auto lingerProbability = permissions.GetSetting<float>(settings, "telematics.command." + commandString + ".linger.probability").GetOrElse(0);
    if (hash % 100 < lingerProbability * 100) {
        auto prelinger = permissions.GetSetting<TDuration>(settings, "telematics.command." + commandString + ".linger.pre").GetOrElse(TDuration::Zero());
        auto postlinger = permissions.GetSetting<TDuration>(settings, "telematics.command." + commandString + ".linger.post").GetOrElse(TDuration::Zero());
        command->SetPrelinger(prelinger);
        command->SetPostlinger(postlinger);
    }

    auto notifyInvocation = permissions.GetSetting<bool>(settings, "telematics.command." + commandString + ".notify_invocation").GetOrElse(false);
    if (notifyInvocation) {
        auto notifier = description->GetNotifier() ? server->GetNotifier(description->GetNotifier()) : nullptr;
        if (notifier) {
            auto objectInfo = api->GetCarsData()->GetObject(carId);
            auto snapshot = server->GetSnapshotsManager().GetSnapshot(carId);
            auto message = TStringBuilder()
                << "command " << commandString << " has been invoked:" << Endl
                << "author: " << permissions.GetHRReport() << Endl
            ;
            if (objectInfo) {
                message << "car: " << objectInfo->GetHRReport() << Endl;
            } else {
                message << "car_id: " << carId << Endl;
            }
            for (auto&& sensor : snapshot.GetSensors()) {
                message << "telematics." << sensor.GetName() << ": " << sensor.GetJsonValue().GetStringRobust() << Endl;
            }
            notifier->Notify(message);
        }
    }

    auto waitConnectionTimeout = permissions.GetSetting<TDuration>(settings, "telematics.wait_connection_timeout").GetOrElse(TDuration::Zero());
    if (waitConnectionTimeout) {
        command->SetWaitConnectionTimeout(waitConnectionTimeout);
    }

    const auto& from = *tag;
    const auto& to = command;
    if (command->ShouldEvolveTag()) {
        if (!tagsManager.EvolveTag(from, command, permissions, server, session)) {
            return nullptr;
        }
    } else {
        const TEvolutionContext* evolutionContext = nullptr;
        if (!from->OnBeforeEvolve(from, to, permissions, server, session, evolutionContext)) {
            return nullptr;
        }
        if (!to->OnAfterEvolve(from, to, permissions, server, session, evolutionContext)) {
            return nullptr;
        }
    }

    return command;
}

NThreading::TFuture<NDrive::TCommonCommandResponse> TTelematicsCommandTag::Command(const TString& carId, TCommand command, const TRuntimeOptions& options, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto promise = NThreading::NewPromise<NDrive::TCommonCommandResponse>();
    auto callback = [promise](const NDrive::TTelematicsClient::THandler& handler) mutable {
        auto response = MakeCommandResponse(handler);
        bool set = promise.TrySetValue(std::move(response));
        Y_ASSERT(set);
    };
    auto timeout = options.Timeout;
    auto tag = MakeAtomicShared<TTelematicsCommandTag>(std::move(command), timeout, std::move(callback));
    tag->SetEvolveTag(options.EvolveTag);
    tag->SetExternalCommandId(options.ExternalCommandId);
    auto result = Command(carId, tag, permissions, server, session);
    if (!result) {
        return {};
    }

    return promise.GetFuture();
}

TExpected<TTelematicsCommandTag::TCommand, yexception> TTelematicsCommandTag::ParseCommand(const NJson::TJsonValue& description, const TString& objectId, const NDrive::IServer& server) {
    auto panic = [&](TCommand&& command, NDrive::NVega::TCommandRequest::TPanic::EType type) {
        auto api = server.GetDriveAPI();
        auto prefix = "telematics.command." + ToString(command.Code);
        auto enable = server.GetSettings().GetValue<bool>(prefix + ".use_panic").GetOrElse(true);
        if (api && enable) {
            auto expectedTag = api->GetTagsManager().GetDeviceTags().GetTagFromCache(objectId, TTelematicsFirmwareTag::Type(), TInstant::Zero());
            if (expectedTag) {
                auto expectedFirmwareInfo = TTelematicsFirmwareTag::GetTaggedInfo(*expectedTag, server);
                if (expectedFirmwareInfo) {
                    if (expectedFirmwareInfo->IsFeatureSupported(NDrive::NVega::TFirmwareInfo::Panic)) {
                        NDrive::NVega::TCommandRequest::TPanic panic;
                        panic.Type = type;
                        panic.Time = std::min<ui32>(
                            std::numeric_limits<ui8>::max(),
                            server.GetSettings().GetValue<ui32>(prefix + ".duration").GetOrElse(10)
                        );

                        TCommand result;
                        result.Argument.Set(panic);
                        result.Code = NDrive::NVega::ECommandCode::YADRIVE_PANIC;
                        return result;
                    }
                } else {
                    DEBUG_LOG << "cannot get FirmwareInfo for " << objectId << ": " << expectedFirmwareInfo.GetError() << Endl;
                }
            } else {
                DEBUG_LOG << "cannot get " << TTelematicsFirmwareTag::Type() << " for " << objectId << ": " << expectedTag.GetError() << Endl;
            }
        }
        return command;
    };

    auto parsed = NDrive::ParseCommand(description);
    if (!parsed) {
        return parsed;
    }

    auto result = *parsed;
    switch (result.Code) {
    case NDrive::NVega::ECommandCode::BLINKER_FLASH:
        result = panic(std::move(result), NDrive::NVega::TCommandRequest::TPanic::BLINK);
        break;
    case NDrive::NVega::ECommandCode::HORN:
        result = panic(std::move(result), NDrive::NVega::TCommandRequest::TPanic::HORN);
        break;
    case NDrive::NVega::ECommandCode::HORN_AND_BLINK:
        result = panic(std::move(result), NDrive::NVega::TCommandRequest::TPanic::HORN_AND_BLINK);
        break;
    case NDrive::NVega::ECommandCode::YADRIVE_WARMING:
        if (auto d = server.GetSettings().GetValue<TDuration>("telematics.command." + ToString(result.Code) + ".duration"); result.Argument.IsNull() && d) {
            auto duration = std::clamp(*d, TDuration::Minutes(1), TDuration::Minutes(60));
            NDrive::NVega::TCommandRequest::TWarming warming;
            warming.Time = duration.Minutes();
            result.Argument.Set(warming);
        }
        break;
    default:
        break;
    }
    return result;
}

TString TTelematicsCommandTag::GenerateId(const TString& carId, const TCommand& command, TStringBuf reqId, const NDrive::IServer& server) {
    auto imei = Yensured(server.GetDriveAPI())->GetIMEI(carId);
    auto parsedReqId = NUtil::ParseReqId(reqId);
    auto result = TStringBuilder() << imei;
    if (parsedReqId.Timestamp) {
        result << '-' << parsedReqId.Timestamp;
    }
    if (parsedReqId.Rnd) {
        result << '-' << parsedReqId.Rnd;
    }
    result << '-' << command.Code;
    return result;
}

TString TTelematicsCommandTag::GetUserErrorDescription(TStringBuf message, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server) {
    return GetUserErrorDescription(NDrive::ParseError(message), locale, permissions, server);
}

TString TTelematicsCommandTag::GetUserErrorDescription(NDrive::ETelematicsNotification notification, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server) {
    TString result;
    auto localization = server.GetLocalization();
    if (localization) {
        auto id = TStringBuilder() << "telematics" << '.' << notification << '.' << "message";
        auto resourceId = permissions.GetSetting<TString>(server.GetSettings(), id).GetOrElse(id);
        auto defaultValue = (locale == LegacyLocale) ? MakeMaybe(std::move(result)) : Nothing();
        result = localization->GetLocalString(locale, resourceId, defaultValue);
    }
    if (!result) {
        auto language = enum_cast<ELanguage>(locale);
        result = NDrive::GetUserErrorDescription(notification, language);
    }
    return result;
}

TString TTelematicsCommandTag::GetUserErrorTitle(TStringBuf message, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server) {
    return GetUserErrorTitle(NDrive::ParseError(message), locale, permissions, server);
}

TString TTelematicsCommandTag::GetUserErrorTitle(NDrive::ETelematicsNotification notification, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server) {
    TString result;
    auto localization = server.GetLocalization();
    if (localization) {
        auto id = TStringBuilder() << "telematics" << '.' << notification << '.' << "title";
        auto resourceId = permissions.GetSetting<TString>(server.GetSettings(), id).GetOrElse(id);
        result = localization->GetLocalString(locale, resourceId, result);
    }
    return result;
}

TExpected<bool> TTelematicsCommandTag::ValidateExternalCommand(const TString& carId, const TCommand& command, const NDrive::TExternalCommandResult& result, const TUserPermissions& permissions, const NDrive::IServer& server) {
    Y_UNUSED(carId);
    const auto& settings = server.GetSettings();
    auto encodedKey = permissions.GetSetting<TString>(settings, "telematics.ble.validation_key");
    if (!encodedKey) {
        return MakeUnexpected(yexception() << "no validation_key registered");
    }
    auto key = NPoly1305::TryParseKey(*encodedKey);
    if (!key) {
        return PassUnexpected(key);
    }
    auto nonceLifetime = permissions.GetSetting<TDuration>(settings, "telematics.ble.nonce_lifetime").GetOrElse(TDuration::Minutes(1));
    auto nonceValidTimestamp = Now() - nonceLifetime;
    for (auto&& signature : result.Signatures) {
        if (!MatchExternalCommandCode(signature.Request.Code, command.Code)) {
            return MakeUnexpected(yexception() << "command code mismatch: " << signature.Request.Code << ' ' << command.Code);
        }
        if (signature.Request.Nonce != signature.Response.Nonce) {
            return MakeUnexpected(yexception() << "nonce mismatch");
        }
        if (signature.Request.Id != signature.Response.Id) {
            return MakeUnexpected(yexception() << "id mismatch");
        }
        if (signature.Response.Result != NDrive::NVega::TCommandResponse::PROCESSED) {
            return MakeUnexpected(yexception() << "invalid result: " << signature.Response.Result);
        }
        if (!signature.Response.ValidateMac(*key)) {
            return MakeUnexpected(yexception() << "invalid mac");
        }
        auto compositeNonce = NDrive::Decode(signature.Request.Nonce);
        if (compositeNonce.Timestamp < nonceValidTimestamp) {
            return MakeUnexpected(yexception() << "nonce expired: " << compositeNonce.Timestamp.Seconds());
        }
        return true;
    }
    return false;
}

bool TTelematicsCommandTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* /*eContext*/) const {
    Y_UNUSED(permissions);
    auto tag = dynamic_cast<TTelematicsCommandTag*>(toTag.Get());
    if (tag) {
        return tag->Invoke(fromTag.GetObjectId(), server, session);
    } else {
        session.SetErrorInfo("telematics_command", "cannot evolve into tags other than TelematicsCommandTag", EDriveSessionResult::IncorrectRequest);
        return false;
    }
}

NDrive::TScheme TTelematicsDeferredCommandTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("code", "Command code").InitVariants<NDrive::NVega::ECommandCode>();
    result.Add<TFSDuration>("default_duration", "Command default duration");
    result.Add<TFSJson>("parameter_setting", "Command param setting");
    result.Add<TFSVariants>("tag_name_on_success", "Add tag after successful command execution").SetVariants(
        NContainer::Keys(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
    );
    return result;
}

NJson::TJsonValue TTelematicsDeferredCommandTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    result["code"] = ToString(Code);
    result["default_duration"] = NJson::ToJson(NJson::Nullable(DefaultDuration));
    result["tag_name_on_success"] = NJson::ToJson(TagNameOnSuccess);
    result["parameter_setting"] = ParameterSetting;
    return result;
}

bool TTelematicsDeferredCommandTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    if (!TTagDescription::DoDeserializeMetaFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["default_duration"], DefaultDuration) &&
        NJson::ParseField(value["tag_name_on_success"], TagNameOnSuccess) &&
        NJson::ParseField(value["code"], NJson::Stringify(Code), true) &&
        NJson::ParseField(value["parameter_setting"], ParameterSetting);
}

NDrive::TScheme TTelematicsDeferredCommandTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSDuration>("duration", "Duration of the command");
    return result;
}

bool TTelematicsDeferredCommandTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return
        NJson::ParseField(value["since"], Since) &&
        NJson::ParseField(value["until"], Until) &&
        NJson::ParseField(value["duration"], Duration);
}

void TTelematicsDeferredCommandTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    if (Since) {
        value["since"] = NJson::ToJson(Since);
    }
    if (Until != TInstant::Max()) {
        value["until"] = NJson::ToJson(Until);
    }
    if (Duration) {
        value["duration"] = NJson::ToJson(Duration);
    }
}

template <class T>
ui64 CalcHashImpl(ui16 id, const T& object) {
    return
        FnvHash<ui64>(&id, sizeof(id)) ^
        object.CalcHash();
}

template <class T>
ui64 CalcHashImpl(const TMap<ui16, T>& sensors) {
    TMaybe<ui64> result;
    for (auto&& [id, object] : sensors) {
        auto hash = CalcHashImpl(id, object);
        if (result) {
            result = *result ^ hash;
        } else {
            result = hash;
        }
    }
    return result.GetOrElse(0);
}

NJson::TJsonValue TTelematicsConfigurationTraits::TCustom::ToJson() const {
    NJson::TJsonValue result;

    result["model"] = Model;
    result["tags_filter"] = TagsFilter.ToString();
    result["sensors"] = NJson::ToJson(NJson::KeyValue(Sensors));
    result["apns"] = NJson::ToJson(NJson::KeyValue(APNs));
    result["server_settings"] = NJson::ToJson(NJson::KeyValue(Servers));
    result["can_sensors"] = NJson::ToJson(NJson::KeyValue(CanParameters));
    result["logic_scripts"] = NJson::ToJson(NJson::KeyValue(LogicScripts));
    result["logic_commands"] = NJson::ToJson(NJson::KeyValue(LogicCommands));
    result["logic_checks"] = NJson::ToJson(NJson::KeyValue(LogicChecks));
    result["logic_triggers"] = NJson::ToJson(NJson::KeyValue(LogicTriggers));
    result["logic_notifies"] = NJson::ToJson(NJson::KeyValue(LogicNotifies));

    if (Values) {
        result["values"] = NJson::ToJson(Values);
    }

    return result;
}

ui64 TTelematicsConfigurationTraits::CalcHash(TStringBuf imei) {
    return FnvHash<ui64>(imei);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TAPNParameter>& apn) {
    return CalcHashImpl(apn);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TSensorTranslation>& sensors) {
    return CalcHashImpl(sensors);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TServerSettingsParameter>& servers) {
    return CalcHashImpl(servers);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::THardPasswordParameter>& pins) {
    return CalcHashImpl(pins);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TUsePasswordParameter>& usePins) {
    return CalcHashImpl(usePins);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TWiredPasswordParameter>& pins) {
    return CalcHashImpl(pins);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TUseWiredPasswordParameter>& usePins) {
    return CalcHashImpl(usePins);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TCanParameter>& canParameters) {
    return CalcHashImpl(canParameters);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TLogicScript>& scripts) {
    return CalcHashImpl(scripts);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TLogicCommand>& commands) {
    return CalcHashImpl(commands);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TLogicCheck>& checks) {
    return CalcHashImpl(checks);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TLogicTrigger>& triggers) {
    return CalcHashImpl(triggers);
}

ui64 TTelematicsConfigurationTraits::CalcHash(const TMap<ui16, NDrive::NVega::TLogicNotify>& notifies) {
    return CalcHashImpl(notifies);
}

ui64 TTelematicsConfigurationTraits::TCustom::CalcHash() const {
    return
        TTelematicsConfigurationTraits::CalcHash(APNs) ^
        TTelematicsConfigurationTraits::CalcHash(Sensors) ^
        TTelematicsConfigurationTraits::CalcHash(Servers) ^
        TTelematicsConfigurationTraits::CalcHash(CanParameters) ^
        TTelematicsConfigurationTraits::CalcHash(LogicScripts) ^
        TTelematicsConfigurationTraits::CalcHash(LogicCommands) ^
        TTelematicsConfigurationTraits::CalcHash(LogicChecks) ^
        TTelematicsConfigurationTraits::CalcHash(LogicTriggers) ^
        TTelematicsConfigurationTraits::CalcHash(LogicNotifies);
}

TString TTelematicsConfigurationTraits::GeneratePin(const TString& objectId, TInstant timestamp) {
    auto days = timestamp.Days();
    auto hash = FnvHash<ui64>(objectId) ^ FnvHash<ui64>(&days, sizeof(days));
    auto copy = hash;

    constexpr size_t size = sizeof(hash) + 4;
    char buf[size];
    for (char& i : buf) {
        i = copy & 0xff;
        copy >>= 5;
    }
    auto result = Base64Encode(TStringBuf(buf, sizeof(buf) - 1));
    result.resize(std::min<size_t>(result.size(), 15));
    return result;
}

void TTelematicsConfigurationTraits::Merge(const TCustom& custom) {
    auto comparator = [&custom](const TCustom& c) {
        return custom == c;
    };
    auto it = std::find_if(Customs.begin(), Customs.end(), comparator);

    if (it != Customs.end()) {
        *it = custom;
    } else {
        Customs.push_back(custom);
    }
}

IEntityTagsManager::TOptionalTag TTelematicsConfigurationTag::Instance(const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    return GetTelematicsTag<TTelematicsConfigurationTag>(objectId, server, session);
}

TTelematicsConfigurationTag::TExpectedPinInfo TTelematicsConfigurationTag::GetPin(const TString& objectId, const NDrive::IServer* server, ui16 serverId, TInstant deadline) {
    return WrapUnexpected<yexception>(GetPinImpl, objectId, server, serverId, deadline);
}

TTelematicsConfigurationTag::TExpectedPinInfo TTelematicsConfigurationTag::GetPinByIMEI(const TString& imei, const NDrive::IServer* server, ui16 serverId, TInstant deadline) {
    return WrapUnexpected<yexception>(GetPinByIMEIImpl, imei, server, serverId, deadline);
}

namespace {
    NDrive::TScheme GetSensorTranslationScheme() {
        NDrive::TScheme result;
        result.Add<TFSNumeric>("on_change", "Transmit sensor value on change more than X");
        result.Add<TFSNumeric>("on_period", "Transmit sensor value every Y seconds");
        result.Add<TFSNumeric>("on_track", "Transmit sensor value with movements");
        return result;
    }
}

NDrive::TScheme TTelematicsConfigurationTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    using namespace NDrive::NVega;

    if (!server) {
        return {};
    }
    const TDriveAPI* api = server->GetDriveAPI();
    if (!api) {
        return {};
    }
    const TModelsDB* models = api->GetModelsData();
    if (!models) {
        return {};
    }

    NDrive::TScheme apnScheme;
    apnScheme.Add<TFSString>("APN", "APN");
    apnScheme.Add<TFSString>("user", "User");
    apnScheme.Add<TFSString>("password", "Password");

    NDrive::TScheme canSensorScheme;
    canSensorScheme.Add<TFSVariants>("can_channel", "Can channel").InitVariants<TCanParameter::ECanChannel>().SetRequired(true);
    canSensorScheme.Add<TFSVariants>("can_id_type", "Can ID type").InitVariants<TCanParameter::EIdentifierType>().SetRequired(true);
    canSensorScheme.Add<TFSVariants>("sensor_type", "Sesnor type").InitVariants<ECommonValueType>().SetRequired(true);
    canSensorScheme.Add<TFSString>("sensor_name", "Sensor name");
    canSensorScheme
        .Add<TFSNumeric>("sensor_id", "Sensor ID")
        .SetMin(CUSTOM_CAN_SENSOR_FIRST_ID)
        .SetMax(CUSTOM_CAN_SENSOR_LAST_ID)
        .SetDefault(CUSTOM_CAN_SENSOR_FIRST_ID)
        .SetRequired(true);

    {
        NDrive::TScheme settings;
        settings.Add<TFSNumeric>("can_id", "Can identifier (PGN)").SetRequired(true);
        settings.Add<TFSNumeric>("reset_value", "Value after reset").SetDefault(0);
        {
            NDrive::TScheme resetSettings;
            resetSettings.Add<TFSNumeric>("reset_seconds", "Reset seconds");
            resetSettings.Add<TFSBoolean>("reset_by_ignition_off", "Reset by ignitation off").SetDefault(false);
            resetSettings.Add<TFSBoolean>("fast_bool", "Fast changing boolean value").SetDefault(false);
            settings.Add<TFSStructure>("reset_settings", "Reset settings").SetStructure(std::move(resetSettings));
        }
        {
            NDrive::TScheme filter;
            {
                NDrive::TScheme mask;
                mask.Add<TFSNumeric>("mask", "Mask").SetDefault(0).SetRequired(true);
                mask.Add<TFSNumeric>("value", "Value").SetDefault(0).SetRequired(true);
                filter.Add<TFSStructure>("mask", "Mask").SetStructure(std::move(mask));
            }
            {
                NDrive::TScheme borders;
                borders.Add<TFSNumeric>("minimum", "Min").SetDefault(0).SetRequired(true);
                borders.Add<TFSNumeric>("maximum", "Max").SetDefault(Max<ui16>()).SetRequired(true);
                filter.Add<TFSStructure>("borders", "Borders").SetStructure(std::move(borders));
            }
            settings.Add<TFSStructure>("filter", "Filter settings").SetStructure(std::move(filter));
        }
        {
            NDrive::TScheme bits;
            {
                NDrive::TScheme destination;
                destination.Add<TFSNumeric>("position", "Position").SetDefault(0).SetRequired(true);
                bits.Add<TFSStructure>("destination", "Destination settings").SetStructure(std::move(destination));
            }
            {
                NDrive::TScheme length;
                length.Add<TFSNumeric>("length", "Length").SetDefault(0).SetRequired(true);
                length.Add<TFSBoolean>("is_signed", "Is signed").SetDefault(false);
                bits.Add<TFSStructure>("length", "Length settings").SetStructure(std::move(length));
            }
            settings.Add<TFSStructure>("bits", "Bits settings").SetStructure(std::move(bits));
        }
        {
            NDrive::TScheme positions;
            positions
                .Add<TFSVariants>("bit_order", "Bit order")
                .InitVariants<TCanParameter::TSettings::TPositions::EOrder>()
                .SetDefault(ToString(TCanParameter::TSettings::TPositions::O_BIG_ENDIAN));

            positions.Add<TFSBoolean>("invert", "Invert").SetDefault(false);
            positions.Add<TFSNumeric>("byte_index", "Byte index").SetDefault(0);
            positions.Add<TFSNumeric>("start_bit", "Start bit").SetDefault(0);
        }
        settings.Add<TFSNumeric>("scale", "Scale").SetDefault(0);
        settings.Add<TFSNumeric>("offset", "Offset").SetDefault(0);
        canSensorScheme.Add<TFSStructure>("settings", "Settings").SetStructure(std::move(settings));
    }

    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSArray>("apns").SetElement(KeyValueScheme(apnScheme));
    result.Add<TFSArray>("sensors").SetElement(KeyValueScheme(std::move(GetSensorTranslationScheme())));
    {
        NDrive::TScheme customs;
        customs.Add<TFSVariants>("model", "Car model restriction").SetVariants(models->GetModelCodes());
        customs.Add<TFSString>("tags_filter", "Tags filter");
        customs.Add<TFSArray>("apns", "APNs").SetElement(KeyValueScheme(apnScheme));
        customs.Add<TFSArray>("sensors", "Sensors").SetElement(KeyValueScheme(std::move(GetSensorTranslationScheme())));
        customs.Add<TFSArray>("can_sensors", "Can sensors").SetElement(KeyValueScheme(canSensorScheme));
        customs.Add<TFSArray>("logic_scripts", "Logic script").SetElement(KeyValueScheme(GetLogicScriptScheme()));
        customs.Add<TFSArray>("logic_commands", "Logic commands").SetElement(KeyValueScheme(GetLogicCommandScheme()));
        customs.Add<TFSArray>("logic_checks", "Logic checks").SetElement(KeyValueScheme(GetLogicCheckScheme()));
        customs.Add<TFSArray>("logic_triggers", "Logic triggers").SetElement(KeyValueScheme(GetLogicTriggerScheme()));
        customs.Add<TFSArray>("logic_notifies", "Logic notifies").SetElement(KeyValueScheme(GetLogicNotifyScheme()));

        {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("address", "Host:Port");
            scheme.Add<TFSNumeric>("period", "InteractionPeriod");
            scheme.Add<TFSVariants>("protocol", "Protocol").InitVariants<NDrive::NVega::TServerSettingsParameter::EProtocol>();
            customs.Add<TFSArray>("server_settings", "Servers").SetElement(KeyValueScheme(std::move(scheme)));
        }
        result.Add<TFSArray>("custom_sensors", "Customs").SetElement(std::move(customs));
    }

    {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("hostport", "Host:Port");
        scheme.Add<TFSString>("password", "Password");
        scheme.Add<TFSNumeric>("id", "Id");
        scheme.Add<TFSNumeric>("interaction_period", "InteractionPeriod");
        scheme.Add<TFSBoolean>("generate_password", "GeneratePassword");
        scheme.Add<TFSVariants>("protocol", "Protocol").InitVariants<NDrive::NVega::TServerSettingsParameter::EProtocol>();
        result.Add<TFSArray>("server_settings", "Servers").SetElement(std::move(scheme));
    }
    {
        NDrive::TScheme wired;
        wired.Add<TFSString>("password", "Password");
        wired.Add<TFSBoolean>("generate_password", "GeneratePassword");
        result.Add<TFSStructure>("wired_settings", "Wired").SetStructure(wired);
    }
    {
        NDrive::TScheme sensorMeta;
        {
            NDrive::TScheme sensorId;
            sensorId.Add<TFSNumeric>("id", "Id");
            sensorId.Add<TFSNumeric>("sub_id", "SubId");
            sensorMeta.Add<TFSStructure>("sensor_id", "SensorId").SetStructure(sensorId);
        }
        {
            NDrive::TScheme data;
            data.Add<TFSString>("display_name", "DisplayName");
            data.Add<TFSNumeric>("min_value", "MinValue");
            data.Add<TFSNumeric>("max_value", "MaxValue");
            data.Add<TFSArray>("critical_values", "CriticalValues").SetElement<TFSString>();
            data.Add<TFSString>("icon", "Icon");
            data.Add<TFSString>("dim", "Dim");
            data.Add<TFSString>("format_pattern", "FormatPattern");
            {
                NDrive::TScheme criticalCodes;
                criticalCodes.Add<TFSString>("code", "Code");
                criticalCodes.Add<TFSString>("decode", "Decode");
                data.Add<TFSArray>("decoding","Decoding").SetElement(std::move(criticalCodes));
            }
            sensorMeta.Add<TFSStructure>("data", "Data").SetStructure(data);
        }
        result.Add<TFSArray>("sensor_meta", "SensorMeta").SetElement(std::move(sensorMeta));
    }
    result.Add<TFSNumeric>("generate_pin_rollout_fraction", "GeneratePinRolloutFraction");
    return result;
}

NJson::TJsonValue TTelematicsConfigurationTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    result["generate_pin_rollout_fraction"] = GeneratePinRolloutFraction;

    NJson::InsertNonNull(result, "apns", NJson::KeyValue(GetAPNs()));
    NJson::InsertNonNull(result, "sensors", NJson::KeyValue(GetSensors()));
    NJson::InsertNonNull(result, "server_settings", ServerSettings);
    NJson::InsertNonNull(result, "wired_settings", WiredSettings);
    NJson::InsertNonNull(result, "custom_sensors", GetCustoms());
    NJson::InsertNonNull(result, "sensor_meta", NJson::KeyValue(GetSensorMeta(), "sensor_id", "data"));

    return result;
}

const TMap<NDrive::TSensorId, TTelematicsConfigurationTag::TSensorMeta>& TTelematicsConfigurationTag::TDescription::GetSensorMeta() const {
    return SensorMeta;
}

bool TTelematicsConfigurationTag::TDescription::IsSensorValueCritical(const NDrive::TSensorId& id, const TMaybe<NDrive::TSensor>& sensor) const {
    if (!SensorMeta.contains(id)) {
        return false;
    }

    TString stringValue;
    TMaybe<double> numericValue;
    const TSensorMeta& sensorMeta = SensorMeta.at(id);
    const TString nullCase = "null";

    if (sensor) {
        if (std::holds_alternative<ui64>(sensor->Value)) {
            numericValue = std::get<ui64>(sensor->Value);
            stringValue = ToString(std::get<ui64>(sensor->Value));
        } else if (std::holds_alternative<double>(sensor->Value)) {
            numericValue = std::get<double>(sensor->Value);
            stringValue = ToString(std::get<double>(sensor->Value));
        } else if (std::holds_alternative<TString>(sensor->Value)) {
            stringValue = std::get<TString>(sensor->Value);
        } else {
            stringValue = nullCase; // parsing error
        }
        if (numericValue) {
            if (sensorMeta.MinValue && *numericValue < *sensorMeta.MinValue) {
                return true;
            }
            if (sensorMeta.MaxValue && *numericValue > *sensorMeta.MaxValue) {
                return true;
            }
        }
    } else {
        stringValue = nullCase;
    }

    for (auto&& i : sensorMeta.CriticalValues) {
        if (numericValue) { // when "0" is critical and "30" non-critical
            if (stringValue == i) {
                return true;
            }
        } else {
            if (stringValue.Contains(i)) { // example: value = "P0043,P0037" and "P0037" is critical
                return true;
            }
            if (i == nullCase && stringValue.empty()) {
                return true;
            }
        }
    }

    return false;
}

bool TTelematicsConfigurationTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    bool parsed =
        NJson::ParseField(value["generate_pin_rollout_fraction"], GeneratePinRolloutFraction) &&
        NJson::ParseField(value["apns"], NJson::KeyValue(MutableAPNs())) &&
        NJson::ParseField(value["server_settings"], ServerSettings) &&
        NJson::ParseField(value["wired_settings"], WiredSettings) &&
        NJson::ParseField(value["sensors"], NJson::KeyValue(MutableSensors())) &&
        NJson::ParseField(value["custom_sensors"], MutableProtectedCustoms()) &&
        NJson::ParseField(value["sensor_meta"], NJson::KeyValue(SensorMeta, "sensor_id", "data"));
    if (!parsed) {
        return false;
    }
    for (auto&& i : ServerSettings) {
        ui16 id = i.Id;

        NDrive::NVega::TServerSettingsParameter server;
        server.HostPort.Set(i.HostPort);
        server.InteractionPeriod = i.InteractionPeriod;
        server.Protocol = i.Protocol;
        MutableServers()[id] = std::move(server);

        MutableProtectedPins()[id].Value.Set(i.Password);
        MutableProtectedUsePins()[id].Mode = i.Password ? NDrive::NVega::TUsePasswordParameter::ENABLE : NDrive::NVega::TUsePasswordParameter::DISABLE;
        if (i.GeneratePassword) {
            MutableProtectedGeneratePinIds().insert(id);
        }
    }
    if (WiredSettings) {
        ui16 id = 0;
        MutableProtectedWiredPin()[id].Value.Set(WiredSettings->Password);
        MutableProtectedUseWiredPin()[id].Mode = WiredSettings->Password ? NDrive::NVega::TUsePasswordParameter::ENABLE : NDrive::NVega::TUsePasswordParameter::DISABLE;
        if (WiredSettings->GeneratePassword) {
            MutableProtectedGenerateWiredPinIds().insert(id);
        }
    }
    return true;
}

namespace {
    template <class T>
    TMap<ui16, T> Patch(const TMap<ui16, T>& base, const TMap<ui16, T>& patch) {
        auto result = base;
        for (auto&&[id, object] : patch) {
            result[id] = object;
        }
        return result;
    }

    TVector<TTelematicsConfigurationTag::TCustom> Patch(
        const TVector<TTelematicsConfigurationTag::TCustom>& base,
        const TVector<TTelematicsConfigurationTag::TCustom>& patch
    ) {
        TVector<TTelematicsConfigurationTag::TCustom> result = base;

        for (auto&& item : patch) {
            auto comp = [&item](const TTelematicsConfigurationTag::TCustom& custom) {
                return item == custom;
            };

            auto iter = FindIf(result.begin(), result.end(), comp);

            if (iter != result.end()) {
                *iter = item;
            } else {
                result.push_back(item);
            }
        }

        return result;
    }

    auto GetTelematicsConfiguration(
        TStringBuf imei,
        TStringBuf model,
        const TTaggedObject& taggedObject,
        const TTelematicsConfigurationTag::TDescription& description,
        const TTelematicsConfigurationTag& tag
    ) {
        TTelematicsConfigurationTraits::TCustom customs;

        auto pins = Patch(description.GetPins(), tag.GetPins());
        auto usePins = Patch(description.GetUsePins(), tag.GetUsePins());
        auto useWiredPin = Patch(description.GetUseWiredPin(), tag.GetUseWiredPin());
        auto wiredPin = Patch(description.GetWiredPin(), tag.GetWiredPin());
        customs.Patch(model, taggedObject, description, tag);

        ui64 expected =
            TTelematicsConfigurationTag::CalcHash(imei) ^
            TTelematicsConfigurationTag::CalcHash(pins) ^
            TTelematicsConfigurationTag::CalcHash(usePins) ^
            TTelematicsConfigurationTag::CalcHash(useWiredPin) ^
            TTelematicsConfigurationTag::CalcHash(wiredPin) ^
            customs.CalcHash();
        return std::make_tuple(
            expected,
            std::move(pins),
            std::move(usePins),
            std::move(useWiredPin),
            std::move(wiredPin),
            std::move(customs)
        );
    }
}

NDrive::NVega::TSettings TTelematicsConfigurationTraits::TCustom::GetSettings() const {
    NDrive::NVega::TSettings settings;

    Apply(APNs, settings);
    Apply(Sensors, settings);
    Apply(Servers, settings);
    Apply(CanParameters, settings);

    Apply(LogicScripts, settings);
    Apply(LogicCommands, settings);
    Apply(LogicChecks, settings);
    Apply(LogicTriggers, settings);
    Apply(LogicNotifies, settings);

    return std::move(settings);
}

void TTelematicsConfigurationTraits::TCustom::Apply(const NDrive::NVega::TSetting& setting) {
    Apply(setting, APNs);
    Apply(setting, Sensors);
    Apply(setting, Servers);
    Apply(setting, CanParameters);
    Apply(setting, LogicScripts);
    Apply(setting, LogicCommands);
    Apply(setting, LogicChecks);
    Apply(setting, LogicTriggers);
    Apply(setting, LogicNotifies);
}

void TTelematicsConfigurationTraits::TCustom::Apply(const NDrive::TTelematicsClient::TGetParameterResponse& response) {
    Apply(response.Sensor, Values);
}

void TTelematicsConfigurationTraits::TCustom::Patch(
    TStringBuf model,
    const TTaggedObject taggedObject,
    const TTelematicsConfigurationTraits& base,
    const TTelematicsConfigurationTraits& patch
) {
    APNs = ::Patch(base.GetAPNs(), patch.GetAPNs());
    Sensors = ::Patch(base.GetSensors(), patch.GetSensors());
    Servers = ::Patch(base.GetServers(), patch.GetServers());

    auto customs = ::Patch(base.GetCustoms(), patch.GetCustoms());
    for (auto&& custom : customs) {
        if (custom.IsMatching(model, taggedObject)) {
            APNs = ::Patch(APNs, custom.APNs);
            Sensors = ::Patch(Sensors, custom.Sensors);
            Servers = ::Patch(Servers, custom.Servers);
            CanParameters = ::Patch(CanParameters, custom.CanParameters);
            LogicScripts = ::Patch(LogicScripts, custom.LogicScripts);
            LogicCommands = ::Patch(LogicCommands, custom.LogicCommands);
            LogicChecks = ::Patch(LogicChecks, custom.LogicChecks);
            LogicTriggers = ::Patch(LogicTriggers, custom.LogicTriggers);
            LogicNotifies = ::Patch(LogicNotifies, custom.LogicNotifies);
        }
    }
}

namespace {
    template<class T>
    void AppendConfigurationId(TSet<NDrive::TSensorId>& result, const TSet<ui16>& filter) {
        if (!TTelematicsConfigurationTraits::TCustom::CanApply(T::GetId(), filter)) {
            return;
        }

        for (ui64 i = 0; i < T::GetCount(); ++i) {
            result.emplace(T::GetId(), i);
        }
    }
}

void TTelematicsConfigurationTraits::TCustom::PostLoad() {
    TVector<NDrive::NVega::TCanParameter> parameters = MakeVector(NContainer::Values(CanParameters));
    for (auto&& sensor : Values) {
        sensor = NDrive::NVega::TCanParameter::Parse(std::move(sensor), parameters);
    }
}

ui64 TTelematicsConfigurationTag::GetExpectedHash(TStringBuf imei, TStringBuf model, const TTaggedObject& taggedObject, const NDrive::IServer* server) const {
    auto api = server ? server->GetDriveAPI() : nullptr;
    auto dscr = api ? api->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()) : nullptr;
    auto description = std::dynamic_pointer_cast<const TTelematicsConfigurationTag::TDescription>(dscr);
    return GetExpectedHash(imei, model, taggedObject, description.Get());
}

ui64 TTelematicsConfigurationTag::GetExpectedHash(TStringBuf imei, TStringBuf model, const TTaggedObject& taggedObject, const TDescription* d) const {
    auto description = d ? d : &Default<TDescription>();
    return std::get<0>(GetTelematicsConfiguration(imei, model, taggedObject, *description, *this));
}

bool TTelematicsConfigurationTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (ApplyOnAdd) {
        ApplyOnAdd = false;
        ConfigurationHash = 0;
        if (!server) {
            session.SetErrorInfo("TelematicsConfigurationTag::OnBeforeAdd", "null server", EDriveSessionResult::InternalError);
            return false;
        }
        const auto& settings = server->GetSettings();
        auto ignoreCommandFailures = settings.GetValue<bool>("tag.telematics_configuration.ingore_command_failures").GetOrElse(false);

        auto now = Now();
        auto api = server->GetDriveAPI();
        auto carsData = api ? api->GetCarsData() : nullptr;
        auto fetchResult = Checked(carsData)->FetchInfo(objectId, session);
        if (!fetchResult) {
            return false;
        }
        auto object = fetchResult.GetResultPtr(objectId);
        if (!object) {
            session.SetErrorInfo("TelematicsConfigurationTag::OnBeforeAdd", TStringBuilder() << "cannot get object info for " << objectId, EDriveSessionResult::InternalError);
            return false;
        }
        auto imei = object->GetIMEI();
        if (!imei) {
            session.SetErrorInfo("TelematicsConfigurationTag::OnBeforeAdd", TStringBuilder() << "null imei for " << objectId, EDriveSessionResult::InternalError);
            return false;
        }

        auto dscr = Checked(api)->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
        auto description = std::dynamic_pointer_cast<const TTelematicsConfigurationTag::TDescription>(dscr);
        if (!description) {
            description = MakeAtomicShared<TTelematicsConfigurationTag::TDescription>();
        }

        auto deadline = GetDeadline();
        if (deadline < now) {
            auto idBucket = FnvHash<ui32>(objectId) % 100;
            auto pin = PinOverride ? PinOverride : TTelematicsConfigurationTag::GeneratePin(objectId, now);
            const auto generatePinRolloutFraction = description->GetGeneratePinRolloutFraction();
            const auto& generatePinIds = description->GetGeneratePinIds();
            const auto& generateWiredPinIds = description->GetGenerateWiredPinIds();
            for (auto&& i : generatePinIds) {
                if (static_cast<ui32>(generatePinRolloutFraction * 100) <= idBucket) {
                    break;
                }
                SetPin(i, pin);
            }
            for (auto&& i : generateWiredPinIds) {
                if (static_cast<ui32>(generatePinRolloutFraction * 100) <= idBucket) {
                    break;
                }
                SetWiredPin(i, pin);
            }
            deadline = now + TDuration::Days(3) + TDuration::Days(4) * RandomNumber<float>();
        }

        auto currentModel = object->GetModel();
        auto taggedObject = server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);

        if (!taggedObject) {
            ERROR_LOG << GetName() << ": cannot get tagged object for " << objectId << Endl;
            return false;
        }

        if (!currentModel) {
            NOTICE_LOG << GetName() << ": empty model for " << objectId << Endl;
        }

        auto&& [
            expected,
            pins,
            usePins,
            useWiredPin,
            wiredPin,
            customs
        ] = GetTelematicsConfiguration(imei, currentModel, *taggedObject, *description, *this);

        const NDrive::TTelematicsClient& client = server->GetTelematicsClient();
        NDrive::NVega::TSettings telematicSettings = customs.GetSettings();
        auto apply = [&](auto&& values) {
            for (auto&& [subid, object] : values) {
                NDrive::NVega::TSetting setting;
                setting.Id = object.GetId();
                setting.SubId = subid;

                TBuffer payload = object.Dump();
                setting.Payload.reserve(payload.Size());
                setting.Payload.insert(setting.Payload.end(), payload.Begin(), payload.End());

                telematicSettings.push_back(std::move(setting));
            }
        };
        apply(pins);
        apply(usePins);
        apply(wiredPin);
        apply(useWiredPin);

        auto data = telematicSettings.Dump();
        auto handler = client.Upload(imei, "SETTINGS", std::move(data));

        auto commit = session.Committed();
        auto upload = handler.GetFuture();

        auto eventLogState = NDrive::TEventLog::CaptureState();
        auto finalize = [
            deadline,
            eventLogState = std::move(eventLogState),
            expected = expected,
            handler = std::move(handler),
            ignoreCommandFailures,
            name = GetName(),
            objectId,
            userId,
            server
        ](const NThreading::TFuture<void>& w) {
            NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
            try {
                w.GetValue();

                if (ignoreCommandFailures) {
                    auto status = handler.GetStatus();
                    Y_ENSURE(
                        status == NDrive::TTelematicsClient::EStatus::Success || status == NDrive::TTelematicsClient::EStatus::Failure,
                        status << ' at ' << handler.GetId()
                    );
                } else {
                    handler.WaitAndEnsureSuccess();
                }

                const auto& deviceTagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
                auto session = deviceTagsManager.BuildSession();
                auto expectedTags = deviceTagsManager.RestoreEntityTags(objectId, { name }, session);
                Y_ENSURE(expectedTags, "cannot restore tag " << name << ": " << session.GetStringReport());
                Y_ENSURE(expectedTags->size() == 1, "cannot restore tag " << name << ": tags count is " << expectedTags->size());
                auto& dbTag = expectedTags->at(0);
                auto tag = dbTag.MutableTagAs<TTelematicsConfigurationTag>();
                Y_ENSURE(tag, "cannot cat tag " << dbTag.GetTagId() << " as TelematicsConfigurationTag");
                tag->SetConfigurationHash(expected, deadline);
                Y_ENSURE(
                    deviceTagsManager.UpdateTagData(dbTag, userId, session),
                    "cannot update tag " << dbTag.GetTagId() << " data: " << session.GetStringReport()
                );
                Y_ENSURE(session.Commit(), "cannot commit transaction: " << session.GetStringReport());
                NDrive::TEventLog::Log("FinalizeTelematicsConfiguration", NJson::TMapBuilder
                    ("handlers", NJson::ToJson(NContainer::Scalar(handler)))
                    ("object_id", objectId)
                );
            } catch (const std::exception& e) {
                NDrive::TEventLog::Log("FinalizeTelematicsConfigurationError", NJson::TMapBuilder
                    ("deadline", NJson::ToJson(deadline))
                    ("expected", expected)
                    ("handlers", NJson::ToJson(NContainer::Scalar(handler)))
                    ("name", name)
                    ("object_id", objectId)
                    ("exception", FormatExc(e))
                );
            }
        };
        NThreading::WaitAll(upload, commit).Subscribe(std::move(finalize));
    }
    return true;
}

template <class K, class T>
bool TTelematicsConfigurationTag::TryMapFromJson(const NJson::TJsonValue& value, TMap<K, T>& result, TMessagesCollector* errors) {
    if (value.IsDefined()) {
        if (!value.IsMap()) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "sensors " + value.GetStringRobust() + " is not an map");
            }
            return false;
        }
        for (auto&&[sid, sobject] : value.GetMapSafe()) {
            K id;
            T object;
            if (!TryFromString(sid, id)) {
                if (errors) {
                    errors->AddMessage(__LOCATION__, "id " + sid + " is not ui16");
                }
                return false;
            }
            if (!NJson::TryFromJson(sobject, object)) {
                if (errors) {
                    errors->AddMessage(__LOCATION__, "cannot parse setting " + sobject.GetStringRobust());
                }
                return false;
            }
            result.emplace(id, std::move(object));
        }
        return true;
    } else {
        return true;
    }
}

namespace {
    TString GetMaxAuxFuelVolumeName(size_t i) {
        TString name = "max_aux_fuel_volume";
        if (i) {
            name.append('_');
            name.append(ToString(i));
        }
        return name;
    }
}

NDrive::TScheme TTelematicsConfigurationTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("apply_on_add", "Apply during OnBeforeAdd").SetDefault(ApplyOnAdd);
    result.Add<TFSNumeric>("aux_fuel_level_transmission_period", "Period of aux fuel level transmission");
    for (size_t i = 0; i < MaxAuxFuelVolume.size(); ++i) {
        result.Add<TFSNumeric>(GetMaxAuxFuelVolumeName(i), "Volume of aux fuel tank #" + ToString(i));
    }
    return result;
}

bool TTelematicsConfigurationTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!NJson::ParseField(value["apply_on_add"], ApplyOnAdd)) {
        return false;
    }
    if (!NJson::ParseField(value["pin_override"], PinOverride)) {
        return false;
    }
    if (value.Has("sensors_hash")) {
        const auto& sensorsHash = value["sensors_hash"];
        if (!sensorsHash.IsUInteger()) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "sensors_hash " + sensorsHash.GetStringRobust() + " is not an integer");
            }
            return false;
        }
        ConfigurationHash = sensorsHash.GetUIntegerRobust();
    }
    if (value.Has("deadline")) {
        const auto& deadline = value["deadline"];
        if (!deadline.IsUInteger()) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "deadline " + deadline.GetStringRobust() + " is not an integer");
            }
            return false;
        }
        Deadline = TInstant::MicroSeconds(deadline.GetUInteger());
    }
    if (value.Has("aux_fuel_level_transmission_period")) {
        const auto& auxFuelLevelTransmissionPeriod = value["aux_fuel_level_transmission_period"];
        if (!auxFuelLevelTransmissionPeriod.IsUInteger()) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "aux_fuel_level_transmission_period" + auxFuelLevelTransmissionPeriod.GetStringRobust() + " is not an unsigned integer");
            }
            return false;
        }
        AuxFuelLevelTransmissionPeriod = auxFuelLevelTransmissionPeriod.GetUInteger();
    }
    for (size_t i = 0; i < MaxAuxFuelVolume.size(); ++i) {
        auto name = GetMaxAuxFuelVolumeName(i);
        if (!value.Has(name)) {
            continue;
        }
        const auto& maxAuxFuelVolume = value[name];
        if (!maxAuxFuelVolume.IsDouble()) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "max_aux_fuel_volume" + maxAuxFuelVolume.GetStringRobust() + " is not a double");
            }
            return false;
        }
        MaxAuxFuelVolume[i] = maxAuxFuelVolume.GetDouble();
        if (MaxAuxFuelVolume[i] <= 0) {
            if (errors) {
                errors->AddMessage(__LOCATION__, "max_aux_fuel_volume" + maxAuxFuelVolume.GetStringRobust() + " is negative");
            }
            return false;
        }
        NDrive::NVega::TSensorCalibration::TTable table;
        table.push_back({ 0, 0, 0 });
        table.push_back({ 4095, MaxAuxFuelVolume[i], 0 });
        NDrive::NVega::TSensorCalibration calibator;
        calibator.SetTable(std::move(table));

        auto id = NDrive::NVega::AuxFuelLevel<1>() + i;
        MutableProtectedCalibrators()[id] = std::move(calibator);
        if (AuxFuelLevelTransmissionPeriod > 0) {
            NDrive::NVega::TSensorTranslation translation;
            translation.OnPeriod = AuxFuelLevelTransmissionPeriod;
            MutableSensors()[id] = translation;
        }
    }
    if (!TryMapFromJson(value["apns"], MutableAPNs(), errors)) {
        return false;
    }
    if (!TryMapFromJson(value["sensors"], MutableSensors(), errors)) {
        return false;
    }
    const auto& customs= value["custom_sensors"];
    if (customs.IsDefined() && !NJson::TryFromJson(customs, MutableProtectedCustoms())) {
        if (errors) {
            errors->AddMessage(__LOCATION__, "cannot deserialize custom sensor from " + customs.GetStringRobust());
        }
        return false;
    }
    if (!TryMapFromJson(value["servers"], MutableServers(), errors)) {
        return false;
    }
    if (!TryMapFromJson(value["pins"], MutableProtectedPins(), errors)) {
        return false;
    }
    if (!TryMapFromJson(value["wired_pin"], MutableProtectedWiredPin(), errors)) {
        return false;
    }
    for (auto&&[id, pin] : GetPins()) {
        const auto& value = pin.Value.Get();
        const bool use = !value.empty();

        MutableProtectedUsePins()[id].Mode = use ? NDrive::NVega::TUsePasswordParameter::ENABLE : NDrive::NVega::TUsePasswordParameter::DISABLE;
    }
    for (auto&&[id, pin] : GetWiredPin()) {
        const auto& value = pin.Value.Get();
        const bool use = !value.empty();

        MutableProtectedUseWiredPin()[id].Mode = use ? NDrive::NVega::TUseWiredPasswordParameter::ENABLE : NDrive::NVega::TUseWiredPasswordParameter::DISABLE;
    }
    if (!TryMapFromJson(value["calibrators"], MutableProtectedCalibrators(), errors)) {
        return false;
    }
    return TBase::DoSpecialDataFromJson(value, errors);
}

void TTelematicsConfigurationTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    value.InsertValue("apply_on_add", ApplyOnAdd);
    value.InsertValue("sensors_hash", ConfigurationHash);
    value.InsertValue("deadline", Deadline.MicroSeconds());
    if (AuxFuelLevelTransmissionPeriod) {
        value.InsertValue("aux_fuel_level_transmission_period", AuxFuelLevelTransmissionPeriod);
    }
    if (PinOverride) {
        value.InsertValue("pin_override", PinOverride);
    }
    for (size_t i = 0; i < MaxAuxFuelVolume.size(); ++i) {
        auto v = MaxAuxFuelVolume[i];
        if (v > 0) {
            value.InsertValue(GetMaxAuxFuelVolumeName(i), v);
        }
    }

    auto& apns = value.InsertValue("apns", NJson::JSON_MAP);
    for (auto&&[id, apn] : GetAPNs()) {
        apns.InsertValue(ToString(id), NJson::ToJson(apn));
    }
    auto& sensors = value.InsertValue("sensors", NJson::JSON_MAP);
    for (auto&&[id, setting] : GetSensors()) {
        sensors.InsertValue(ToString(id), NJson::ToJson(setting));
    }
    auto& customSensors = value.InsertValue("custom_sensors", NJson::JSON_ARRAY);
    for (auto&& setting : GetCustoms()) {
        customSensors.AppendValue(NJson::ToJson(setting));
    }
    auto& servers = value.InsertValue("servers", NJson::JSON_MAP);
    for (auto&&[id, server] : GetServers()) {
        servers.InsertValue(ToString(id), NJson::ToJson(server));
    }
    auto& pins = value.InsertValue("pins", NJson::JSON_MAP);
    for (auto&&[id, pin] : GetPins()) {
        pins.InsertValue(ToString(id), NJson::ToJson(pin));
    }
    auto& wiredPins = value.InsertValue("wired_pin", NJson::JSON_MAP);
    for (auto&&[id, pin] : GetWiredPin()) {
        wiredPins.InsertValue(ToString(id), NJson::ToJson(pin));
    }
    auto& calibrators = value.InsertValue("calibrators", NJson::JSON_MAP);
    for (auto&&[id, calibrator] : GetCalibrators()) {
        calibrators.InsertValue(ToString(id), NJson::ToJson(calibrator));
    }
}

NDrive::TTelematicsClient::THandler TTelematicsConfigurationTag::GetActualConfiguration(
    const TString& objectId,
    const NDrive::IServer* server,
    TDuration timeout/* = TDuration::Minutes(1)*/
) {
    if (!server) {
        return {};
    }

    auto api = server->GetDriveAPI();
    if (!api) {
        return {};
    }

    auto object = api->GetTagsManager().GetDeviceTags().GetObject(objectId);
    if (!object) {
        return {};
    }

    const auto deviceInfos = Yensured(api->GetCarsData())->FetchInfo({ objectId }, TInstant::Zero());
    auto deviceInfo = deviceInfos.GetResultPtr(objectId);

    auto imei = deviceInfo->GetIMEI();
    if (!imei) {
        return {};
    }

    const NDrive::TTelematicsClient& client = server->GetTelematicsClient();

    auto taggedObject = server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);
    auto tag = taggedObject->GetFirstTagByClass<TTelematicsConfigurationTag>();
    if (!tag) {
        tag = TDBTag();
    }

    auto configurationTag = tag.GetTagAs<TTelematicsConfigurationTag>();
    if (!configurationTag) {
        ERROR_LOG << Type() << ": cannot cast telematics configuration tag for " << objectId << Endl;
        return {};
    }

    auto dscr = Checked(api)->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
    auto description = std::dynamic_pointer_cast<const TTelematicsConfigurationTag::TDescription>(dscr);
    if (!description) {
        description = MakeAtomicShared<TTelematicsConfigurationTag::TDescription>();
    }

    if (!taggedObject) {
        ERROR_LOG << Type() << ": cannot get tagged object for " << objectId << Endl;
        return {};
    }

    return client.Download(imei, "BASE", timeout);
}

NDrive::TTelematicsClient::THandlers TTelematicsConfigurationTag::GetActualValues(
    const TString& objectId,
    const TSet<NDrive::TSensorId>& ids,
    const NDrive::IServer* server,
    TDuration timeout/* = TDuration::Minutes(1)*/
) {
    if (!server) {
        return {};
    }

    auto api = server->GetDriveAPI();
    if (!api) {
        return {};
    }

    const auto deviceInfos = Yensured(api->GetCarsData())->FetchInfo({ objectId }, TInstant::Zero());
    auto deviceInfo = deviceInfos.GetResultPtr(objectId);

    auto imei = deviceInfo->GetIMEI();
    if (!imei) {
        return {};
    }

    const NDrive::TTelematicsClient& client = server->GetTelematicsClient();
    NDrive::TTelematicsClient::THandlers handlers;

    auto apply = [&](NDrive::TSensorId id) {
        auto handler = client.Command(
            imei,
            NDrive::NVega::TCommand::GetParameter(id),
            timeout
        );
        handlers.push_back(std::move(handler));
    };

    for (const auto& id : ids) {
        apply(id);
    }

    return std::move(handlers);
}

TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo> TTelematicsFirmwareTag::TDescription::GetFirmwareInfos() const {
    return Firmwares;
}

NDrive::TScheme TTelematicsFirmwareTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    if (!server) {
        return {};
    }
    const TDriveAPI* api = server->GetDriveAPI();
    if (!api) {
        return {};
    }
    const TModelsDB* models = api->GetModelsData();
    if (!models) {
        return {};
    }

    NDrive::TScheme result;
    NDrive::TScheme& firmware = result.Add<TFSArray>("firmware").SetElement<NDrive::TScheme>();
    firmware.Add<TFSVariants>("model", "Device model").SetVariants(models->GetModelCodes()).SetRequired(true);
    firmware.Add<TFSVariants>("type", "Telematics type").InitVariants<NDrive::NVega::TFirmwareInfo::EType>().SetRequired(true);
    firmware.Add<TFSString>("tags_filter", "Tags filter");
    firmware.Add<TFSVariants>("full_name", "Firmware full name").SetVariants(TTelematicsFirmwareTag::List(*server)).SetRequired(true);
    firmware.Add<TFSNumeric>("percent", "Percent of all cars");
    firmware.Add<TFSBoolean>("ignore_firmware_model", "Ignore firmware model").SetDefault(false);
    result.Add<TFSBoolean>("ignore_car_status", "Ignore car status").SetDefault(false);
    result.Add<TFSStructure>("update_time_restrictions", "Schedule to update").SetStructure(TTimeRestrictionsPool<TTimeRestriction>::GetScheme());
    return result;
}

NJson::TJsonValue TTelematicsFirmwareTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result;
    result["firmware"] = NJson::ToJson(Firmwares);
    result["ignore_car_status"] = NJson::ToJson(IgnoreCarStatus);
    result["update_time_restrictions"] = NJson::ToJson(UpdateTimeRestrictions);
    return result;
}

bool TTelematicsFirmwareTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    bool result =
        NJson::ParseField(value["firmware"], Firmwares) &&
        NJson::ParseField(value["ignore_car_status"], IgnoreCarStatus) &&
        NJson::ParseField(value["update_time_restrictions"], UpdateTimeRestrictions);
    if (!result) {
        return false;
    }
    std::sort(Firmwares.begin(), Firmwares.end());
    return true;
}

namespace {
    TString GetFirmwareDataPath(const TString& name) {
        return name + "/data";
    }
    TString GetFirmwareInfoPath(const TString& name) {
        return name + "/info";
    }
    TBuffer GetFirmwareData(const TString& name, const NDrive::IServer& server) noexcept(false) {
        auto storage = server.GetVersionedStorage(TTelematicsFirmwareTag::Type());
        Y_ENSURE(storage, "storage " << TTelematicsFirmwareTag::Type() << " is missing");
        auto path = GetFirmwareDataPath(name);
        Y_ENSURE(storage->ExistsNode(path), "node " << path << " does not exist");
        TString data;
        Y_ENSURE(storage->GetValue(path, data), "cannot get data from " << path);
        Y_ENSURE(!data.empty(), "empty data from " << path);
        return { data.data(), data.size() };
    }
    NDrive::NVega::TFirmwareInfo GetFirmwareInfo(const TString& name, const NDrive::IServer& server) noexcept(false) {
        auto storage = server.GetVersionedStorage(TTelematicsFirmwareTag::Type());
        Y_ENSURE(storage, "storage " << TTelematicsFirmwareTag::Type() << " is missing");
        auto path = GetFirmwareInfoPath(name);
        Y_ENSURE(storage->ExistsNode(path), "node " << path << " does not exist");
        TString data;
        Y_ENSURE(storage->GetValue(path, data), "cannot get data from " << path);
        auto description = NJson::ReadJsonFastTree(data);
        auto info = NJson::FromJson<NDrive::NVega::TFirmwareInfo>(description);
        return info;
    }
    TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo> GetFirmwareInfoFromDescription(const NDrive::IServer& server) noexcept(false) {
        auto tagDescription = Yensured(server.GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(TTelematicsFirmwareTag::Type());
        auto telematicsFirmwareTagDescription = dynamic_cast<const TTelematicsFirmwareTag::TDescription*>(tagDescription.Get());
        Y_ENSURE(telematicsFirmwareTagDescription, "description for " << TTelematicsFirmwareTag::Type() << " is not found");

        return telematicsFirmwareTagDescription->GetFirmwareInfos();
    }
    NDrive::NVega::TFirmwareInfo GetCurrentFirmwareInfo(const TConstDBTag& tag, const NDrive::IServer& server) noexcept(false) {
        const TDriveAPI* api = Yensured(server.GetDriveAPI());
        const NDrive::ISensorApi* sensors = Yensured(server.GetSensorApi());

        const TString& id = tag.GetObjectId();
        const TString& imei = api->GetIMEI(id);
        Y_ENSURE(imei, "IMEI for " << id << " is not found");
        const auto fw = sensors->GetSensor(imei, VEGA_MCU_FIRMWARE_VERSION).ExtractValue(TDuration::MilliSeconds(100));
        Y_ENSURE(fw, "VEGA_MCU_FIRMWARE_VERSION for " << id << " is not found");
        NDrive::NVega::TFirmwareInfo result = NDrive::NVega::ParseFirmwareInfo(fw->ConvertTo<TString>());

        const auto deviceInfos = Yensured(api->GetCarsData())->FetchInfo({ id }, TInstant::Zero());
        const TDriveCarInfo* deviceInfo = deviceInfos.GetResultPtr(id);
        Y_ENSURE(deviceInfo, "device " << id << " is not found");
        result.Model = deviceInfo->GetModel();

        return result;
    }
    NDrive::NVega::TFirmwareInfo GetTaggedFirmwareInfo(const TConstDBTag& tag, const NDrive::IServer& server) noexcept(false) {
        Y_ENSURE(tag);
        auto telematicsFirmware = tag.GetTagAs<TTelematicsFirmwareTag>();
        Y_ENSURE(telematicsFirmware, "cannot cast " << tag.GetTagId() << " as TelematicsFirmwareTag");
        if (telematicsFirmware->HasFirmware()) {
            return telematicsFirmware->GetFirmware();
        }
        if (telematicsFirmware->GetName()) {
            return GetFirmwareInfo(telematicsFirmware->GetName(), server);
        }
        return GetCurrentFirmwareInfo(tag, server);
    }
    NDrive::NVega::TFirmwareInfo GetOnlyTaggedFirmwareInfo(const TConstDBTag& tag, const NDrive::IServer& server) noexcept(false) {
        Y_ENSURE(tag);
        auto telematicsFirmware = tag.GetTagAs<TTelematicsFirmwareTag>();
        Y_ENSURE(telematicsFirmware, "cannot cast " << tag.GetTagId() << " as TelematicsFirmwareTag");
        if (telematicsFirmware->HasFirmware()) {
            return telematicsFirmware->GetFirmware();
        }
        if (telematicsFirmware->GetName()) {
            return GetFirmwareInfo(telematicsFirmware->GetName(), server);
        }
        return { };
    }
    std::tuple<TString, bool, TMaybe<TTimeRestrictionsPool<TTimeRestriction>>, TInstant> GetFirmwareUpdateInfo(const TConstDBTag& tag, const NDrive::IServer& server) noexcept(false) {
        Y_ENSURE(tag);
        auto telematicsFirmware = tag.GetTagAs<TTelematicsFirmwareTag>();
        Y_ENSURE(telematicsFirmware, "cannot cast " << tag.GetTagId() << " as TelematicsFirmwareTag");

        auto description = Yensured(server.GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(TTelematicsFirmwareTag::Type());
        auto telematicsFirmwareTagDescription = dynamic_cast<const TTelematicsFirmwareTag::TDescription*>(description.Get());
        Y_ENSURE(telematicsFirmwareTagDescription, "description for " << TTelematicsFirmwareTag::Type() << " is not found");

        TString name = telematicsFirmware->GetName();
        bool ignoreCarStatus = telematicsFirmware->GetIgnoreCarStatus();
        TMaybe<TTimeRestrictionsPool<TTimeRestriction>> updateTimeRestrictions = telematicsFirmware->OptionalUpdateTimeRestrictions();
        TInstant lastUpdate = telematicsFirmware->GetLastUpdate();

        if (!name) {
            ignoreCarStatus = telematicsFirmwareTagDescription->GetIgnoreCarStatus();
            updateTimeRestrictions = telematicsFirmwareTagDescription->OptionalUpdateTimeRestrictions();
        }

        return std::make_tuple(std::move(name), ignoreCarStatus, std::move(updateTimeRestrictions), lastUpdate);
    }
}

bool TTelematicsFirmwareTag::Add(const NDrive::NVega::TFirmwareInfo& info, TBuffer&& data, const NDrive::IServer& server) {
    auto storage = server.GetVersionedStorage(Type());
    if (!storage) {
        return false;
    }
    const TString& name = info.GetName();
    if (!name) {
        return false;
    }

    TString serializedInfo = NJson::ToJson(info).GetStringRobust();
    TString serializedData;
    data.AsString(serializedData);

    return
        storage->SetValue(GetFirmwareInfoPath(name), serializedInfo) &&
        storage->SetValue(GetFirmwareDataPath(name), serializedData);
}

bool TTelematicsFirmwareTag::Remove(const TString& name, const NDrive::IServer& server) {
    auto storage = server.GetVersionedStorage(Type());
    if (!storage) {
        return false;
    }
    return
        storage->RemoveNode(GetFirmwareInfoPath(name)) &&
        storage->RemoveNode(GetFirmwareDataPath(name));
}

TExpected<NDrive::NVega::TFirmwareInfo, yexception> TTelematicsFirmwareTag::GetCurrentInfo(const TConstDBTag& tag, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetCurrentFirmwareInfo, tag, server);
}

TExpected<NDrive::NVega::TFirmwareInfo, yexception> TTelematicsFirmwareTag::GetTaggedInfo(const TConstDBTag& tag, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetTaggedFirmwareInfo, tag, server);
}

TExpected<NDrive::NVega::TFirmwareInfo, yexception> TTelematicsFirmwareTag::GetOnlyTaggedInfo(const TConstDBTag& tag, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetOnlyTaggedFirmwareInfo, tag, server);
}

TExpected<NDrive::NVega::TFirmwareInfo, yexception> TTelematicsFirmwareTag::GetInfo(const TString& name, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetFirmwareInfo, name, server);
}

TExpected<TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo>, yexception> TTelematicsFirmwareTag::GetInfoFromDescription(const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetFirmwareInfoFromDescription, server);
}

TExpected<TBuffer, yexception> TTelematicsFirmwareTag::GetData(const TString& name, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetFirmwareData, name, server);
}

TExpected<std::tuple<TString, bool, TMaybe<TTimeRestrictionsPool<TTimeRestriction>>, TInstant>, yexception> TTelematicsFirmwareTag::GetUpdateInfo(const TConstDBTag& tag, const NDrive::IServer& server) {
    return WrapUnexpected<yexception>(GetFirmwareUpdateInfo, tag, server);
}

TSet<TString> TTelematicsFirmwareTag::List(const NDrive::IServer& server) {
    auto storage = server.GetVersionedStorage(Type());
    if (!storage) {
        return {};
    }
    TVector<TString> nodes;
    if (!storage->GetNodes({}, nodes, /*withDirs=*/true)) {
        ERROR_LOG << "cannot list nodes in " << Type() << Endl;
        return {};
    }
    return MakeSet(nodes);
}

NDrive::TScheme TTelematicsFirmwareTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    if (server) {
        result.Add<TFSVariants>("name", "Firmware to apply").SetVariants(List(*server));
        result.Add<TFSBoolean>("ignore_car_status", "Ignore car status").SetDefault(false);
        result.Add<TFSStructure>("update_time_restrictions", "Schedule to update").SetStructure(TTimeRestrictionsPool<TTimeRestriction>::GetScheme());
        result.Add<TFSNumeric>("last_update", "Timestamp of last firmware update").SetDefault(0);
    }
    return result;
}

bool TTelematicsFirmwareTag::OnAfterAdd(const TDBTag& self, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (server && Name) {
        if (!List(*server).contains(Name)) {
            session.SetErrorInfo("TelematicsFirmwareTag::OnAfterAdd", Name + " is not registered", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        auto info = GetInfo(Name, *server);
        if (!info) {
            session.SetErrorInfo("TelematicsFirmwareTag::GetInfo", TString(info.GetError().AsStrBuf()), EDriveSessionResult::InternalError);
            return false;
        }
        auto current = GetCurrentInfo(self, *server);
        if (!current) {
            session.SetErrorInfo("TelematicsFirmwareTag::GetCurrentInfo", TString(current.GetError().AsStrBuf()), EDriveSessionResult::IncorrectCarSignal);
            return false;
        }
        if (!current->IsCompatibleWith(*info)) {
            session.SetErrorInfo("TelematicsFirmwareTag::OnAfterAdd", Name + " is not compatible with " + current->GetName(), EDriveSessionResult::IncorrectRequest);
            return false;
        }
    }
    return true;
}

bool TTelematicsFirmwareTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return
        TBase::DoSpecialDataFromJson(value, errors) &&
        NJson::ParseField(value["firmware_info"], FirmwareInfo) &&
        NJson::ParseField(value["firmware_upload_handler"], FirmwareUploadHandler) &&
        NJson::ParseField(value["name"], Name) &&
        NJson::ParseField(value["ignore_car_status"], IgnoreCarStatus) &&
        NJson::ParseField(value["update_time_restrictions"], UpdateTimeRestrictions) &&
        NJson::ParseField(value["last_update"], LastUpdate);
}

void TTelematicsFirmwareTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    if (Name) {
        value["name"] = NJson::ToJson(Name);
    }
    if (FirmwareInfo) {
        value["firmware_info"] = NJson::ToJson(FirmwareInfo);
    }
    if (FirmwareUploadHandler) {
        value["firmware_upload_handler"] = NJson::ToJson(FirmwareUploadHandler);
    }
    value["ignore_car_status"] = IgnoreCarStatus;
    if (UpdateTimeRestrictions) {
        value["update_time_restrictions"] = NJson::ToJson(UpdateTimeRestrictions);
    }
    if (LastUpdate) {
        value["last_update"] = NJson::ToJson(LastUpdate);
    }
}

namespace {

}

TTelematicsSensorTraits::TFuelTransformation::TFuelTransformation() {
    Type = TTelematicsSensorTraits::ETransformationType::FuelCalibration;
}

NDrive::TMultiSensor TTelematicsSensorTraits::TFuelTransformation::Transform(NDrive::TMultiSensor&& sensors) const {
    auto sensor = NDrive::ISensorApi::GetSensorFamily(sensors, FromId);

    if (sensor) {
        auto timestamp = sensor->Timestamp + TDuration::Seconds(1);

        sensor->Id = ToId;
        sensor->Value = Calibration.Get(*sensor);
        sensor->Timestamp = timestamp;

        sensors = NDrive::ISensorApi::Merge(std::move(sensors), { std::move(*sensor) });
    }

    return std::move(sensors);
}

NDrive::TMultiSensor TTelematicsSensorTag::Transform(const TString& objectId, NDrive::TMultiSensor&& sensors, const NDrive::IServer& server) {
    auto object = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(objectId);
    if (!object) {
        return std::move(sensors);
    }

    auto tag = object->GetFirstTagByClass<TTelematicsSensorTag>();
    if (!tag) {
        return std::move(sensors);
    }

    try {
        auto sensorTag = tag.GetTagAs<TTelematicsSensorTag>();
        if (!sensorTag) {
            return std::move(sensors);
        }

        auto dscr = Checked(server.GetDriveAPI())->GetTagsManager().GetTagsMeta().GetDescriptionByName(sensorTag->GetName());
        auto description = std::dynamic_pointer_cast<const TTelematicsSensorTag::TDescription>(dscr);
        if (!description) {
            description = MakeAtomicShared<TTelematicsSensorTag::TDescription>();
        }

        for (auto&& transform : description->GetTransformations()) {
            sensors = transform->Transform(std::move(sensors));
        }

        for (auto&& transform : sensorTag->GetTransformations()) {
            sensors = transform->Transform(std::move(sensors));
        }

        return std::move(sensors);
    } catch (...) {
        ERROR_LOG << Type() << ": cannot transform sensors " << CurrentExceptionMessage() << Endl;
    }

    return std::move(sensors);
}

NDrive::TScheme TTelematicsSensorTraits::ITransformation::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSVariants>("type", "Transformation type").InitVariants<TTelematicsSensorTraits::ETransformationType>().SetRequired(true);
    result.Add<TFSNumeric>("from_id", "From id").SetRequired(true);
    result.Add<TFSNumeric>("to_id", "To id").SetRequired(true);
    result.Add<TFSJson>("transformation", "Transformation special data").SetRequired(true);
    return result;
}

NJson::TJsonValue TTelematicsSensorTraits::ITransformation::ToJson() const {
    NJson::TJsonValue result;
    result["from_id"] = FromId;
    result["to_id"] = ToId;
    result["type"] = ToString(Type);
    return result;
}

bool TTelematicsSensorTraits::ITransformation::TryFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["from_id"], FromId) &&
        NJson::ParseField(value["to_id"], ToId);
}

NJson::TJsonValue TTelematicsSensorTraits::TFuelTransformation::ToJson() const {
    NJson::TJsonValue result = ITransformation::ToJson();
    result["transformation"] = NJson::ToJson(Calibration);
    return result;
}

bool TTelematicsSensorTraits::TFuelTransformation::TryFromJson(const NJson::TJsonValue& value) {
    return ITransformation::TryFromJson(value) && NJson::ParseField(value["transformation"], Calibration);
}

NDrive::TScheme TTelematicsSensorTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);

    result
        .Add<TFSArray>("transformations", "Transformation settings")
        .SetElement(TTelematicsSensorTraits::ITransformation::GetScheme());

    return result;
}

NJson::TJsonValue TTelematicsSensorTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    result["transformations"] = NJson::ToJson(Transformations);
    return result;
}

bool TTelematicsSensorTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    if (!TTagDescription::DoDeserializeMetaFromJson(value)) {
        return false;
    }
    return NJson::ParseField(value["transformations"], Transformations);
}

NDrive::TScheme TTelematicsSensorTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);

    result
        .Add<TFSArray>("transformations", "Transformation settings")
        .SetElement(TTelematicsSensorTraits::ITransformation::GetScheme());

    return result;
}

bool TTelematicsSensorTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return
        TBase::DoSpecialDataFromJson(value, errors) &&
        NJson::ParseField(value["transformations"], Transformations);
}

void TTelematicsSensorTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    value["transformations"] = NJson::ToJson(Transformations);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TExternalCommandResult::TSignature& object) {
    NJson::TJsonValue result;
    result["request"] = NJson::ToJson(object.Request);
    result["response"] = NJson::ToJson(object.Response);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TExternalCommandResult& object) {
    NJson::TJsonValue result;
    result["signatures"] = NJson::ToJson(object.Signatures);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TTelematicsConfigurationTag::TServerSettings& object) {
    NJson::TJsonValue result;
    result["hostport"] = object.HostPort;
    result["password"] = NJson::ToJson(NJson::Nullable(object.Password));
    result["id"] = object.Id;
    result["interaction_period"] = object.InteractionPeriod;
    result["generate_password"] = object.GeneratePassword;
    result["protocol"] = NJson::ToJson(NJson::Stringify(object.Protocol));
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TTelematicsConfigurationTag::TWiredSettings& object) {
    NJson::TJsonValue result;
    result["password"] = NJson::ToJson(NJson::Nullable(object.Password));
    result["generate_password"] = object.GeneratePassword;
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const TTelematicsSensorTraits::ETransformationType& object) {
    return ToString(object);
}

template<>
NJson::TJsonValue NJson::ToJson(const THolder<TTelematicsSensorTag::ITransformation>& object) {
    NJson::TJsonValue result;

    if (object) {
        result = object->ToJson();
        result["type"] = NJson::ToJson(object->GetType());
    }

    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const TTelematicsFirmwareTag::TDescription::TFirmwareInfo& object) {
    NJson::TJsonValue result = NJson::ToJson(object.GetInfo());
    result["tags_filter"] = object.GetTagsFilter().ToString();
    result["ignore_firmware_model"] = object.GetIgnoreFirmwareModel();
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TExternalCommandResult::TSignature& result) {
    return
        NJson::ParseField(value["request"], result.Request) &&
        NJson::ParseField(value["response"], result.Response);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TExternalCommandResult& result) {
    return
        NJson::ParseField(value["signatures"], result.Signatures);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsConfigurationTag::TServerSettings& result) {
    return
        NJson::ParseField(value["hostport"], result.HostPort) &&
        NJson::ParseField(value["password"], result.Password) &&
        NJson::ParseField(value["id"], result.Id) &&
        NJson::ParseField(value["interaction_period"], result.InteractionPeriod) &&
        NJson::ParseField(value["generate_password"], result.GeneratePassword) &&
        NJson::ParseField(value["protocol"], NJson::Stringify(result.Protocol));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsConfigurationTag::TSensorMeta& result) {
    return
        NJson::ParseField(value["display_name"], result.DisplayName) &&
        NJson::ParseField(value["min_value"], result.MinValue) &&
        NJson::ParseField(value["max_value"], result.MaxValue) &&
        NJson::ParseField(value["critical_values"], result.CriticalValues) &&
        NJson::ParseField(value["icon"], result.Icon) &&
        NJson::ParseField(value["dim"], result.Dim) &&
        NJson::ParseField(value["format_pattern"], result.FormatPattern) &&
        NJson::ParseField(value["decoding"], NJson::KeyValue(result.Decoding, "code", "decode"), true);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsConfigurationTag::TWiredSettings& result) {
    return
        NJson::ParseField(value["password"], result.Password) &&
        NJson::ParseField(value["generate_password"], result.GeneratePassword);
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsSensorTraits::ETransformationType& object) {
    return TryFromString(value.GetStringRobust(), object);
}

namespace {
    THolder<TTelematicsSensorTraits::ITransformation> CreateTransformation(TTelematicsSensorTraits::ETransformationType type) {
        switch (type) {
        case TTelematicsSensorTraits::ETransformationType::FuelCalibration:
            return TTelematicsSensorTraits::ITransformation::Create<TTelematicsSensorTraits::TFuelTransformation>();
        default:
            return nullptr;
        }
    }
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, THolder<TTelematicsSensorTraits::ITransformation>& object) {
    TTelematicsSensorTraits::ETransformationType type;
    if (!NJson::ParseField(value["type"], type)) {
        return false;
    }

    object = CreateTransformation(type);
    if (!object) {
        return false;
    }

    return object->TryFromJson(value);
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTelematicsFirmwareTag::TDescription::TFirmwareInfo& object) {
    if (!NJson::TryFromJson(value, object.MutableInfo())) {
        return false;
    }

    if (value.Has("tags_filter")) {
        TString tagsFilter;
        if (!NJson::TryFromJson(value["tags_filter"], tagsFilter)) {
            return false;
        }
        object.SetTagsFilter(TTagsFilter::BuildFromString(tagsFilter));
    }

    return NJson::ParseField(value["ignore_firmware_model"], object.MutableIgnoreFirmwareModel());
}

template struct TExpectedSizeOf<TTelematicsCommandTag, 120>;

ITag::TFactory::TRegistrator<TTelematicsCommandTag> TelematicsCommandTagRegistrator(TTelematicsCommandTag::Type());
TTagDescription::TFactory::TRegistrator<TTelematicsCommandTag::TDescription> TelematicsCommandTagDescriptionRegistrator(TTelematicsCommandTag::Type());

ITag::TFactory::TRegistrator<TTelematicsDeferredCommandTag> TelematicsDeferredCommandTagRegistrator(TTelematicsDeferredCommandTag::Type());
TTagDescription::TFactory::TRegistrator<TTelematicsDeferredCommandTag::TDescription> TelematicsDeferredCommandTagDescriptionRegistrator(TTelematicsDeferredCommandTag::Type());

ITag::TFactory::TRegistrator<TTelematicsConfigurationTag> TelematicsConfigurationTagRegistrator(TTelematicsConfigurationTag::Type());
TTagDescription::TFactory::TRegistrator<TTelematicsConfigurationTag::TDescription> TelematicsConfigurationTagDescriptionRegistrator(TTelematicsConfigurationTag::Type());

ITag::TFactory::TRegistrator<TTelematicsFirmwareTag> TelematicsFirmwareTagRegistrator(TTelematicsFirmwareTag::Type());
TTagDescription::TFactory::TRegistrator<TTelematicsFirmwareTag::TDescription> TelematicsFirmwareTagDescriptionRegistrator(TTelematicsFirmwareTag::Type());

ITag::TFactory::TRegistrator<TTelematicsSensorTag> TelematicsSensorRegistrator(TTelematicsSensorTag::Type());
TTagDescription::TFactory::TRegistrator<TTelematicsSensorTag::TDescription> TelematicsSensorDescriptionRegistrator(TTelematicsSensorTag::Type());
