#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/api/server/interop.h>
#include <drive/telematics/common/firmware.h>
#include <drive/telematics/protocol/actions.h>
#include <drive/telematics/protocol/calibration.h>
#include <drive/telematics/protocol/settings.h>
#include <drive/telematics/server/tasks/lite.h>

#include <rtline/library/time_restriction/time_restriction.h>

namespace NDrive {
    enum class ETelematicsNotification : ui32;

    struct TExternalCommandResult {
    public:
        struct TSignature {
            NDrive::NVega::TCommandRequestSigned Request;
            NDrive::NVega::TCommandResponseSigned Response;
        };
        using TSignatures = TVector<TSignature>;

    public:
        TSignatures Signatures;
    };

    bool ContainsActionInTag(
        const TString& userId,
        const TDriveAPI* driveApi,
        const THttpStatusManagerConfig& configHttpStatus,
        const TString& carId,
        NDrive::TEntitySession& session,
        TStringBuf actionName
    );
}

class TTelematicsCommandTagTraits {
public:
    using TCommand = NDrive::NVega::TCommand;

    struct TRuntimeContext {
        NDrive::TTelematicsClient::THandlerCallback Callback2;
        NDrive::TTelematicsClient::THandler Handler2;

        TDuration CallbackTimeout;
        TDuration Timeout;
        TDuration WaitConnectionTimeout;
        TDuration DataLifetime;
        TDuration Prelinger;
        TDuration Postlinger;
        TString ExternalCommandId;
        bool EvolveTag = true;
    };

public:
    TTelematicsCommandTagTraits() = default;
    TTelematicsCommandTagTraits(TCommand command, TDuration timeout, TDuration callbackTimeout, IDistributedTask::TPtr callback);
    TTelematicsCommandTagTraits(TCommand command, TDuration timeout, NDrive::TTelematicsClient::THandlerCallback&& callback);

    NDrive::NVega::ECommandCode GetCommand() const {
        return Command.Code;
    }
    const TString& GetHandlerId() const {
        return HandlerId;
    }
    TDuration GetTimeout() const {
        return GetContext().Timeout;
    }
    bool ShouldEvolveTag() const {
        return GetContext().EvolveTag;
    }

    void SetEvolveTag(bool value);
    void SetExternalCommandId(const TString& value);
    void SetPrelinger(TDuration value);
    void SetPostlinger(TDuration value);
    void SetWaitConnectionTimeout(TDuration value);

protected:
    const TRuntimeContext& GetContext() const;
    TRuntimeContext& GetContext();

    void SetHandler(NDrive::TTelematicsClient::THandler&& handler);

    bool Invoke(const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session);
    bool Invoke2(const TString& objectId, const TString& imei, const NDrive::IServer* server, NDrive::TEntitySession& session);

    bool FromJson(const NJson::TJsonValue& value);
    void ToJson(NJson::TJsonValue& value) const;

    void SetCommand(NDrive::NVega::ECommandCode value) {
        Command = value;
    }
    void SetHandlerId(const TString& value) {
        HandlerId = value;
    }

private:
    TCommand Command;
    TString HandlerId;

    THolder<TRuntimeContext> Context;
};

class TTelematicsCommandTag
    : public ISerializableTag<NDrive::NProto::TTelematicsTag, /*variativeRead = */true>
    , public TTelematicsCommandTagTraits
{
private:
    using TBase = ISerializableTag<NDrive::NProto::TTelematicsTag, true>;

public:
    class TDescription: public TTagDescription {
    public:
        using TTagDescription::TTagDescription;

        const TString& GetNotifier() const {
            return Notifier;
        }

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        TString Notifier;
    };

    struct TRuntimeOptions {
        TString ExternalCommandId;
        TDuration Timeout;
        bool EvolveTag = true;

        TRuntimeOptions(TDuration timeout)
            : Timeout(timeout)
        {
        }
    };

public:
    static TString Type() {
        return "telematics";
    }
    static NThreading::TFuture<NDrive::TCommonCommandResponse> Command(const TString& carId, TCommand command, const TRuntimeOptions& options, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session);
    static TAtomicSharedPtr<TTelematicsCommandTag> Command(const TString& carId, TAtomicSharedPtr<TTelematicsCommandTag> command, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session);
    static TExpected<TCommand> ParseCommand(const NJson::TJsonValue& description, const TString& objectId, const NDrive::IServer& server);
    static TString GenerateId(const TString& carId, const TCommand& command, TStringBuf reqId, const NDrive::IServer& server);
    static TString GetUserErrorDescription(TStringBuf message, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server);
    static TString GetUserErrorDescription(NDrive::ETelematicsNotification notification, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server);
    static TString GetUserErrorTitle(TStringBuf message, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server);
    static TString GetUserErrorTitle(NDrive::ETelematicsNotification notification, ELocalization locale, const TUserPermissions& permissions, const NDrive::IServer& server);
    static TExpected<bool> ValidateExternalCommand(const TString& carId, const TCommand& command, const NDrive::TExternalCommandResult& result, const TUserPermissions& permissions, const NDrive::IServer& server);

public:
    TTelematicsCommandTag()
        : TBase(Type())
    {
    }
    TTelematicsCommandTag(TCommand command, TDuration timeout, NDrive::TTelematicsClient::THandlerCallback&& callback = nullptr)
        : TBase(Type())
        , TTelematicsCommandTagTraits(std::move(command), timeout, std::move(callback))
    {
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual TString GetHRDescription() const override {
        return Type() + ":" + ToString(GetCommand());
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual bool OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

protected:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TTelematicsDeferredCommandTag: public IJsonSerializableTag {
private:
    using TBase = IJsonSerializableTag;

public:
    class TDescription : public TTagDescription {
    public:
        using TTagDescription::TTagDescription;

        NDrive::NVega::ECommandCode GetCode() const {
            return Code;
        }

        TDuration GetDefaultDuration() const {
            return DefaultDuration;
        }

        const NJson::TJsonValue& GetParameterSetting() const {
            return ParameterSetting;
        }

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        NDrive::NVega::ECommandCode Code = NDrive::NVega::ECommandCode::UNKNOWN;
        TDuration DefaultDuration = TDuration::Zero();
        NJson::TJsonValue ParameterSetting;

    private:
        R_OPTIONAL(TString, TagNameOnSuccess);
    };

public:
    static TString Type() {
        return "telematics_deferred_command";
    }

public:
    using TBase::TBase;

    TDuration GetDuration() const {
        return Duration;
    }

    TInstant GetSince() const {
        return Since;
    }

    TInstant GetUntil() const {
        return Until;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

protected:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    TInstant Since = TInstant::Zero();
    TInstant Until = TInstant::Max();
    TDuration Duration = TDuration::Zero();
};

class TTelematicsConfigurationTraits {
public:
    using TAPNs = TMap<ui16, NDrive::NVega::TAPNParameter>;
    using TSensors = TMap<ui16, NDrive::NVega::TSensorTranslation>;
    using TServers = TMap<ui16, NDrive::NVega::TServerSettingsParameter>;
    using TCanParameters = TMap<ui16, NDrive::NVega::TCanParameter>;
    using TPins = TMap<ui16, NDrive::NVega::THardPasswordParameter>;
    using TUsePins = TMap<ui16, NDrive::NVega::TUsePasswordParameter>;
    using TWiredPin = TMap<ui16, NDrive::NVega::TWiredPasswordParameter>;
    using TUseWiredPin = TMap<ui16, NDrive::NVega::TUseWiredPasswordParameter>;
    using TCalibrators = TMap<NDrive::TSensorId, NDrive::NVega::TSensorCalibration>;
    using TLogicScripts = TMap<ui16, NDrive::NVega::TLogicScript>;
    using TLogicCommands = TMap<ui16, NDrive::NVega::TLogicCommand>;
    using TLogicChecks = TMap<ui16, NDrive::NVega::TLogicCheck>;
    using TLogicTriggers = TMap<ui16, NDrive::NVega::TLogicTrigger>;
    using TLogicNotifies = TMap<ui16, NDrive::NVega::TLogicNotify>;

public:
    class TCustom {
    public:
        NJson::TJsonValue ToJson() const;
        ui64 CalcHash() const;

        [[nodiscard]] NDrive::NVega::TSettings GetSettings() const;
        void Apply(const NDrive::NVega::TSetting& setting);
        void Apply(const NDrive::TTelematicsClient::TGetParameterResponse& response);
        void Patch(TStringBuf model, const TTaggedObject taggedObject, const TTelematicsConfigurationTraits& base, const TTelematicsConfigurationTraits& patch);
        void PostLoad();

        inline bool operator==(const TCustom& other) const {
            return Model == other.Model && TagsFilter.ToString() == other.TagsFilter.ToString();
        }

        bool IsMatching(TStringBuf model, const TTaggedObject& object) const {
            return (!Model || Model == model) && (!TagsFilter || TagsFilter.IsMatching(object));
        }

        inline static bool CanApply(ui16 id, const TSet<ui16>& filter) {
            return filter.empty() || filter.contains(id);
        }

    private:
        R_FIELD(TString, Model);
        R_FIELD(TTagsFilter, TagsFilter);
        R_FIELD(TAPNs, APNs);
        R_FIELD(TSensors, Sensors);
        R_FIELD(TServers, Servers);
        R_FIELD(TCanParameters, CanParameters);
        R_FIELD(TVector<NDrive::TSensor>, Values);
        R_FIELD(TLogicScripts, LogicScripts);
        R_FIELD(TLogicCommands, LogicCommands);
        R_FIELD(TLogicChecks, LogicChecks);
        R_FIELD(TLogicTriggers, LogicTriggers);
        R_FIELD(TLogicNotifies, LogicNotifies);

    private:
        template <class T>
        void Apply(
            const TMap<ui16, T>& data,
            NDrive::NVega::TSettings& settings
        ) const {
            for (auto&& [subid, object] : data) {
                NDrive::NVega::TSetting setting;
                setting.Id = object.GetId();
                setting.SubId = subid;

                TBuffer payload = object.Dump();
                setting.Payload.reserve(payload.Size());
                setting.Payload.insert(setting.Payload.end(), payload.Begin(), payload.End());
                settings.push_back(setting);
            }
        }

        template <class T>
        void Apply(const NDrive::TSensor& sensor, TVector<T>& data) const {
            if (!NDrive::NVega::IsSettingId(sensor.Id)) {
                data.push_back(sensor);
            }
        }

        template <class T>
        void Apply(const NDrive::NVega::TSetting& setting, TMap<ui16, T>& data) const {
            if (setting.Id != T::GetId()) {
                return;
            }

            T result;
            const auto& payload = setting.Payload;

            TBuffer buffer{payload.data(), payload.size()};
            result.Parse(buffer);
            if (!result.IsEmpty()) {
                data[setting.SubId] = result;
            }
        }

    };

public:
    static TString Type() {
        return "telematics_configuration";
    }

    static ui64 CalcHash(TStringBuf imei);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TAPNParameter>& apn);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TSensorTranslation>& sensors);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TServerSettingsParameter>& servers);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::THardPasswordParameter>& pins);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TUsePasswordParameter>& usePins);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TWiredPasswordParameter>& pins);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TUseWiredPasswordParameter>& usePins);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TCanParameter>& canParameters);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TLogicScript>& scripts);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TLogicCommand>& commands);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TLogicCheck>& checks);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TLogicTrigger>& triggers);
    static ui64 CalcHash(const TMap<ui16, NDrive::NVega::TLogicNotify>& notifies);

    static TString GeneratePin(const TString& objectId, TInstant timestamp);

public:
    void Merge(const TCustom& custom);

protected:
    R_FIELD(TAPNs, APNs);
    R_FIELD(TSensors, Sensors);
    R_FIELD(TServers, Servers);
    R_READONLY(TPins, Pins);
    R_READONLY(TUsePins, UsePins);
    R_READONLY(TWiredPin, WiredPin);
    R_READONLY(TUseWiredPin, UseWiredPin);
    R_READONLY(TSet<ui16>, GeneratePinIds);
    R_READONLY(TSet<ui16>, GenerateWiredPinIds);
    R_READONLY(TVector<TCustom>, Customs);
    R_READONLY(TCalibrators, Calibrators);
};

class TTelematicsConfigurationTag
    : public IJsonSerializableTag
    , public TTelematicsConfigurationTraits
{
public:
    struct TServerSettings {
        TString HostPort;
        TString Password;
        ui16 Id = 0;
        ui16 InteractionPeriod = 0;
        bool GeneratePassword = false;
        NDrive::NVega::TServerSettingsParameter::EProtocol Protocol = NDrive::NVega::TServerSettingsParameter::DISABLED;
    };

    struct TSensorMeta {
        TString DisplayName;
        TVector<TString> CriticalValues;
        TMaybe<double> MinValue;
        TMaybe<double> MaxValue;
        TMaybe<TString> Icon;
        TMaybe<TString> Dim;
        TMaybe<TString> FormatPattern;
        TMap<TString, TString> Decoding;
    };

    struct TWiredSettings {
        TString Password;
        bool GeneratePassword = false;
    };

    class TDescription
        : public TTagDescription
        , public TTelematicsConfigurationTraits
    {
    public:
        inline TDescription() = default;

        float GetGeneratePinRolloutFraction() const {
            return GeneratePinRolloutFraction;
        }

        bool IsSensorValueCritical(const NDrive::TSensorId& id, const TMaybe<NDrive::TSensor>& sensor) const;
        const TMap<NDrive::TSensorId, TSensorMeta>& GetSensorMeta() const;
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        TVector<TServerSettings> ServerSettings;
        TMaybe<TWiredSettings> WiredSettings;
        TMap<NDrive::TSensorId, TSensorMeta> SensorMeta;
        float GeneratePinRolloutFraction = 0;
    };

    struct TPinInfo {
        TString Imei;
        TString Value;
        TString Method;

        TPinInfo(const TString& imei, const TString& value, const TString& method)
            : Imei(imei)
            , Value(value)
            , Method(method)
        {
        }
    };
    using TExpectedPinInfo = TExpected<TPinInfo, yexception>;

private:
    using TBase = IJsonSerializableTag;

public:
    static IEntityTagsManager::TOptionalTag Instance(const TString& objectId, const NDrive::IServer* server, NDrive::TEntitySession& session);
    static TExpectedPinInfo GetPin(const TString& objectId, const NDrive::IServer* server, ui16 serverId, TInstant deadline);
    static TExpectedPinInfo GetPinByIMEI(const TString& imei, const NDrive::IServer* server, ui16 serverId, TInstant deadline);

    static NDrive::TTelematicsClient::THandler GetActualConfiguration(
        const TString& objectId,
        const NDrive::IServer* server,
        TDuration timeout = TDuration::Minutes(1)
    );

    static NDrive::TTelematicsClient::THandlers GetActualValues(
        const TString& objectId,
        const TSet<NDrive::TSensorId>& ids,
        const NDrive::IServer* server,
        TDuration timeout = TDuration::Minutes(1)
    );

public:
    TTelematicsConfigurationTag()
        : IJsonSerializableTag(Type())
    {
        MaxAuxFuelVolume.fill(0);
    }

    ui64 GetExpectedHash(TStringBuf imei, TStringBuf model, const TTaggedObject& taggedObject, const NDrive::IServer* server) const;
    ui64 GetExpectedHash(TStringBuf imei, TStringBuf model, const TTaggedObject& taggedObject, const TDescription* description) const;

    ui64 GetConfigurationHash() const {
        return ConfigurationHash;
    }
    TInstant GetDeadline() const {
        return Deadline;
    }
    void SetApplyOnAdd(bool value) {
        ApplyOnAdd = value;
    }
    void SetPinOverride(const TString& value) {
        PinOverride = value;
    }
    void SetPin(ui16 id, TStringBuf value) {
        MutableProtectedPins()[id].Value.Set(value);
        MutableProtectedUsePins()[id].Mode = value.empty() ? NDrive::NVega::TUsePasswordParameter::DISABLE : NDrive::NVega::TUsePasswordParameter::ENABLE;
    }
    void SetWiredPin(ui16 id, TStringBuf value) {
        MutableProtectedWiredPin()[id].Value.Set(value);
        MutableProtectedUseWiredPin()[id].Mode = value.empty() ? NDrive::NVega::TUseWiredPasswordParameter::DISABLE : NDrive::NVega::TUseWiredPasswordParameter::ENABLE;
    }
    void SetConfigurationHash(ui64 value, TInstant deadline) {
        ConfigurationHash = value;
        Deadline = deadline;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }
    virtual TString GetHRDescription() const override {
        return Type();
    }
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

    template <class K, class T>
    bool TryMapFromJson(const NJson::TJsonValue& value, TMap<K, T>& result, TMessagesCollector* errors);

private:
    ui64 ConfigurationHash = 0;
    TInstant Deadline = TInstant::Zero();
    TString PinOverride;
    ui16 AuxFuelLevelTransmissionPeriod = 0;
    std::array<float, 10> MaxAuxFuelVolume;
    bool ApplyOnAdd = true;
};

class TTelematicsFirmwareTag : public IJsonSerializableTag {
public:
    class TDescription : public TTagDescription {
    public:
        class TFirmwareInfo final {
        public:
            TFirmwareInfo() = default;
            TFirmwareInfo(const NDrive::NVega::TFirmwareInfo& info)
                : Info(info)
            {
            }

            bool operator<(const TFirmwareInfo& other) const {
                auto tagsFilter = TagsFilter.ToString();
                auto otherTagsFilter = other.TagsFilter.ToString();
                return std::tie(tagsFilter, Info) < std::tie(otherTagsFilter, other.Info);
            }

            bool operator==(const TFirmwareInfo& other) const {
                return
                    TagsFilter.ToString() == other.TagsFilter.ToString() &&
                    Info.IsCompatibleWith(other.Info);
            }

            bool IsCompatibleWith(const TTaggedDevice& object, const NDrive::NVega::TFirmwareInfo& info) const {
                return Info.IsCompatibleWith(info) && (!TagsFilter || TagsFilter.IsMatching(object));
            }

            TString DebugString() const {
                return TagsFilter.ToString() + " " + Info.Model + " " + ToString(Info.Type);
            }

        private:
            R_FIELD(TTagsFilter, TagsFilter);
            R_FIELD(NDrive::NVega::TFirmwareInfo, Info);
            R_FIELD(bool, IgnoreFirmwareModel, false);
        };

    public:
        TVector<TFirmwareInfo> GetFirmwareInfos() const;
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        TVector<TFirmwareInfo> Firmwares;
        R_OPTIONAL(TTimeRestrictionsPool<TTimeRestriction>, UpdateTimeRestrictions);
        R_FIELD(bool, IgnoreCarStatus, false);
    };

private:
    using TBase = IJsonSerializableTag;

public:
    static TString Type() {
        return "telematics_firmware";
    }

    static bool Add(const NDrive::NVega::TFirmwareInfo& info, TBuffer&& data, const NDrive::IServer& server);
    static bool Remove(const TString& name, const NDrive::IServer& server);
    static TExpected<NDrive::NVega::TFirmwareInfo, yexception> GetCurrentInfo(const TConstDBTag& tag, const NDrive::IServer& server);
    static TExpected<NDrive::NVega::TFirmwareInfo, yexception> GetTaggedInfo(const TConstDBTag& tag, const NDrive::IServer& server);
    static TExpected<NDrive::NVega::TFirmwareInfo, yexception> GetOnlyTaggedInfo(const TConstDBTag& tag, const NDrive::IServer& server);
    static TExpected<NDrive::NVega::TFirmwareInfo, yexception> GetInfo(const TString& name, const NDrive::IServer& server);
    static TExpected<TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo>, yexception> GetInfoFromDescription(const NDrive::IServer& server);
    static TExpected<TBuffer, yexception> GetData(const TString& name, const NDrive::IServer& server);
    static TSet<TString> List(const NDrive::IServer& server);
    static TExpected<std::tuple<TString, bool, TMaybe<TTimeRestrictionsPool<TTimeRestriction>>, TInstant>, yexception> GetUpdateInfo(const TConstDBTag& tag, const NDrive::IServer& server);

public:
    TTelematicsFirmwareTag(const TString& name = {})
        : TBase(Type())
        , Name(name)
    {
    }

    const TString& GetName() const {
        return Name;
    }
    void SetName(const TString& value) {
        Name = value;
    }
    const NDrive::NVega::TFirmwareInfo& GetFirmware() const {
        return *FirmwareInfo;
    }
    bool HasFirmware() const {
        return FirmwareInfo.Defined();
    }
    void SetFirmware(const NDrive::NVega::TFirmwareInfo& info, const TString& uploadHandler) {
        FirmwareInfo = info;
        FirmwareUploadHandler = uploadHandler;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }
    virtual TString GetHRDescription() const override {
        return Type();
    }
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    TString Name;

    TMaybe<NDrive::NVega::TFirmwareInfo> FirmwareInfo;
    TString FirmwareUploadHandler;
    R_OPTIONAL(TTimeRestrictionsPool<TTimeRestriction>, UpdateTimeRestrictions);
    R_FIELD(bool, IgnoreCarStatus, false);
    R_FIELD(TInstant, LastUpdate);
};

class TTelematicsSensorTraits {
public:
    enum class ETransformationType: ui8 {
        FuelCalibration = 0 /* "fuel_calibration" */,
    };

    class ITransformation {
    public:
        virtual ~ITransformation() = default;

        ETransformationType GetType() const {
            return Type;
        }

        virtual NDrive::TMultiSensor Transform(NDrive::TMultiSensor&& sensors) const = 0;

        virtual NJson::TJsonValue ToJson() const;
        virtual bool TryFromJson(const NJson::TJsonValue& value);

        static NDrive::TScheme GetScheme();

        template<class T, typename... Args>
        static THolder<T> Create(Args&&... args) {
            static_assert(std::is_base_of<ITransformation, T>::value, "not base of ITransformation");
            return MakeHolder<T>(std::forward<Args>(args)...);
        }

    protected:
        ETransformationType Type;
        ui16 FromId = 0;
        ui16 ToId = 0;
    };

    class TFuelTransformation: public ITransformation {
    public:
        using TBase = ITransformation;

    public:
        TFuelTransformation();
        virtual ~TFuelTransformation() = default;

        NDrive::TMultiSensor Transform(NDrive::TMultiSensor&& sensors) const override;


        NJson::TJsonValue ToJson() const override;
        bool TryFromJson(const NJson::TJsonValue& value) override;

    private:
        NDrive::NVega::TSensorCalibration Calibration;
    };

public:
    const TVector<THolder<ITransformation>>& GetTransformations() const {
        return Transformations;
    }

protected:
    TVector<THolder<ITransformation>> Transformations;
};

class TTelematicsSensorTag
    : public IJsonSerializableTag
    , public TTelematicsSensorTraits
{
public:
    using TBase = IJsonSerializableTag;

public:
    class TDescription
        : public TTagDescription
        , public TTelematicsSensorTraits
    {
    public:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;
    };

public:
    TTelematicsSensorTag(const TString& name = {})
        : TBase(Type())
        , Name(name)
    {
    }

    static NDrive::TMultiSensor Transform(const TString& objectId, NDrive::TMultiSensor&& sensors, const NDrive::IServer& server);

public:
    static TString Type() {
        return "telematics_sensor";
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }
    virtual TString GetHRDescription() const override {
        return Type();
    }
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

private:
    TString Name;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;
};
