#include "transformation.h"

#include "chargable.h"

#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/proto/drive.pb.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/algorithm/type_traits.h>
#include <rtline/util/types/string_pool.h>

static Y_THREAD(TStringPool) TagNamePool;

const TString TTransformationTag::TypeName = "transformation";
TTransformationTag::TFactory::TRegistrator<TTransformationTag> TTransformationTag::Registrator(TypeName);

TBlob TTransformationTag::DoSerializeSpecialData(NDrive::NProto::TTagHeader& h) const {
    h.SetIsTagProto(true);
    NDrive::NProto::TTransformationTag protoTransformation;
    protoTransformation.SetHandler(CommandId);
    protoTransformation.SetFrom(From);
    protoTransformation.SetTo(To);
    protoTransformation.SetRollbackTimestamp(RollbackTimestamp.Seconds());
    TDropPolicyProvider::SerializeToProto(protoTransformation);
    if (ServicingInfo) {
        auto servicingInfoProto = protoTransformation.MutableServicingInfo();
        servicingInfoProto->SetStart(ServicingInfo->Start.Seconds());
        servicingInfoProto->SetFinish(ServicingInfo->Finish.Seconds());
        servicingInfoProto->SetMileage(ServicingInfo->Mileage);
    }
    return TBlob::FromString(protoTransformation.SerializeAsString());
}

bool TTransformationTag::DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& h, const TBlob& data) {
    if (h.HasIsTagProto() && h.GetIsTagProto()) {
        NDrive::NProto::TTransformationTag protoTransformation;
        if (!protoTransformation.ParseFromArray(data.AsCharPtr(), data.Size())) {
            return false;
        }
        if (!protoTransformation.HasFrom() || !protoTransformation.HasTo()) {
            return false;
        }
        CommandId = protoTransformation.GetHandler();
        From = TlsRef(TagNamePool).Get(protoTransformation.GetFrom());
        To = TlsRef(TagNamePool).Get(protoTransformation.GetTo());
        if (!TDropPolicyProvider::DeserializeFromProto(protoTransformation)) {
            return false;
        }
        if (protoTransformation.HasRollbackTimestamp()) {
            RollbackTimestamp = TInstant::Seconds(protoTransformation.GetRollbackTimestamp());
        }
        if (protoTransformation.HasServicingInfo()) {
            const auto& servicingInfoProto = protoTransformation.GetServicingInfo();
            TServicingInfo servicingInfo;
            servicingInfo.Start = TInstant::Seconds(servicingInfoProto.GetStart());
            servicingInfo.Finish = TInstant::Seconds(servicingInfoProto.GetFinish());
            servicingInfo.Mileage = servicingInfoProto.GetMileage();
            SetServicingInfo(std::move(servicingInfo));
        }
    } else {
        TStringBuf jsonStr(data.AsCharPtr(), data.Size());
        NJson::TJsonValue jsonValue;
        if (!NJson::ReadJsonFastTree(jsonStr, &jsonValue)) {
            return false;
        }
        if (!jsonValue.Has("from") || !jsonValue.Has("to")) {
            return false;
        }
        if (jsonValue.Has("rollback_timestamp")) {
            double timestamp;
            if (jsonValue["rollback_timestamp"].GetDouble(&timestamp)) {
                RollbackTimestamp = TInstant::Seconds(timestamp);
            } else {
                return false;
            }
        }

        From = TlsRef(TagNamePool).Get(jsonValue["from"].GetStringRobust());
        To = TlsRef(TagNamePool).Get(jsonValue["to"].GetStringRobust());
    }
    return true;
}

void TTransformationTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    json.InsertValue("transformation_command_id", CommandId);
    json.InsertValue("transformation_from", From);
    json.InsertValue("transformation_to", To);
    json.InsertValue("transformation_rollback_timestamp", RollbackTimestamp.SecondsFloat());
    json.InsertValue("transformation_fee", GetFinishOfferFee());
    json.InsertValue("transformation_fee_policy", ::ToString(GetFinishOfferFeePolicy()));
    json.InsertValue("transformation_alert_landing_id", GetAlertLandingId());
    if (ServicingInfo) {
        json.InsertValue("servicing_start", ServicingInfo->Start.Seconds());
        json.InsertValue("servicing_finish", ServicingInfo->Finish.Seconds());
        json.InsertValue("servicing_mileage", ServicingInfo->Mileage);
    }
}

bool TTransformationTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/, const TEvolutionContext* /*eContext*/) const {
    const TTransformationTag* transformation = fromTag.GetTagAs<TTransformationTag>();
    if (!transformation) {
        return false;
    }
    TChargableTag* chargableTo = dynamic_cast<TChargableTag*>(toTag.Get());
    const TString& from = transformation->GetFrom();
    if (chargableTo && chargableTo->GetName() == "old_state_reservation" && from != "old_state_reservation" && from != "old_state_acceptance") {
        CopyTo(*chargableTo);
    }
    return true;
}

bool TTransformationTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(self);
    Y_UNUSED(userId);
    Y_UNUSED(server);
    if (!GetPerformer().empty()) {
        session.SetErrorInfo("TransformationTag::OnBeforeRemove", "cannot remove transformation with performer");
        return false;
    }
    return true;
}

template <bool Successful>
ITag::TPtr FinishTransformation(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const NDrive::ITag::TEvolutionContext* eContext) {
    const auto& tagManager = Yensured(server)->GetDriveDatabase().GetTagsManager();
    auto evlog = NDrive::GetThreadEventLogger();
    {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "TransformationFinish")
                ("from", NJson::ToJson(from))
                ("to", NJson::ToJson(to))
                ("successful", Successful)
                ("user_id", permissions.GetUserId())
            );
        }
        if constexpr (Successful) {
            auto servicingInfoSource = from.GetTagAs<IServicingInfoHolder>();
            auto servicingInfo = servicingInfoSource ? servicingInfoSource->GetServicingInfo() : nullptr;
            auto servicingInfoRecipient = std::dynamic_pointer_cast<IServicingInfoHolder>(to);
            if (servicingInfo && servicingInfoRecipient) {
                servicingInfoRecipient->SetServicingInfo(MakeCopy(*servicingInfo));
            }
            if (!tagManager.GetDeviceTags().EvolveTag(from, to, permissions, server, session, eContext)) {
                return nullptr;
            }
        } else {
            if (!tagManager.GetDeviceTags().DirectEvolveTag(permissions.GetUserId(), from, to, session)) {
                return nullptr;
            }
        }
    }
    return to;
}

template <bool Successful>
ITag::TPtr TransformationFinish(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, ui32 attempts, const NDrive::ITag::TEvolutionContext* evolutionContext) {
    const auto& tagManager = Yensured(server)->GetDriveDatabase().GetTagsManager();
    for (ui32 attempt = 0; attempt <= attempts; ++attempt) try {
        auto session = tagManager.GetDeviceTags().BuildTx<NSQL::Writable>();
        auto transformed = FinishTransformation<Successful>(from, to, permissions, server, session, evolutionContext);
        if (!transformed) {
            session.Check();
        }
        if (!session.Commit()) {
            session.Check();
        }
        return transformed;
    } catch (TCodedException& e) {
        NJson::TJsonValue ev = NJson::TMapBuilder
            ("attempt", attempt)
            ("error", e.GetReport())
            ("object_id", from.GetObjectId())
            ("user_id", permissions.GetUserId())
            ("tag_id", from.GetTagId())
        ;
        NDrive::TEventLog::Log("EvolveFailure", ev);
        auto evlog = NDrive::GetThreadEventLogger();
        if (evlog) {
            ev["event"] = "EvolveFailure";
            evlog->AddEvent(std::move(ev));
        }
        if (attempt < attempts) {
            Sleep(TDuration::MilliSeconds(10));
            continue;
        }
        throw;
    }
    return nullptr;
}

ITag::TPtr TTransformationTag::Complete(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) {
    return FinishTransformation<true>(from, to, permissions, server, session, eContext);
}

ITag::TPtr TTransformationTag::Rollback(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) {
    return FinishTransformation<false>(from, to, permissions, server, session, eContext);
}

TExpected<ITag::TPtr, TCodedException> TTransformationTag::Complete(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, ui32 attempts, const TEvolutionContext* eContext) {
    return WrapUnexpected<TCodedException>(TransformationFinish<true>, from, to, permissions, server, attempts, eContext);
}

TExpected<ITag::TPtr, TCodedException> TTransformationTag::Rollback(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, ui32 attempts, const TEvolutionContext* eContext) {
    return WrapUnexpected<TCodedException>(TransformationFinish<false>, from, to, permissions, server, attempts, eContext);
}

template struct TExpectedSizeOf<TTransformationTag, 160>;
