#pragma once

#include "servicing.h"

#include <drive/backend/data/common/common.h>
#include <drive/backend/actions/evolution.h>
#include <drive/backend/areas/drop_object_features.h>

#include <rtline/util/types/expected.h>

class TTransformationTag
    : public TCommonTag
    , public TDropPolicyProvider
    , public IServicingInfoHolder
{
private:
    using TBase = TCommonTag;

private:
    static TFactory::TRegistrator<TTransformationTag> Registrator;

private:
    R_FIELD(TString, CommandId);
    R_FIELD(TString, From);
    R_FIELD(TString, To);
    R_FIELD(TInstant, RollbackTimestamp, TInstant::Max());

protected:
    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& h) const override;
    bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& h, const TBlob& data) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;

    bool OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;
    bool OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

public:
    static ITag::TPtr Complete(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext);
    static ITag::TPtr Rollback(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext);

    static TExpected<ITag::TPtr, TCodedException> Complete(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, ui32 attempts, const TEvolutionContext* eContext);
    static TExpected<ITag::TPtr, TCodedException> Rollback(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, ui32 attempts, const TEvolutionContext* eContext);

public:
    static const TString TypeName;

public:
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual TString GetHRDescription() const override {
        return From + " -> " + To;
    }

    TTransformationTag()
        : TBase(TypeName)
    {
    }

    TTransformationTag(const TString& from, const TString& to)
        : TBase(TypeName)
        , From(from)
        , To(to)
    {
    }

    TString GetPerformGroup() const override {
        return To;
    }
    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    const TServicingInfo* GetServicingInfo() const override {
        return ServicingInfo.Get();
    }
    void SetServicingInfo(TServicingInfo&& value) override {
        ServicingInfo = std::move(value);
    }

private:
    TOptionalServicingInfo ServicingInfo;
};
