#include "user_insurance.h"

#include "chargable.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>

#include <rtline/util/algorithm/ptr.h>


ITag::TFactory::TRegistrator<TUserKaskoTag> TUserKaskoTag::Registrator(TUserKaskoTag::Type());
TTagDescription::TFactory::TRegistrator<TUserKaskoTag::TDescription> TUserKaskoTag::TDescription::Registrator(TUserKaskoTag::Type());


NDrive::TScheme TUserKaskoTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSString>("kasko_package_name", "Тариф КАСКО для заказа").SetRequired(true);
    scheme.Add<TFSText>("kasko_package_params", "Параметры пакета (json)");
    scheme.Add<TFSString>("score_tag_name", "Тег скоринга").SetDefault(ScoreTagName).SetRequired(true);
    auto& item = scheme.Add<TFSArray>("discount_map", "Распределение скидки").SetElement<NDrive::TScheme>();
    item.Add<TFSNumeric>("percentile").SetMin(0).SetMax(100).SetRequired(true);
    item.Add<TFSNumeric>("discount").SetMin(0).SetMax(100).SetRequired(true);
    return scheme;
}

NJson::TJsonValue TUserKaskoTag::TDescription::DoSerializeMetaToJson() const {
    auto json = TBase::DoSerializeMetaToJson();
    json.InsertValue("kasko_package_name", KaskoPackageName);
    if (KaskoPackageParams.IsDefined()) {
        json.InsertValue("kasko_package_params", KaskoPackageParams.GetStringRobust());
    }
    json.InsertValue("score_tag_name", ScoreTagName);
    if (!DiscountMap.empty()) {
        auto& map = json.InsertValue("discount_map", NJson::JSON_ARRAY);
        for (auto&& [perc, discount] : DiscountMap) {
            map.AppendValue(NJson::TMapBuilder
                ("percentile", perc)
                ("discount", discount)
            );
        }
    }
    return json;
}

bool TUserKaskoTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue &jsonMeta) {
    for (auto&& item : jsonMeta["discount_map"].GetArray()) {
        std::pair<ui32, ui32> val;
        if (!NJson::ParseField(item, "percentile", val.first, /* required = */ true)
            || !NJson::ParseField(item, "discount", val.second, /* required = */ true))
        {
            return false;
        }
        DiscountMap.insert(val);
    }
    if (jsonMeta.Has("kasko_package_params")) {
        NJson::TJsonValue params;
        if (!NJson::ReadJsonFastTree(jsonMeta["kasko_package_params"].GetString(), &params)) {
            return false;
        }
        KaskoPackageParams = params;
    }
    return TBase::DoDeserializeMetaFromJson(jsonMeta)
        && NJson::ParseField(jsonMeta, "score_tag_name", ScoreTagName, /* required = */ false)
        && NJson::ParseField(jsonMeta, "kasko_package_name", KaskoPackageName, /* required = */ true);
}

ui32 TUserKaskoTag::TDescription::GetDiscount(const ui32 percentile) const {
    for (auto&& [perc, discount] : DiscountMap) {
        if (percentile < perc) {
            return discount;
        }
    }
    return 0;
}


void TUserKaskoTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    if (KaskoData) {
        json.InsertValue("insurance", KaskoData->SerializeToJson());
    }
}

bool TUserKaskoTag::DoSpecialDataFromJson(const NJson::TJsonValue& /*json*/, TMessagesCollector* /*errors*/) {
    return false;
}

TUserKaskoTag::TProto TUserKaskoTag::DoSerializeSpecialDataToProto() const {
    auto proto = TBase::DoSerializeSpecialDataToProto();
    if (KaskoData) {
        *proto.MutableKaskoData() = KaskoData->SerializeToProto();
    }
    return proto;
}

bool TUserKaskoTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (proto.HasKaskoData()) {
        NDrive::NRenins::TKaskoData kaskoData;
        if (!kaskoData.DeserializeFromProto(proto.GetKaskoData())) {
            return false;
        }
        KaskoData = kaskoData;
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}
