#include "user_origin.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/tags/tags_manager.h>

TMaybe<TString> TUserOriginTag::GetExternalUserId(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    auto optionalTag = Get(userId, server, session);
    if (!optionalTag) {
        return {};
    }
    auto originTag = std::dynamic_pointer_cast<TUserOriginTag>(optionalTag->GetData());
    if (!originTag && *optionalTag) {
        session.SetErrorInfo("UserOriginTag::GetExternalUserId", "cannot cast as UserOriginTag " + optionalTag->GetTagId());
        return {};
    }
    if (!originTag) {
        return TString();
    }
    return originTag->GetExternalUserId();
}

TMaybe<TDBTag> TUserOriginTag::Get(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto& userTagManager = server.GetDriveDatabase().GetTagsManager().GetUserTags();
    const auto expectedTags = userTagManager.RestoreEntityTags(userId, { Type() }, session);
    if (!expectedTags) {
        return {};
    }
    if (expectedTags->empty()) {
        return TDBTag();
    }
    const auto& expectedTag = expectedTags->front();
    return expectedTag;
}

bool TUserOriginTag::Set(const TString& userId, const TString& origin, const TString& externalUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    auto optionalTag = Get(userId, server, session);
    if (!optionalTag) {
        return false;
    }
    auto originTag = std::dynamic_pointer_cast<TUserOriginTag>(optionalTag->GetData());
    if (!originTag && *optionalTag) {
        session.SetErrorInfo("UserOriginTag::GetExternalUserId", "cannot cast as UserOriginTag " + optionalTag->GetTagId());
        return false;
    }
    if (originTag && originTag->GetOrigin() == origin && originTag->GetExternalUserId() == externalUserId) {
        return true;
    }

    auto tag = MakeAtomicShared<TUserOriginTag>(Type());
    tag->SetOrigin(origin);
    tag->SetExternalUserId(externalUserId);
    auto addedTag = server.GetDriveDatabase().GetTagsManager().GetUserTags().AddTag(tag, userId, userId, &server, session);
    return addedTag.Defined();
}

bool TUserOriginTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return
        NJson::ParseField(value["origin"], Origin) &&
        NJson::ParseField(value["external_user_id"], ExternalUserId);
}

void TUserOriginTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::InsertNonNull(value, "origin", Origin);
    NJson::InsertNonNull(value, "external_user_id", ExternalUserId);
}

ITag::TFactory::TRegistrator<TUserOriginTag> TUserOriginTag(TUserOriginTag::Type());
