#include "user_tags.h"

#include "container_tag.h"
#include "billing_tags.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/registrar/config.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user_documents.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

#include <rtline/util/instant_model.h>

const TString TAdditionalRolesUserTag::TypeName = "additional_roles_user_tag";
ITag::TFactory::TRegistrator<TAdditionalRolesUserTag> TAdditionalRolesUserTag::Registrator(TAdditionalRolesUserTag::TypeName);
TTagDescription::TFactory::TRegistrator<TAdditionalRolesUserTag::TDescription> AdditionalRolesUserTagDescriptionRegistrator(TAdditionalRolesUserTag::TypeName);

const TString TIntroViewsInfoTag::TypeName = "intro_views";
ITag::TFactory::TRegistrator<TIntroViewsInfoTag> TIntroViewsInfoTag::Registrator(TIntroViewsInfoTag::TypeName);

const TString TStaffHierarchyTag::TypeName = "staff_hierarchy";
ITag::TFactory::TRegistrator<TStaffHierarchyTag> TStaffHierarchyTag::Registrator(TStaffHierarchyTag::TypeName);
TStaffHierarchyTag::TDescription::TFactory::TRegistrator<TStaffHierarchyTag::TDescription> TStaffHierarchyTag::TDescription::Registrator(TStaffHierarchyTag::TypeName);

const TString TTemporaryActionTag::TypeName = "temp_action";
ITag::TFactory::TRegistrator<TTemporaryActionTag> TTemporaryActionTag::Registrator(TTemporaryActionTag::TypeName);
TTemporaryActionTag::TDescription::TFactory::TRegistrator<TTemporaryActionTag::TDescription> TTemporaryActionTag::DescriptionRegistrator(TTemporaryActionTag::TypeName);

const TString TSimpleUserTag::TypeName = "simple_user_tag";
ITag::TFactory::TRegistrator<TSimpleUserTag> TSimpleUserTag::Registrator(TSimpleUserTag::TypeName);

const TString TUniqueUserTag::TypeName = "unique_user_tag";
ITag::TFactory::TRegistrator<TUniqueUserTag> TUniqueUserTag::Registrator(TUniqueUserTag::TypeName);

const TString TRewriteUserTag::TypeName = "rewrite_user_tag";
ITag::TFactory::TRegistrator<TRewriteUserTag> TRewriteUserTag::Registrator(TRewriteUserTag::TypeName);

const TString TAdjustUserTag::TypeName = "adjust_event_tag";
ITag::TFactory::TRegistrator<TAdjustUserTag> TAdjustUserTag::Registrator(TAdjustUserTag::TypeName);
TTagDescription::TFactory::TRegistrator<TAdjustUserTag::TDescription> TAdjustUserTag::TDescription::Registrator(TAdjustUserTag::TypeName);

TMaybe<TUserProblemTag::EBlockedStatus> TUserProblemTag::ShouldBeBlocked(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto& userTagManager = server.GetDriveDatabase().GetTagsManager().GetUserTags();
    auto taggedObject = userTagManager.RestoreObject(userId, session);
    if (!taggedObject) {
        return {};
    }
    ui64 banThreshold = server.GetSettings().GetValueDef<ui64>("user.bans.points_threshold", 6);
    ui64 pointsSum = 0;
    for (auto&& tag : taggedObject->GetTags()) {
        auto impl = tag.GetTagAs<TUserProblemTag>();
        if (!impl) {
            continue;
        }
        auto descr = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(impl->GetName());
        auto descrImpl = dynamic_cast<const TUserProblemTag::TDescription*>(descr.Get());
        pointsSum += descrImpl ? descrImpl->GetPoint() : 0;
        if (pointsSum >= banThreshold) {
            break;
        }
    }
    return (pointsSum >= banThreshold ? TUserProblemTag::EBlockedStatus::Blocked : TUserProblemTag::EBlockedStatus::Ok);
}

bool TUserProblemTag::EnsureNotBlocked(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    auto blockedStatus = ShouldBeBlocked(userId, server, session);
    if (!blockedStatus) {
        return false;
    }
    switch (*blockedStatus) {
    case TUserProblemTag::EBlockedStatus::Blocked:
        session.SetErrorInfo("UserProblemTag::EnsureNotBlocked", "user should be blocked", EDriveSessionResult::UserShouldBeBlocked);
        return false;
    case TUserProblemTag::EBlockedStatus::Ok:
        return true;
    }
}

bool TUserProblemTag::SendMessage(const TString& userId, const TString& message, const NDrive::NChat::IMessage::EMessageType type, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const {
    TString chatId;
    TString chatTopic;
    IChatRobot::ParseTopicLink(GetTopicLink(server), chatId, chatTopic);

    if (chatId) {
        IChatRobot::TPtr chatRobot = server.GetChatRobot(chatId);
        if (!chatRobot) {
            if (chatSession) {
                chatSession->SetErrorInfo(GetName(), chatId + " chat robot not configured", EDriveSessionResult::InconsistencySystem);
            } else {
                ERROR_LOG << chatId << " chat robot not configured" << Endl;
            }
            return false;
        }

        NDrive::NChat::TMessage chatMessage;
        chatMessage.SetType(type);
        chatMessage.SetText(message);

        if (chatSession) {
            if (!chatRobot->SendArbitraryMessage(userId, chatTopic, robotId, chatMessage, *chatSession)) {
                return false;
            }
        } else {
            auto session = chatRobot->BuildChatEngineSession();
            if (!chatRobot->SendArbitraryMessage(userId, chatTopic, robotId, chatMessage, session) || !session.Commit()) {
                ERROR_LOG << "can't send message for " << userId << Endl;
                return false;
            }
        }
    }
    return true;
}

bool TUserProblemTag::MoveToStep(const TString& userId, const TString& stepName, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const {
    TString chatId;
    TString chatTopic;
    IChatRobot::ParseTopicLink(GetTopicLink(server), chatId, chatTopic);

    if (chatId) {
        IChatRobot::TPtr chatRobot = server.GetChatRobot(chatId);
        if (!chatRobot) {
            if (chatSession) {
                chatSession->SetErrorInfo(GetName(), chatId + " chat robot not configured", EDriveSessionResult::InconsistencySystem);
            } else {
                ERROR_LOG << chatId << " chat robot not configured" << Endl;
            }
            return false;
        }

        if (!chatRobot->MoveToStep(userId, chatTopic, stepName, chatSession, true, "", robotId)) {
            return false;
        }
    }
    return true;
}

const TString TUserProblemTag::TypeName = "user_problem_tag";
ITag::TFactory::TRegistrator<TUserProblemTag> TUserProblemTag::Registrator(TUserProblemTag::TypeName);
TUserProblemTag::TDescription::TFactory::TRegistrator<TUserProblemTag::TDescription> TUserProblemTag::TDescription::Registrator(TUserProblemTag::TypeName);

const TString TRegistrationUserTag::TypeName = "user_registration_tag";
ITag::TFactory::TRegistrator<TRegistrationUserTag> TRegistrationUserTag::Registrator(TRegistrationUserTag::TypeName);
TRegistrationUserTag::TDescription::TFactory::TRegistrator<TRegistrationUserTag::TDescription> TRegistrationUserTag::TDescription::Registrator(TRegistrationUserTag::TypeName);

bool TRegistrationUserTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto api = server ? server->GetDriveAPI() : nullptr;
    const auto& tagsMeta = Yensured(api)->GetTagsManager().GetTagsMeta();
    auto tagDescription = tagsMeta.GetDescriptionByName(GetName());
    auto description = tagDescription ? tagDescription->GetAs<TDescription>() : nullptr;
    if (!description) {
        session.SetErrorInfo("RegistrationUserTag::OnAfterAdd", "cannot get description");
        return false;
    }
    const auto& objectId = self.GetObjectId();
    const auto& status = description->GetStatus();
    const auto& usersData = *Yensured(api->GetUsersData());
    const auto& chatRobotId = ChatRobot ? ChatRobot : description->GetChatRobot();
    const auto& chatItemId = ChatItem ? ChatItem : description->GetChatItem();

    if (description->GetCheckActiveSession()) {
        TInstant actuality = Now();
        for (auto&& duplicateId : DuplicateIds) {
            TVector<ISession::TConstPtr> userSessions;
            if (server->GetDriveAPI()->GetCurrentUserSessions(duplicateId, userSessions, actuality) && userSessions.size() > 0) {
                const auto& activeSessionTagName = description->GetActiveSessionTagName();
                if (activeSessionTagName) {
                    auto tag = tagsMeta.CreateTagAs<TUserProblemTag>(activeSessionTagName, "Есть активная поездка");
                    if (!tag) {
                        session.SetErrorInfo("RegistrationUserTag::OnAfterAdd", "cannot create UserProblemTag " + activeSessionTagName);
                        return false;
                    }
                    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, objectId, server, session)) {
                        return false;
                    }
                }
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple({self.GetTagId()}, userId, session, true)) {
                    return false;
                }
                return true;
            }
        }
    }

    auto fetchResult = usersData.FetchInfo(objectId, session);
    if (!fetchResult) {
        return false;
    }
    auto userPtr = fetchResult.GetResultPtr(objectId);
    if (!userPtr) {
        session.SetErrorInfo("RegistrationUserTag::OnAfterAdd", "cannot get user data");
        return false;
    }
    auto user = *userPtr;

    if (description->GetTransferDataFromDuplicates()) {
        const auto& registrationManager = *Yensured(server->GetUserRegistrationManager());
        for (auto&& duplicateId : DuplicateIds) {
            auto dupFetchResult = usersData.FetchInfo(duplicateId, session);
            if (!dupFetchResult) {
                return false;
            }
            auto duplicate = dupFetchResult.GetResultPtr(duplicateId);
            if (!duplicate) {
                continue;
            }
            if (!registrationManager.MoveData(user, duplicate, userId, session, description->GetTransferTraits())) {
                return false;
            }
        }
    }

    if (chatItemId && chatRobotId) {
        auto chatEngine = server->GetChatEngine();
        auto chatRobot = server->GetChatRobot(chatRobotId);
        if (!chatRobot) {
            session.SetErrorInfo("chat_robot", "not_configured", EDriveSessionResult::InternalError);
            return false;
        }

        NDrive::TEntitySession* sessionPtr = nullptr;
        if (Yensured(api)->GetDatabaseName() == Yensured(chatEngine)->GetDatabaseName()) {
            sessionPtr = &session;
        }

        if (!chatRobot->MoveToStep(objectId, "", chatItemId, sessionPtr, true)) {
            session.SetErrorInfo("chat_robot", "can't set state", EDriveSessionResult::InternalError);
            return false;
        }
    }

    if (status) {
        user.UpdateStatus(status);
        auto updatedUser = usersData.UpdateUser(std::move(user), userId, session);
        if (!updatedUser) {
            return false;
        }
    }
    return true;
}

bool TRegistrationUserTag::OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const {
    Y_UNUSED(eContext);
    auto impl = std::dynamic_pointer_cast<TRegistrationUserTag>(toTag);
    if (impl) {
        auto self = fromTag;
        self.SetData(toTag);
        if (!OnAfterAdd(self, permissions.GetUserId(), server, tx)) {
            return false;
        }
    }
    return true;
}

void TRegistrationUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::InsertNonNull(value, "chat_item", ChatItem);
    NJson::InsertNonNull(value, "chat_robot", ChatRobot);
    NJson::InsertNonNull(value, "duplicate_ids", DuplicateIds);
}

bool TRegistrationUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return
        NJson::ParseField(value["chat_item"], ChatItem) &&
        NJson::ParseField(value["chat_robot"], ChatRobot) &&
        NJson::ParseField(value["duplicate_ids"], DuplicateIds) &&
        TBase::DoSpecialDataFromJson(value, errors);
}

TRegistrationUserTag::TProto TRegistrationUserTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TRegistrationUserTagData proto = TBase::DoSerializeSpecialDataToProto();
    if (ChatItem) {
        proto.SetChatItem(ChatItem);
    }
    if (ChatRobot) {
        proto.SetChatRobot(ChatRobot);
    }
    for (const auto& id : DuplicateIds) {
        proto.AddDuplicateIds(id);
    }
    return proto;
}

bool TRegistrationUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    ChatItem = proto.GetChatItem();
    ChatRobot = proto.GetChatRobot();
    for (const auto& id : proto.GetDuplicateIds()) {
        DuplicateIds.emplace(id);
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TRegistrationUserTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSBoolean>("no_cleanup", "Оставлять тег между повторными впусками людей");
    result.Add<TFSString>("registration_chat_item", "Состояние, в которое переводить чат");
    result.Add<TFSVariants>("registration_chat_id", "id чатового робота, в который отправлять сообщения").SetVariants(NContainer::Keys(server->GetChatRobots()));
    result.Add<TFSVariants>("status", "Статус, в который перевести пользователя").SetVariants({
        NDrive::UserStatusActive,
        NDrive::UserStatusBlocked,
        NDrive::UserStatusFastRegistered,
        NDrive::UserStatusOnboarding,
        NDrive::UserStatusRejected,
        NDrive::UserStatusScreening,
    });
    result.Add<TFSBoolean>("check_active_session", "Проверять наличие у дубликатов активных поездок");
    result.Add<TFSString>("active_session_tag_name", "Навешивать тег при наличии активной поездки");

    {
        auto tab = result.StartTabGuard("transfer");
        result.Add<TFSBoolean>("transfer_data_from_dup", "Переносить данные с дубликата").SetDefault(false);
        result.Add<TFSBoolean>("transfer_tags", "Переносить теги").SetDefault(true);
        result.Add<TFSBoolean>("transfer_roles", "Переносить роли").SetDefault(true);
        result.Add<TFSBoolean>("transfer_bonuses", "Переносить бонусы").SetDefault(true);
        result.Add<TFSBoolean>("transfer_first_ride", "Переносить факт первой поездки").SetDefault(true);

    }
    return result;
}

NJson::TJsonValue TRegistrationUserTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    NJson::InsertField(jsonMeta, "comment", Comment);
    NJson::InsertField(jsonMeta, "no_cleanup", NoCleanup);
    NJson::InsertField(jsonMeta, "registration_chat_item", ChatItem);
    NJson::InsertField(jsonMeta, "registration_chat_id", ChatRobot);
    NJson::InsertField(jsonMeta, "status", NJson::Stringify(Status));

    NJson::InsertField(jsonMeta, "check_active_session", CheckActiveSession);
    NJson::InsertField(jsonMeta, "active_session_tag_name", ActiveSessionTagName);

    NJson::InsertField(jsonMeta, "transfer_data_from_dup", TransferDataFromDuplicates);
    NJson::InsertField(jsonMeta, "transfer_tags", !!(TransferTraits & NDataTransfer::Tags));
    NJson::InsertField(jsonMeta, "transfer_roles", !!(TransferTraits & NDataTransfer::Roles));
    NJson::InsertField(jsonMeta, "transfer_bonuses", !!(TransferTraits & NDataTransfer::Bonuses));
    NJson::InsertField(jsonMeta, "transfer_first_ride", !!(TransferTraits & NDataTransfer::FirstRide));
    return jsonMeta;
}

bool TRegistrationUserTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    auto parseTransferTrait = [this](
            const NJson::TJsonValue& jsonMeta,
            const TString& fieldName,
            NDataTransfer::TDataTransferTraits trait) -> bool {
        bool present = false;
        if (!NJson::ParseField(jsonMeta, fieldName, present, false)) {
            return false;
        }
        if (present) {
            TransferTraits |= trait;
        } else  {
            TransferTraits &= ~trait;
        }
        return true;
    };
    return
        NJson::ParseField(jsonMeta, "comment", Comment) &&
        NJson::ParseField(jsonMeta, "no_cleanup", NoCleanup, false) &&
        NJson::ParseField(jsonMeta, "registration_chat_item", ChatItem) &&
        NJson::ParseField(jsonMeta, "registration_chat_id", ChatRobot) &&
        NJson::ParseField(jsonMeta, "check_active_session", CheckActiveSession, false) &&
        NJson::ParseField(jsonMeta, "active_session_tag_name", ActiveSessionTagName) &&
        NJson::ParseField(jsonMeta, "transfer_data_from_dup", TransferDataFromDuplicates, false) &&
        parseTransferTrait(jsonMeta, "transfer_tags", NDataTransfer::Tags) &&
        parseTransferTrait(jsonMeta, "transfer_roles", NDataTransfer::Roles) &&
        parseTransferTrait(jsonMeta, "transfer_bonuses", NDataTransfer::Bonuses) &&
        parseTransferTrait(jsonMeta, "transfer_first_ride", NDataTransfer::FirstRide) &&
        NJson::ParseField(jsonMeta["status"], Status);
}

const TString TPassportTrackTag::TypeName = "passport_track_tag";
ITag::TFactory::TRegistrator<TPassportTrackTag> TPassportTrackTag::Registrator(TPassportTrackTag::TypeName);


const TString TConnectionUserTag::TypeName = "user_connection_tag";
ITag::TFactory::TRegistrator<TConnectionUserTag> TConnectionUserTag::Registrator(TConnectionUserTag::TypeName);

const TString TReferralConnectionTag::TypeName = "referral_connection_tag";
ITag::TFactory::TRegistrator<TReferralConnectionTag> TReferralConnectionTag::Registrator(TReferralConnectionTag::TypeName);
TTagDescription::TFactory::TRegistrator<TReferralConnectionTagDescription> TReferralConnectionTag::RegistratorDescription(TReferralConnectionTag::TypeName);

bool TReferralConnectionTag::OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    const TDriveAPI* driveApi = server->GetDriveAPI();
    if (!driveApi) {
        session.SetErrorInfo(GetName(), "No api configured", EDriveSessionResult::InternalError);
        return false;
    }

    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        session.SetErrorInfo(GetName(), "Unknown type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TReferralConnectionTagDescription* descriptionImpl = dynamic_cast<const TReferralConnectionTagDescription*>(description.Get());
    if (!descriptionImpl) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (AvailableBalance < 0) {
        AvailableBalance = descriptionImpl->GetCashbackLimit();
    }
    return true;
}

void TReferralConnectionTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    JWRITE(json, "available_balance", AvailableBalance);
    TBase::SerializeSpecialDataToJson(json);
}

bool TReferralConnectionTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    JREAD_INT_OPT(json, "available_balance", AvailableBalance);
    return TBase::DoSpecialDataFromJson(json, errors);
}

TReferralConnectionTag::TProto TReferralConnectionTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TConnectionUserTagData proto = TBase::DoSerializeSpecialDataToProto();
    proto.MutableReferralData()->SetAvailableBalance(AvailableBalance);
    return proto;
}

bool TReferralConnectionTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    AvailableBalance = proto.GetReferralData().GetAvailableBalance();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TReferralConnectionTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme = TBase::GetScheme(server);
    scheme.Add<TFSNumeric>("available_balance", "Доступно для начисления").SetReadOnly(true);
    return scheme;
}

const TString TUserDocumentsReaskTag::TypeName = "documents_reask_tag";
ITag::TFactory::TRegistrator<TUserDocumentsReaskTag> TUserDocumentsReaskTag::Registrator(TUserDocumentsReaskTag::TypeName);

NDrive::TScheme TUserDocumentsReaskTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("ps", "Паспорт, селфи");
    result.Add<TFSBoolean>("pb", "Паспорт, страница с фотографией");
    result.Add<TFSBoolean>("pr", "Паспорт, регистрация");
    result.Add<TFSBoolean>("lf", "Права, лицевая");
    result.Add<TFSBoolean>("lb", "Права, оборот");
    result.Add<TFSBoolean>("ls", "Права, селфи");
    result.Add<TFSVariants>("chat_id", "Чат регистрации").SetVariants(NContainer::Keys(server->GetChatRobots())).SetEditable(true);
    NDrive::TScheme& resubmitOverrides = result.Add<TFSArray>("resubmit_overrides", "Сообщения перезапроса").SetElement<NDrive::TScheme>();
    resubmitOverrides.Add<TFSString>("localization", "Ключ локализации");
    resubmitOverrides.Add<TFSString>("node", "Группа нод");
    resubmitOverrides.Add<TFSVariants>("resubmit_item", "Тип документа").InitVariants<NUserDocument::EType>();
    return result;
}

void TUserDocumentsReaskTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    NJson::InsertField(json, "lb", LicenseBack);
    NJson::InsertField(json, "lf", LicenseFront);
    NJson::InsertField(json, "ls", LicenseSelfie);
    NJson::InsertField(json, "pb", PassportBiographical);
    NJson::InsertField(json, "pr", PassportRegistration);
    NJson::InsertField(json, "ps", PassportSelfie);
    NJson::InsertField(json, "chat_id", ChatId);
    NJson::InsertField(json, "resubmit_overrides", ResubmitOverrides);
    TBase::SerializeSpecialDataToJson(json);
}

bool TUserDocumentsReaskTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return NJson::ParseField(json["pb"], PassportBiographical, false, errors) &&
        NJson::ParseField(json["pr"], PassportRegistration, false, errors) &&
        NJson::ParseField(json["ps"], PassportSelfie, false, errors) &&
        NJson::ParseField(json["lb"], LicenseBack, false, errors) &&
        NJson::ParseField(json["lf"], LicenseFront, false, errors) &&
        NJson::ParseField(json["ls"], LicenseSelfie, false, errors) &&
        NJson::ParseField(json["chat_id"], ChatId, false, errors) &&
        NJson::ParseField(json["resubmit_overrides"], ResubmitOverrides, false, errors) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TUserDocumentsReaskTag::TProto TUserDocumentsReaskTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TUserDocumentsReaskTagData proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetLicenseBack(LicenseBack);
    proto.SetLicenseFront(LicenseFront);
    proto.SetLicenseSelfie(LicenseSelfie);
    proto.SetPassportBiographical(PassportBiographical);
    proto.SetPassportRegistration(PassportRegistration);
    proto.SetPassportSelfie(PassportSelfie);
    proto.SetChatId(ChatId);
    TDocumentResubmitOverride::AddToProto(ResubmitOverrides, proto);
    return proto;
}

bool TUserDocumentsReaskTag::DoDeserializeSpecialDataFromProto(const TUserDocumentsReaskTag::TProto& proto) {
    LicenseBack = proto.GetLicenseBack();
    LicenseFront = proto.GetLicenseFront();
    LicenseSelfie = proto.GetLicenseSelfie();
    PassportBiographical = proto.GetPassportBiographical();
    PassportRegistration = proto.GetPassportRegistration();
    PassportSelfie = proto.GetPassportSelfie();
    ChatId = proto.GetChatId();
    auto optionalOverrides = TDocumentResubmitOverride::GetFromProto(proto);
    if (!optionalOverrides) {
        return false;
    }
    SetResubmitOverrides(std::move(*optionalOverrides));
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TUserDocumentsReaskTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!server->GetUserRegistrationManager()) {
        session.SetErrorInfo("registration_manager", "not_cofigured", EDriveSessionResult::InternalError);
        return false;
    }

    auto tagImpl = self.GetTagAs<TUserDocumentsReaskTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag_impl", "null", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    ui32 mask = tagImpl->GetReaskMask();
    if (!mask) {
        session.SetErrorInfo("user documents chat reask", "no documents reasked", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    if (!server->GetUserRegistrationManager()->ReaskDocumentPhotos(self.GetObjectId(), mask, session, userId, ChatId, ResubmitOverrides)) {
        session.AddErrorMessage("user documents chat reask", "failed or not yet allowed");
        return false;
    }

    return server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple({self.GetTagId()}, "robot-frontend-explicit", session, true);
}

ui32 TUserDocumentsReaskTag::GetReaskMask() const {
    ui32 mask = 0;
    if (LicenseBack) {
        mask |= NUserDocument::EType::LicenseBack;
    }
    if (LicenseFront) {
        mask |= NUserDocument::EType::LicenseFront;
    }
    if (LicenseSelfie) {
        mask |= NUserDocument::EType::LicenseSelfie;
    }
    if (PassportBiographical) {
        mask |= NUserDocument::EType::PassportBiographical;
    }
    if (PassportRegistration) {
        mask |= NUserDocument::EType::PassportRegistration;
    }
    if (PassportSelfie) {
        mask |= NUserDocument::EType::PassportSelfie;
    }
    return mask;
}

const TString TUserRobotCallTag::TypeName = "user_robot_call_tag";
ITag::TFactory::TRegistrator<TUserRobotCallTag> TUserRobotCallTag::Registrator(TUserRobotCallTag::TypeName);
TUserRobotCallTag::TDescription::TFactory::TRegistrator<TUserRobotCallTag::TDescription> TUserRobotCallTag::TDescription::Registrator(TUserRobotCallTag::TypeName);

bool TUserRobotCallTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& voiceChoices = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName())->GetAs<TDescription>()->GetVoices();
    if (!voiceChoices.size()) {
        session.SetErrorInfo("voices", "empty", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    const auto& voice = voiceChoices[rand() % voiceChoices.size()];

    auto userFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(self.GetObjectId(), session);
    if (!userFR) {
        return false;
    }
    auto userPtr = userFR.GetResultPtr(self.GetObjectId());
    if (!userPtr) {
        session.SetErrorInfo("user", "not_found", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    if (!userPtr->GetPhone()) {
        session.SetErrorInfo("user", "no_phone_number", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    TString sessionId;
    if (!server->GetDriveAPI()->GetOctopusClient() || !server->GetDriveAPI()->GetOctopusClient()->Call(userPtr->GetPhone(), voice.HelloPrompt, voice.GoodbyePrompt, sessionId)) {
        session.SetErrorInfo("call", "failed", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    TDBTag selfCopy = self.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    if (!selfCopy) {
        session.SetErrorInfo("tag", "incorrect deep copy", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto tagImpl = selfCopy.MutableTagAs<TUserRobotCallTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag", "incorrect_type", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    tagImpl->SetCallSessionId(sessionId);

    return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(selfCopy, userId, session);
}

const TString TExternalPromoTag::TypeName = "external_promo_tag";
ITag::TFactory::TRegistrator<TExternalPromoTag> TExternalPromoTag::Registrator(TExternalPromoTag::TypeName);
TExternalPromoTag::TDescription::TFactory::TRegistrator<TExternalPromoTag::TDescription> TExternalPromoTag::TDescription::Registrator(TExternalPromoTag::TypeName);

bool TExternalPromoTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto& tagsMeta = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto description = tagsMeta.GetDescriptionByName(GetName());
    if (!description) {
        session.SetErrorInfo(GetName(), "Unknown type", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    const TExternalPromoTag::TDescription* descriptionImpl = description->GetAs<TExternalPromoTag::TDescription>();
    if (!descriptionImpl) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (!descriptionImpl->GetPromoTable()) {
        session.SetErrorInfo(GetName(), "Promo table is undefined", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    auto promoTable = session->GetDatabase().GetTable(descriptionImpl->GetPromoTable());
    if (!promoTable) {
        session.SetErrorInfo(GetName(), "Incorrect promo table", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    TString selectCondition = "type='" + descriptionImpl->GetType() + "' AND reserved=FALSE";
    TString condition = "id in (SELECT id FROM " + descriptionImpl->GetPromoTable() + " WHERE "+ selectCondition + " LIMIT 1 FOR UPDATE)";

    auto transaction = session.GetTransaction();
    TRecordsSet recordSet;
    TQueryResultPtr queryResult = promoTable->UpdateRow(condition, "reserved=TRUE", transaction, &recordSet);
    if (queryResult->IsSucceed()) {
        if (queryResult->GetAffectedRows() != 1 || recordSet.size() != 1) {
            session.SetErrorInfo("select promo code", "can't reserve " + descriptionImpl->GetType() + " promo code", EDriveSessionResult::ResourceLocked);
            return false;
        }
    } else {
        session.SetErrorInfo("select promo code", "UpdateRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    TString code = recordSet.begin()->Get("code");
    if (!code) {
        session.SetErrorInfo("select promo code", "incorrect promo code", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    TDBTag selfCopy = self.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    if (!selfCopy) {
        session.SetErrorInfo("tag", "incorrect deep copy", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto tagImpl = selfCopy.MutableTagAs<TExternalPromoTag>();
    if (!tagImpl) {
        session.SetErrorInfo("tag", "incorrect_type", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    tagImpl->SetCode(code);
    tagImpl->SetDeeplink(recordSet.begin()->Get("deeplink"));

    return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(selfCopy, userId, session);
}

bool TIntroViewInfo::IncrementViews(const TDuration intervalIgnore) {
    if (ModelingNow() - Last >= intervalIgnore) {
        ++Views;
        Last = ModelingNow();
        return true;
    } else {
        return false;
    }
}

const TString TUserChatShowTag::TypeName = "user_chat_show_tag";
ITag::TFactory::TRegistrator<TUserChatShowTag> TUserChatShowTag::Registrator(TUserChatShowTag::TypeName);

const TString TUserChatNotificationTag::TypeName = "user_chat_notification";
ITag::TFactory::TRegistrator<TUserChatNotificationTag> TUserChatNotificationTag::Registrator(TUserChatNotificationTag::TypeName);

const TString TUserIsolationPassTag::TypeName = "isolation_pass_tag";
ITag::TFactory::TRegistrator<TUserIsolationPassTag> TUserIsolationPassTag::Registrator(TUserIsolationPassTag::TypeName);

TUserIsolationPassTag::TUserIsolationPassTag(const TCovidPassClient::TPassData& passData, const EPassApiType& apiType, const TInstant validationTime, const TString& carId, const TString& chatId)
    : ApiResponse(passData.GetApiResponse())
    , ApiType(apiType)
    , Valid(passData.GetIsValid())
    , Token(passData.GetPassToken())
    , ValidUntil(passData.GetValidUntil())
    , ValidationTime(validationTime)
    , CarId(carId)
    , ChatId(chatId)
{}

NDrive::TScheme TUserIsolationPassTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSJson>("api_response", "Ответ оператора").SetReadOnly(true);
    result.Add<TFSVariants>("api_type", "Оператор").SetReadOnly(true);
    result.Add<TFSBoolean>("is_valid", "Валиден").SetDefault(false);
    result.Add<TFSString>("pass_string", "Токен").SetRequired(true);
    result.Add<TFSNumeric>("valid_until", "Срок действия").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSNumeric>("validation_timestamp", "Время проверки").SetVisual(TFSNumeric::EVisualType::DateTime).SetDefault(ModelingNow().Seconds());
    result.Add<TFSString>("car_id", "Идентификатор автомобиля").SetRequired(true);
    result.Add<TFSString>("chat_id", "Идентификатор чата получения пропуска").SetRequired(true);
    return result;
}

void TUserIsolationPassTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "api_response", ApiResponse);
    JWRITE_ENUM(json, "api_type", ApiType);
    JWRITE(json, "is_valid", Valid);
    JWRITE(json, "pass_string", Token);
    JWRITE_INSTANT(json, "valid_until", ValidUntil);
    JWRITE_INSTANT(json, "validation_timestamp", ValidationTime);
    JWRITE(json, "car_id", CarId);
    JWRITE(json, "chat_id", ChatId);
}

bool TUserIsolationPassTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (json.Has("api_response")) {
        ApiResponse = json["api_response"];
    }
    JREAD_FROM_STRING_OPT(json, "api_type", ApiType);
    JREAD_BOOL(json, "is_valid", Valid);
    JREAD_STRING(json, "pass_string", Token);
    JREAD_INSTANT_OPT(json, "valid_until", ValidUntil);
    JREAD_INSTANT_OPT(json, "validation_timestamp", ValidationTime);
    JREAD_STRING(json, "car_id", CarId);
    JREAD_STRING_OPT(json, "chat_id", ChatId);
    return TBase::DoSpecialDataFromJson(json, errors);
}

TUserIsolationPassTag::TProto TUserIsolationPassTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TUserIsolationPassTag proto = TBase::DoSerializeSpecialDataToProto();

    proto.SetApiResponse(ApiResponse.GetStringRobust());
    proto.SetApiType(::ToString(ApiType));
    proto.SetValidFlag(Valid);
    proto.SetToken(Token);
    proto.SetValidUntil(ValidUntil.Seconds());
    proto.SetValidationTime(ValidationTime.Seconds());
    proto.SetCarId(CarId);
    proto.SetChatId(ChatId);
    return proto;
}

bool TUserIsolationPassTag::DoDeserializeSpecialDataFromProto(const TUserIsolationPassTag::TProto& proto) {
    if (!ReadJsonFastTree(proto.GetApiResponse(), &ApiResponse)) {
        return false;
    }
    if (!TryFromString(proto.GetApiType(), ApiType)) {
        return false;
    }
    Valid = proto.GetValidFlag();
    Token = proto.GetToken();
    ValidUntil = TInstant::Seconds(proto.GetValidUntil());
    ValidationTime = TInstant::Seconds(proto.GetValidationTime());
    CarId = proto.GetCarId();
    ChatId = proto.GetChatId();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

TUserChatShow TUserChatShowTag::Get(const TString& userStatus) const {
    TUserChatShow result;
    result.Id = TopicLink;
    result.Deeplink = Deeplink;
    result.IsClosable = userStatus == NDrive::UserStatusActive ? true : IsClosable;
    result.ShowInRiding = ShowInRiding;
    return result;
}

NDrive::TScheme TUserChatShowTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("topic_link", "id диалога в чате");
    result.Add<TFSString>("deeplink", "Диплинк экрана для показа");
    result.Add<TFSBoolean>("is_closable", "Чат сворачиваемый?").SetDefault(true);
    result.Add<TFSBoolean>("show_in_riding", "Показывать в поездке").SetDefault(false);
    result.Add<TFSBoolean>("remove_on_view", "Снимать тег после просмотра чата").SetDefault(false);
    return result;
}

void TUserChatShowTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "topic_link", TopicLink);
    NJson::InsertField(json, "deeplink", Deeplink);
    NJson::InsertField(json, "is_closable", IsClosable);
    NJson::InsertField(json, "show_in_riding", ShowInRiding);
    NJson::InsertField(json, "remove_on_view", RemoveOnView);
}

bool TUserChatShowTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return
        NJson::ParseField(json, "topic_link", TopicLink) &&
        NJson::ParseField(json, "deeplink", Deeplink) &&
        NJson::ParseField(json, "is_closable", IsClosable, true) &&
        NJson::ParseField(json, "show_in_riding", ShowInRiding, true) &&
        NJson::ParseField(json, "remove_on_view", RemoveOnView, true) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TUserChatShowTag::TProto TUserChatShowTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetTopicLink(TopicLink);
    proto.SetDeeplink(Deeplink);
    proto.SetIsClosable(IsClosable);
    proto.SetShowInRiding(ShowInRiding);
    proto.SetRemoveOnView(RemoveOnView);
    return proto;
}

bool TUserChatShowTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    TopicLink = proto.GetTopicLink();
    Deeplink = proto.GetDeeplink();
    IsClosable = proto.GetIsClosable();
    ShowInRiding = proto.GetShowInRiding();
    RemoveOnView = proto.GetRemoveOnView();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

template <>
NJson::TJsonValue NJson::ToJson(const TUserChatShow& object) {
    NJson::TJsonValue result;
    result["id"] = object.Id;
    result["deeplink"] = object.Deeplink;
    result["is_closable"] = object.IsClosable;
    result["show_in_riding"] = object.ShowInRiding;
    return result;
}

bool TServiceTask::IsGarage() const {
    return TagId.StartsWith("garage");
}

template<>
NJson::TJsonValue NJson::ToJson(const TServiceTask& task) {
    NJson::TJsonValue result;
    result["tag_id"] = task.GetTagId();
    result["start_time"] = task.GetStartTime().Seconds();
    result["end_time"] = task.GetEndTime().Seconds();
    result["car_id"] = task.GetCarId();
    result["car_number"] = task.GetCarNumber();
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const TServiceRoute& route) {
    NJson::TJsonValue result;
    result["routing_task_id"] = route.GetRoutingTaskId();
    result["tasks"] = ToJson(route.GetTasks());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TServiceTask& task) {
    return NJson::ParseField(value["tag_id"], task.MutableTagId(), true)
        && NJson::ParseField(value["start_time"], task.MutableStartTime(), true)
        && NJson::ParseField(value["end_time"], task.MutableEndTime(), true)
        && NJson::ParseField(value["car_id"], task.MutableCarId())
        && NJson::ParseField(value["car_number"], task.MutableCarNumber());
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TServiceRoute& route) {
    return NJson::ParseField(value["tasks"], route.MutableTasks())
        && NJson::ParseField(value["routing_task_id"], route.MutableRoutingTaskId());
}

const TString TServiceRouteTag::TypeName = "service_route_tag";
const TString TServiceRouteTag::ServiceRouteTagName = "service_route_tag";
ITag::TFactory::TRegistrator<TServiceRouteTag> TServiceRouteTag::Registrator(TServiceRouteTag::TypeName);

NDrive::TScheme TServiceRouteTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    NDrive::TScheme route;
    route.Add<TFSString>("tasks", "Описание заданий");
    route.Add<TFSString>("routing_task_id", "id задачи Маршрутизации");
    result.Add<TFSStructure>("route", "Маршрут").SetStructure(route);
    return result;
}

void TServiceRouteTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json["route"] = NJson::ToJson(Route);
    json["route"]["tasks"] = json["route"]["tasks"].GetStringRobust();
}

bool TServiceRouteTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    NJson::TJsonValue routeJson = json["route"];
    bool successfulTasksParse = NJson::ReadJsonFastTree(json["route"]["tasks"].GetStringRobust(), &routeJson["tasks"]);
    return TBase::DoSpecialDataFromJson(json, errors)
        && successfulTasksParse
        && NJson::ParseField(routeJson, Route);
}

TServiceRouteTag::TProto TServiceRouteTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TServiceRouteTag proto = TBase::DoSerializeSpecialDataToProto();
    for (const auto& task : Route.GetTasks()) {
        NDrive::NProto::TServiceRouteTag::TServiceTask* protoInfo = proto.AddTask();
        protoInfo->SetTagId(task.GetTagId());
        protoInfo->SetStartTime(task.GetStartTime().Seconds());
        protoInfo->SetEndTime(task.GetEndTime().Seconds());
        protoInfo->SetCarId(task.GetCarId());
        protoInfo->SetCarNumber(task.GetCarNumber());
    }
    proto.SetRoutingTaskId(Route.GetRoutingTaskId());
    return proto;
}

bool TServiceRouteTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (const auto& protoTask : proto.GetTask()) {
        auto& task = Route.MutableTasks().emplace_back();
        task.SetTagId(protoTask.GetTagId());
        task.SetStartTime(TInstant::Seconds(protoTask.GetStartTime()));
        task.SetEndTime(TInstant::Seconds(protoTask.GetEndTime()));
        task.SetCarId(protoTask.GetCarId());
        task.SetCarNumber(protoTask.GetCarNumber());
    }
    Route.MutableRoutingTaskId() = proto.GetRoutingTaskId();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TServiceZonesTag::TypeName = "service_zones_tag";
const TString TServiceZonesTag::ServiceZoneAreaType = "service_zone";
const TString TServiceZonesTag::AvailableZonesTagNameSettings = "routing.available_zones_tag_name";
const TString TServiceZonesTag::DefaultAvailableZonesTagName = "tech_available_zones";
ITag::TFactory::TRegistrator<TServiceZonesTag> TServiceZonesTag::Registrator(TServiceZonesTag::TypeName);

NDrive::TScheme TServiceZonesTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    auto areasSet = server->GetDriveAPI()->GetAreasDB()->GetAreaIds(ServiceZoneAreaType);
    result.Add<TFSVariants>("available_zones", "Доступные зоны").SetMultiSelect(true).SetVariants(areasSet);
    result.Add<TFSVariants>("additional_tag_names", "Дополнительные теги").SetVariants(
        NContainer::Keys(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
    ).SetMultiSelect(true);
    return result;
}

void TServiceZonesTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json["available_zones"] = NJson::ToJson(AvailableZones);
    json["additional_tag_names"] = NJson::ToJson(AdditionalTagNames);
}

bool TServiceZonesTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return NJson::ParseField(json["available_zones"], AvailableZones)
        && NJson::ParseField(json["additional_tag_names"], AdditionalTagNames)
        && TBase::DoSpecialDataFromJson(json, errors);
}

TServiceZonesTag::TProto TServiceZonesTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TServiceZonesTag proto = TBase::DoSerializeSpecialDataToProto();
    for (const auto& zone : AvailableZones) {
        proto.AddAvailableZones(zone);
    }
    for (const auto& tagName : AdditionalTagNames) {
        proto.AddAdditionalTagNames(tagName);
    }
    return proto;
}

bool TServiceZonesTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (const auto& zone : proto.GetAvailableZones()) {
        AvailableZones.push_back(zone);
    }
    for (const auto& tagName : proto.GetAdditionalTagNames()) {
        AdditionalTagNames.push_back(tagName);
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TUserPhotoResendTag::TypeName = "user_photo_resend_tag";
ITag::TFactory::TRegistrator<TUserPhotoResendTag> TUserPhotoResendTag::Registrator(TUserPhotoResendTag::TypeName);

NDrive::TScheme TUserPhotoResendTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSArray>("photo_ids", "ID документов").SetElement<TFSString>();
    return result;
}

void TUserPhotoResendTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json["photo_ids"] = NJson::ToJson(PhotoIds);
}

bool TUserPhotoResendTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return
        NJson::ParseField(json["photo_ids"], PhotoIds) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TUserPhotoResendTag::TProto TUserPhotoResendTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TUserPhotoResendTag proto = TBase::DoSerializeSpecialDataToProto();
    for (const auto& id : PhotoIds) {
        proto.AddPhotoIds(id);
    }
    return proto;
}

bool TUserPhotoResendTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (const auto& id : proto.GetPhotoIds()) {
        PhotoIds.emplace(id);
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TSelfieVerificationResultTag::TypeName = "selfie_verification_result_tag";
ITag::TFactory::TRegistrator<TSelfieVerificationResultTag> TSelfieVerificationResultTag::Registrator(TSelfieVerificationResultTag::TypeName);
TSelfieVerificationResultTag::TDescription::TFactory::TRegistrator<TSelfieVerificationResultTag::TDescription> TSelfieVerificationResultTag::TDescription::Registrator(TSelfieVerificationResultTag::TypeName);

NDrive::TScheme TSelfieVerificationResultTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("face_on_selfie", "Найдено лицо на селфи").SetReadOnly(true);
    result.Add<TFSBoolean>("face_on_passport", "Найдено лицо на селфи с паспортом").SetReadOnly(true);
    result.Add<TFSNumeric>("similarity", "Схожесть лиц");
    return result;
}

void TSelfieVerificationResultTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "face_on_selfie", FoundFaceOnSelfie);
    NJson::InsertField(json, "face_on_passport", FoundFaceOnPassport);
    NJson::InsertField(json, "similarity", Similarity);
}

bool TSelfieVerificationResultTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (errors) {
        return
            NJson::ParseField(json, "face_on_selfie", FoundFaceOnSelfie, false, *errors) &&
            NJson::ParseField(json, "face_on_passport", FoundFaceOnPassport, false, *errors) &&
            NJson::ParseField(json, "similarity", Similarity, false, *errors) &&
            TBase::DoSpecialDataFromJson(json, errors);
    }
    return
        NJson::ParseField(json, "face_on_selfie", FoundFaceOnSelfie, false) &&
        NJson::ParseField(json, "face_on_passport", FoundFaceOnPassport, false) &&
        NJson::ParseField(json, "similarity", Similarity, false) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TSelfieVerificationResultTag::TProto TSelfieVerificationResultTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TSelfieVerificationResultTag proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetFoundFaceOnSelfie(FoundFaceOnSelfie);
    proto.SetFoundFaceOnPassport(FoundFaceOnPassport);
    proto.SetSimilarity(Similarity);
    return proto;
}

bool TSelfieVerificationResultTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    FoundFaceOnSelfie = proto.GetFoundFaceOnSelfie();
    FoundFaceOnPassport = proto.GetFoundFaceOnPassport();
    Similarity = proto.GetSimilarity();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TSelfieVerificationResultTag::IsEntryAllowed(const NDrive::IServer& server) const {
    auto baseDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    Y_ENSURE(baseDescription);
    auto description = baseDescription->GetAs<TDescription>();
    Y_ENSURE(description);
    if (FoundFaceOnSelfie && FoundFaceOnPassport) {
        return Similarity >= description->GetMinAllowedSimilarity();
    }
    return (FoundFaceOnSelfie || description->GetAllowOnBadSelfie()) && (FoundFaceOnPassport || description->GetAllowOnBadPassport());
}

const TString TUserSubscriptionTag::TypeName = "user_subscription_tag";
ITag::TFactory::TRegistrator<TUserSubscriptionTag> TUserSubscriptionTag::Registrator(TUserSubscriptionTag::TypeName);

void TUserSubscriptionTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "status", ::ToString(Status));
    NJson::InsertField(json, "auto_payment", AutoPayment);
    NJson::InsertField(json, "used_taxi_rides_count", UsedTaxiRidesCount);
    NJson::InsertField(json, "last_billing_operation", LastBillingOperation);
}

bool TUserSubscriptionTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return
        NJson::ParseField(json, "status", NJson::Stringify(Status), false) &&
        NJson::ParseField(json, "auto_payment", AutoPayment, false) &&
        NJson::ParseField(json, "used_taxi_rides_count", UsedTaxiRidesCount, false) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TUserSubscriptionTag::TProto TUserSubscriptionTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TUserSubscriptionTag proto = TBase::DoSerializeSpecialDataToProto();
    if (Status != TUserSubscriptionTag::ESubscriptionStatus::Undefined) {
        proto.SetStatus(static_cast<NDrive::NProto::TUserSubscriptionTag::ESubscriptionStatus>(Status));
    }
    proto.SetAutoPayment(AutoPayment);
    proto.SetUsedTaxiRidesCount(UsedTaxiRidesCount);
    proto.SetLastBillingOperation(LastBillingOperation);
    return proto;
}

bool TUserSubscriptionTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    AutoPayment = proto.GetAutoPayment();
    Status = static_cast<TUserSubscriptionTag::ESubscriptionStatus>(proto.GetStatus());
    UsedTaxiRidesCount = proto.GetUsedTaxiRidesCount();
    LastBillingOperation = proto.GetLastBillingOperation();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TUserSubscriptionTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("used_taxi_rides_count", "Количество использованных поездок на такси").SetDefault(0);
    result.Add<TFSBoolean>("auto_payment", "Автооплата").SetDefault(true);
    result.Add<TFSVariants>("status", "Статус абонемента").SetVariants({
        TUserSubscriptionTag::ESubscriptionStatus::Active,
        TUserSubscriptionTag::ESubscriptionStatus::Pending,
    });
    return result;
}

NDrive::TScheme TUserSubscriptionTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSJson>("description", "Описание тега");
    result.Add<TFSJson>("status", "Описание статуса");
    result.Add<TFSJson>("more_info", "Подробное описание абонемента");
    result.Add<TFSJson>("taxi_description", "Описание для поездок на такси");
    result.Add<TFSNumeric>("priority", "Приоритет подписки").SetDefault(0);
    {
        TSet<TString> rolesIds;
        auto roles = server->GetDriveAPI()->GetRolesManager()->GetRoles();
        for (const auto& role : roles) {
            rolesIds.insert(role.GetRoleId());
        }
        result.Add<TFSVariants>("role_ids", "Идентификаторы ролей с плюшками").SetVariants(rolesIds).SetMultiSelect(true);
    }
    auto tagNames = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TOperationTag::TypeName });
    result.Add<TFSVariants>("debit_tag", "Тег списания").SetVariants(tagNames);
    result.Add<TFSVariants>("credit_tag", "Тег начисления бонусов").SetVariants(tagNames);
    result.Add<TFSNumeric>("max_taxi_rides", "Количество разрешённых поездок на такси").SetDefault(0);
    result.Add<TFSDuration>("pending_duration", "Время ожидания платежа");

    auto tagsInPoint = server->GetDriveAPI()->GetAreasDB()->GetAreaTags(TInstant::Zero());
    result.Add<TFSVariants>("tags_in_point", "Теги зоны доступности такси", 100000).SetVariants(std::move(tagsInPoint)).MulVariantsLeft({"", "!"}).SetMultiSelect(true);
    result.Add<TFSBoolean>("is_autopayment_visible", "Разрешать включать автооплату").SetDefault(true);
    return result;
}

NJson::TJsonValue TUserSubscriptionTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "description", Description);
    NJson::InsertField(json, "status", Status);
    NJson::InsertField(json, "priority", Priority);
    NJson::InsertField(json, "role_ids", RoleIds);
    NJson::InsertField(json, "debit_tag", DebitTag);
    NJson::InsertField(json, "credit_tag", CreditTag);
    NJson::InsertField(json, "max_taxi_rides", MaxTaxiRides);
    NJson::InsertField(json, "tags_in_point", TagsInPoint);
    NJson::InsertField(json, "pending_duration", PendingDuration);
    NJson::InsertField(json, "more_info", MoreInfo);
    NJson::InsertField(json, "is_autopayment_visible", IsAutoPaymentVisible);
    return json;
}

bool TUserSubscriptionTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "description", Description) &&
        NJson::ParseField(json, "status", Status) &&
        NJson::ParseField(json, "priority", Priority, true) &&
        NJson::ParseField(json, "role_ids", RoleIds) &&
        NJson::ParseField(json, "debit_tag", DebitTag) &&
        NJson::ParseField(json, "credit_tag", CreditTag) &&
        NJson::ParseField(json, "max_taxi_rides", MaxTaxiRides, false) &&
        NJson::ParseField(json, "tags_in_point", TagsInPoint) &&
        NJson::ParseField(json, "pending_duration", PendingDuration) &&
        NJson::ParseField(json, "more_info", MoreInfo) &&
        NJson::ParseField(json, "is_autopayment_visible", IsAutoPaymentVisible, false) &&
        TBase::DoDeserializeMetaFromJson(json);
}

TTagDescription::TFactory::TRegistrator<TUserSubscriptionTag::TDescription> TUserSubscriptionTag::TDescription::Registrator(TUserSubscriptionTag::TypeName);

template <>
NJson::TJsonValue NJson::ToJson<TUserSubscriptionTag::TDescription::TBenefit>(const TUserSubscriptionTag::TDescription::TBenefit& benefit) {
    NJson::TJsonValue json;
    json.InsertValue("name", benefit.Name);
    json.InsertValue("description", benefit.Description);
    json.InsertValue("icon", benefit.Icon);
    json.InsertValue("type", benefit.Type);
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson<TUserSubscriptionTag::TDescription::TTaxiDescription>(const TUserSubscriptionTag::TDescription::TTaxiDescription& taxi) {
    NJson::TJsonValue json;
    json.InsertValue("description", taxi.Description);
    json.InsertValue("icon", taxi.Icon);
    json.InsertValue("bg_color", taxi.BgColor);
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson<TUserSubscriptionTag::TDescription::TMoreInfo>(const TUserSubscriptionTag::TDescription::TMoreInfo& info) {
    NJson::TJsonValue json;
    json.InsertValue("name", info.Name);
    json.InsertValue("link", info.Link);
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson<TUserSubscriptionTag::TDescription::TSubscriptionDescription>(const TUserSubscriptionTag::TDescription::TSubscriptionDescription& description) {
    NJson::TJsonValue json;
    json.InsertValue("name", description.Name);
    json.InsertValue("description", description.Description);
    json.InsertValue("price", description.Price);
    json.InsertValue("bg_color", description.BgColor);

    NJson::TJsonValue jsonBenefits;
    for (const auto& benefit :  description.Benefits) {
        jsonBenefits.AppendValue(ToJson(benefit));
    }
    json.InsertValue("details", jsonBenefits);
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson<TUserSubscriptionTag::TDescription::TStatus>(const TUserSubscriptionTag::TDescription::TStatus& status) {
    NJson::TJsonValue json;
    json.InsertValue("name", status.Name);
    json.InsertValue("details", status.Details);
    json.InsertValue("bg_color", status.BgColor);
    json.InsertValue("color", status.Color);
    return json;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserSubscriptionTag::TDescription::TBenefit& benefit) {
    return NJson::ParseField(value["name"], benefit.Name) &&
        NJson::ParseField(value["description"], benefit.Description) &&
        NJson::ParseField(value["icon"], benefit.Icon) &&
        NJson::ParseField(value["type"], benefit.Type);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserSubscriptionTag::TDescription::TTaxiDescription& taxi) {
    return NJson::ParseField(value["description"], taxi.Description) &&
        NJson::ParseField(value["icon"], taxi.Icon) &&
        NJson::ParseField(value["bg_color"], taxi.BgColor);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserSubscriptionTag::TDescription::TMoreInfo& info) {
    return NJson::ParseField(value["name"], info.Name) &&
        NJson::ParseField(value["link"], info.Link);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserSubscriptionTag::TDescription::TSubscriptionDescription& description) {
    return  NJson::ParseField(value["name"], description.Name) &&
        NJson::ParseField(value["description"], description.Description) &&
        NJson::ParseField(value["details"], description.Benefits) &&
        NJson::ParseField(value["price"], description.Price) &&
        NJson::ParseField(value["bg_color"], description.BgColor);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserSubscriptionTag::TDescription::TStatus& status) {
    return  NJson::ParseField(value["name"], status.Name) &&
        NJson::ParseField(value["details"], status.Details) &&
        NJson::ParseField(value["bg_color"], status.BgColor) &&
        NJson::ParseField(value["color"], status.Color);
}

TDBActions TUserSubscriptionTag::GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool /*getPotential*/) const {
    if (Status != TUserSubscriptionTag::ESubscriptionStatus::Active) {
        return {};
    }

    auto description = tagsManager.GetTagsMeta().GetDescriptionByName(GetName());
    auto descriptionImpl = description->GetAs<TUserSubscriptionTag::TDescription>();
    if (!descriptionImpl) {
        return {};
    }

    TDBActions result;
    auto actionIds = rolesManager.GetRolesInfoDB().GetActions(descriptionImpl->GetRoleIds());
    TMap<TString, TDBAction> additionalActions = rolesManager.GetActions(actionIds);
    for (auto&& action : additionalActions) {
        auto impl = action.second->Clone();
        auto& sourceContext = Yensured(impl)->MutableSourceContext();
        sourceContext.SetTagId(self.GetTagId());
        if (Yensured(self.GetData())->HasSLAInstant()) {
            sourceContext.SetUntil(self.GetData()->GetSLAInstant());
        }
        result.push_back(std::move(impl));
    }
    return result;
}

TString TUserSubscriptionTag::GetUniqueTaxiRideToken(const TConstDBTag& self) const {
    auto token = TStringBuilder() << self.GetTagId() << "-" << UsedTaxiRidesCount + 1 << "-" << GetSLAInstant().Seconds();
    return token;
}

struct TAddTagContext {
    TString TagName;
    TString SubscriptionId;
    ui32 Amount;
    TString HistoryUserId;
    TString UserId;
    TString Source;
};

TMaybe<TString> AddOperationTag(const TAddTagContext& context, const NDrive::IServer* server, NDrive::TEntitySession& tx, TInstant deadline = TInstant::Zero()) {
    const auto& tagsManager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    auto operationTag = tagsManager.GetTagsMeta().CreateTag(context.TagName);
    if (!operationTag) {
        tx.SetErrorInfo(context.Source, "fail to create tag (" + context.TagName + ") for: " + context.SubscriptionId);
        return {};
    }
    auto newTag = std::dynamic_pointer_cast<TOperationTag>(operationTag);
    if (!newTag) {
        tx.SetErrorInfo(context.Source, "fail to cast tag " + context.TagName + " to TOperationTag for: " + context.SubscriptionId);
        return {};
    }
    newTag->SetAmount(context.Amount);
    if (deadline) {
        newTag->SetAmountDeadline(deadline);
    }
    auto resultDbTag = tagsManager.GetUserTags().AddTag(operationTag, context.HistoryUserId, context.UserId, server, tx);
    if (!resultDbTag) {
        return {};
    }
    if (resultDbTag->empty()) {
        tx.SetErrorInfo(context.Source, "fail to add tag (" + context.TagName + "), empty result for:" + context.SubscriptionId);
        return {};
    }
    return resultDbTag->front().GetTagId();
}

bool CheckBillOperationActuality(
    const TString& source,
    const IDriveTagsManager& tagsManager,
    const TString& billingOperation,
    const TString& subscriptionId,
    NDrive::TEntitySession& tx
) {
    const auto dbTags = tagsManager.GetUserTags().RestoreTags({ subscriptionId }, tx);
    if (!dbTags) {
        return false;
    }
    if (dbTags->empty() || !dbTags->front().Is<TUserSubscriptionTag>()) {
        tx.SetErrorInfo(source, "consistency error in tag: " + subscriptionId);
        return false;
    }
    if (billingOperation != dbTags->front().GetTagAs<TUserSubscriptionTag>()->GetLastBillingOperation()) {
        tx.SetErrorInfo(source, "state is already changed in tag: " + subscriptionId);
        return false;
    }
    return true;
}

TMaybe<ui64> GeLasActivationEventId(const TString& subscriptionId, const IDriveTagsManager& tagsManager, NDrive::TEntitySession& tx) {
    const auto lastEventIdFetchOptions = IEntityTagsManager::TQueryOptions(0, true).SetActions({EObjectHistoryAction::Add, EObjectHistoryAction::UpdateData});
    auto events = tagsManager.GetUserTags().GetEventsByTag(subscriptionId, tx, 0, {}, lastEventIdFetchOptions);
    if (!events) {
        return {};
    }
    TInstant lastSLA = TInstant::Zero();
    ui64 slaEventId = 0;
    for (auto&& event : *events) {
        if (event.GetHistoryAction() == EObjectHistoryAction::Add) {
            return event.GetHistoryEventId();
        }
        if (event.GetHistoryAction() == EObjectHistoryAction::UpdateData) {
            if (lastSLA && lastSLA != event->GetSLAInstant()) {
                return slaEventId;
            } else {
                lastSLA = event->GetSLAInstant();
                slaEventId = event.GetHistoryEventId();
            }
        }
    }
    return slaEventId;
}

TMaybe<TUserSubscriptionTag::EPaymentStatus> TUserSubscriptionTag::CheckPayment(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    auto tagImpl = self.MutableTagAs<TUserSubscriptionTag>();
    if (!tagImpl) {
        tx.SetErrorInfo("CheckPayment", "fail to get data from tag: " + self.GetTagId());
        return {};
    }
    if (!tagImpl->GetLastBillingOperation()) {
        return EPaymentStatus::Unknown;
    }
    auto lock = tx.TryLock(self.GetTagId());
    if (!lock || !*lock) {
        tx.SetErrorInfo("CheckPayment", "fail to lock tag: " + self.GetTagId());
        return {};
    }
    const auto& tagsManager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    if (!CheckBillOperationActuality("CheckPayment", tagsManager, tagImpl->GetLastBillingOperation(), self.GetTagId(), tx)) {
        return {};
    }
    return CheckPaymentImpl(self, userId, server, tx);
}

TMaybe<TUserSubscriptionTag::EPaymentStatus> TUserSubscriptionTag::CheckPaymentImpl(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    auto tagImpl = self.MutableTagAs<TUserSubscriptionTag>();
    if (!tagImpl) {
        tx.SetErrorInfo("CheckPaymentImpl", "fail to get data from tag: " + self.GetTagId());
        return {};
    }
    const auto& tagsManager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    const auto dbTags = tagsManager.GetUserTags().RestoreTags({ tagImpl->GetLastBillingOperation() }, tx);
    if (!dbTags) {
        return {};
    }
    if (!dbTags->empty()) {
        return EPaymentStatus::Processing;
    }

    TString debitTag;
    TString creditTag;
    ui32 amount = 0;
    {
        const auto selfDescription = Yensured(tagImpl->GetDescriptionAs<TDescription>(*server, tx));
        debitTag = selfDescription->GetDebitTag();
        creditTag = selfDescription->GetCreditTag();
        amount = selfDescription->GetDescription().Price;
    }

    const auto lastEventId = GeLasActivationEventId(self.GetTagId(), tagsManager, tx);
    if (!lastEventId) {
        return {};
    }

    EPaymentStatus result = EPaymentStatus::Unknown;
    TTagEventsManager::TQueryOptions options(0, true);
    options.SetTagIds({ tagImpl->GetLastBillingOperation() });
    options.SetActions({ EObjectHistoryAction::Remove });
    const auto events = tagsManager.GetUserTags().GetEventsByObject(self.GetObjectId(), tx, *lastEventId, {}, options);
    if (!events) {
        return {};
    }
    for (const auto& event : *events) {
        const auto tag = event.GetTagAs<TOperationTag>();
        if (!tag) {
            continue;
        }
        if (tag->GetResolution() != ToString(TBillingTag::EOperationResolution::Finished)) {
            return EPaymentStatus::Failed;
        }
        if (tag->GetName() == creditTag) {
            return EPaymentStatus::Completed;
        }
        if (tag->GetName() == debitTag) {
            TAddTagContext context;
            context.Source = "CheckPayment";
            context.TagName = creditTag;
            context.Amount = amount;
            context.UserId = self.GetObjectId();
            context.SubscriptionId = self.GetTagId();
            context.HistoryUserId = userId;
            const auto billingOperationId = tagImpl->HasSLAInstant()
                ? AddOperationTag(context, server, tx, tagImpl->GetSLAInstantUnsafe())
                : AddOperationTag(context, server, tx);
            if (!billingOperationId) {
                return {};
            }
            tagImpl->SetLastBillingOperation(*billingOperationId);
            if (!tagsManager.GetUserTags().UpdateTagData(self, userId, tx)) {
                return {};
            }
            return EPaymentStatus::Processing;
        }
    }
    return result;
}

bool TUserSubscriptionTag::ApplyPayment(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    auto lock = tx.TryLock(self.GetTagId());
    if (!lock || !*lock) {
        tx.SetErrorInfo("ApplyPayment", "fail to lock tag: " + self.GetTagId());
        return false;
    }
    const auto& tagsManager = Yensured(Yensured(server)->GetDriveAPI())->GetTagsManager();
    auto tagImpl = self.MutableTagAs<TUserSubscriptionTag>();
    if (!tagImpl) {
        tx.SetErrorInfo("ApplyPayment", "fail to get data from tag: " + self.GetTagId());
        return false;
    }
    if (!CheckBillOperationActuality("ApplyPayment", tagsManager, tagImpl->GetLastBillingOperation(), self.GetTagId(), tx)) {
        return false;
    }
    if (tagImpl->GetLastBillingOperation()) {
        auto checkResult = CheckPaymentImpl(self, userId, server, tx);
        if (!checkResult) {
            return false;
        }
        if (*checkResult == EPaymentStatus::Completed || *checkResult == EPaymentStatus::Processing) {
            return true;
        }
    }
    TAddTagContext context;
    context.Source = "ApplyPayment";
    const auto selfDescription = Yensured(tagImpl->GetDescriptionAs<TDescription>(*server, tx));
    context.TagName = selfDescription->GetDebitTag();
    context.Amount = selfDescription->GetDescription().Price;
    context.UserId = self.GetObjectId();
    context.SubscriptionId = self.GetTagId();
    context.HistoryUserId = userId;
    const auto billingOperationId = AddOperationTag(context, server, tx);
    if (!billingOperationId) {
        return false;
    }
    tagImpl->SetLastBillingOperation(*billingOperationId);
    return tagsManager.GetUserTags().UpdateTagData(self, userId, tx);
}

bool TUserSubscriptionTag::HasTaxiRides(const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    const auto& selfDescription = *Yensured(GetDescriptionAs<TUserSubscriptionTag::TDescription>(*server, tx));
    return UsedTaxiRidesCount < selfDescription.GetMaxTaxiRides();
}

bool TUserSubscriptionTag::IsTaxiAvailableByFilter(TMaybe<TGeoCoord> userLocation, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    if (!userLocation) {
        return false;
    }
    const auto& selfDescription = *Yensured(GetDescriptionAs<TUserSubscriptionTag::TDescription>(*Yensured(server), tx));
    auto expectedTags = selfDescription.GetTagsInPoint();
    if (!expectedTags.empty()) {
        auto tagsInPoint = Yensured(Yensured(server->GetDriveAPI())->GetAreasDB())->GetTagsInPoint(*userLocation);
        return TBaseAreaTagsFilter::FilterWeak(tagsInPoint, expectedTags);
    }
    return true;
}

const TString TTaxiPromocodeUserTag::TypeName = "taxi_promocode_user_tag";
ITag::TFactory::TRegistrator<TTaxiPromocodeUserTag> TTaxiPromocodeUserTag::Registrator(TTaxiPromocodeUserTag::TypeName);

void TTaxiPromocodeUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "promocode", Promocode);
}

bool TTaxiPromocodeUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return
        NJson::ParseField(json, "promocode", Promocode, false) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TTaxiPromocodeUserTag::TProto TTaxiPromocodeUserTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TTaxiPromocodeUserTag proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetPromocode(Promocode);
    return proto;
}

bool TTaxiPromocodeUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    Promocode = proto.GetPromocode();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TTaxiPromocodeUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("promocode", "Промокод для поездки на такси");
    return result;
}

bool TTaxiPromocodeUserTag::GeneratePromocode(const NDrive::IServer* server, const  NDrive::TTaxiPromocodesClient::TRequestData& data) {
    auto taxiClient = Yensured(server->GetTaxiPromocodesClient());
    auto result = taxiClient->GeneratePromocode(data);
    if (!result.Initialized()) {
        NDrive::TEventLog::Log("GeneratePromocodeError", NJson::TMapBuilder
            ("error", "cannot generate taxi promocode. Response is not initialized!")
            ("request_data", data.GetGenerateRequestData().GetStringRobust())
        );
        return false;
    }
    if (!result.Wait(taxiClient->GetTimeout()) || !result.HasValue()) {
        NDrive::TEventLog::Log("GeneratePromocodeError", NJson::TMapBuilder
            ("error", "cannot generate taxi promocode")
            ("request_data", data.GetGenerateRequestData().GetStringRobust())
            ("exception_info", NThreading::GetExceptionInfo(result).GetStringRobust())
        );
        return false;
    }
    Promocode = result.GetValue();
    return true;
}

TString TTaxiPromocodeUserTag::GetDeepLink() const {
    if (!Promocode) {
        return "";
    }
    auto link = TStringBuilder() << "yandextaxi://route?&coupon=" << Promocode;
    return link;
}

TTagDescription::TFactory::TRegistrator<TTaxiPromocodeUserTag::TDescription> TTaxiPromocodeUserTag::TDescription::Registrator(TTaxiPromocodeUserTag::TypeName);

NDrive::TScheme TTaxiPromocodeUserTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("series_id", "Id серии промокода").SetRequired(true);
    return result;
}

NJson::TJsonValue TTaxiPromocodeUserTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "series_id", SeriesId);
    return json;
}

bool TTaxiPromocodeUserTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "series_id", SeriesId, false) &&
        TBase::DoDeserializeMetaFromJson(json);
}
