#pragma once

#include "notifications_abstract.h"
#include "permission_tags.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/common/temporary_tags.h>
#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/chat/message.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/registrar/ifaces.h>

#include <drive/library/cpp/covid_pass/client.h>
#include <drive/library/cpp/octopus/client.h>
#include <drive/library/cpp/taxi/promocodes/client.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/string/vector.h>

enum class EPassApiType {
    Moscow /* "moscow" */,
    MoscowRegion /* "moscow_region" */,
    Gosuslugi /* "gosuslugi" */,
    Undefined /* "undefined" */,
};

class TAdditionalRolesUserTag: public TAdditionalRolesTag {
private:
    using TBase = TAdditionalRolesTag;

public:
    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

public:
    static const TString TypeName;

private:
    static TFactory::TRegistrator<TAdditionalRolesUserTag> Registrator;
};

class TIntroViewInfo {
private:
    R_FIELD(TString, LandingId);
    R_FIELD(ui32, Views, 0);
    R_FIELD(TInstant, Last, TInstant::Zero());

public:
    bool IncrementViews(const TDuration intervalIgnore);

    NDrive::NProto::TIntroViewInfo SerializeToProto() const {
        NDrive::NProto::TIntroViewInfo result;
        result.SetLandingId(LandingId);
        result.SetViews(Views);
        result.SetLast(Last.Seconds());
        return result;
    }

    bool DeserializeFromProto(const NDrive::NProto::TIntroViewInfo& info) {
        LandingId = info.GetLandingId();
        Views = info.GetViews();
        Last = TInstant::Seconds(info.GetLast());
        return true;
    }

    static NDrive::TScheme GetScheme() {
        NDrive::TScheme result;
        result.Add<TFSNumeric>("Views").SetMin(0);
        result.Add<TFSNumeric>("Last").SetMin(0);
        result.Add<TFSString>("LandingId");
        return result;
    }
};

class TIntroViewsInfoTag: public INativeSerializationTag<NDrive::NProto::TIntroViewsInfoTag> {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TIntroViewsInfoTag>;

private:
    static TFactory::TRegistrator<TIntroViewsInfoTag> Registrator;

public:
    static const TString TypeName;

private:
    TMap<TString, TIntroViewInfo> Views;

public:
    using TBase::TBase;
    bool IncrementViews(const TString& landingId, const TDuration intervalIgnore) {
        auto it = Views.find(landingId);
        if (it == Views.end()) {
            TIntroViewInfo newInfo;
            newInfo.SetLandingId(landingId);
            newInfo.IncrementViews(intervalIgnore);
            Views.emplace(landingId, std::move(newInfo));
            return true;
        } else {
            return it->second.IncrementViews(intervalIgnore);
        }
    }

    TMaybe<TIntroViewInfo> GetInfoByLanding(const TString& landingId) const {
        auto it = Views.find(landingId);
        if (it == Views.end()) {
            return {};
        } else {
            return it->second;
        }
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSArray>("Views").SetElement(TIntroViewInfo::GetScheme());
        return result;
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        for (auto&& i : Views) {
            *proto.AddViews() = i.second.SerializeToProto();
        }
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        for (auto&& i : proto.GetViews()) {
            TIntroViewInfo info;
            if (!info.DeserializeFromProto(i)) {
                return false;
            }
            Views.emplace(info.GetLandingId(), std::move(info));
        }
        return true;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TTagDescription::TPtr GetMetaDescription(const TString& type) const override {
        TTagDescription::TPtr result = TBase::GetMetaDescription(type);
        if (result) {
            result->SetGrouppingTags({"user"});
        }
        return result;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }
};

class TTemporaryActionTag: public ITemporaryActionTag {
private:
    using TBase = ITemporaryActionTag;

private:
    static TFactory::TRegistrator<TTemporaryActionTag> Registrator;
    static ITemporaryActionTag::TDescription::TFactory::TRegistrator<ITemporaryActionTag::TDescription> DescriptionRegistrator;

public:
    static const TString TypeName;

public:
    using TBase::TBase;

public:
    virtual NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::User;
    }
};

class TSimpleUserTag: public ISerializableTag<NDrive::NProto::TCommonTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCommonTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TSimpleUserTag> Registrator;

public:
    TSimpleUserTag() = default;

    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }
};

class TUniqueUserTag: public TSimpleUserTag {
private:
    using TBase = TSimpleUserTag;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUniqueUserTag> Registrator;

public:
    using TBase::TBase;

protected:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }
};

class TRewriteUserTag: public TSimpleUserTag {
public:
    static const TString TypeName;
    static TFactory::TRegistrator<TRewriteUserTag> Registrator;

protected:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }
};

class TAdjustUserTag: public ISerializableTag<NDrive::NProto::TTimestampTagData> {
private:
    R_READONLY(TInstant, Timestamp);

private:
    using TBase = ISerializableTag<NDrive::NProto::TTimestampTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TAdjustUserTag> Registrator;

    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(TString, Token);
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSString>("token").SetRequired(true);
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
            JWRITE(jsonMeta, "token", Token);
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_STRING(jsonMeta, "token", Token);
            return true;
        }
    };

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("event_time", "Время события").SetVisual(TFSNumeric::EVisualType::DateTime);
        return result;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetTimestamp(Timestamp.Seconds());
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (proto.HasTimestamp()) {
            Timestamp = TInstant::Seconds(proto.GetTimestamp());
        }
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        TJsonProcessor::WriteInstant(json, "event_time", Timestamp, TInstant::Zero());
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_INSTANT_OPT(json, "event_time", Timestamp);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    bool OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) override {
        if (Timestamp == TInstant::Zero()) {
            Timestamp = ModelingNow();
        }
        return true;
    }
};

class TStaffHierarchyTag: public INativeSerializationTag<NDrive::NProto::TStaffHierarchyTag> {
private:
    R_FIELD(i32, Rank, 0);

private:
    static TFactory::TRegistrator<TStaffHierarchyTag> Registrator;

private:
    using TBase = INativeSerializationTag<NDrive::NProto::TStaffHierarchyTag>;

public:
    static const TString TypeName;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(TString, Department);
        R_FIELD(TString, HeadDepartment);
        R_FIELD(TString, DisplayName);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSString>("display_name").SetRequired(true);
            result.Add<TFSString>("department_name", "Идентификатор подразделения").SetRequired(true);
            result.Add<TFSString>("head_department_name", "идентификатор головного подразделения");
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
            JWRITE(jsonMeta, "display_name", DisplayName);
            JWRITE(jsonMeta, "department_name", Department);
            JWRITE_DEF(jsonMeta, "head_department_name", HeadDepartment, "");
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_STRING_OPT(jsonMeta, "display_name", DisplayName);
            JREAD_STRING(jsonMeta, "department_name", Department);
            JREAD_STRING_OPT(jsonMeta, "head_department_name", HeadDepartment);
            return true;
        }
    };

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("Rank", "Ранг в подразделении (чем больше тем выше положение)").SetDefault(0);
        return result;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TStaffHierarchyTag proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetRank(Rank);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        Rank = proto.GetRank();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }
};

class TUserProblemTag: public ISerializableTag<NDrive::NProto::TUserProblemTagData> {
public:
    enum class ELinkType {
        Startrek /* "st" */,
        Photo /* "photo" */,
        Video /* "video" */,
        Other /* "other" */,
    };

    enum class EBlockedStatus {
        Blocked /* "blocked" */,
        Ok /* "ok" */,
    };

    class TLink {
    private:
        R_FIELD(TString, Uri);
        R_FIELD(ELinkType, Type, ELinkType::Other);

    public:
        static NDrive::TScheme GetScheme() {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("uri");
            scheme.Add<TFSVariants>("type").SetMultiSelect(false).InitVariants<ELinkType>().SetDefault(::ToString(ELinkType::Other));
            return scheme;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& info) {
            JREAD_STRING_OPT(info, "uri", Uri);
            JREAD_FROM_STRING_OPT(info, "type", Type);
            return true;
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result = NJson::JSON_NULL;
            JWRITE_DEF(result, "uri", Uri, "");
            JWRITE_ENUM(result, "type", Type);
            return result;
        }
    };

private:
    using TBase = ISerializableTag<NDrive::NProto::TUserProblemTagData>;

private:
    R_FIELD(TVector<TLink>, Links);
    R_FIELD(TString, SessionId);
    R_FIELD(TString, CarNumber);
    R_FIELD(TString, ChatTopic);

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUserProblemTag> Registrator;

public:
    static TMaybe<TUserProblemTag::EBlockedStatus> ShouldBeBlocked(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session);
    static bool EnsureNotBlocked(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session);

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(TString, PushText);
        R_FIELD(bool, Bonus, false);
        R_FIELD(bool, Fees, false);
        R_FIELD(i32, Point, 0);

        R_FIELD(TString, ChatMessagesGroup);
        R_FIELD(ui32, ChatMessagesGroupPriority, 0);

        R_FIELD(bool, IsNeutral, false);
        R_FIELD(bool, DropOnReentry, false);
        R_FIELD(bool, InstantBan, false);
        R_FIELD(TDuration, BanDuration, TDuration::Zero());

        R_FIELD(TString, BanChat);
        R_FIELD(TString, BanDeeplink);

        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSString>("push_text", "Текст оповещающего пуша");
            result.Add<TFSBoolean>("bonus", "Начисление бонусов").SetDefault(false);
            result.Add<TFSBoolean>("fees", "Начисление штрафа").SetDefault(false);
            result.Add<TFSNumeric>("point", "Балл");

            result.Add<TFSString>("chat_messages_group", "Группа сообщений в чате");
            result.Add<TFSNumeric>("chat_messages_group_priority", "Приоритет группы сообщений в чате для выбора нужной");

            result.Add<TFSBoolean>("is_neutral", "Не учитывать при подсчете скора для профилей-клонов");
            result.Add<TFSBoolean>("drop_on_reentry", "Снимать при повторном впуске в сервис");
            result.Add<TFSBoolean>("instant_ban", "Банить не дожидаясь конца аренды");
            result.Add<TFSString>("ban_duration", "Длительность бана");

            result.Add<TFSString>("ban_chat", "Чат бана");
            result.Add<TFSString>("ban_deeplink", "Диплинк бана");

            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
            NJson::InsertNonNull(jsonMeta, "push_text", PushText);
            NJson::InsertNonNull(jsonMeta, "bonus", Bonus);
            NJson::InsertNonNull(jsonMeta, "fees", Fees);
            NJson::InsertNonNull(jsonMeta, "point", Point);
            if (ChatMessagesGroup) {
                NJson::InsertField(jsonMeta, "chat_messages_group", ChatMessagesGroup);
                NJson::InsertField(jsonMeta, "chat_messages_group_priority", ChatMessagesGroupPriority);
            }
            NJson::InsertField(jsonMeta, "is_neutral", IsNeutral);
            NJson::InsertField(jsonMeta, "drop_on_reentry", DropOnReentry);
            NJson::InsertField(jsonMeta, "instant_ban", InstantBan);
            NJson::InsertField(jsonMeta, "ban_duration", NJson::Hr(BanDuration));
            NJson::InsertField(jsonMeta, "ban_chat", BanChat);
            NJson::InsertField(jsonMeta, "ban_deeplink", BanDeeplink);

            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            return
                NJson::ParseField(jsonMeta, "push_text", PushText) &&
                NJson::ParseField(jsonMeta, "bonus", Bonus, false) &&
                NJson::ParseField(jsonMeta, "fees", Fees, false) &&
                NJson::ParseField(jsonMeta, "point", Point, false) &&
                NJson::ParseField(jsonMeta, "chat_messages_group", ChatMessagesGroup, false) &&
                NJson::ParseField(jsonMeta, "chat_messages_group_priority", ChatMessagesGroupPriority, false) &&
                NJson::ParseField(jsonMeta, "is_neutral", IsNeutral, false) &&
                NJson::ParseField(jsonMeta, "drop_on_reentry", DropOnReentry, false) &&
                NJson::ParseField(jsonMeta, "instant_ban", InstantBan, false) &&
                NJson::ParseField(jsonMeta, "ban_chat", BanChat, false) &&
                NJson::ParseField(jsonMeta, "ban_deeplink", BanDeeplink, false) &&
                NJson::ParseField(jsonMeta, "ban_duration", NJson::Hr(BanDuration));
        }
    };

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSArray>("links", "Ссылки").SetElement(TLink::GetScheme());
        result.Add<TFSString>("session_id", "Идентификатор сессии").SetVisual(TFSString::EVisualType::GUID);
        result.Add<TFSString>("car_number", "Номер машины");
        result.Add<TFSString>("topic_link", "Идентификатор чата");
        return result;
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        JWRITE_DEF(json, "session_id", SessionId, "");
        JWRITE_DEF(json, "car_number", CarNumber, "");
        JWRITE_DEF(json, "topic_link", ChatTopic, "");
        NJson::TJsonValue& links = json.InsertValue("links", NJson::JSON_ARRAY);
        for (auto&& i : Links) {
            links.AppendValue(i.SerializeToJson());
        }
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_STRING_OPT(json, "session_id", SessionId);
        JREAD_STRING_OPT(json, "car_number", CarNumber);
        JREAD_STRING_OPT(json, "topic_link", ChatTopic);
        if (json.Has("links")) {
            const NJson::TJsonValue::TArray* arr;
            if (!json["links"].GetArrayPointer(&arr)) {
                return false;
            }
            for (auto&& i : *arr) {
                TLink link;
                if (!link.DeserializeFromJson(i)) {
                    return false;
                }
                Links.emplace_back(std::move(link));
            }
        }
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TUserProblemTagData proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetSessionId(SessionId);
        proto.SetCarNumber(CarNumber);
        proto.SetChatTopic(ChatTopic);
        for (auto&& i : Links) {
            NDrive::NProto::TLink link;
            link.SetUri(i.GetUri());
            link.SetType(::ToString(i.GetType()));
            *proto.AddLinks() = link;
        }
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        for (auto&& i : proto.GetLinks()) {
            TLink link;
            ELinkType type;
            if (!TryFromString(i.GetType(), type)) {
                return false;
            }
            link.SetUri(i.GetUri());
            link.SetType(type);
            Links.emplace_back(std::move(link));
        }
        SessionId = proto.GetSessionId();
        CarNumber = proto.GetCarNumber();
        ChatTopic = proto.GetChatTopic();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        if (server->GetUserRegistrationManager() && server->GetUserRegistrationManager()->IsSubjectToBan(self.GetObjectId(), session)) {
            return server->GetUserRegistrationManager()->BanUser(self.GetObjectId(), userId, NBans::EReason::Auto, session);
        }
        return true;
    }

    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        if (server->GetUserRegistrationManager() && server->GetUserRegistrationManager()->IsSubjectToFinishBan(self.GetObjectId(), session)) {
            return server->GetUserRegistrationManager()->UnbanUser(self.GetObjectId(), userId, session);
        }
        return true;
    }

    const TUserProblemTag::TDescription* GetDescription(const ITagsMeta& tagsMeta) const {
        return tagsMeta.GetDescriptionByName(GetName())->GetAs<TUserProblemTag::TDescription>();
    }

    ui32 GetPoints(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetPoint() : 0;
    }

    TString GetBanChat(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetBanChat() : "";
    }

    TString GetBanDeeplink(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetBanDeeplink() : "";
    }

    bool GetIsNeutral(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetIsNeutral() : false;
    }

    bool GetInstantBan(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetInstantBan() : false;
    }

    bool GetDropOnReentry(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetDropOnReentry() : false;
    }

    ui32 GetChatMessageGroupPriority(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetChatMessagesGroupPriority() : 0;
    }

    TString GetChatMessagesGroupId(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetChatMessagesGroup() : "";
    }

    TVector<TString> GetStartrekIssues(const TString& queue) const {
        TVector<TString> queues;
        if (queue) {
            queues.push_back(queue);
        }
        return GetStartrekIssues(queues);
    }

    TVector<TString> GetStartrekIssues(const TVector<TString>& queues = {}) const {
        TVector<TString> issues;
        for (const auto& link : GetLinks()) {
            if (link.GetType() != TUserProblemTag::ELinkType::Startrek) {
                continue;
            }
            TString issue = link.GetUri().substr(link.GetUri().find_last_of("/") + 1);
            if (queues) {
                bool match = false;
                for (const TString& queue : queues) {
                    if (issue.StartsWith(queue + "-")) {
                        match = true;
                        break;
                    }
                }
                if (!match) {
                    continue;
                }
            }
            issues.emplace_back(std::move(issue));
        }
        return issues;
    }

    virtual TString GetTopicLink(const NDrive::IServer& /*server*/) const {
        return ChatTopic;
    }

    bool SendMessage(const TString& userId, const TString& message, const NDrive::NChat::IMessage::EMessageType type, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const;
    bool MoveToStep(const TString& userId, const TString& stepName, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const;
};

class TRegistrationUserTag: public ISerializableTag<NDrive::NProto::TRegistrationUserTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TRegistrationUserTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TRegistrationUserTag> Registrator;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    private:
        R_FIELD(TString, Comment);
        R_FIELD(bool, NoCleanup, false);

        R_FIELD(TString, ChatItem);
        R_FIELD(TString, ChatRobot);
        R_FIELD(TString, Status);
        R_FIELD(TString, ActiveSessionTagName);

        R_FIELD(bool, CheckActiveSession, false);

        R_FIELD(bool, TransferDataFromDuplicates, false);
        R_FIELD(NDataTransfer::TDataTransferTraits, TransferTraits, NDataTransfer::TransferAll);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const override;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

private:
    R_FIELD(TString, ChatItem);
    R_FIELD(TString, ChatRobot);
    R_FIELD(TSet<TString>, DuplicateIds);
};

class TPassportTrackTag: public ISerializableTag<NDrive::NProto::TPassportTrackTagData> {
private:
    R_FIELD(TString, TrackId);

private:
    using TBase = ISerializableTag<NDrive::NProto::TPassportTrackTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TPassportTrackTag> Registrator;

public:
    using TBase::TBase;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        json["track_id"] = TrackId;
        TBase::SerializeSpecialDataToJson(json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_STRING(json, "track_id", TrackId);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TPassportTrackTagData proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetTrackId(TrackId);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        TrackId = proto.GetTrackId();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSString>("track_id", "Идентификатор пользователя").SetReadOnly(true);
        return scheme;
    }
};

class TConnectionUserTag: public ISerializableTag<NDrive::NProto::TConnectionUserTagData> {
private:
    R_FIELD(TString, ConnectedUserId);

private:
    using TBase = ISerializableTag<NDrive::NProto::TConnectionUserTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TConnectionUserTag> Registrator;

public:
    using TBase::TBase;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        json["connected_user_id"] = ConnectedUserId;
        TBase::SerializeSpecialDataToJson(json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_STRING(json, "connected_user_id", ConnectedUserId);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TConnectionUserTagData proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetConnectedUserId(ConnectedUserId);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        ConnectedUserId = proto.GetConnectedUserId();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSString>("connected_user_id", "Идентификатор пользователя").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
        return scheme;
    }
};


enum class EReferralType {
    Bonuses /* "bonuses" */,
    YandexPlus /* "yandex_plus" */,
};

class TReferralConnectionTagDescription: public TTagDescription {
private:
    R_READONLY(ui32, CashbackLimit, 70000);

private:
    using TBase = TTagDescription;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("cashback_limit", "Ограничение на размер начисления");
        return result;
    }

    virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
        NJson::TJsonValue result;
        JWRITE(result, "cashback_limit", CashbackLimit);
        return result;
    }
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
        JREAD_UINT_OPT(jsonMeta, "cashback_limit", CashbackLimit);
        return true;
    }
};

class TReferralConnectionTag: public TConnectionUserTag {
private:
    using TBase = TConnectionUserTag;

private:
    R_FIELD(i32, AvailableBalance, -1);

public:
    using TBase::TBase;

    virtual bool OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    static const TString TypeName;

private:
    static TFactory::TRegistrator<TReferralConnectionTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TReferralConnectionTagDescription> RegistratorDescription;
};

class TUserDocumentsReaskTag: public ISerializableTag<NDrive::NProto::TUserDocumentsReaskTagData> {
private:
    R_FIELD(bool, LicenseBack, false);
    R_FIELD(bool, LicenseFront, false);
    R_FIELD(bool, LicenseSelfie, false);
    R_FIELD(bool, PassportBiographical, false);
    R_FIELD(bool, PassportSelfie, false);
    R_FIELD(bool, PassportRegistration, false);
    R_FIELD(TString, ChatId);
    R_FIELD(TVector<TDocumentResubmitOverride>, ResubmitOverrides);

private:
    using TBase = ISerializableTag<NDrive::NProto::TUserDocumentsReaskTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUserDocumentsReaskTag> Registrator;

public:
    using TBase::TBase;

    ui32 GetReaskMask() const;

protected:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }
};

class TUserRobotCallTag: public ISerializableTag<NDrive::NProto::TUserRobotCallTagData> {
private:
    R_FIELD(TString, CallSessionId);
    R_FIELD(NRobotPhoneCall::ECallStatus, CallStatus);

private:
    using TBase = ISerializableTag<NDrive::NProto::TUserRobotCallTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUserRobotCallTag> Registrator;

public:
    using TBase::TBase;

    struct TRobotVoiceSettings {
        TString HelloPrompt;
        TString GoodbyePrompt;

        TRobotVoiceSettings() = default;

        TRobotVoiceSettings(const TString& helloPrompt, const TString& goodbyePrompt)
            : HelloPrompt(helloPrompt)
            , GoodbyePrompt(goodbyePrompt)
        {
        }

        NJson::TJsonValue SerializeToJson() const {
            return HelloPrompt + "," + GoodbyePrompt;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& json) {
            if (!json.IsString()) {
                return false;
            }
            TVector<TString> tokens = SplitString(json.GetString(), ",");
            if (tokens.size() != 2) {
                return false;
            }
            HelloPrompt = tokens[0];
            GoodbyePrompt = tokens[1];
            return true;
        }
    };

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

        R_FIELD(TVector<TRobotVoiceSettings>, Voices);
        R_FIELD(TDuration, MaxWaitTime, TDuration::Seconds(60));

        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSArray>("voices", "Голоса, по два разделенных запятой id").SetElement<TFSString>();
            result.Add<TFSDuration>("max_wait_time", "Сколько максимально ждать результата").SetDefault(TDuration::Seconds(60));
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
            JWRITE_DURATION(jsonMeta, "max_wait_time", MaxWaitTime);

            NJson::TJsonValue voices = NJson::JSON_ARRAY;
            for (auto&& voice : Voices) {
                voices.AppendValue(voice.SerializeToJson());
            }
            jsonMeta["voices"] = std::move(voices);

            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_DURATION(jsonMeta, "max_wait_time", MaxWaitTime);

            if (jsonMeta.Has("voices") && jsonMeta["voices"].IsArray()) {
                Voices.clear();
                for (auto&& voiceJson : jsonMeta["voices"].GetArray()) {
                    TRobotVoiceSettings voice;
                    if (!voice.DeserializeFromJson(voiceJson)) {
                        return false;
                    }
                    Voices.push_back(std::move(voice));
                }
            } else {
                return false;
            }

            return true;
        }
    };

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        json["call_session_id"] = CallSessionId;
        json["call_status"] = (ui32)CallStatus;
        TBase::SerializeSpecialDataToJson(json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_STRING_OPT(json, "call_session_id", CallSessionId);
        ui32 callStatusInt = 0;
        JREAD_UINT_OPT(json, "call_status", callStatusInt);
        CallStatus = (NRobotPhoneCall::ECallStatus)callStatusInt;
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TUserRobotCallTagData proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetCallSessionId(CallSessionId);
        proto.SetCallStatus((ui32)CallStatus);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        CallSessionId = proto.GetCallSessionId();
        CallStatus = NRobotPhoneCall::ECallStatus(proto.GetCallStatus());
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
};

class TExternalPromoTag: public ISerializableTag<NDrive::NProto::TExternalPromoTagData> {
private:
    R_FIELD(TString, Code);
    R_FIELD(TString, Deeplink);

private:
    using TBase = ISerializableTag<NDrive::NProto::TExternalPromoTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TExternalPromoTag> Registrator;

    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

        R_FIELD(TString, Type);
        R_FIELD(TInstant, Deadline, TInstant::Max());
        R_FIELD(i32, Priority, 0);
        R_FIELD(TString, PromoTable, "external_promo_codes");
        R_FIELD(TString, Title);
        R_FIELD(TString, Icon);
        R_FIELD(TString, Description);
        R_FIELD(TString, DetailedDescription);

        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSString>("type", "Тип промо").SetRequired(true);
            result.Add<TFSNumeric>("deadline", "Дата окончания").SetVisual(TFSNumeric::EVisualType::DateTime);
            result.Add<TFSNumeric>("priority", "Приоритет кнопки").SetDefault(0);
            result.Add<TFSString>("promo_table", "Таблица с промо").SetDefault("external_promo_codes");
            result.Add<TFSString>("title", "Заголовок").SetRequired(true);
            result.Add<TFSString>("icon", "Иконка").SetRequired(true);
            result.Add<TFSString>("description", "Описание").SetRequired(true);
            result.Add<TFSText>("detailed_description", "Детальное описание");
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
            JWRITE(jsonMeta, "type", Type);
            JWRITE(jsonMeta, "promo_table", PromoTable);
            JWRITE(jsonMeta, "priority", Priority);
            JWRITE_INSTANT(jsonMeta, "deadline", Deadline);
            JWRITE(jsonMeta, "title", Title);
            JWRITE(jsonMeta, "icon", Icon);
            JWRITE(jsonMeta, "description", Description);
            JWRITE(jsonMeta, "detailed_description", DetailedDescription);
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_STRING(jsonMeta, "type", Type);
            JREAD_STRING_OPT(jsonMeta, "promo_table", PromoTable);
            JREAD_INT_OPT(jsonMeta, "priority", Priority);
            JREAD_INSTANT_OPT(jsonMeta, "deadline", Deadline);
            JREAD_STRING(jsonMeta, "title", Title);
            JREAD_STRING(jsonMeta, "icon", Icon);
            JREAD_STRING(jsonMeta, "description", Description);
            JREAD_STRING_OPT(jsonMeta, "detailed_description", DetailedDescription);
            return TBase::DoDeserializeMetaFromJson(jsonMeta);
        }
    };

protected:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSString>("code", "Промокод");
        result.Add<TFSString>("deeplink");
        return result;
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        JWRITE(json, "code", Code);
        JWRITE_DEF(json, "deeplink", Deeplink, "");
        TBase::SerializeSpecialDataToJson(json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_STRING_OPT(json, "code", Code);
        JREAD_STRING_OPT(json, "deeplink", Deeplink);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TExternalPromoTagData proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetCode(Code);
        proto.SetDeeplink(Deeplink);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        Code = proto.GetCode();
        Deeplink = proto.GetDeeplink();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
};

struct TUserChatShow {
    TString Id;
    TString Deeplink;
    bool IsClosable = false;
    bool ShowInRiding = false;
};

class TUserChatShowTag: public ISerializableTag<NDrive::NProto::TUserChatShowTagData> {
private:
    R_FIELD(TString, TopicLink);
    R_FIELD(TString, Deeplink);
    R_FIELD(bool, IsClosable, true);
    R_FIELD(bool, ShowInRiding, true);
    R_FIELD(bool, RemoveOnView, false);

private:
    using TBase = ISerializableTag<NDrive::NProto::TUserChatShowTagData>;
    using TProto = NDrive::NProto::TUserChatShowTagData;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUserChatShowTag> Registrator;

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    TUserChatShow Get(const TString& userStatus) const;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
};

class TUserChatNotificationTag: public ISerializableTag<NDrive::NProto::TUserChatNotificationTagData> {
private:
    R_FIELD(TInstant, LastNotifiedAt);
    R_FIELD(ui64, LastMessageId, 0);

private:
    using TBase = ISerializableTag<NDrive::NProto::TUserChatNotificationTagData>;
    using TProto = NDrive::NProto::TUserChatNotificationTagData;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUserChatNotificationTag> Registrator;

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("last_notified_at", "UTC timestamp последней нотификации");
        result.Add<TFSNumeric>("last_message_id", "id последнего сообщения, повлекшего нотификацию").SetDefault(true);
        return result;
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        JWRITE_INSTANT(json, "last_notified_at", LastNotifiedAt);
        JWRITE(json, "last_message_id", LastMessageId);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_INSTANT(json, "last_notified_at", LastNotifiedAt);
        JREAD_UINT(json, "last_message_id", LastMessageId);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetLastNotifiedAt(LastNotifiedAt.Seconds());
        proto.SetLastMessageId(LastMessageId);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        LastNotifiedAt = TInstant::Seconds(proto.GetLastNotifiedAt());
        LastMessageId = proto.GetLastMessageId();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }
};

class TUserIsolationPassTag: public ISerializableTag<NDrive::NProto::TUserIsolationPassTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TUserIsolationPassTag>;
    using TDescription = TTagDescription;

private:
    R_FIELD(NJson::TJsonValue, ApiResponse);
    R_FIELD(EPassApiType, ApiType, EPassApiType::Undefined);
    R_FIELD(bool, Valid, false);
    R_FIELD(TString, Token);
    R_FIELD(TInstant, ValidUntil);
    R_FIELD(TInstant, ValidationTime);
    R_FIELD(TString, CarId);
    R_FIELD(TString, ChatId);

private:
    static TFactory::TRegistrator<TUserIsolationPassTag> Registrator;

public:
    static const TString TypeName;

public:
    using TBase::TBase;
    TUserIsolationPassTag(const TCovidPassClient::TPassData& passData, const EPassApiType& apiType, const TInstant validationTime, const TString& carId, const TString& chatId);

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TServiceTask {
public:
    bool IsGarage() const;

public:
    R_FIELD(TString, TagId);
    R_FIELD(TInstant, StartTime);
    R_FIELD(TInstant, EndTime);
    R_FIELD(TString, CarId);
    R_FIELD(TString, CarNumber);
};

class TServiceRoute {
public:
    R_FIELD(TString, RoutingTaskId);
    R_FIELD(TVector<TServiceTask>, Tasks);
};

class TServiceRouteTag : public ISerializableTag<NDrive::NProto::TServiceRouteTag> {
public:
    using TBase = ISerializableTag<NDrive::NProto::TServiceRouteTag>;

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

public:
    static const TString ServiceRouteTagName;
    static const TString TypeName;

private:
    static TFactory::TRegistrator<TServiceRouteTag> Registrator;

private:
    R_FIELD(TServiceRoute, Route);
};

class TServiceZonesTag : public ISerializableTag<NDrive::NProto::TServiceZonesTag> {
public:
    using TBase = ISerializableTag<NDrive::NProto::TServiceZonesTag>;

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

public:
    static const TString TypeName;
    static const TString ServiceZoneAreaType;
    static const TString AvailableZonesTagNameSettings;
    static const TString DefaultAvailableZonesTagName;

private:
    static TFactory::TRegistrator<TServiceZonesTag> Registrator;

private:
    R_FIELD(TVector<TString>, AvailableZones);
    R_FIELD(TVector<TString>, AdditionalTagNames);
};

class TUserPhotoResendTag: public ISerializableTag<NDrive::NProto::TUserPhotoResendTag> {
public:
    using TBase = ISerializableTag<NDrive::NProto::TUserPhotoResendTag>;

private:
    R_FIELD(TSet<TString>, PhotoIds);

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

public:
    static const TString TypeName;

private:
    static TFactory::TRegistrator<TUserPhotoResendTag> Registrator;
};

class TSelfieVerificationResultTag: public ISerializableTag<NDrive::NProto::TSelfieVerificationResultTag> {
public:
    using TBase = ISerializableTag<NDrive::NProto::TSelfieVerificationResultTag>;

private:
    R_FIELD(bool, FoundFaceOnSelfie, false);
    R_FIELD(bool, FoundFaceOnPassport, false);
    R_FIELD(double, Similarity, 0);

public:
    static const TString TypeName;
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(double, MinAllowedSimilarity, 0);
        R_FIELD(bool, AllowOnBadSelfie, false);
        R_FIELD(bool, AllowOnBadPassport, false);

        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSBoolean>("allow_on_bad_selfie", "Считать результат с нераспознанным селфи корректным");
            result.Add<TFSBoolean>("allow_on_bad_passport", "Считать результат с нераспознанным паспортным селфи корректным");
            result.Add<TFSNumeric>("min_similarity", "Минимальная допустимая схожесть лиц");
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
            NJson::InsertField(jsonMeta, "allow_on_bad_selfie", AllowOnBadSelfie);
            NJson::InsertField(jsonMeta, "allow_on_bad_passport", AllowOnBadPassport);
            NJson::InsertField(jsonMeta, "min_similarity", MinAllowedSimilarity);
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            return
                NJson::ParseField(jsonMeta, "allow_on_bad_selfie", AllowOnBadSelfie, false) &&
                NJson::ParseField(jsonMeta, "allow_on_bad_passport", AllowOnBadPassport, false) &&
                NJson::ParseField(jsonMeta, "min_similarity", MinAllowedSimilarity, false);
        }
    };

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    bool IsEntryAllowed(const NDrive::IServer& server) const noexcept(false);

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

private:
    static TFactory::TRegistrator<TSelfieVerificationResultTag> Registrator;
};

class TUserSubscriptionTag
    : public ISerializableTag<NDrive::NProto::TUserSubscriptionTag>
    , public IUserActionTag
{
public:
    using TBase = ISerializableTag<NDrive::NProto::TUserSubscriptionTag>;

    using TPtr = TAtomicSharedPtr<TUserSubscriptionTag>;

public:
    class TDescription : public TTagDescription {
    public:
        using TBase = TTagDescription;

        struct TBenefit {
            TString Name;
            TString Description;
            TString Icon;
            TString Type;
        };
        struct TSubscriptionDescription {
            TString Name;
            TString Description;
            TVector<TBenefit> Benefits;
            ui32 Price = 0;
            TString BgColor;
        };
        struct TStatus {
            TString Name;
            TString Details;
            TString Color;
            TString BgColor;
        };
        struct TTaxiDescription {
            TString Description;
            TString Icon;
            TString BgColor;
        };
        struct TMoreInfo {
            TString Name;
            TString Link;
        };

    private:
        R_FIELD(TSubscriptionDescription, Description);
        R_FIELD(TStatus, Status);
        R_FIELD(TTaxiDescription, TaxiDescription);
        R_FIELD(i32, Priority, 0);
        R_FIELD(TSet<TString>, RoleIds);
        R_FIELD(TString, DebitTag);
        R_FIELD(TString, CreditTag);
        R_FIELD(ui32, MaxTaxiRides, 0);
        R_FIELD(TSet<TString>, TagsInPoint);
        R_FIELD(TDuration, PendingDuration, TDuration::Hours(0));
        R_FIELD(TMoreInfo, MoreInfo);
        R_FIELD(bool, IsAutoPaymentVisible, true);
    private:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& json) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

    enum class ESubscriptionStatus {
        Undefined = 0 /* "undefined" */,
        Active = 1 /* "active" */,
        Pending = 2 /* "pending" */,
    };

    enum class EPaymentStatus {
        Unknown,
        Processing,
        Completed,
        Failed,
    };

public:
    using TBase::TBase;

    static  bool ApplyPayment(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx);
    static TMaybe<EPaymentStatus> CheckPayment(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx);

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    TDBActions GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool getPotential) const override;

    bool HasTaxiRides(const NDrive::IServer* server, NDrive::TEntitySession& tx) const;

    bool IsTaxiAvailableByFilter(TMaybe<TGeoCoord> userLocation, const NDrive::IServer* server, NDrive::TEntitySession& tx) const;

    TString GetUniqueTaxiRideToken(const TConstDBTag& self) const;

private:
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    static TMaybe<EPaymentStatus> CheckPaymentImpl(TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx);
public:
    static const TString TypeName;

private:
    static TFactory::TRegistrator<TUserSubscriptionTag> Registrator;

private:
    R_FIELD(TUserSubscriptionTag::ESubscriptionStatus, Status, TUserSubscriptionTag::ESubscriptionStatus::Undefined);
    R_FIELD(bool, AutoPayment, false);
    R_FIELD(ui32, UsedTaxiRidesCount, 0);
    R_FIELD(TString, LastBillingOperation);
};

class TTaxiPromocodeUserTag : public ISerializableTag<NDrive::NProto::TTaxiPromocodeUserTag> {
public:
    using TBase = ISerializableTag<NDrive::NProto::TTaxiPromocodeUserTag>;
    using TBase::TBase;

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    bool GeneratePromocode(const NDrive::IServer* server, const NDrive::TTaxiPromocodesClient::TRequestData& data);

    TString GetDeepLink() const;

private:
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

public:
    static const TString TypeName;

    class TDescription : public TTagDescription {
    public:
        using TBase = TTagDescription;

    private:
        R_FIELD(TString, SeriesId);

    private:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& json) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

private:
    static TFactory::TRegistrator<TTaxiPromocodeUserTag> Registrator;

private:
    R_FIELD(TString, Promocode);
};
