#include "config.h"

#include <drive/backend/database/drive/landing.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/database/drive/takeout.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/areas/drop_object_features.h>
#include <drive/backend/areas/location.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/drivematics/zone/config/zone_config.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/images/database.h>
#include <drive/backend/incident/manager.h>
#include <drive/backend/incident/renins_claims/client.h>
#include <drive/backend/insurance/ingos/client.h>
#include <drive/backend/insurance/renins/client.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/major/client.h>
#include <drive/backend/notifications/mail/binder.h>
#include <drive/backend/offers/abstract.h>
#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/checkers.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/offer_price.h>
#include <drive/backend/roles/config.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags_search.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/backend/users/login.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/autocode/client.h>
#include <drive/library/cpp/datasync/client.h>
#include <drive/library/cpp/disk/client.h>
#include <drive/library/cpp/element/client.h>
#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/octopus/client.h>
#include <drive/library/cpp/passport/client.h>
#include <drive/library/cpp/samsara/client.h>
#include <drive/library/cpp/solomon/client.h>
#include <drive/library/cpp/staff/client.h>
#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/support_ai/client.h>
#include <drive/library/cpp/takeout/client.h>
#include <drive/library/cpp/threading/future.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/polyline.h>
#include <rtline/library/metasearch/simple/config.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/network/neh_request.h>
#include <rtline/util/types/accessor.h>

#include <util/stream/file.h>

TDriveAPIConfig::TDriveAPIConfig() {
}

TDriveAPIConfig::~TDriveAPIConfig() {
}

void TDriveAPIConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    const TYandexConfig::TSectionsMap children = section->GetAllChildren();

    GeoAreasFreshness = section->GetDirectives().Value("GeoAreasFreshness", GeoAreasFreshness);
    OffersStorageName = section->GetDirectives().Value("OffersStorageName", OffersStorageName);
    AssertCorrectConfig(!!OffersStorageName, "Incorrect offers storage %s", OffersStorageName.data());
    HeadAppFreshness = section->GetDirectives().Value<TDuration>("HeadAppFreshness", HeadAppFreshness);
    NeedAttachmentManager = section->GetDirectives().Value<bool>("NeedAttachmentManager", NeedAttachmentManager);
    NeedDatasyncQueueClient = section->GetDirectives().Value<bool>("NeedDatasyncQueueClient", NeedDatasyncQueueClient);
    OfferLivetime = section->GetDirectives().Value("OfferLivetime", OfferLivetime);

    {
        auto featuresIt = children.find("RolesFeatures");
        if (featuresIt != children.end()) {
            RolesFeaturesConfig.Init(featuresIt->second);
        } else {
            RolesFeaturesConfig.Init(nullptr);
        }
    }

    {
        auto it = children.find("StateFilters");
        if (it != children.end()) {
            StateFiltersConfig.Init(it->second);
        }
    }

    {
        auto it = children.find("AreasConfig");
        if (it != children.end()) {
            AreasConfig.Init(it->second);
        }
    }

    {
        auto it = children.find("LandingsConfig");
        if (it != children.end()) {
            LandingsConfig.Init(it->second);
        }
    }

    {
        auto it = children.find("HeadAppHistoryConfig");
        if (it != children.end()) {
            HeadAppHistoryConfig.Init(it->second);
        } else {
            HeadAppHistoryConfig.SetMaxHistoryDeep(TDuration::Minutes(10)).SetDeep(TDuration::Minutes(10));
        }
    }

    {
        auto it = children.find("ActionPropositions");
        if (it != children.end()) {
            ActionPropositionsConfig.Init(it->second);
        }
    }
    {
        auto it = children.find("RolesHistory");
        if (it != children.end()) {
            RolesHistoryConfig.Init(it->second);
        }
    }

    TTagsManagerConfig::Init(section);

    DBName = section->GetDirectives().Value("DBName", DBName);
    AssertCorrectConfig(!!DBName, "Incorrect DBName value for drive_api");

    {
        auto searchIt = children.find("Search");
        if (searchIt != children.end()) {
            TVector<TString> entities;
            if (searchIt->second->GetDirectives().FillArray("Entities", entities)) {
                for (auto&& i : entities) {
                    NDrive::ESearchEntity entity;
                    AssertCorrectConfig(TryFromString<NDrive::ESearchEntity>(i, entity), "Incorrect entity in config: %s", i.data());
                    SearchEntities |= (NDrive::TSearchEntities)entity;
                }
            } else {
                const bool isSearchEnabled = searchIt->second->GetDirectives().Value("IsEnabled", false);
                if (isSearchEnabled) {
                    SearchEntities |= NDrive::AllSearchEntities;
                }
            }
        }
        INFO_LOG << "SearchEntities: " << SearchEntities << Endl;
    }

    {
        auto usersIt = children.find("Users");
        if (usersIt != children.end()) {
            UsersConfig.Reset(new TUsersDBConfig);
            UsersConfig->Init(usersIt->second);
        }
    }

    {
        auto carsIt = children.find("Cars");
        if (carsIt != children.end()) {
            CarsConfig.Reset(new TCarsDBConfig);
            CarsConfig->Init(carsIt->second);
        }
    }

    {
        auto billingIt = children.find("Billing");
        AssertCorrectConfig(billingIt != children.end(), "No Billing Configured");
        BillingConfig.Reset(new TBillingConfig);
        BillingConfig->Init(billingIt->second, requestPolicy);
    }
    {
        auto mdsIt = children.find("MDS");
        if (mdsIt == children.end()) {
            mdsIt = children.find("S3");
        }
        if (mdsIt != children.end()) {
            MDSClientConfig.Reset(new TS3ClientConfig);
            MDSClientConfig->Init(mdsIt->second, requestPolicy);
        }
    }
    {
        auto autocodeIt = children.find("Autocode");
        if (autocodeIt != children.end()) {
            AutocodeClientConfig.Reset(new NDrive::NAutocode::TAutocodeClientConfig);
            AutocodeClientConfig->Init(autocodeIt->second, requestPolicy);
        }
    }
    {
        auto majorIt = children.find("Major");
        if (majorIt != children.end()) {
            MajorClientConfig.Reset(new TMajorClientConfig);
            MajorClientConfig->Init(majorIt->second);
        }
    }
    {
        auto elementIt = children.find("ElementFines");
        if (elementIt != children.end()) {
            ElementFinesClientConfig.Reset(new NDrive::TFinesApiConfig);
            ElementFinesClientConfig->Init(elementIt->second);
        }
    }
    {
        auto diskIt = children.find("YandexDisk");
        if (diskIt != children.end()) {
            YandexDiskClientConfig.Reset(new TYandexDiskClientConfig);
            YandexDiskClientConfig->Init(diskIt->second, requestPolicy);
        }
    }
    {
        auto startrekIt = children.find("Startrek");
        if (startrekIt != children.end()) {
            StartrekClientConfig.Reset(new TStartrekClientConfig);
            StartrekClientConfig->Init(startrekIt->second, requestPolicy);
        }
    }
    {
        if (auto solomonIt = children.find("Solomon"); solomonIt != children.end()) {
            SolomonClientConfig.Reset(new TSolomonClientConfig);
            SolomonClientConfig->Init(solomonIt->second, requestPolicy);
        }
        if (auto solomonIt = children.find("DeprecatedSolomon"); solomonIt != children.end()) {
            DeprecatedSolomonClientConfig.Reset(new TSolomonClientConfig);
            DeprecatedSolomonClientConfig->Init(solomonIt->second, requestPolicy);
        }
    }
    {
        auto samsaraIt = children.find("Samsara");
        if (samsaraIt != children.end()) {
            SamsaraClientConfig.Reset(new TSamsaraClientConfig());
            SamsaraClientConfig->Init(samsaraIt->second, requestPolicy);
        }
    }
    {
        auto staffClientIt = children.find("StaffClient");
        if (staffClientIt != children.end()) {
            StaffClientConfig.Reset(new TStaffClientConfig());
            StaffClientConfig->Init(staffClientIt->second, requestPolicy);
        }
    }
    {
        auto finesManagerIt = children.find("FinesManager");
        if (finesManagerIt != children.end()) {
            FinesManagerConfig.Reset(new NDrive::NFine::TFinesManagerConfig());
            FinesManagerConfig->Init(finesManagerIt->second);
        }
    }
    {
        auto reninsClaimIt = children.find("ReninsClaim");
        if (reninsClaimIt != children.end()) {
            ReninsClaimConfig.Reset(new NDrive::NRenins::TReninsClaimClientConfig);
            ReninsClaimConfig->Init(reninsClaimIt->second, requestPolicy);
        }
    }
    {
        auto incidentsManagerIt = children.find("IncidentsManager");
        if (incidentsManagerIt != children.end()) {
            IncidentsManagerConfig.Reset(new NDrive::TIncidentsManagerConfig);
            IncidentsManagerConfig->Init(incidentsManagerIt->second);
        }
    }
    {
        auto reninsIt = children.find("Renins");
        if (reninsIt != children.end()) {
            ReninsConfig.Reset(new NDrive::TReninsClientConfig);
            ReninsConfig->Init(reninsIt->second, requestPolicy);
        }
    }
    {
        auto reninsIt = children.find("Ingos");
        if (reninsIt != children.end()) {
            IngosConfig.Reset(new NDrive::TIngosClientConfig);
            IngosConfig->Init(reninsIt->second, requestPolicy);
        }
    }
    {
        auto passportIt = children.find("PassportClient");
        if (passportIt != children.end()) {
            PassportClientConfig.Reset(new TPassportClientConfig);
            PassportClientConfig->Init(passportIt->second, requestPolicy);
        }
    }
    {
        auto zoneIt = children.find("ZoneDb");
        ZoneConfig.Reset(new NDrivematics::TZoneConfig);
        if (zoneIt != children.end()) {
            ZoneConfig->Init(zoneIt->second);
        }
    }

    PrivateDataClientType = section->GetDirectives().Value("PrivateDataClientType", PrivateDataClientType);
    if (PrivateDataClientType == "datasync") {
        auto datasyncIt = children.find("Datasync");
        AssertCorrectConfig(datasyncIt != children.end(), "DataSync config is missing although it is selected in PrivateDataClientType or by default");

        DatasyncConfig.Reset(new TDatasyncClientConfig);
        DatasyncConfig->Init(datasyncIt->second);
    }

    TakeoutClientType = section->GetDirectives().Value("TakeoutClientType", TakeoutClientType);
    if (TakeoutClientType == "takeout") {
        auto takeoutIt = children.find("Takeout");
        AssertCorrectConfig(takeoutIt != children.end(), "Takeout config is missing although it is selected in TakeoutClientType");

        TakeoutConfig.Reset(new TTakeoutClientConfig);
        TakeoutConfig->Init(takeoutIt->second);
    }

    OctopusClientType = section->GetDirectives().Value("OctopusClientType", OctopusClientType);
    if (OctopusClientType == "octopus") {
        auto octopusIt = children.find("Octopus");
        AssertCorrectConfig(octopusIt != children.end(), "Octopus config is missing although it is selected in OctopusClientType or by default");

        OctopusConfig.Reset(new TOctopusClientConfig);
        OctopusConfig->Init(octopusIt->second);
    }

    {
        auto documentPhotosIt = children.find("DocumentPhotos");
        if (documentPhotosIt != children.end()) {
            DocumentPhotosConfig.Reset(new TDocumentPhotosConfig);
            DocumentPhotosConfig->Init(documentPhotosIt->second);
        }
    }

    {
        auto geocoderIt = children.find("Geocoder");
        if (geocoderIt != children.end()) {
            GeocoderOptions.Reset(new NDrive::TGeocoder::TOptions);
            GeocoderOptions->Init(geocoderIt->second);
        }
    }
}

void TDriveAPIConfig::ToString(IOutputStream& os) const {
    os << "OffersStorageName: " << OffersStorageName << Endl;
    os << "OfferLivetime: " << OfferLivetime << Endl;
    TTagsManagerConfig::ToString(os);
    if (SearchEntities) {
        os << "<Search>" << Endl;
        TSet<TString> entities;
        for (auto&& i : GetEnumAllValues<NDrive::ESearchEntity>()) {
            if ((NDrive::TSearchEntities)i & SearchEntities) {
                entities.emplace(::ToString(i));
            }
        }
        os << "Entities: " << JoinSeq(", ", entities) << Endl;
        os << "</Search>" << Endl;
    }
    os << "PrivateDataClientType: " << PrivateDataClientType << Endl;
    os << "TakeoutClientType: " << TakeoutClientType << Endl;
    os << "HeadAppFreshness:" << HeadAppFreshness << Endl;

    os << "<AreasConfig>" << Endl;
    AreasConfig.ToString(os);
    os << "</AreasConfig>" << Endl;

    os << "<LandingsConfig>" << Endl;
    LandingsConfig.ToString(os);
    os << "</LandingsConfig>" << Endl;

    os << "<StateFilters>" << Endl;
    StateFiltersConfig.ToString(os);
    os << "</StateFilters>" << Endl;

    os << "<ActionPropositions>" << Endl;
    ActionPropositionsConfig.ToString(os);
    os << "</ActionPropositions>" << Endl;

    os << "<RolesHistory>" << Endl;
    RolesHistoryConfig.ToString(os);
    os << "</RolesHistory>" << Endl;

    os << "<HeadAppHistoryConfig>" << Endl;
    HeadAppHistoryConfig.ToString(os);
    os << "</HeadAppHistoryConfig>" << Endl;

    os << "<RolesFeatures>" << Endl;
    RolesFeaturesConfig.ToString(os);
    os << "</RolesFeatures>" << Endl;

    os << "DBName: " << DBName << Endl;

    if (BillingConfig) {
        os << "<Billing>" << Endl;
        BillingConfig->ToString(os);
        os << "</Billing>" << Endl;
    }

    if (MDSClientConfig) {
        os << "<MDS>" << Endl;
        MDSClientConfig->ToString(os);
        os << "</MDS>" << Endl;
    }

    if (AutocodeClientConfig) {
        os << "<Autocode>" << Endl;
        AutocodeClientConfig->ToString(os);
        os << "</Autocode>" << Endl;
    }

    if (MajorClientConfig) {
        os << "<Major>" << Endl;
        MajorClientConfig->ToString(os);
        os << "</Major>" << Endl;
    }

    if (ElementFinesClientConfig) {
        os << "<ElementFines>" << Endl;
        ElementFinesClientConfig->ToString(os);
        os << "</ElementFines>" << Endl;
    }

    if (YandexDiskClientConfig) {
        os << "<YandexDisk>" << Endl;
        YandexDiskClientConfig->ToString(os);
        os << "</YandexDisk>" << Endl;
    }

    if (StartrekClientConfig) {
        os << "<Startrek>" << Endl;
        StartrekClientConfig->ToString(os);
        os << "</Startrek>" << Endl;
    }

    if (SolomonClientConfig) {
        os << "<Solomon>" << Endl;
        SolomonClientConfig->ToString(os);
        os << "</Solomon>" << Endl;
    }

    if (DatasyncConfig) {
        os << "<Datasync>" << Endl;
        DatasyncConfig->ToString(os);
        os << "</Datasync>" << Endl;
    }

    if (TakeoutConfig) {
        os << "<Takeout>" << Endl;
        TakeoutConfig->ToString(os);
        os << "</Takeout>" << Endl;
    }

    if (ReninsClaimConfig) {
        os << "<ReninsClaim>" << Endl;
        ReninsClaimConfig->ToString(os);
        os << "</ReninsClaim>" << Endl;
    }

    if (IncidentsManagerConfig) {
        os << "<IncidentsManager>" << Endl;
        IncidentsManagerConfig->ToString(os);
        os << "</IncidentsManager>" << Endl;
    }

    if (ReninsConfig) {
        os << "<Renins>" << Endl;
        ReninsConfig->ToString(os);
        os << "</Renins>" << Endl;
    }

    if (PassportClientConfig) {
        os << "<PassportClient>" << Endl;
        PassportClientConfig->ToString(os);
        os << "</PassportClient>" << Endl;
    }

    if (DocumentPhotosConfig) {
        os << "<DocumentPhotos>" << Endl;
        DocumentPhotosConfig->ToString(os);
        os << "</DocumentPhotos>" << Endl;
    }

    if (GeocoderOptions) {
        os << "<Geocoder>" << Endl;
        GeocoderOptions->ToString(os);
        os << "</Geocoder>" << Endl;
    }

    if (ZoneConfig) {
        os << "<ZoneDb>" << Endl;
        ZoneConfig->ToString(os);
        os << "</ZoneDb>" << Endl;
    }
}
