#include "consistency.h"

#include <rtline/library/json/merge.h>
#include <rtline/library/json/field.h>

TConsistency::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    Timestamp = GetFieldDecodeIndex("timestamp", decoderBase);
    EntityName = GetFieldDecodeIndex("entity_name", decoderBase);
    ReferenceObjectId = GetFieldDecodeIndex("entity_id", decoderBase);

    ZoneId = GetFieldDecodeIndex("zone_id", decoderBase);
    NamedFilterId = GetFieldDecodeIndex("named_filters_id", decoderBase);
    SignalConfigurationId = GetFieldDecodeIndex("signal_configuration_id", decoderBase);
    NotifierId = GetFieldDecodeIndex("notifier_id", decoderBase);
    ModelId = GetFieldDecodeIndex("models_id", decoderBase);
}

bool TConsistency::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);

    READ_DECODER_VALUE(decoder, values, EntityName);
    READ_DECODER_VALUE(decoder, values, ReferenceObjectId);

    READ_DECODER_VALUE_INSTANT(decoder, values, Timestamp);

    if (decoder.GetZoneId() > -1) {
        TString temp;
        READ_DECODER_VALUE_TEMP(decoder, values, temp, ZoneId);
        if (temp) {
            ZoneId = temp;
        }
    }
    if (decoder.GetNamedFilterId() > -1) {
        TString temp;
        READ_DECODER_VALUE_TEMP(decoder, values, temp, NamedFilterId);
        if (temp) {
            NamedFilterId = temp;
        }
    }
    if (decoder.GetSignalConfigurationId() > -1) {
        TString temp;
        READ_DECODER_VALUE_TEMP(decoder, values, temp, SignalConfigurationId);
        if (temp) {
            SignalConfigurationId = temp;
        }
    }
    if (decoder.GetNotifierId() > -1) {
        TString temp;
        READ_DECODER_VALUE_TEMP(decoder, values, temp, NotifierId);
        if (temp) {
            NotifierId = temp;
        }
    }
    if (decoder.GetModelId() > -1) {
        TString temp;
        READ_DECODER_VALUE_TEMP(decoder, values, temp, ModelId);
        if (temp) {
            ModelId = temp;
        }
    }
    return true;
}

bool TConsistency::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    TMessagesCollector errors;
    return NJson::TryFieldsFromJson(jsonInfo, GetFields(), &errors) &&
        true;
}

NStorage::TTableRecord TConsistency::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    auto json = NJson::FieldsToJson(GetFields());
    json["timestamp"] = GetTimestamp().Seconds();
    Y_ENSURE(result.DeserializeFromJson(json));
    return result;
}

bool TConsistency::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const ITagsHistoryContext* /*context*/) {
    return DeserializeFromJson(record.SerializeToJson());
}

[[nodiscard]] bool TConsistencyStorage::AddConnectivity(const TVector<TConsistency>& references, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TConsistency>* containerExt) const {
    if (!references) {
        return true;
    }
    TInstant now = ModelingNow();
    for (auto reference : references) {
        reference.SetTimestamp(now);
        reference.SetId("uuid_generate_v4()");
        if (!TBase::Insert(reference, session, containerExt)) {
            return false;
        }
    }
    return true;
}

template <typename TObjectsContainer>
[[nodiscard]] bool TConsistencyStorage::RemoveConnectivity(const TObjectsContainer& references, NDrive::TEntitySession& session) const {
    if (!references) {
        return true;
    }
    auto restoredObjects = TBase::FetchInfo(session,
        NSQL::TQueryOptions().AddGenericCondition("entity_name", references[0].GetEntityName())
            .AddGenericCondition("entity_id", references[0].GetReferenceObjectId())
    );
    if (!restoredObjects) {
        return false;
    }
    TSet<TString> idsToRemove = MakeSet<TString>(IterateKeys(restoredObjects));
    if (idsToRemove.empty()) {
        return true;
    }
    return TBase::Remove(idsToRemove, session);
}

[[nodiscard]] bool TConsistencyStorage::UpdateConnectivity(const TVector<TConsistency>& objects, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TConsistency>* containerExt) const {
    return
        RemoveConnectivity(objects, session) &&
        AddConnectivity(objects, session, containerExt);
}
