#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/tags/tags.h>

#include <library/cpp/iterator/iterate_keys.h>

#include <rtline/library/json/merge.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

class TConsistency {
public:
    using TId = TString;

private:
    R_FIELD(TString, Id, "uuid_generate_v4()");
    R_FIELD(TInstant, Timestamp);

    R_FIELD(TString, EntityName);
    R_FIELD(TString, ReferenceObjectId);

    R_OPTIONAL(TString, ZoneId);
    R_OPTIONAL(TString, NamedFilterId);
    R_OPTIONAL(TString, SignalConfigurationId);
    R_OPTIONAL(TString, NotifierId);
    R_OPTIONAL(TString, ModelId);

public:
    DECLARE_FIELDS(
        Field(Id, "id"),
        Field(NJson::Seconds(Timestamp), "timestamp"),
        Field(EntityName, "entity_name"),
        Field(ReferenceObjectId, "entity_id"),
        Field(ZoneId, "zone_id"),
        Field(NamedFilterId, "named_filters_id"),
        Field(SignalConfigurationId, "signal_configuration_id"),
        Field(NotifierId, "notifier_id"),
        Field(ModelId, "models_id")
    );

public:
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Timestamp, -1);
        R_FIELD(i32, EntityName, -1);
        R_FIELD(i32, ReferenceObjectId, -1);
        R_FIELD(i32, ZoneId, -1);
        R_FIELD(i32, NamedFilterId, -1);
        R_FIELD(i32, SignalConfigurationId, -1);
        R_FIELD(i32, NotifierId, -1);
        R_FIELD(i32, ModelId, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

public:
    TConsistency() = default;

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const ITagsHistoryContext* /*context*/);
    NStorage::TTableRecord SerializeToTableRecord() const;

    bool Parse(const NStorage::TTableRecord& record) {
        return DeserializeFromTableRecord(record, nullptr);
    }

public:

    bool operator < (const TConsistency& item) const {
        return Id < item.GetId();
    }

    explicit operator bool() const {
        return HasZoneId() || HasNamedFilterId() || HasSignalConfigurationId() || HasNotifierId() || HasModelId();
    }
};

class TConsistencyConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TConsistency& object) {
        return BuildCondition(object.GetId());
    }
};

class TConsistencyStorage : public TDatabaseEntityManager<TConsistency> {
    using TBase = TDatabaseEntityManager<TConsistency>;

public:
    using TBase::TBase;

public:
    TString GetTableName() const override {
        return "entity_references";
    }

    TString GetMainId(const TConsistency& e) const override {
        return e.GetId();
    }


public:
    [[nodiscard]] bool AddConnectivity(const TVector<TConsistency>& references, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TConsistency>* containerExt = nullptr) const;
    template <class TObjectsContainer>
    [[nodiscard]] bool RemoveConnectivity(const TObjectsContainer& references, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool UpdateConnectivity(const TVector<TConsistency>& objects, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TConsistency>* containerExt = nullptr) const;
};
