#include "landing.h"

#include <drive/backend/offers/offers/standart.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/json_processing.h>

TLanding::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("landing_id", decoderBase);
    Enabled = GetFieldDecodeIndex("landing_enabled", decoderBase);
    Revision = GetFieldDecodeIndex("revision", decoderBase);

    JsonLanding = GetFieldDecodeIndex("landing_json", decoderBase);
    if (JsonLanding < 0 ) {
        JsonLanding = GetFieldDecodeIndex("landing_text", decoderBase);
    }

    Meta = GetFieldDecodeIndex("landing_meta", decoderBase);
    if (Meta < 0) {
        Priority = GetFieldDecodeIndex("landing_priority", decoderBase);
        ShouldSubstitute = GetFieldDecodeIndex("landing_json_substitute", decoderBase);
        Deadline = GetFieldDecodeIndex("landing_deadline", decoderBase);
        ChatId = GetFieldDecodeIndex("landing_chat_id", decoderBase);
        ChatTitle = GetFieldDecodeIndex("landing_chat_title", decoderBase);
        Preview = GetFieldDecodeIndex("landing_preview", decoderBase);
        TimestampOverride = GetFieldDecodeIndex("landing_timestamp_override", decoderBase);//
        ChatIcon = GetFieldDecodeIndex("landing_chat_icon", decoderBase);
        ChatMessagesGroup = GetFieldDecodeIndex("landing_chat_messages_group", decoderBase);
        ChatEnabled = GetFieldDecodeIndex("landing_chat_enabled", decoderBase);
        EventTagName = GetFieldDecodeIndex("landing_event_tag_name", decoderBase);
        ChatDeadline = GetFieldDecodeIndex("landing_chat_deadline", decoderBase);
        GeoTags = GetFieldDecodeIndex("landing_geo_tags", decoderBase);
        CheckAuditoryConditionInList = GetFieldDecodeIndex("landing_check_auditory_condition_in_list", decoderBase);
        AuditoryConditionRaw = GetFieldDecodeIndex("landing_auditory_condition", decoderBase);
        PayloadPatch = GetFieldDecodeIndex("payload_patch", decoderBase);
    }
}

bool TLanding::DeserializeMeta(const NJson::TJsonValue& jsonMeta) {
    if (!NJson::ParseField(jsonMeta, "deadline", Deadline) ||
        !NJson::ParseField(jsonMeta, "priority", Priority, false) ||
        !NJson::ParseField(jsonMeta, "json_substitute", ShouldSubstitute, false) ||
        !NJson::ParseField(jsonMeta, "chat_id", ChatId) ||
        !NJson::ParseField(jsonMeta, "chat_title", ChatTitle) ||
        !NJson::ParseField(jsonMeta, "preview", Preview) ||
        !NJson::ParseField(jsonMeta, "chat_icon", ChatIcon) ||
        !NJson::ParseField(jsonMeta, "chat_messages_group", ChatMessagesGroup) ||
        !NJson::ParseField(jsonMeta, "chat_deadline", ChatDeadline) ||
        !NJson::ParseField(jsonMeta, "chat_enabled", ChatEnabled, false) ||
        !NJson::ParseField(jsonMeta, "timestamp_override", TimestampOverride) ||
        !NJson::ParseField(jsonMeta, "check_auditory_condition_in_list", CheckAuditoryConditionInList, false) ||
        !NJson::ParseField(jsonMeta, "event_tag_name", EventTagName)) {
        return false;
    }

    if (jsonMeta.Has("geo_tags")) {
        const NJson::TJsonValue::TArray* arr;
        if (!jsonMeta["geo_tags"].GetArrayPointer(&arr)) {
            return false;
        }
        for (auto&& i : *arr) {
            if (!i.IsString()) {
                return false;
            }
            GeoTags.emplace(i.GetString());
        }
    }

    if (jsonMeta.Has("auditory_condition")) {
        AuditoryConditionRaw = jsonMeta["auditory_condition"];
        AuditoryCondition = ICondition::DeserializeFromJson(AuditoryConditionRaw);
    }
    if (jsonMeta.Has("payload_patch")) {
        PayloadPatch = jsonMeta["payload_patch"];
        if (!PayloadPatch.IsMap()) {
            return false;
        }
    }

    return true;
}

bool TLanding::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE_DEF(decoder, values, Enabled, true);
    READ_DECODER_VALUE_DEF(decoder, values, Revision, 0);

    if (!Id) {
        return false;
    }

    READ_DECODER_VALUE_JSON(decoder, values, JsonLanding, JsonLanding);

    if (decoder.GetMeta() > 0) {
        NJson::TJsonValue jsonMeta;
        READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
        return DeserializeMeta(jsonMeta);
    } else {
        READ_DECODER_VALUE(decoder, values, Priority);
        READ_DECODER_VALUE(decoder, values, ShouldSubstitute);
        READ_DECODER_VALUE(decoder, values, ChatId);
        READ_DECODER_VALUE(decoder, values, ChatTitle);
        READ_DECODER_VALUE(decoder, values, Preview);
        READ_DECODER_VALUE(decoder, values, ChatIcon);
        READ_DECODER_VALUE(decoder, values, ChatMessagesGroup);
        READ_DECODER_VALUE(decoder, values, ChatEnabled);
        READ_DECODER_VALUE(decoder, values, EventTagName);
        READ_DECODER_VALUE(decoder, values, CheckAuditoryConditionInList);

        {
            TString deadline;
            READ_DECODER_VALUE_TEMP(decoder, values, deadline, Deadline);
            if (!deadline.empty()) {
                Deadline = TInstant::Seconds(FromString<ui64>(deadline));
            }
        }
        {
            TString deadline;
            READ_DECODER_VALUE_TEMP(decoder, values, deadline, ChatDeadline);
            if (!deadline.empty()) {
                ChatDeadline = TInstant::Seconds(FromString<ui64>(deadline));
            }
        }
        {
            TString deadline;
            READ_DECODER_VALUE_TEMP(decoder, values, deadline, TimestampOverride);
            if (!deadline.empty()) {
                TimestampOverride = TInstant::Seconds(FromString<ui64>(deadline));
            }
        }
        {
            if (decoder.GetGeoTags() > 0)
            {
                TString geoTags;
                READ_DECODER_VALUE_TEMP(decoder, values, geoTags, GeoTags);

                GeoTags = StringSplitter(geoTags).Split(',').SkipEmpty();
            }
        }
        {
            NJson::TJsonValue auditoryConditionRaw;
            READ_DECODER_VALUE_JSON(decoder, values, auditoryConditionRaw, AuditoryConditionRaw);

            if (auditoryConditionRaw.IsDefined()) {
                AuditoryConditionRaw = auditoryConditionRaw;
                AuditoryCondition = ICondition::DeserializeFromJson(AuditoryConditionRaw);
            }
        }
        /* uncomment if it need
        {
            NJson::TJsonValue payloadPatch;
            TString payloadPatchStr;

            if (decoder.GetPayloadPatch() > 0) {
                READ_DECODER_VALUE_TEMP(decoder, values, payloadPatchStr, PayloadPatch);
                NJson::ReadJsonTree(payloadPatchStr, &payloadPatch);
            }
            if (payloadPatch.IsDefined()) {
                if (!payloadPatch.IsMap()) {
                    return false;
                }
                PayloadPatch = payloadPatch;
            }
        }
        */
    }
    return true;
}

bool TLanding::DesereializeRequestJson(const NJson::TJsonValue& requestData, TMessagesCollector& errors) {
    if (!NJson::ParseField(requestData, "landing_id", Id, true, errors) ||
        !NJson::ParseField(requestData, "landing_priority", Priority, false, errors) ||
        !NJson::ParseField(requestData, "landing_enabled", Enabled, false, errors) ||
        !NJson::ParseField(requestData, "json_substitute", ShouldSubstitute, false, errors) ||
        !NJson::ParseField(requestData, "revision", Revision, false, errors) ||
        !NJson::ParseField(requestData, "landing_deadline", Deadline, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_id", ChatId, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_title", ChatTitle, false, errors) ||
        !NJson::ParseField(requestData, "landing_preview", Preview, false, errors) ||
        !NJson::ParseField(requestData, "landing_timestamp_override", TimestampOverride, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_icon", ChatIcon, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_messages_group", ChatMessagesGroup, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_deadline", ChatDeadline, false, errors) ||
        !NJson::ParseField(requestData, "landing_chat_enabled", ChatEnabled, false, errors) ||
        !NJson::ParseField(requestData, "landing_check_auditory_condition_in_list", CheckAuditoryConditionInList, false, errors) ||
        !NJson::ParseField(requestData, "landing_event_tag_name", EventTagName, false, errors)) {
        return false;
    }

    if (!requestData["landing_json"].IsMap()) {
        errors.AddMessage("LandingDeserialize", "incorrect landing json");
    }
    SetJsonLanding(requestData["landing_json"]);

    TString geoTags;
    if (!NJson::ParseField(requestData, "landing_geo_tags", geoTags)) {
        errors.AddMessage("LandingDeserialize", "incorrect landing_geo_tags");
        return false;
    }
    StringSplitter(geoTags).SplitBySet(", ").SkipEmpty().Collect(&GeoTags);

    const auto& conditionJson = requestData["landing_auditory_condition"];
    if (conditionJson.IsDefined()) {
        if (!conditionJson.IsMap()) {
            errors.AddMessage("LandingDeserialize", "landing_auditory_condition is not map");
            return false;
        }
        ICondition::TPtr condition = ICondition::DeserializeFromJson(conditionJson, &errors);
        if (condition) {
            SetAuditoryCondition(condition);
            SetAuditoryConditionRaw(conditionJson);
        } else {
            return false;
        }
    }

    const auto& payloadPatch = requestData["payload_patch"];
    if (payloadPatch.IsDefined()) {
        if (!payloadPatch.IsMap()) {
            errors.AddMessage("LandingDeserialize", "payload_patch is not map");
            return false;
        }
        SetPayloadPatch(payloadPatch);
    }

    return true;
}

NJson::TJsonValue TLanding::SerializeMeta() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (Deadline != TInstant::Max()) {
        NJson::InsertField(result, "deadline", Deadline.Seconds());
    }
    if (GeoTags.size()) {
        NJson::TJsonValue& geoTagsJson = result.InsertValue("geo_tags", NJson::JSON_ARRAY);
        for (auto&& i : GeoTags) {
            geoTagsJson.AppendValue(i);
        }
    }
    NJson::InsertField(result, "priority", Priority);
    NJson::InsertField(result, "json_substitute", ShouldSubstitute);
    NJson::InsertNonNull(result, "chat_id", ChatId);
    NJson::InsertNonNull(result, "chat_icon", ChatIcon);
    NJson::InsertNonNull(result, "chat_title", ChatTitle);
    NJson::InsertNonNull(result, "preview", Preview);
    NJson::InsertNonNull(result, "chat_messages_group", ChatMessagesGroup);
    if (ChatDeadline != TInstant::Max()) {
        NJson::InsertField(result, "chat_deadline", ChatDeadline.Seconds());
    }
    NJson::InsertField(result, "chat_enabled", ChatEnabled);
    NJson::InsertNonNull(result, "timestamp_override", TimestampOverride.Seconds());
    NJson::InsertField(result, "check_auditory_condition_in_list", CheckAuditoryConditionInList);
    if (AuditoryConditionRaw.IsDefined()) {
        NJson::InsertField(result, "auditory_condition", AuditoryConditionRaw);
    }
    if (PayloadPatch.IsDefined()) {
        NJson::InsertField(result, "payload_patch", PayloadPatch);
    }

    NJson::InsertNonNull(result, "event_tag_name", EventTagName);
    return result;
}

NStorage::TTableRecord TLanding::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("landing_id", Id)
        .Set("landing_text", JsonLanding.GetStringRobust())
        .Set("landing_enabled", Enabled)
        .Set("revision", Revision)
        .Set("landing_meta", SerializeMeta());
    return result;
}

NJson::TJsonValue TLanding::GetAdminReport() const {
    NJson::TJsonValue result;
    result["landing_id"] = Id;
    result["landing_enabled"] = Enabled;
    result["landing_json"] = JsonLanding;
    result.InsertValue("landing_priority", Priority);
    result["landing_json_substitute"] = ShouldSubstitute;
    result["revision"] = Revision;

    NJson::InsertField(result, "landing_chat_enabled", ChatEnabled);
    NJson::InsertNonNull(result, "landing_chat_id", ChatId);
    NJson::InsertNonNull(result, "landing_chat_title", ChatTitle);
    NJson::InsertNonNull(result, "landing_preview", Preview);
    NJson::InsertNonNull(result, "landing_timestamp_override", TimestampOverride.Seconds());
    NJson::InsertNonNull(result, "landing_chat_icon", ChatIcon);
    NJson::InsertNonNull(result, "landing_chat_messages_group", ChatMessagesGroup);
    NJson::InsertNonNull(result, "landing_event_tag_name", EventTagName);
    NJson::InsertField(result, "landing_check_auditory_condition_in_list", CheckAuditoryConditionInList);
    if (AuditoryConditionRaw.IsDefined()) {
        NJson::InsertField(result, "landing_auditory_condition", AuditoryConditionRaw);
    }
    if (GeoTags.size()) {
        NJson::InsertField(result, "landing_geo_tags", JoinSeq(",", GeoTags));
    }
    if (Deadline != TInstant::Max()) {
        NJson::InsertField(result, "landing_deadline", Deadline.Seconds());
    }
    if (ChatDeadline != TInstant::Max()) {
        NJson::InsertField(result, "landing_chat_deadline", ChatDeadline.Seconds());
    }
    if (PayloadPatch.IsDefined()) {
        NJson::InsertField(result, "payload_patch", PayloadPatch);
    }
    return result;
}

NJson::TJsonValue TLanding::GetPublicReport(ELocalization locale, const TLandingContext* context) const {
    NJson::TJsonValue result = context ? UnescapeWithContext(JsonLanding, *context, locale) : JsonLanding;
    if (result.Has("payload_patch")) {
        result["payload_patch"] = result["payload_patch"].GetStringRobust();
    }
    result.InsertValue("id", Id);
    result.InsertValue("priority", Priority);
    result.InsertValue("preview", Preview);
    if (Deadline != TInstant::Max()) {
        result.InsertValue("deadline", Deadline.Seconds());
    }
    return result;
}

NJson::TJsonValue TLanding::UnescapeWithContext(const NJson::TJsonValue& raw, const TLandingContext& context, ELocalization locale) const {
    TString landingString = raw.GetStringRobust();
    if (context.GetOffer()) {
        landingString = context.GetOffer()->FormDescriptionElement(landingString, locale, context.GetServer()->GetLocalization());
    }
    NJson::TJsonValue jsonNew;
    if (!NJson::ReadJsonFastTree(landingString, &jsonNew)) {
        ERROR_LOG << "Cannot parse landing after transformation: " << landingString << Endl;
        return raw;
    }
    return jsonNew;
}


bool TLandingsDB::InitLandingUserRequest(const TVector<TString>& landingIds, const TString& requestSource, NDrive::TEntitySession& session, const TLandingContext* context, EDriveSessionResult result) const {
    TVector<TLanding> landings;
    if (!GetCustomObjectsFromCache(landings, MakeSet(landingIds))) {
        return false;
    }
    for (auto&& i : landings) {
        auto locale = DefaultLocale;
        TUnistatSignalsCache::SignalAdd("landing-request-user-choice", i.GetId(), 1);
        session.SetErrorLanding(requestSource, i.GetPublicReport(locale, context), result);
        return true;
    }
    return false;
}

NJson::TJsonValue TLandingAcceptance::GetReport() const {
    NJson::TJsonValue result;
    result["landing_id"] = Id;
    result["user_id"] = UserId;
    result["accept_comment"] = Comment;
    return result;
}

bool TLandingAcceptance::DeserializeFromTableRecord(const NStorage::TTableRecord& record) {
    Id = record.Get("landing_id");
    UserId = record.Get("user_id");
    if (!Id || !UserId) {
        return false;
    }
    record.TryGet("accept_comment", Comment);
    {
        auto lastAcceptedAtStr = record.Get("last_accepted_at");
        ui64 lastAcceptedAtInt = 0;
        if (lastAcceptedAtStr && !TryFromString(lastAcceptedAtStr, lastAcceptedAtInt)) {
            return false;
        } else {
            LastAcceptedAt = TInstant::Seconds(lastAcceptedAtInt);
        }
    }
    return true;
}

NStorage::TTableRecord TLandingAcceptance::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("landing_id", Id).Set("user_id", UserId).Set("accept_comment", Comment);
    if (LastAcceptedAt != TInstant::Zero()) {
        result.Set("last_accepted_at", LastAcceptedAt.Seconds());
    }
    return result;
}


bool TUserLandings::Parse(const TString& id, const TVector<NStorage::TTableRecord>& records) {
    SetUserId(id);
    for (auto&& rec : records) {
        TLandingAcceptance acceptance;
        if (!acceptance.DeserializeFromTableRecord(rec)) {
            return false;
        }
        Landings.emplace_back(std::move(acceptance));
    }
    return true;
}
