#pragma once
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/database/transaction/tx.h>

#include <drive/backend/chat_robots/configuration/chat_script.h>
#include <drive/backend/offers/abstract.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class TLandingContext {
    R_FIELD(ICommonOffer::TPtr, Offer);

public:
    TLandingContext(const NDrive::IServer* server)
        : Server(server)
    {
    }

    const NDrive::IServer* GetServer() const {
        return Server;
    }

private:
    const NDrive::IServer* Server;
};

class TLanding {
    R_FIELD(TString, Id);
    R_FIELD(NJson::TJsonValue, JsonLanding, NJson::EJsonValueType::JSON_MAP);
    R_FIELD(bool, Enabled, true);
    R_FIELD(TSet<TString>, GeoTags);
    R_FIELD(TInstant, Deadline, TInstant::Max());
    R_FIELD(int, Priority, 0);
    R_FIELD(bool, ShouldSubstitute, true);
    R_FIELD(TString, EventTagName);
    R_FIELD(ui32, Revision, 0);

    R_FIELD(TString, ChatId);
    R_FIELD(TString, ChatIcon);
    R_FIELD(TString, ChatTitle);
    R_FIELD(TString, Preview);
    R_FIELD(TString, ChatMessagesGroup);
    R_FIELD(TInstant, TimestampOverride);
    R_FIELD(bool, ChatEnabled, true);
    R_FIELD(TInstant, ChatDeadline, TInstant::Max());

    R_FIELD(NJson::TJsonValue, AuditoryConditionRaw);
    R_FIELD(ICondition::TPtr, AuditoryCondition);
    R_FIELD(bool, CheckAuditoryConditionInList, false);

    R_FIELD(NJson::TJsonValue, PayloadPatch, NJson::JSON_NULL);

public:

    using TId = TString;

    TLanding() = default;

    bool operator !() const {
        return !Id;
    }
    TMaybe<ui32> OptionalRevision() const {
        return Revision ? Revision : TMaybe<ui32>();
    }

    static TString GetHistoryTableName() {
        return "drive_landings_history";
    }

    static TString GetTableName() {
        return "drive_landings";
    }

    static TString GetPropositionsTableName() {
        return "drive_landings_propositions";
    }

    const TString& GetInternalId() const {
        return GetId();
    }
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, JsonLanding, -1);
        R_FIELD(i32, Enabled, -1);
        R_FIELD(i32, Revision, -1);
        R_FIELD(i32, Meta, -1);

        R_FIELD(i32, Priority, -1);
        R_FIELD(i32, ShouldSubstitute, -1);
        R_FIELD(i32, Deadline, -1);
        R_FIELD(i32, ChatId, -1);
        R_FIELD(i32, ChatTitle, -1);
        R_FIELD(i32, Preview, -1);
        R_FIELD(i32, TimestampOverride, -1);
        R_FIELD(i32, ChatIcon, -1);
        R_FIELD(i32, ChatMessagesGroup, -1);
        R_FIELD(i32, ChatEnabled, -1);
        R_FIELD(i32, EventTagName, -1);
        R_FIELD(i32, ChatDeadline, -1);
        R_FIELD(i32, GeoTags, -1);
        R_FIELD(i32, CheckAuditoryConditionInList, -1);
        R_FIELD(i32, AuditoryConditionRaw, -1);
        R_FIELD(i32, PayloadPatch, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeMeta(const NJson::TJsonValue& jsonMeta);

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    bool DesereializeRequestJson(const NJson::TJsonValue& requestData, TMessagesCollector& errors);

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record) {
        return TBaseDecoder::DeserializeFromTableRecord(*this, record);
    }

    bool Parse(const NStorage::TTableRecord& record) {
        return DeserializeFromTableRecord(record);
    }

    NJson::TJsonValue SerializeMeta() const;

    NStorage::TTableRecord SerializeToTableRecord() const;

    NJson::TJsonValue GetAdminReport() const;

    NJson::TJsonValue BuildJsonReport() const {
        return GetAdminReport();
    }

    NJson::TJsonValue GetPublicReport(ELocalization locale, const TLandingContext* context = nullptr) const;
    NJson::TJsonValue UnescapeWithContext(const NJson::TJsonValue& raw, const TLandingContext& context, ELocalization locale) const;

    bool operator<(const TLanding& item) const {
        if (Priority == item.Priority) {
            return Id < item.Id;
        } else {
            return Priority < item.Priority;
        }
    }
};

class TLandingConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "landing_id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("landing_id", id);
        return trCondition;
    }

    template <class T>
    static NStorage::TTableRecord BuildCondition(const T& object) {
        return BuildCondition(object.GetId());
    }
};

class TLandingsDB: public TDBEntitiesManagerWithPropositions<TLanding, TLandingConditionConstructor> {
private:
    using TBase = TDBEntitiesManagerWithPropositions<TLanding, TLandingConditionConstructor>;
public:
    using TBase::TBase;
    bool InitLandingUserRequest(const TVector<TString>& landings, const TString& requestSource, NDrive::TEntitySession& session, const TLandingContext* context = nullptr, EDriveSessionResult result = EDriveSessionResult::RequestForUser) const;
};

class TLandingAcceptance {
    R_FIELD(TString, Id);
    R_FIELD(TString, UserId);
    R_FIELD(TString, Comment);
    R_FIELD(TInstant, LastAcceptedAt, TInstant::Zero());
public:
    TLandingAcceptance() = default;

    NJson::TJsonValue GetReport() const;

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record);

    NStorage::TTableRecord SerializeToTableRecord() const;

    bool Parse(const NStorage::TTableRecord& record) {
        return DeserializeFromTableRecord(record);
    }
};

class TUserLandings {
    R_FIELD(TString, UserId);
    R_READONLY(TVector<TLandingAcceptance>, Landings);
public:
    bool Parse(const TString& id, const TVector<NStorage::TTableRecord>& records);
};

class TLandingAcceptanceDB: public TCachedEntityManager<TUserLandings, false> {
private:
    using TBase = TCachedEntityManager<TUserLandings, false>;

public:
    using TBase::TBase;

    virtual TString GetMainId(const TUserLandings& e) const override {
        return e.GetUserId();
    }

    virtual TString GetTableName() const override {
        return "user_landings";
    }

    TString GetColumnName() const override {
        return "user_id";
    }

    virtual void UpdateUniqueCondition(NStorage::TTableRecord& /*unique*/, const TUserLandings& /*info*/) const override {
        throw TWithBackTrace<yexception>() << "unexpected call";
    }
};
