#include "named_filters.h"

#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TNamedFiltersStorage::TNamedFiltersStorage(const IHistoryContext& context, const THistoryConfig& hConfig)
    : TBase(context, hConfig)
{
}

bool TNamedFilter::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, DisplayName);
    if (DisplayName.empty()) {
        return false;
    }
    ui32 revision;
    if (decoder.GetRevision() > -1) {
        READ_DECODER_VALUE_TEMP(decoder, values, revision, Revision);
        Revision = revision;
    }
    READ_DECODER_VALUE(decoder, values, Description);
    READ_DECODER_VALUE(decoder, values, FilterType);
    if (FilterType.empty()) {
        return false;
    }
    READ_DECODER_VALUE(decoder, values, Priority);

    {
        NJson::TJsonValue filterJson;
        READ_DECODER_VALUE_JSON(decoder, values, filterJson, Filter);
        if (!Filter.DeserializeFromJson(filterJson) || Filter.IsEmpty()) {
            return false;
        }
    }
    return true;
}

NStorage::TTableRecord TNamedFilter::SerializeToTableRecord() const {
    NStorage::TTableRecord record;
    record.Set("id", Id);
    record.Set("display_name", DisplayName);
    record.Set("description", Description);
    record.Set("type", FilterType);
    record.Set("filter", Filter.SerializeToJson().GetStringRobust());
    record.Set("priority", Priority);
    if (HasRevision()) {
        record.Set("revision", *Revision);
    }
    return record;
}

NJson::TJsonValue TNamedFilter::GetReport() const {
    NJson::TJsonValue report;
    report["id"] = Id;
    report["display_name"] = DisplayName;
    report["description"] = Description;
    report["type"] = FilterType;
    report["filter"] = Filter.SerializeToJson();
    report["priority"] = Priority;
    if (HasRevision()) {
        report["revision"] = Revision.GetRef();
    }
    return report;
}

NJson::TJsonValue TNamedFilter::GetReport(TVector<TNamedFilter>&& objectContainer) {
    NJson::TJsonArray report;
    std::sort(objectContainer.begin(), objectContainer.end());
    for (auto&& l : objectContainer) {
        report.AppendValue(l.GetReport());
    }
    return report;
}

TNamedFilter::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    DisplayName = GetFieldDecodeIndex("display_name", decoderBase);
    Description = GetFieldDecodeIndex("description", decoderBase);
    FilterType = GetFieldDecodeIndex("type", decoderBase);
    Filter = GetFieldDecodeIndex("filter", decoderBase);
    Priority = GetFieldDecodeIndex("priority", decoderBase);
    Revision = GetFieldDecodeIndex("revision", decoderBase);
}

TSet<TString> TNamedFiltersStorage::GetNamedFilters(const TInstant reqActuality) const {
    TSet<TString> namedFilter;
    const auto action = [&namedFilter](const auto& stateId, const auto& /* entity */) { namedFilter.insert(stateId); };
    Y_ENSURE(ForObjectsMap(action, reqActuality));
    return namedFilter;
}

TTagsFilters TNamedFiltersStorage::GetdFiltersByIds(const TSet<TString>& ids, const TInstant reqActuality) const {
    TTagsFilters result;
    auto action = [&result](const TNamedFilter& namedFilter) -> void {
        result.push_back(namedFilter.GetFilter());
    };
    Y_ENSURE(
          ForObjectsList(action, reqActuality, &ids) &&
          result.size() == ids.size()
        , "cannot GetdFiltersByIds"
    );
    return result;
}
