#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>

#include <drive/backend/tags/tags_filter.h>

#include <rtline/util/types/accessor.h>

class TNamedFilter {
    R_FIELD(TString, Id, "uuid_generate_v4()");
    R_FIELD(TString, DisplayName);
    R_FIELD(TString, Description);
    R_FIELD(TString, FilterType);
    R_FIELD(TTagsFilter, Filter);
    R_FIELD(i32, Priority, 0);
    R_OPTIONAL(ui32, Revision);

public:
    using TId = TString;

    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, DisplayName, -1);
        R_FIELD(i32, Description, -1);
        R_FIELD(i32, FilterType, -1);
        R_FIELD(i32, Filter, -1);
        R_FIELD(i32, Priority, -1);
        R_FIELD(i32, Revision, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

public:
    TNamedFilter() = default;

    static TString GetTableName() {
        return "named_filters";
    }

    static TString GetHistoryTableName() {
        return GetTableName() + "_history";
    }

    TString GetInternalId() const {
        return GetId();
    }

    const TString GetName() {
        return GetId();
    }

    bool operator!() const {
        return !Id;
    }

    bool operator==(const TNamedFilter& item) const {
        return GetId() == item.GetId();
    }

    bool operator<(const TNamedFilter& item) const {
        return std::tie(Priority, Id) < std::tie(item.Priority, item.Id);
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    bool DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& /*errors*/, const IHistoryContext* hContext) {
        return DeserializeWithDecoder(decoder, values, hContext);
    }

    template <class TContainer>
    static TSet<TString> GetIds(const TContainer& namedFilters) {
        TSet<TString> result;
        for (const auto& namedFilter : namedFilters) {
            result.insert(namedFilter.GetId());
        }
    }

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue GetReport() const;
    static NJson::TJsonValue GetReport(TVector<TNamedFilter>&& objectContainer);

};

class TNamedFiltersConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TNamedFilter& object) {
        return BuildCondition(object.GetId());
    }
};

class TNamedFiltersStorage : public TDBEntitiesManager<TNamedFilter, TNamedFiltersConditionConstructor> {
    using TBase = TDBEntitiesManager<TNamedFilter, TNamedFiltersConditionConstructor>;

public:
    TNamedFiltersStorage(const IHistoryContext& context, const THistoryConfig& hConfig);

    TSet<TString> GetNamedFilters(const TInstant reqActuality = TInstant::Zero()) const;
    TTagsFilters GetdFiltersByIds(const TSet<TString>& ids, const TInstant reqActuality = TInstant::Zero()) const;
};
