#include "private_data.h"

#include <drive/backend/logging/events.h>

#include <rtline/library/storage/abstract.h>
#include <rtline/util/algorithm/type_traits.h>

// Datasync-based

IPrivateDataAcquisitionCallback::IPrivateDataAcquisitionCallback(ui16 persdataResponses)
    : ExpectedResponses(persdataResponses)
    , ReceivedResponces(0)
    , EventLogState(NDrive::TEventLog::CaptureState())
{
}

IPrivateDataAcquisitionCallback::IPrivateDataAcquisitionCallback()
    : IPrivateDataAcquisitionCallback(0)
{
}

void IPrivateDataAcquisitionCallback::SetExpectedResponses(ui16 responcesCount) {
    ExpectedResponses = responcesCount;
    if (ExpectedResponses == 0) {
        ProcessAllResponses();
    }
}

void IPrivateDataAcquisitionCallback::OnPassportReceipt(const TString& revision, const TUserPassportData& passport) {
    NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
    DoOnPassportReceipt(revision, passport);
    RegisterResponse();
}

void IPrivateDataAcquisitionCallback::OnDrivingLicenseReceipt(const TString& revision, const TUserDrivingLicenseData& drivingLicense) {
    NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
    DoOnDrivingLicenseReceipt(revision, drivingLicense);
    RegisterResponse();
}

void IPrivateDataAcquisitionCallback::OnUnsuccessfulResponse(EPrivateDataType documentType, const TString& revision, const NNeh::THttpAsyncReport* response) {
    NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
    if (response) {
        NJson::TJsonValue report;
        if (response->HasReport()) {
            report = NJson::ToJson(NJson::JsonString(response->GetReportSafe()));
        }
        NDrive::TEventLog::Log("PrivateDataAcquisitionError", NJson::TMapBuilder
            ("document_type", NJson::ToJson(NJson::Stringify(documentType)))
            ("revision", revision)
            ("code", response->GetHttpCode())
            ("report", std::move(report))
        );
    }
    DoOnUnsuccessfulResponse(documentType, revision);
    RegisterResponse();
}

void IPrivateDataAcquisitionCallback::RegisterResponse() {
    if (++ReceivedResponces == ExpectedResponses) {
        ProcessAllResponses();
    }
}


IPrivateDataJsonCallback::IPrivateDataJsonCallback(NUserReport::TReportTraits reportTraits)
    : IPrivateDataAcquisitionCallback()
    , ReportTraits(reportTraits)
    , PrivateDataReport(NJson::TJsonValue::UNDEFINED)
{}

void IPrivateDataJsonCallback::SetBaseReport(NJson::TJsonValue&& json) {
    TGuard<TMutex> g(Mutex);
    ResultReport = json;
}

void IPrivateDataJsonCallback::DoOnPassportReceipt(const TString& revision, const TUserPassportData& passport) {
    TGuard<TMutex> g(Mutex);
    NJson::TJsonValue passportJson(passport.SerializeToJson(ReportTraits));
    passportJson.InsertValue("revision", revision);
    PrivateDataReport["passport"].AppendValue(std::move(passportJson));
}

void IPrivateDataJsonCallback::DoOnDrivingLicenseReceipt(const TString& revision, const TUserDrivingLicenseData& drivingLicense) {
    TGuard<TMutex> g(Mutex);
    NJson::TJsonValue drivingLicenseJson(drivingLicense.SerializeToJson(ReportTraits));
    drivingLicenseJson.InsertValue("revision", revision);
    PrivateDataReport["driving_license"].AppendValue(std::move(drivingLicenseJson));
}

void IPrivateDataJsonCallback::DoOnUnsuccessfulResponse(EPrivateDataType /*documentType*/, const TString& /*revision*/) {}

void IPrivateDataJsonCallback::ProcessAllResponses() {
    {
        TGuard<TMutex> g(Mutex);
        if (PrivateDataReport != NJson::TJsonValue::UNDEFINED) {
            ResultReport.InsertValue("documents", PrivateDataReport);
        }
    }
    DoProcessAllResponses();
}

bool IPrivateDataStorage::GetDrivingLicenseSync(const TUserIdInfo& user, const TString& revision, TUserDrivingLicenseData& drivingLicense) const {
    auto asyncResult = GetDrivingLicense(user, revision);
    asyncResult.Wait();
    if (!asyncResult.HasValue()) {
        ERROR_LOG << "cannot GetDrivingLicenseSync: " << NThreading::GetExceptionMessage(asyncResult);
        return false;
    }

    drivingLicense = asyncResult.ExtractValue();
    return true;
}

void IPrivateDataStorage::GetDrivingLicense(const TUserIdInfo& user, const TString& revision, IPrivateDataAcquisitionCallback::TPtr callback) const {
    auto asyncResult = GetDrivingLicense(user, revision);
    if (callback) {
        asyncResult.Subscribe([callback, revision](const auto& r) {
            if (r.HasValue()) {
                callback->OnDrivingLicenseReceipt(revision, r.GetValue());
            } else {
                callback->OnUnsuccessfulResponse(EPrivateDataType::DrivingLicense, revision, nullptr);
            }
        });
    }
}

bool IPrivateDataStorage::GetPassportSync(const TUserIdInfo& user, const TString& revision, TUserPassportData& passport) const {
    auto asyncResult = GetPassport(user, revision);
    asyncResult.Wait();
    if (!asyncResult.HasValue()) {
        ERROR_LOG << "cannot GetDrivingLicenseSync: " << NThreading::GetExceptionMessage(asyncResult);
        return false;
    }

    passport = asyncResult.ExtractValue();
    return true;
}

void IPrivateDataStorage::GetPassport(const TUserIdInfo& user, const TString& revision, IPrivateDataAcquisitionCallback::TPtr callback) const {
    auto asyncResult = GetPassport(user, revision);
    if (callback) {
        asyncResult.Subscribe([callback, revision](const auto& r) {
            if (r.HasValue()) {
                callback->OnPassportReceipt(revision, r.GetValue());
            } else {
                callback->OnUnsuccessfulResponse(EPrivateDataType::Passport, revision, nullptr);
            }
        });
    }
}

TDatabasePrivateDataStorage::TDatabasePrivateDataStorage(const TAtomicSharedPtr<NRTLine::IVersionedStorage>& storage)
    : Storage(storage)
{
}

TDatabasePrivateDataStorage::~TDatabasePrivateDataStorage() {
}

NThreading::TFuture<NJson::TJsonValue> TDatabasePrivateDataStorage::Get(const TString& key, const TString& fallbackKey) const {
    auto k = NThreading::MakeFuture(key);
    return k.Apply([this, fallbackKey](const NThreading::TFuture<TString>& k) -> NJson::TJsonValue {
        const auto& key = k.GetValue();

        TString data;
        bool acquired = Yensured(Storage)->GetValue(key, data);
        if (!acquired) {
            acquired = Yensured(Storage)->GetValue(fallbackKey, data);
        }
        if (!acquired) {
            throw TDatasyncNotFound() << "cannot GetValue " << key;
        }
        return NJson::ReadJsonFastTree(data);
    });
}

NThreading::TFuture<TUserDrivingLicenseData> TDatabasePrivateDataStorage::GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const {
    auto key = TStringBuilder() << user.GetUserId() << ':' << "driving_license" << ':' << revision;
    auto fallbackKey = TStringBuilder() << user.GetUid() << ':' << "driving_license" << ':' << revision;
    return Get(key, fallbackKey).Apply([](const NThreading::TFuture<NJson::TJsonValue>& j) {
        TUserDrivingLicenseData result;
        Y_ENSURE(result.ParseFromDatasync(j.GetValue()), "cannot ParseFromDatasync: " << j.GetValue().GetStringRobust());
        return result;
    });
}

NThreading::TFuture<TUserPassportData> TDatabasePrivateDataStorage::GetPassport(const TUserIdInfo& user, const TString& revision) const {
    auto key = TStringBuilder() << user.GetUserId() << ':' << "passport" << ':' << revision;
    auto fallbackKey = TStringBuilder() << user.GetUid() << ':' << "passport" << ':' << revision;
    return Get(key, fallbackKey).Apply([](const NThreading::TFuture<NJson::TJsonValue>& j) {
        TUserPassportData result;
        Y_ENSURE(result.ParseFromDatasync(j.GetValue()), "cannot ParseFromDatasync: " << j.GetValue().GetStringRobust());
        return result;
    });
}

NThreading::TFuture<void> TDatabasePrivateDataStorage::Set(const TString& key, NJson::TJsonValue&& data) const {
    auto k = NThreading::MakeFuture(key);
    return k.Apply([this, data = std::move(data)](const NThreading::TFuture<TString>& k) {
        const auto& key = k.GetValue();
        bool inserted = Yensured(Storage)->SetValue(key, data.GetStringRobust());
        Y_ENSURE(inserted, "cannot SetValue " << key);
    });
}

NThreading::TFuture<void> TDatabasePrivateDataStorage::UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const {
    auto key = TStringBuilder() << user.GetUserId() << ':' << "driving_license" << ':' << revision;
    auto legacyKey = TStringBuilder() << user.GetUid() << ':' << "driving_license" << ':' << revision;
    auto data = payload.SerializeToDatasyncJson();
    auto legacyResult = Set(legacyKey, MakeCopy(data));
    auto result = Set(key, std::move(data));
    return NThreading::WaitAll(legacyResult, result);
}

NThreading::TFuture<void> TDatabasePrivateDataStorage::UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const {
    auto key = TStringBuilder() << user.GetUserId() << ':' << "passport" << ':' << revision;
    auto legacyKey = TStringBuilder() << user.GetUid() << ':' << "passport" << ':' << revision;
    auto data = payload.SerializeToDatasyncJson();
    auto legacyResult = Set(legacyKey, MakeCopy(data));
    auto result = Set(key, std::move(data));
    return NThreading::WaitAll(legacyResult, result);
}

TDatasyncPrivateDataStorage::TPrivateDataGetReportWrapper::TPrivateDataGetReportWrapper(TAtomicSharedPtr<IPrivateDataAcquisitionCallback> privateDataCallback, EPrivateDataType documentType, const TString& revision)
    : PrivateDataCallback(privateDataCallback)
    , DocumentType(documentType)
    , Revision(revision)
    , EventLogState(NDrive::TEventLog::CaptureState())
{
}

void TDatasyncPrivateDataStorage::TPrivateDataGetReportWrapper::OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) {
    NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
    for (auto&& report : reports) {
        if (report.GetHttpCode() / 100 != 2) {
            PrivateDataCallback->OnUnsuccessfulResponse(DocumentType, Revision, &report);
        } else {
            NJson::TJsonValue jsonReport;
            if (!NJson::ReadJsonFastTree(report.GetReportSafe(), &jsonReport)) {
                PrivateDataCallback->OnUnsuccessfulResponse(DocumentType, Revision, &report);
                return;
            }
            if (DocumentType == EPrivateDataType::Passport) {
                TUserPassportData passport;
                if (!passport.ParseFromDatasync(jsonReport)) {
                    PrivateDataCallback->OnUnsuccessfulResponse(DocumentType, Revision, &report);
                    return;
                }
                PrivateDataCallback->OnPassportReceipt(Revision, passport);
            } else if (DocumentType == EPrivateDataType::DrivingLicense) {
                TUserDrivingLicenseData drivingLicense;
                if (!drivingLicense.ParseFromDatasync(jsonReport)) {
                    PrivateDataCallback->OnUnsuccessfulResponse(DocumentType, Revision, &report);
                    return;
                }
                PrivateDataCallback->OnDrivingLicenseReceipt(Revision, drivingLicense);
            }
        }
    }
}

TDatasyncPrivateDataStorage::TPrivateDataUpdateReportWrapper::TPrivateDataUpdateReportWrapper(TAtomicSharedPtr<IPrivateDataUpdateCallback> privateDataCallback, const TString& revision, EPrivateDataType documentType, const NJson::TJsonValue& payload)
    : PrivateDataCallback(privateDataCallback)
    , Revision(revision)
    , DocumentType(documentType)
    , Payload(payload)
    , EventLogState(NDrive::TEventLog::CaptureState())
{
}

void TDatasyncPrivateDataStorage::TPrivateDataUpdateReportWrapper::OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) {
    NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
    for (auto&& report : reports) {
        if (report.GetHttpCode() / 100 == 2) {
            if (PrivateDataCallback) {
                PrivateDataCallback->OnSuccessfulUpdate(DocumentType, Revision);
            }
        } else {
            NJson::TJsonValue rprt;
            if (report.HasReport()) {
                rprt = NJson::ToJson(NJson::JsonString(report.GetReportSafe()));
            }
            NDrive::TEventLog::Log("PrivateDataUpdateError", NJson::TMapBuilder
                ("document_type", NJson::ToJson(NJson::Stringify(DocumentType)))
                ("revision", Revision)
                ("payload", Payload)
                ("code", report.GetHttpCode())
                ("report", std::move(rprt))
            );
            if (PrivateDataCallback) {
                PrivateDataCallback->OnUnsuccessfulUpdate(DocumentType, Revision);
            }
        }
    }
}

TDatasyncPrivateDataStorage::TDatasyncPrivateDataStorage(const TDatasyncClient& datasync)
    : Datasync(datasync)
{
}

void TDatasyncPrivateDataStorage::GetPassport(const TUserIdInfo& user, const TString& revision, TAtomicSharedPtr<IPrivateDataAcquisitionCallback> callback) const {
    auto uid = user.GetPassportUid();
    auto callbackWrapper = THolder<TPrivateDataGetReportWrapper>(new TPrivateDataGetReportWrapper(callback, EPrivateDataType::Passport, revision));
    Datasync.PerformCollectionRequest("documents_unverified", revision, uid, std::move(callbackWrapper));
}

NThreading::TFuture<TUserPassportData> TDatasyncPrivateDataStorage::GetPassport(const TUserIdInfo& user, const TString& revision) const {
    auto uid = user.GetPassportUid();
    return Datasync.Get("documents_unverified", revision, uid).Apply([uid, revision](const NThreading::TFuture<TDatasyncClient::TResponse>& r) {
        auto report = r.GetValue();
        Y_ENSURE_EX(
            report.GetCode() != 404,
            TDatasyncNotFound() << "passport for (" << uid << ", " << revision << ") not found"
        );
        Y_ENSURE_EX(
            report.GetCode() / 100 == 2,
            TDatasyncException() << "passport data fetch was unsuccessful for " << uid << " " << revision << ": " << report.GetCode() << " " << report.GetError()
        );
        TUserPassportData passport;
        Y_ENSURE_EX(
            passport.ParseFromDatasync(report.GetValue()),
            TDatasyncException() << "unable to parse passport data for " << uid << " " << revision << ": " << report.GetValue().GetStringRobust()
        );
        return passport;
    });
}

bool TDatasyncPrivateDataStorage::GetPassportSync(const TUserIdInfo& user, const TString& revision, TUserPassportData& passport) const {
    auto uid = user.GetPassportUid();
    auto response = Datasync.PerformSyncCollectionRequest("documents_unverified", revision, uid);
    if (response.Code() / 100 != 2) {
        ERROR_LOG << "DATASYNC RESPONSE CODE: " << response.Code() << Endl;
        NDrive::TEventLog::Log("GetPassportSyncError", NJson::TMapBuilder
            ("code", response.Code())
            ("content", NJson::ToJson(NJson::JsonString(response.Content())))
            ("revision", revision)
            ("uid", uid)
        );
        return false;
    }
    auto rawContent = response.Content();
    NJson::TJsonValue payloadJson;
    return (NJson::ReadJsonFastTree(rawContent, &payloadJson) && passport.ParseFromDatasync(payloadJson));
}

void TDatasyncPrivateDataStorage::GetDrivingLicense(const TUserIdInfo& user, const TString& revision, TAtomicSharedPtr<IPrivateDataAcquisitionCallback> callback) const {
    auto uid = user.GetPassportUid();
    auto callbackWrapper = THolder<TPrivateDataGetReportWrapper>(new TPrivateDataGetReportWrapper(callback, EPrivateDataType::DrivingLicense, revision));
    Datasync.PerformCollectionRequest("driving_license_unverified", revision, uid, std::move(callbackWrapper));
}

NThreading::TFuture<TUserDrivingLicenseData> TDatasyncPrivateDataStorage::GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const {
    auto uid = user.GetPassportUid();
    return Datasync.Get("driving_license_unverified", revision, uid).Apply([uid, revision](const NThreading::TFuture<TDatasyncClient::TResponse>& r) {
        auto report = r.GetValue();
        Y_ENSURE_EX(
            report.GetCode() != 404,
            TDatasyncNotFound() << "license for (" << uid << ", " << revision << ") not found"
        );
        Y_ENSURE_EX(
            report.GetCode() / 100 == 2,
            TDatasyncException() << "driving license data fetch was unsuccessful for " << uid << " " << revision << ": " << report.GetCode() << " " << report.GetError() << " " << report.GetValue().GetStringRobust()
        );
        TUserDrivingLicenseData drivingLicense;
        Y_ENSURE_EX(
            drivingLicense.ParseFromDatasync(report.GetValue()),
            TDatasyncException() << "unable to parse driving license data for " << uid << " " << revision << ": " << report.GetValue().GetStringRobust()
        );
        return drivingLicense;
    });
}

bool TDatasyncPrivateDataStorage::GetDrivingLicenseSync(const TUserIdInfo& user, const TString& revision, TUserDrivingLicenseData& drivingLicense) const {
    auto uid = user.GetPassportUid();
    auto response = Datasync.PerformSyncCollectionRequest("driving_license_unverified", revision, uid);
    if (response.Code() / 100 != 2) {
        ERROR_LOG << "DATASYNC RESPONSE CODE: " << response.Code() << Endl;
        NDrive::TEventLog::Log("GetDrivingLicenseSyncError", NJson::TMapBuilder
            ("code", response.Code())
            ("content", NJson::ToJson(NJson::JsonString(response.Content())))
            ("revision", revision)
            ("uid", uid)
        );
        return false;
    }
    auto rawContent = response.Content();
    NJson::TJsonValue payloadJson;
    return (NJson::ReadJsonFastTree(rawContent, &payloadJson) && drivingLicense.ParseFromDatasync(payloadJson));
}

NThreading::TFuture<void> TDatasyncPrivateDataStorage::UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const {
    auto uid = user.GetPassportUid();
    return Datasync.Update("documents_unverified", revision, uid, payload.SerializeToDatasyncJson()).Apply([uid, revision](const NThreading::TFuture<TDatasyncClient::TResponse>& r) -> NThreading::TFuture<void> {
        if (r.HasException() || ! r.HasValue()) {
            return NThreading::TExceptionFuture<TDatasyncException>() << "TDatasyncPrivateDataStorage::UpdatePassport cannot update passport: " << NThreading::GetExceptionMessage(r);
        }
        auto report = r.GetValue();
        if (report.GetCode() == 404) {
            return NThreading::TExceptionFuture<TDatasyncNotFound>() << "TDatasyncPrivateDataStorage::UpdatePassport cannot find passport for " << uid
                << ", revision: " << revision << ", code: " << report.GetCode() << ", error: " << report.GetError() << ", message: " << report.GetValue().GetStringRobust();
        }
        if (report.GetCode() / 100 != 2) {
            return NThreading::TExceptionFuture<TDatasyncException>() << "TDatasyncPrivateDataStorage::UpdatePassport cannot update passport for " << uid
                << ", revision: " << revision << ", code: " << report.GetCode() << ", error: " << report.GetError() << ", message: " << report.GetValue().GetStringRobust();
        }
        return NThreading::MakeFuture();
    });
}

NThreading::TFuture<void> TDatasyncPrivateDataStorage::UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const {
    auto uid = user.GetPassportUid();
    return Datasync.Update("driving_license_unverified", revision, uid, payload.SerializeToDatasyncJson()).Apply([uid, revision](const NThreading::TFuture<TDatasyncClient::TResponse>& r) -> NThreading::TFuture<void> {
        if (r.HasException() || ! r.HasValue()) {
            return NThreading::TExceptionFuture<TDatasyncException>() << "TDatasyncPrivateDataStorage::UpdateDrivingLicense cannot update driving license: " << NThreading::GetExceptionMessage(r);
        }
        auto report = r.GetValue();
        if (report.GetCode() == 404) {
            return NThreading::TExceptionFuture<TDatasyncNotFound>() << "TDatasyncPrivateDataStorage::UpdateDrivingLicense cannot find license for " << uid
                << ", revision: " << revision << ", code: " << report.GetCode() << ", error: " << report.GetError() << ", message: " << report.GetValue().GetStringRobust();
        }
        if (report.GetCode() / 100 != 2) {
            return NThreading::TExceptionFuture<TDatasyncException>() << "TDatasyncPrivateDataStorage::UpdateDrivingLicense cannot update driving license for " << uid
                << ", revision: " << revision << ", code: " << report.GetCode() << ", error: " << report.GetError() << ", message: " << report.GetValue().GetStringRobust();
        }
        return NThreading::MakeFuture();
    });
}

// Fake

TString TFakePrivateDataStorage::GetDocumentKey(const TUserIdInfo& user, std::string_view revision) const {
    const auto& id = user.GetUid() ? user.GetUid() : user.GetUserId();
    return TStringBuilder() << "(" << id << ", " << revision << ")";
}

NThreading::TFuture<TUserPassportData> TFakePrivateDataStorage::GetPassport(const TUserIdInfo& user, const TString& revision) const {
    const auto& key = GetDocumentKey(user, revision);
    DEBUG_LOG << "Get from fake passport storage : " << key << Endl;
    if (PassportStorage.contains(key) && !GetIsFaultyUnsafe()) {
        const auto& passport = PassportStorage[key];
        return NThreading::MakeFuture(passport);
    } else {
        return NThreading::TExceptionFuture<TDatasyncNotFound>() << "missing key: " << key;
    }
}

NThreading::TFuture<TUserDrivingLicenseData> TFakePrivateDataStorage::GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const {
    const auto& key = GetDocumentKey(user, revision);
    DEBUG_LOG << "Get from fake driving_license storage : " << key << Endl;
    if (DrivingLicenseStorage.contains(key) && !GetIsFaultyUnsafe()) {
        const auto& drivingLicense = DrivingLicenseStorage[key];
        return NThreading::MakeFuture(drivingLicense);
    } else {
        return NThreading::TExceptionFuture<TDatasyncNotFound>() << "missing key: " << key;
    }
}

NThreading::TFuture<void> TFakePrivateDataStorage::UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const {
    const auto& key = GetDocumentKey(user, revision);
    DEBUG_LOG << "Put into fake passport storage : " << key << Endl;
    if (!GetIsFaultyUnsafe()) {
        PassportStorage[key] = payload;
        return NThreading::MakeFuture();
    } else {
        return NThreading::TExceptionFuture<TDatasyncException>() << "TFakePrivateDataStorage::UpdatePassport cannot update passport";
    }
}

NThreading::TFuture<void> TFakePrivateDataStorage::UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const {
    const auto& key = GetDocumentKey(user, revision);
    DEBUG_LOG << "Put into fake driving license storage : " << key << Endl;
    if (!GetIsFaultyUnsafe()) {
        DrivingLicenseStorage[key] = payload;
        return NThreading::MakeFuture();
    } else {
        return NThreading::TExceptionFuture<TDatasyncException>() << "TFakePrivateDataStorage::UpdateDrivingLicense cannot update driving license";
    }
}
