#pragma once

#include <drive/backend/logging/events.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/datasync/client.h>

enum EPrivateDataType {
    Passport,
    DrivingLicense
};

struct TDatasyncException : public virtual yexception {
};

struct TDatasyncNotFound : public virtual TDatasyncException {
};

class IPrivateDataAcquisitionCallback {
public:
    using TPtr = TAtomicSharedPtr<IPrivateDataAcquisitionCallback>;

public:
    IPrivateDataAcquisitionCallback(ui16 persdataResponses);
    IPrivateDataAcquisitionCallback();

    virtual ~IPrivateDataAcquisitionCallback() = default;

    void SetExpectedResponses(ui16 responcesCount);

    void OnPassportReceipt(const TString& revision, const TUserPassportData& passport);
    void OnDrivingLicenseReceipt(const TString& revision, const TUserDrivingLicenseData& drivingLicense);
    void OnUnsuccessfulResponse(EPrivateDataType documentType, const TString& revision, const NNeh::THttpAsyncReport* response);

private:
    virtual void DoOnPassportReceipt(const TString& revision, const TUserPassportData& passport) = 0;
    virtual void DoOnDrivingLicenseReceipt(const TString& revision, const TUserDrivingLicenseData& drivingLicense) = 0;
    virtual void DoOnUnsuccessfulResponse(EPrivateDataType documentType, const TString& revision) = 0;
    virtual void ProcessAllResponses() = 0;

    void RegisterResponse();

private:
    ui16 ExpectedResponses;
    std::atomic<ui16> ReceivedResponces;
    NDrive::TEventLog::TState EventLogState;
};

class IPrivateDataJsonCallback : public IPrivateDataAcquisitionCallback {
public:
    IPrivateDataJsonCallback(NUserReport::TReportTraits reportTraits);

    void SetBaseReport(NJson::TJsonValue&& json);

    virtual void DoOnPassportReceipt(const TString& revision, const TUserPassportData& passport) override;
    virtual void DoOnDrivingLicenseReceipt(const TString& revision, const TUserDrivingLicenseData& drivingLicense) override;
    virtual void DoOnUnsuccessfulResponse(EPrivateDataType /*documentType*/, const TString& /*revision*/) override;
    virtual void ProcessAllResponses() override;

private:
    virtual void DoProcessAllResponses() = 0;

protected:
    NJson::TJsonValue ResultReport;

private:
    TMutex Mutex;
    NUserReport::TReportTraits ReportTraits;
    NJson::TJsonValue PrivateDataReport;
};

class IPrivateDataUpdateCallback {
public:
    virtual void OnSuccessfulUpdate(EPrivateDataType documentType, const TString& revision) = 0;
    virtual void OnUnsuccessfulUpdate(EPrivateDataType documentType, const TString& revision) = 0;
    virtual ~IPrivateDataUpdateCallback() noexcept(false) {
    }
};

class IPrivateDataStorage {
public:
    virtual NThreading::TFuture<TUserPassportData> GetPassport(const TUserIdInfo& user, const TString& revision) const = 0;
    virtual bool GetPassportSync(const TUserIdInfo& user, const TString& revision, TUserPassportData& passport) const;
    virtual void GetPassport(const TUserIdInfo& user, const TString& revision, IPrivateDataAcquisitionCallback::TPtr callback) const;

    virtual NThreading::TFuture<TUserDrivingLicenseData> GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const = 0;
    virtual bool GetDrivingLicenseSync(const TUserIdInfo& user, const TString& revision, TUserDrivingLicenseData& drivingLicense) const;
    virtual void GetDrivingLicense(const TUserIdInfo& user, const TString& revision, IPrivateDataAcquisitionCallback::TPtr callback) const;

    virtual NThreading::TFuture<void> UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const = 0;
    virtual NThreading::TFuture<void> UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const = 0;

    virtual ~IPrivateDataStorage() {
    }
};

class TDatabasePrivateDataStorage : public IPrivateDataStorage {
public:
    TDatabasePrivateDataStorage(const TAtomicSharedPtr<NRTLine::IVersionedStorage>& storage);
    ~TDatabasePrivateDataStorage();

    virtual NThreading::TFuture<TUserPassportData> GetPassport(const TUserIdInfo& user, const TString& revision) const override;
    virtual NThreading::TFuture<TUserDrivingLicenseData> GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const override;

    virtual NThreading::TFuture<void> UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const override;
    virtual NThreading::TFuture<void> UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const override;

private:
    NThreading::TFuture<NJson::TJsonValue> Get(const TString& key, const TString& fallbackKey) const;
    NThreading::TFuture<void> Set(const TString& key, NJson::TJsonValue&& data) const;

private:
    TAtomicSharedPtr<NRTLine::IVersionedStorage> Storage;
};

class TDatasyncPrivateDataStorage : public IPrivateDataStorage {
private:
    class TPrivateDataGetReportWrapper : public NNeh::THttpAsyncReport::ICallback {
    private:
        TAtomicSharedPtr<IPrivateDataAcquisitionCallback> PrivateDataCallback;
        EPrivateDataType DocumentType;
        TString Revision;
        NDrive::TEventLog::TState EventLogState;

    public:
        TPrivateDataGetReportWrapper(TAtomicSharedPtr<IPrivateDataAcquisitionCallback> privateDataCallback, EPrivateDataType documentType, const TString& revision);
        virtual void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override final;
    };

    class TPrivateDataUpdateReportWrapper : public NNeh::THttpAsyncReport::ICallback {
    private:
        TAtomicSharedPtr<IPrivateDataUpdateCallback> PrivateDataCallback;
        TString Revision;
        EPrivateDataType DocumentType;
        NJson::TJsonValue Payload;
        NDrive::TEventLog::TState EventLogState;

    public:
        TPrivateDataUpdateReportWrapper(TAtomicSharedPtr<IPrivateDataUpdateCallback> privateDataCallback, const TString& revision, EPrivateDataType documentType, const NJson::TJsonValue& payload);
        virtual void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override final;
    };

    const TDatasyncClient& Datasync;

public:
    TDatasyncPrivateDataStorage(const TDatasyncClient& datasync);
    virtual void GetPassport(const TUserIdInfo& user, const TString& revision, TAtomicSharedPtr<IPrivateDataAcquisitionCallback> callback) const override final;
    virtual NThreading::TFuture<TUserPassportData> GetPassport(const TUserIdInfo& user, const TString& revision) const override final;
    virtual bool GetPassportSync(const TUserIdInfo& user, const TString& revision, TUserPassportData& passport) const override final;
    virtual void GetDrivingLicense(const TUserIdInfo& user, const TString& revision, TAtomicSharedPtr<IPrivateDataAcquisitionCallback> callback) const override final;
    virtual NThreading::TFuture<TUserDrivingLicenseData> GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const override final;
    virtual bool GetDrivingLicenseSync(const TUserIdInfo& user, const TString& revision, TUserDrivingLicenseData& drivingLicense) const override final;
    virtual NThreading::TFuture<void> UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const override final;
    virtual NThreading::TFuture<void> UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const override final;
};

class TFakePrivateDataStorage : public IPrivateDataStorage {
    R_OPTIONAL(bool, IsFaulty, {}, mutable);

private:
    mutable TMap<TString, TUserPassportData> PassportStorage;
    mutable TMap<TString, TUserDrivingLicenseData> DrivingLicenseStorage;

private:
    TString GetDocumentKey(const TUserIdInfo& user, std::string_view revision) const;

public:
    TFakePrivateDataStorage()
        : IsFaulty(false)
    {
    }

    virtual NThreading::TFuture<TUserPassportData> GetPassport(const TUserIdInfo& user, const TString& revision) const override final;
    virtual NThreading::TFuture<TUserDrivingLicenseData> GetDrivingLicense(const TUserIdInfo& user, const TString& revision) const override final;
    virtual NThreading::TFuture<void> UpdatePassport(const TUserIdInfo& user, const TString& revision, const TUserPassportData& payload) const override final;
    virtual NThreading::TFuture<void> UpdateDrivingLicense(const TUserIdInfo& user, const TString& revision, const TUserDrivingLicenseData& payload) const override final;
};
