#include "drive_api.h"

#include "config.h"

#include <drive/backend/database/consistency/consistency.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/database/drive/named_filters.h>
#include <drive/backend/database/drive/takeout.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/areas/drop_object_features.h>
#include <drive/backend/areas/location.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/backend/drivematics/zone/config/zone_config.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshots/tag.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/images/database.h>
#include <drive/backend/incident/manager.h>
#include <drive/backend/incident/renins_claims/client.h>
#include <drive/backend/insurance/ingos/client.h>
#include <drive/backend/insurance/renins/client.h>
#include <drive/backend/insurance/task/history.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/major/client.h>
#include <drive/backend/notifications/mail/binder.h>
#include <drive/backend/offers/abstract.h>
#include <drive/backend/offers/discount.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/checkers.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/actions/offer_price.h>
#include <drive/backend/roles/config.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/sessions/manager/additional_service.h>
#include <drive/backend/sessions/manager/dedicated_fleet.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags_search.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/backend/users/login.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/autocode/client.h>
#include <drive/library/cpp/datasync/client.h>
#include <drive/library/cpp/datasync_queue/client.h>
#include <drive/library/cpp/disk/client.h>
#include <drive/library/cpp/element/client.h>
#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/octopus/client.h>
#include <drive/library/cpp/passport/client.h>
#include <drive/library/cpp/samsara/client.h>
#include <drive/library/cpp/solomon/client.h>
#include <drive/library/cpp/staff/client.h>
#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/takeout/client.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/trust/lpm.h>
#include <drive/library/cpp/yang/client.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/polyline.h>
#include <rtline/library/json/proto/adapter.h>
#include <rtline/library/storage/ydb/structured.h>
#include <rtline/library/metasearch/simple/config.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/network/neh_request.h>
#include <rtline/util/types/accessor.h>

#include <util/system/env.h>

using TDurationGuard = TTimeGuardImpl<false, TLOG_INFO, true>;

TDriveAPI::~TDriveAPI() {
}

TString TDriveAPI::GetDatabaseName() const {
    return Config.GetDBName();
}

TModelsDB& TDriveAPI::GetModelsDB() const {
    return *Yensured(Models);
}

TMaybe<TIntroViewInfo> TDriveAPI::GetUserLandingAlertViewsInfo(const TString& userId, const TString& landingAlertId, NDrive::TEntitySession& session) const {
    auto taggedUser = GetTagsManager().GetUserTags().RestoreObject(userId, session);
    if (taggedUser) {
        TVector<TDBTag> tags = taggedUser->GetTagsByClass<TIntroViewsInfoTag>();
        const TIntroViewsInfoTag* introTag = nullptr;
        for (auto&& t : tags) {
            introTag = t.GetTagAs<TIntroViewsInfoTag>();
            if (!introTag) {
                session.SetErrorInfo("GetUserLandingAlertViewsInfo", "cannot cast " + t.GetTagId() + " to IntroViewsInfoTag");
                return {};
            }
            TMaybe<TIntroViewInfo> viewsInfo = introTag->GetInfoByLanding(landingAlertId);
            if (!!viewsInfo) {
                return viewsInfo;
            }
        }
    }
    return {};
}

bool TDriveAPI::IncrementUserLandingAlertViewsInfo(const TString& userId, const TString& landingAlertId, const TDuration intervalIgnore, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto taggedUser = GetTagsManager().GetUserTags().RestoreObject(userId, session);
    TDBTag introUserTag;
    if (taggedUser) {
        TVector<TDBTag> tags = taggedUser->GetTagsByClass<TIntroViewsInfoTag>();
        for (auto&& t : tags) {
            introUserTag = t;
            TMaybe<TIntroViewInfo> viewsInfo = t.GetTagAs<TIntroViewsInfoTag>()->GetInfoByLanding(landingAlertId);
            if (!!viewsInfo) {
                break;
            }
        }
    }
    if (!introUserTag) {
        THolder<TIntroViewsInfoTag> tag(new TIntroViewsInfoTag(TIntroViewsInfoTag::TypeName));
        Y_ENSURE_BT(tag->IncrementViews(landingAlertId, intervalIgnore));
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag.Release(), userId, userId, server, session)) {
            return false;
        }
    } else {
        auto tagInfo = introUserTag.MutableTagAs<TIntroViewsInfoTag>();
        if (!tagInfo) {
            session.SetErrorInfo("IncrementUserLandingAlertViewsInfo", "cannot cast " + introUserTag.GetTagId() + " to IntroViewsInfoTag");
            return false;
        }
        if (tagInfo->IncrementViews(landingAlertId, intervalIgnore)) {
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(introUserTag, userId, session)) {
                return false;
            }
        }
    }
    return true;
}

bool TDriveAPI::DetectObjectTagsOperator(const IEntityTagsManager*& tagsManager, const TSet<TString>& tagIds, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const {
    if (!GetTagsManager().GetDeviceTags().RestoreTags(tagIds, tags, session)) {
        return false;
    }
    if (tags.size()) {
        tagsManager = &GetTagsManager().GetDeviceTags();
        return true;
    }
    if (!GetTagsManager().GetUserTags().RestoreTags(tagIds, tags, session)) {
        return false;
    }
    if (tags.size()) {
        tagsManager = &GetTagsManager().GetUserTags();
        return true;
    }
    session.SetErrorInfo("tags_manager_detection", "incorrect tag ids", EDriveSessionResult::IncorrectCarTags);
    return false;
}

bool TDriveAPI::DetectObjectTagsOperator(const IEntityTagsManager*& tagsManager, const TSet<TString>& tagIds, NDrive::TEntitySession& session) const {
    TVector<TDBTag> tags;
    return DetectObjectTagsOperator(tagsManager, tagIds, tags, session);
}

NDrive::TAreaIds TDriveAPI::GetAreaIds(const TGeoCoord& c) const {
    NDrive::TAreaIds result;
    const auto inserter = [&result](const TFullAreaInfo& info) {
        result.insert(info.GetArea().GetIdentifier());
        return true;
    };
    AreasDB->ProcessTagsInPoint(c, inserter, Config.GetGeoAreasFreshness());
    return result;
}

NDrive::TLocationTags TDriveAPI::GetTagsInPoint(const TGeoCoord& c, const TInternalPointContext& internalContext) const {
    NDrive::TLocationTags result;
    const auto inserter = [&result](const TFullAreaInfo& info) {
        result.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
        return true;
    };
    AreasDB->ProcessTagsInPoint(c, inserter, Config.GetGeoAreasFreshness(), internalContext);
    return result;
}

const TInsuranceHistoryManager& TDriveAPI::GetInsuranceHistoryManager() const {
    return *Yensured(InsuranceHistoryManager);
}

TVector<TInsuranceNotification> TDriveAPI::GetActualInsuranceNotifications() const {
    TVector<TInsuranceNotification> result;
    NStorage::ITableAccessor::TPtr table = Database->GetTable("insurance_tasks");
    TRecordsSet records;
    {
        auto transaction = Database->CreateTransaction(true);
        table->GetRows("", records, transaction);
    }
    for (auto&& i : records) {
        TInsuranceNotification notification;
        if (notification.DeserializeFromTableRecord(i, nullptr)) {
            result.emplace_back(notification);
        } else {
            WARNING_LOG << "Cannot parse insurance notification" << Endl;
        }
    }
    return result;
}

bool TDriveAPI::CommitInsuranceNotification(ui64 notificationId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = Database->GetTable("insurance_tasks");

    TRecordsSet affected;
    auto result = table->RemoveRow(NStorage::TRecordBuilder("notification_id", notificationId), session.GetTransaction(), &affected);
    if (!result || !result->IsSucceed()) {
        session.SetErrorInfo("insurance_task", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    for (auto&& record : affected.GetRecords()) {
        TInsuranceNotification notification;
        if (!notification.DeserializeFromTableRecord(record, nullptr)) {
            return false;
        }
        if (!InsuranceHistoryManager->AddHistory(notification, notification.GetUserId(), EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }
    return true;
}

bool TDriveAPI::InsuranceNotification(const TString& sessionId, const TInstant start, const TInstant finish, const TString& userId, const TString& objectId, const TString& historyUser, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = Database->GetTable("insurance_tasks");
    auto transaction = session.GetTransaction();
    NStorage::TTableRecord record;
    record.Set("user_id", userId).Set("object_id", objectId).Set("start", start.Seconds()).Set("finish", finish.Seconds()).Set("session_id", sessionId);

    TRecordsSet records;
    auto result = table->AddRow(record, transaction, "", &records);
    if (!result || !result->IsSucceed() || records.GetRecords().size() != 1) {
        session.SetErrorInfo("insurance_task", "AddRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    const auto& added = records.GetRecords().front();
    TInsuranceNotification notification;
    if (!notification.DeserializeFromTableRecord(added, nullptr)) {
        session.SetErrorInfo("InsuranceNotification", "cannot deserialize from " + added.SerializeToJson().GetStringRobust());
        return false;
    }
    if (!InsuranceHistoryManager->AddHistory(notification, historyUser, EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool TDriveAPI::AcceptLandings(const TUserPermissions& permissions, const TVector<TLandingAcceptance>& acceptances, const TInstant reqActuality, NDrive::TEntitySession& session) const {
    Y_UNUSED(reqActuality);
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = Database->GetTable("user_landings");
    TRecordsSet records;
    TSet<TString> landingsForAccept;
    for (auto&& i : acceptances) {
        NStorage::TTableRecord tr;
        tr.Set("user_id", permissions.GetUserId()).Set("landing_id", i.GetId()).Set("last_accepted_at", i.GetLastAcceptedAt());
        if (i.GetComment()) {
            INFO_LOG << "Setting accept_comment: " << i.GetComment() << Endl;
            tr.Set("accept_comment", i.GetComment());
        }
        NStorage::TTableRecord cond;
        cond.Set("user_id", permissions.GetUserId()).Set("landing_id", i.GetId());
        auto result = table->UpdateRow(cond, tr, transaction);
        if (!result) {
            return false;
        }
        if (result->GetAffectedRows() == 0) {
            auto addResult = table->AddIfNotExists(tr, transaction, cond);
            if (!addResult || addResult->GetAffectedRows() == 0) {
                return false;
            }
        }
        landingsForAccept.emplace(i.GetId());
    }

    auto td = GetTagsManager().GetTagsMeta().GetTagsByType(TLandingUserTag::TypeName);
    TSet<TString> tagNames;
    TMap<TString, TTagDescription::TConstPtr> tdByName;
    for (auto&& i : td) {
        tdByName.emplace(i->GetName(), i);
        tagNames.emplace(i->GetName());
    }

    auto optionalTaggedUser = GetTagsManager().GetUserTags().RestoreObject(permissions.GetUserId(), session);
    if (!optionalTaggedUser) {
        return false;
    }

    TVector<TDBTag> tagsForRemove;
    {
        for (auto&& i : optionalTaggedUser->MutableTags()) {
            auto itTagDescription = tdByName.find(i->GetName());
            if (itTagDescription != tdByName.end()) {
                const TLandingUserTag::TDescription* lDescription = itTagDescription->second->GetAs<TLandingUserTag::TDescription>();
                if (lDescription && landingsForAccept.contains(lDescription->GetLandingId())) {
                    i->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::LandingAccepted));
                    tagsForRemove.emplace_back(i);
                }
            }
        }
    }
    if (!GetTagsManager().GetUserTags().RemoveTagsSimple(tagsForRemove, permissions.GetUserId(), session, true)) {
        session.SetErrorInfo("accept_landings", "info", EDriveSessionResult::TransactionProblem);
        return false;
    }


    return true;
}

bool TDriveAPI::CleanLandings(const TString& userId, const TVector<TString>& landingIds, NDrive::TEntitySession* session) const {
    NStorage::ITransaction::TPtr transaction = session ? session->GetTransaction() : Database->CreateTransaction(false);
    TString landindCondition = "";
    if (landingIds.size()) {
        landindCondition = " AND landing_id IN (" + transaction->Quote(landingIds) + ")";
    }
    if (transaction->Exec("DELETE FROM user_landings WHERE user_id='" + userId + "'" + landindCondition) && (session || transaction->Commit())) {
        return true;
    }

    if (session) {
        session->SetErrorInfo("CleanLandings", "DELETE failed", EDriveSessionResult::TransactionProblem);
    }
    return false;
}

bool TDriveAPI::GetLandingsForUser(const NDrive::IServer* server, const TUserPermissions& permissions, const TGeoCoord* uPosition, TVector<TLanding>& result, NDrive::TEntitySession& session, TVector<TDBTag>& skippedTags, bool includeNewFormat) const {
    auto td = GetTagsManager().GetTagsMeta().GetTagsByType(TLandingUserTag::TypeName);
    TVector<TString> tagNames;
    TMap<TString, TTagDescription::TConstPtr> tdByName;
    for (auto&& i : td) {
        tagNames.emplace_back(i->GetName());
        tdByName.emplace(i->GetName(), i);
    }
    TVector<TDBTag> userTags;
    if (!GetTagsManager().GetUserTags().RestoreTags({ permissions.GetUserId() }, tagNames, userTags, session)) {
        return false;
    }

    TUserAppSettings userAppSettings(permissions.GetUserId());
    auto subscription = userAppSettings.GetLandingSubscription(session);
    if (!subscription && subscription.GetError() != TUserSettingStatus::NotFound) {
        return false;
    }
    bool unsubscribed = subscription ? IsFalse(subscription.GetRef()) : false;

    TSet<TString> userAvailableLandingIds;
    for (auto&& a : permissions.GetActionsActual()) {
        if (!!a) {
            auto checkResult = permissions.CheckActionAcceptable(*a, *server);

            if (checkResult == IDynamicActionChecker::ECheckResult::EOk) {
                userAvailableLandingIds.insert(a->GetLanding().begin(), a->GetLanding().end());
            }

            if (checkResult == IDynamicActionChecker::ECheckResult::EUserFails) {
                userAvailableLandingIds.insert(a->GetContrLanding().begin(), a->GetContrLanding().end());
            }
        }
    }
    TMap<TString, const TLandingUserTag*> landingTags;
    TSet<TString> nonDisposableLandingId;
    for (auto&& i : userTags) {
        auto it = tdByName.find(i->GetName());
        if (it != tdByName.end()) {
            const TLandingUserTag::TDescription* lDescription = it->second->GetAs<TLandingUserTag::TDescription>();
            if (lDescription) {
                if (!unsubscribed || lDescription->GetCommunicationChannelSettings().GetIgnoreSubscriptionStatus()) {
                    userAvailableLandingIds.emplace(lDescription->GetLandingId());
                    const TLandingUserTag* landingTag = i.GetTagAs<TLandingUserTag>();
                    if (landingTag) {
                        landingTags.emplace(lDescription->GetLandingId(), landingTag);
                    }
                } else {
                    INFO_LOG << "GetLandingsForUser: removing landing tag " + i->GetName() + " from " + permissions.GetUserId() + " because user is unsubscribed" << Endl;
                    i->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::UserUnsubscribedFromCommunitation));
                    skippedTags.emplace_back(i);
                }
                if (!lDescription->IsDisposable()) {
                    nonDisposableLandingId.insert(lDescription->GetLandingId());
                }
            }
        }
    }

    NStorage::ITableAccessor::TPtr table = Database->GetTable("user_landings");
    TRecordsSet records;
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    if (!table->GetRows("user_id='" + permissions.GetUserId() + "'", records, transaction)) {
        session.SetErrorInfo("landings", "info", EDriveSessionResult::TransactionProblem);
        return false;
    }

    for (auto&& i : records.GetRecords()) {
        const auto& landingId = i.Get("landing_id");
        if (nonDisposableLandingId.contains(landingId)) {
            continue;
        }
        userAvailableLandingIds.erase(landingId);
    }

    TVector<TLanding> userLandings;
    {
        if (!LandingsDB->GetCustomObjectsFromCache(userLandings, userAvailableLandingIds)) {
            session.SetErrorInfo("GetLandingsForUser", "cannot GetCustomObjectsFromCache from LandingsDB");
            return false;
        }
        const auto predRemove = [](const TLanding& landing) -> bool {
            return !landing.GetEnabled() || landing.GetDeadline() < Now();
        };
        userLandings.erase(std::remove_if(userLandings.begin(), userLandings.end(), predRemove), userLandings.end());
    }
    TSet<TString> landingGeoTags;
    for (auto&& landing : userLandings) {
        landingGeoTags.insert(landing.GetGeoTags().begin(), landing.GetGeoTags().end());
    }

    TSet<TString> userGeoTags;
    if (uPosition) {
        userGeoTags = GetAreasDB()->CheckGeoTags(*uPosition, landingGeoTags, Now() - TDuration::Minutes(1));
    } else {
        userGeoTags.emplace("undefined");
    }

    NJson::TJsonValue report;
    for (auto&& landing : userLandings) {
        if (landing.GetGeoTags().size() && IsIntersectionEmpty(landing.GetGeoTags(), userGeoTags)) {
            continue;
        }
        if (landing.GetChatId().size() && !includeNewFormat) {
            continue;
        }
        if (landing.GetChatId().empty() && landing.GetShouldSubstitute()) {
            auto it = landingTags.find(landing.GetId());
            if (it != landingTags.end()) {
                auto json = it->second->SubstituteTemplate(landing.GetJsonLanding(), server);
                if (json) {
                    landing.SetJsonLanding(json.GetRef());
                }
            }
        }
        result.emplace_back(std::move(landing));
    }
    return true;
}

TMap<TString, TGeoCoord> TDriveAPI::GetObjectFuturePositions() const {
    TMap<TString, TGeoCoord> result;
    auto sb = GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!sb) {
        return result;
    }
    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> currentSessions = sb->GetSessionsActual();
    for (auto&& i : currentSessions) {
        const TBillingSession* bs = dynamic_cast<const TBillingSession*>(i.Get());
        if (bs) {
            const TFixPointOffer* fpOffer = dynamic_cast<const TFixPointOffer*>(bs->GetCurrentOffer().Get());
            if (fpOffer) {
                result.emplace(fpOffer->GetObjectId(), fpOffer->GetFinish());
            }
        }
    }
    return result;
}

const IEntityTagsManager& TDriveAPI::GetEntityTagsManager(NEntityTagsManager::EEntityType entity) const {
    switch (entity) {
    case NEntityTagsManager::EEntityType::Car:
        return GetTagsManager().GetDeviceTags();
    case NEntityTagsManager::EEntityType::Area:
        return GetAreasDB()->GetTagsManager();
    case NEntityTagsManager::EEntityType::Trace:
        return GetTagsManager().GetTraceTags();
    case NEntityTagsManager::EEntityType::User:
        return GetTagsManager().GetUserTags();
    case NEntityTagsManager::EEntityType::Account:
        return GetTagsManager().GetAccountTags();
    case NEntityTagsManager::EEntityType::Undefined:
        return GetTagsManager().GetDeviceTags();
    }
}

TUserPermissions::TPtr TDriveAPI::GetUserPermissions(
    const TString& userId,
    const TUserPermissionsFeatures& userFeatures,
    TInstant reqActuality,
    IReplyContext::TPtr context,
    bool addCustomActions,
    bool useCache,
    bool forceFetchPermissions
) const {
    auto ignoreActuality = SettingsDB ? SettingsDB->GetValue<bool>("user_permissions.ignore_actuality").GetOrElse(false) : false;
    auto actuality = ignoreActuality ? Now() : reqActuality;
    return RolesManager->GetUserPermissions(userId, *TagsManager, Config.GetRolesFeaturesConfig(), userFeatures, actuality, context, AreasDB.Get(), addCustomActions, useCache, HasBillingManager() ? &GetBillingManager().GetAccountsManager() : nullptr, forceFetchPermissions);
}

TSet<TString> TDriveAPI::GetActions(ITag::TConstPtr tag) const {
    auto description = tag ? GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName()) : nullptr;
    if (description) {
        return description->GetAvailableCarActions();
    } else {
        return {};
    }
}

TSet<TString> TDriveAPI::GetActions(const TDBTag& tag) const {
    return GetActions(tag.GetData());
}

TVector<TString> TDriveAPI::GetIMEIs(TConstArrayRef<TString> carIds, const TMap<TString, TDriveCarInfo>& fetchResult) const {
    TVector<TString> result;
    for (auto&& carId : carIds) {
        auto p = fetchResult.find(carId);
        if (p != fetchResult.end()) {
            result.push_back(p->second.GetIMEI());
        } else {
            result.emplace_back();
        }
    }
    return result;
}

TVector<TString> TDriveAPI::GetIMEIs(TConstArrayRef<TString> carIds) const {
    auto carsData = GetCarsData();
    Y_ENSURE(carsData, "CarsDB is missing");
    auto fetchResult = carsData->GetCachedOrFetch(carIds).GetResult();
    return GetIMEIs(carIds, fetchResult);
}

TString TDriveAPI::GetIMEI(const TString& carId, const TMap<TString, TDriveCarInfo>& fetchResult) const {
    auto imeis = GetIMEIs({ carId }, fetchResult);
    if (imeis.size() == 1) {
        return imeis[0];
    } else {
        return {};
    }
}

TString TDriveAPI::GetIMEI(const TString& carId) const {
    auto imeis = GetIMEIs({ carId });
    if (imeis.size() == 1) {
        return imeis[0];
    } else {
        return {};
    }
}

TVector<TString> TDriveAPI::GetCarIdByIMEIs(TConstArrayRef<TString> imeis) const {
    auto fetchResult = Yensured(GetCarsImei())->GetCachedOrFetch(imeis);

    TVector<TString> result;
    result.reserve(imeis.size());
    for (auto&& imei : imeis) {
        auto p = fetchResult.GetResult().find(imei);
        if (p != fetchResult.GetResult().end()) {
            result.push_back(p->second.GetId());
        } else {
            result.emplace_back();
        }
    }
    return result;
}

TString TDriveAPI::GetCarIdByIMEI(const TString& imei) const {
    auto ids = GetCarIdByIMEIs({ imei });
    if (ids.size() == 1) {
        return ids[0];
    } else {
        return {};
    }
}

TVector<TString> TDriveAPI::GetCarIdByNumbers(TConstArrayRef<TString> numbers) const {
    auto fetchResult = Yensured(GetCarNumbers())->GetCachedOrFetch(numbers);

    TVector<TString> result;
    result.reserve(numbers.size());
    for (auto&& number : numbers) {
        auto p = fetchResult.GetResult().find(number);
        if (p != fetchResult.GetResult().end()) {
            result.push_back(p->second.GetId());
        } else {
            result.emplace_back();
        }
    }
    return result;
}

TString TDriveAPI::GetCarIdByNumber(const TString& number) const {
    auto ids = GetCarIdByNumbers({ number });
    if (ids.size() == 1) {
        return ids[0];
    } else {
        return {};
    }
}

bool TDriveAPI::DoStop() {
    if (!StateFiltersDB->Stop()) {
        return false;
    }
    if (!NamedFiltersDB->Stop()) {
        return false;
    }
    if (!AreasDB->Stop()) {
        return false;
    }
    if (!LandingsDB->Stop()) {
        return false;
    }
    if (!CarsByVin->Stop()) {
        return false;
    }
    if (!CarsByImei->Stop()) {
        return false;
    }
    if (!CarNumbers->Stop()) {
        return false;
    }
    if (!Models->Stop()) {
        return false;
    }
    if (!Users->Stop()) {
        return false;
    }
    if (!Cars->Stop()) {
        return false;
    }
    if (!CarGenericAttachments->Stop()) {
        return false;
    }
    if (!CarAttachmentAssignments->Stop()) {
        return false;
    }
    if (!ShortSessionsManager->Stop()) {
        return false;
    }
    if (RolesManager) {
        RolesManager->Stop();
    }
    if (!BillingManager->Stop()) {
        return false;
    }
    return true;
}

bool TDriveAPI::DoStart() {
    try {
        auto threadCount = FromStringWithDefault<size_t>(GetEnv("DRIVE_API_START_THREADS"), 16);
        auto threadPool = CreateThreadPool(threadCount);
        TDurationGuard dgg("StartComponent DriveApiMain");
        TVector<NThreading::TFuture<void>> futures;
        if (Config.GetNeedAttachmentManager()) {
            futures.push_back(NThreading::Async([this]() {
                TDurationGuard dg("StartComponent CarGenericAttachments");
                Y_ENSURE(CarGenericAttachments->Start());
            }, *threadPool));
            futures.push_back(NThreading::Async([this]() {
                TDurationGuard dg("StartComponent CarAttachmentAssignments");
                Y_ENSURE(CarAttachmentAssignments->Start());
            }, *threadPool));
        }
        if (RolesManager) {
            futures.push_back(NThreading::Async([this] {
                TDurationGuard dg("StartComponent RolesManager");
                RolesManager->Start();
            }, *threadPool));
        }
        futures.push_back(NThreading::Async([this]() {
            TDurationGuard dg("StartComponent Cars");
            Y_ENSURE(Cars->Start());
        }, *threadPool));
        futures.push_back(NThreading::Async([this]() {
            TDurationGuard dg("StartComponent Users");
            Y_ENSURE(Users->Start());
        }, *threadPool));
        futures.push_back(NThreading::Async([this]() {
            TDurationGuard dg("StartComponent BillingManager");
            Y_ENSURE(BillingManager->Start());
        }, *threadPool));
        NThreading::WaitAll(futures).GetValueSync();
    } catch (...) {
        ERROR_LOG << "cannot start DriveAPI: " << CurrentExceptionInfo().GetStringRobust() << Endl;
        return false;
    }

    try {
        TDurationGuard dgg("StartComponent DriveApiOther");
        {
            TDurationGuard dg("StartComponent Models");
            Y_ENSURE(Models->SetPeriod(TDuration::Minutes(1)).Start());
        }
        {
            TDurationGuard dg("StartComponent CarNumbers");
            Y_ENSURE(CarNumbers->SetPeriod(TDuration::Minutes(1)).Start());
        }
        {
            TDurationGuard dg("StartComponent CarsByImei");
            Y_ENSURE(CarsByImei->SetPeriod(TDuration::Minutes(1)).Start());
        }
        {
            TDurationGuard dg("StartComponent CarsByVin");
            Y_ENSURE(CarsByVin->SetPeriod(TDuration::Minutes(1)).Start());
        }
        {
            TDurationGuard dg("StartComponent LandingsDB");
            Y_ENSURE(LandingsDB->Start());
        }
        {
            TDurationGuard dg("StartComponent AreasDB");
            Y_ENSURE(AreasDB->Start());
        }
        {
            TDurationGuard dg("StartComponent ShortSessionsManager");
            Y_ENSURE(ShortSessionsManager->Start());
        }
        {
            TDurationGuard dg("StartComponent StateFiltersDB");
            Y_ENSURE(StateFiltersDB->Start());
        }
        {
            TDurationGuard dg("StartComponent NamedFiltersDB");
            Y_ENSURE(NamedFiltersDB->Start());
        }
    } catch (...) {
        ERROR_LOG << "cannot start DriveAPI: " << CurrentExceptionInfo().GetStringRobust() << Endl;
        return false;
    }

    return true;
}

TExpected<TPaymentsData, TString> TDriveAPI::AddClosedBillingInfo(const TBillingCompilation& compilation, const TString& userId, const NDrive::IServer& server, bool strict) const {
    if (!HasBillingManager()) {
        return MakeUnexpected<TString>("no billing manager");
    }
    auto offer = compilation.GetCurrentOffer();
    auto locale = offer ? offer->GetLocale() : DefaultLocale;
    auto finishingBill = compilation.GetBill(locale, &server, nullptr);
    auto pricing = compilation.GetCurrentOfferPricing();
    if (!pricing) {
        return MakeUnexpected<TString>("no current pricing");
    }
    TBillingTask::TCashbacksInfo additionalCashback;
    auto cashbackLimit = pricing->GetBillingCashback();
    ui32 cashback = 0;
    for (auto&& [_, price] : pricing->GetPrices()) {
        if (cashback >= cashbackLimit) {
            break;
        }
        if (price.GetCashback() > 0) {
            auto sum = Min<ui32>((ui32)price.GetCashback(), cashbackLimit - cashback);
            TBillingTask::TCashbackInfo info(sum, price.HasPayload() ? price.GetPayloadRef() : NJson::TJsonValue());
            additionalCashback.MutableParts().emplace_back(info);
            cashback += sum;
        }
    }
    if (!strict) {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto payments = GetBillingManager().GetPaymentsManager().GetFinishedPayments(NContainer::Scalar(compilation.GetSessionId()), tx);
        if (!payments) {
            return MakeUnexpected<TString>("Cannot fetch finished payments " + tx.GetStringReport());
        }
        if (auto sessionPayments = payments->FindPtr(compilation.GetSessionId())) {
            return *sessionPayments;
        }
    }
    return GetBillingManager().AddClosedBillingInfo(compilation.GetSessionId(), userId, compilation.GetBillingSumPrice(), finishingBill.GetCashbackPercent(), additionalCashback);
}

TExpected<bool, TCodedException> TDriveAPI::CloseSession(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, bool strict) const {
    return WrapUnexpected<TCodedException>([&session, &permissions, &server, strict] {
        TChargableTag::CloseSession(session, permissions, server, strict);
        return true;
    });
}

bool TDriveAPI::IncrementCashbackSessions(const TUserPermissions& permissions, NDrive::TEntitySession* session, const NDrive::IServer& server) const {
    if (session) {
        return IncrementCashbackSessions(permissions, *session, server);
    } else {
        auto tx = BuildTx<NSQL::Writable>();
        if (!IncrementCashbackSessions(permissions, tx, server)) {
            ERROR_LOG << "IncrementCashbackSessions: " << tx.GetStringReport() << Endl;
            return false;
        }
        if (!tx.Commit()) {
            ERROR_LOG << "IncrementCashbackSessions: " << tx.GetStringReport() << Endl;
            return false;
        }
        return true;
    }
}

bool TDriveAPI::IncrementCashbackSessions(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer& server) const {
    const TString userId = permissions.GetUserId();
    TDBTag settingsTag = GetUserSettings(userId, session);
    TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
    if (settingsData) {
        auto cashbackCountStr = settingsData->GetField("cashback_count");
        ui32 cashbackCount = 0;
        if (cashbackCountStr) {
            if (!TryFromString(*cashbackCountStr, cashbackCount)) {
                return false;
            }
        }
        try {
            TUserDictionaryTag::SetSettings(settingsTag->GetName(), {std::make_pair("cashback_count", ::ToString(cashbackCount + 1))}, permissions.Self(), "", server, session);
        } catch (const TCodedException& e) {
            ERROR_LOG << "IncrementCashbackSessions: " << e.GetReport() << Endl;
            return false;
        }
        return true;
    }
    return false;
}

bool TDriveAPI::GetCurrentUserSessions(const TString& userId, TVector<TAtomicSharedPtr<const ISession>>& result, TInstant reqActuality, const TSet<TString>& states) const {
    TVector<TDBTag> tags;
    if (!TagsManager->GetDeviceTags().GetPerformedTags(userId, tags, reqActuality)) {
        return false;
    }
    result.clear();

    auto tx = TagsManager->GetDeviceTags().BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
    for (auto&& i : tags) {
        if (i.GetTagAs<TChargableTag>() || i.GetTagAs<TTransformationTag>()) {
            auto optionalSession = SessionManager->GetTagSession(i.GetTagId(), tx);
            if (!optionalSession) {
                return false;
            }
            auto sessionInfo = *optionalSession;
            if (!!sessionInfo && !sessionInfo->GetClosed() && sessionInfo->GetUserId() == userId) {
                if (states && !states.contains(i->GetName())) {
                    continue;
                }
                result.emplace_back(sessionInfo);
            }
        }
    }

    const auto& userTagManager = TagsManager->GetUserTags();
    auto taggedUser = userTagManager.GetCachedObject(userId);
    if (taggedUser) {
        for (auto&& tag : taggedUser->GetTags()) {
            if (tag.Is<TOfferHolderTag>()) {
                auto session = CreateOfferHolderSession(tag, {}, userTagManager, tx);
                if (!session) {
                    continue;
                }
                result.push_back(std::move(session));
            }
        }
    }
    return true;
}

bool TDriveAPI::GetCurrentSessionsObjects(const TString& userId, TVector<TString>& objects, const TInstant reqActuality, const TSet<TString>& states) const {
    TVector<TAtomicSharedPtr<const ISession>> sessions;
    if (!GetCurrentUserSessions(userId, sessions, reqActuality, states)) {
        return false;
    }
    objects.clear();
    for (const auto& session : sessions) {
        const auto& objectId = session->GetObjectId();
        if (objectId) {
            objects.emplace_back(objectId);
        }
    }
    return true;
}

bool TDriveAPI::GetUserSession(const TString& userId, TAtomicSharedPtr<const ISession>& result, const TString& sessionId, TInstant reqActuality) const {
    result = nullptr;
    if (!!sessionId) {
        auto sb = TagsManager->GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", reqActuality);
        auto billingSession = sb ? sb->GetSession(sessionId) : nullptr;
        if (billingSession) {
            result = billingSession;
            return true;
        }
    }
    {
        TVector<TAtomicSharedPtr<const ISession>> sessions;
        if (!GetCurrentUserSessions(userId, sessions, reqActuality)) {
            return false;
        }
        if (sessionId) {
            for (auto&& session : sessions) {
                if (session && session->GetSessionId() == sessionId) {
                    result = session;
                    return true;
                }
            }
        } else {
            if (sessions.size() == 1) {
                result = sessions.front();
            }
        }
    }
    return true;
}

namespace {
    auto GetDatabase(const TDriveAPIConfig& config, const TMap<TString, TDatabasePtr>& databases) {
        const auto& name = config.GetDBName();
        auto i = databases.find(name);
        AssertCorrectConfig(i != databases.end(), "cannot find database %s", name.c_str());
        return i->second;
    }
}

TDriveAPI::TDriveAPI(const TDriveAPIConfig& config, const TMap<TString, TDatabasePtr>& databases, const ISettings* settings, const NDrive::IServer& server, const TAtomicSharedPtr<IDriveTagsManager> tagsManager)
    : TDatabaseSessionConstructor(::GetDatabase(config, databases))
    , Config(config)
    , TagsManager(tagsManager)
    , SessionManager(MakeHolder<TSessionManager>(*TagsManager))
    , AdditionalServiceSessionManager(MakeHolder<TAdditionalServiceSessionManager>(*TagsManager))
    , DedicatedFleetSessionManager(MakeHolder<TDedicatedFleetSessionManager>(*TagsManager))
    , SettingsDB(settings)
{
    if (Config.GetCarsConfig()) {
        TDurationGuard dg("ConstructComponent CarsDB");
        Cars.Reset(new TCarsDB(GetTagsManager().GetContext(), *Config.GetCarsConfig()));
    } else {
        TDurationGuard dg("ConstructComponent CarsDB");
        Cars.Reset(new TCarsDB(GetTagsManager().GetContext()));
    }

    CarsByImei.Reset(new TDBImei(Database));
    CarsByVin.Reset(new TDBVin(Database));
    CarNumbers.Reset(new TCarNumbersDB(Database));

    if (Config.GetUsersConfig()) {
        TDurationGuard dg("ConstructComponent UsersDB");
        Users.Reset(new TUsersDB(GetTagsManager().GetContext(), &GetTagsManager().GetUserTags(), *Config.GetUsersConfig()));
    } else {
        TDurationGuard dg("ConstructComponent UsersDB");
        Users.Reset(new TUsersDB(GetTagsManager().GetContext(), &GetTagsManager().GetUserTags()));
    }

    Models.Reset(new TModelsDB(Database));
    LandingsDB.Reset(new TLandingsDB(GetTagsManager().GetContext(), Config.GetLandingsConfig()));
    UserLandings.Reset(new TLandingAcceptanceDB(Database));

    AreasDB.Reset(new TAreasDB(GetTagsManager().GetContext(), Config.GetAreasConfig()));
    NamedFiltersDB.Reset(new TNamedFiltersStorage(GetTagsManager().GetContext(), THistoryConfig().SetDeep(TDuration::Days(0))));
    StateFiltersDB.Reset(new TStateFiltersDB(GetTagsManager().GetDeviceTags(), GetTagsManager().GetContext(), Config.GetStateFiltersConfig()));
    TakeoutRequests.Reset(new TTakeoutRequestDB(Database));

    SignalsConfigurationsDB.Reset(new TSignalsConfigurationsStorage(GetTagsManager().GetContext()));

    ConsistencyDB.Reset(new TConsistencyStorage(Database));

    LoginsManager.Reset(new TLoginsManager(Database));
    RolesManager.Reset(new TRolesManager(GetTagsManager().GetContext(), *Users, Config.GetActionPropositionsConfig(), Config.GetRolesHistoryConfig()));

    MinimalCompiledRides.Reset(new TMinimalRidingHistoryManager(GetTagsManager().GetContext()));

    ImagesDB.Reset(new TImagesStorage(GetTagsManager().GetContext()));

    InsuranceHistoryManager.Reset(new TInsuranceHistoryManager(GetTagsManager().GetContext()));

    MaintenanceDB = MakeHolder<TMaintenanceDB>(GetTagsManager().GetContext());

    if (Config.GetZoneConfig() && Config.GetZoneConfig()->IsEnable()) {
        ZoneDB.Reset(new TZoneStorage(GetTagsManager().GetContext(), THistoryConfig().SetDeep(Config.GetZoneConfig()->GetDeep())));
    }

    if (Config.GetNeedDatasyncQueueClient()) {
        DatasyncQueueClient = MakeHolder<TDatasyncQueueClient>(Database);
    }

    if (Config.GetNeedAttachmentManager()) {
        TDurationGuard dg("ConstructComponent Car*Attachments");
        CarGenericAttachments.Reset(new TCarGenericAttachmentDB(Database, GetTagsManager().GetContext()));
        CarAttachmentAssignments.Reset(new TCarAttachmentAssignmentDB(Database, *CarGenericAttachments, *Cars, GetTagsManager().GetContext()));
    }

    {
        if (Config.GetPassportClientConfig()) {
            PassportClient = MakeHolder<TPassportClient>(*Config.GetPassportClientConfig(), server.GetTvmClient(Config.GetPassportClientConfig()->GetSelfClientId()));
        }
    }

    {
        TDurationGuard dg("ConstructComponent ShortSessionsManager");
        ShortSessionsManager.Reset(new TShortSessionsManager(GetTagsManager().GetContext(), Config.GetHeadAppHistoryConfig()));
    }
    if (Config.GetNeedAttachmentManager()) {
        TDurationGuard dg("ConstructComponent HeadAccountManager");
        HeadAccountManager.Reset(new THeadAccountManager(Database, *CarAttachmentAssignments, *ShortSessionsManager, GetTagsManager().GetContext(), PassportClient.Get(), Config.GetHeadAppFreshness()));
    }

    {
        TDurationGuard dg("ConstructComponent BillingManager");
        if (Config.GetBillingConfig()) {
            BillingManager = MakeHolder<TBillingManager>(*Config.GetBillingConfig(), GetTagsManager().GetContext(), *Users, server.GetSettings(), &server, &server);
        }
    }
    {
        if (Config.GetMDSClientConfig()) {
            TDurationGuard dg("ConstructComponent MDSClient");
            MDSClient = MakeHolder<TS3Client>(*Config.GetMDSClientConfig());
            if (!MDSClient->Init(true)) {
                ERROR_LOG << "Can't initialize mds client" << Endl;
            }
        }
    }
    {
        if (Config.GetAutocodeClientConfig()) {
            TDurationGuard dg("ConstructComponent AutocodeClient");
            AutocodeClient = MakeHolder<NDrive::NAutocode::TAutocodeClient>(*Config.GetAutocodeClientConfig());
        }
    }
    {
        if (Config.GetMajorClientConfig()) {
            TDurationGuard dg("ConstructComponent MajorClient");
            MajorClient = MakeHolder<TMajorClient>(*Config.GetMajorClientConfig());
        }
    }
    {
        if (Config.GetElementFinesClientConfig()) {
            TDurationGuard dg("ConstructComponent ElementFinesClient");
            ElementFinesClient = MakeHolder<NDrive::TElementFinesClient>(*Config.GetElementFinesClientConfig());
        }
    }
    {
        if (Config.GetYandexDiskClientConfig()) {
            TDurationGuard dg("ConstructComponent YandexDiskClient");
            YandexDiskClient = MakeHolder<TYandexDiskClient>(*Config.GetYandexDiskClientConfig());
        }
    }
    {
        if (Config.GetStartrekClientConfig()) {
            TDurationGuard dg("ConstructComponent StartrekClient");
            StartrekClient = MakeHolder<TStartrekClient>(*Config.GetStartrekClientConfig());
        }
    }
    {
        if (Config.GetSolomonClientConfig()) {
            TDurationGuard dg("ConstructComponent SolomonClient");
            SolomonClient = MakeHolder<TSolomonClient>(*Config.GetSolomonClientConfig());
        }
        if (Config.GetDeprecatedSolomonClientConfig()) {
            TDurationGuard dg("ConstructComponent DeprecatedSolomonClient");
            DeprecatedSolomonClient = MakeHolder<TSolomonClient>(*Config.GetDeprecatedSolomonClientConfig());
        }
    }
    {
        if (Config.GetSamsaraClientConfig()) {
            TDurationGuard dg("ConstructComponent SamsaraClient");
            SamsaraClient = MakeHolder<TSamsaraClient>(*Config.GetSamsaraClientConfig());
        }
    }
    {
        if (Config.GetStaffClientConfig()) {
            TDurationGuard dg("ConstructComponent StaffClient");
            StaffClient = MakeHolder<TStaffClient>(*Config.GetStaffClientConfig());
        }
    }
    {
        if (Config.GetFinesManagerConfig()) {
            TDurationGuard dg("ConstructComponent FinesManager");
            FinesManager = MakeHolder<NDrive::NFine::TFinesManager>(GetTagsManager().GetContext(), *Config.GetFinesManagerConfig(), server);
        }
    }
    {
        if (Config.GetReninsClaimConfig()) {
            TDurationGuard dg("ConstructComponent ReninsClaimClient");
            ReninsClaimClient = MakeHolder<NDrive::NRenins::TReninsClaimClient>(*Config.GetReninsClaimConfig());
        }
    }
    {
        if (Config.GetIncidentsManagerConfig()) {
            TDurationGuard dg("ConstructComponent IncidentsManager");
            IncidentsManager = MakeHolder<NDrive::TIncidentsManager>(*Config.GetIncidentsManagerConfig(), GetTagsManager().GetContext(), server);
        }
    }
    {
        if (Config.GetReninsConfig()) {
            TDurationGuard dg("ConstructComponent ReninsClient");
            ReninsClient = MakeHolder<NDrive::TReninsClient>(*Config.GetReninsConfig());
        }
    }
    {
        if (Config.GetIngosConfig()) {
            TDurationGuard dg("ConstructComponent IngosClient");
            IngosClient = MakeHolder<NDrive::TIngosClient>(*Config.GetIngosConfig());
        }
    }
    {
        if (Config.GetDatasyncConfig()) {
            TDurationGuard dg("ConstructComponent DatasyncClient");
            DatasyncClient = MakeHolder<TDatasyncClient>(*Config.GetDatasyncConfig());
        }
    }

    if (auto options = Config.GetGeocoderOptions()) {
        auto selfClientId = options->TvmSource;
        auto destinationTvmId = options->TvmDestination;
        auto tvmClient = selfClientId ? server.GetTvmClient(selfClientId) : nullptr;
        auto tvmAuth = selfClientId ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, destinationTvmId) : Nothing();
        GeocoderClient = MakeHolder<NDrive::TGeocoder>(*options, std::move(tvmAuth), nullptr);
    }

    if (Config.GetPrivateDataClientType() == "datasync") {
        Y_ENSURE(DatasyncClient);
        PrivateDataClient = MakeHolder<TDatasyncPrivateDataStorage>(*DatasyncClient);
    } else if (Config.GetPrivateDataClientType() == "fake") {
        PrivateDataClient = MakeHolder<TFakePrivateDataStorage>();
    } else if (auto storage = server.GetVersionedStorage(Config.GetPrivateDataClientType())) {
        PrivateDataClient = MakeHolder<TDatabasePrivateDataStorage>(storage);
    } else {
        ERROR_LOG << "no valid private data storage type specified; any usage attempt will fail" << Endl;
    }

    if (Config.GetTakeoutClientType() == "takeout") {
        TakeoutClient.Reset(new TTakeoutClient(*Config.GetTakeoutConfig()));
    } else if (Config.GetTakeoutClientType() == "fake") {
        TakeoutClient.Reset(new TFakeTakeoutClient());
    } else {
        ERROR_LOG << "no valid takeout client type specified" << Endl;
    }

    if (Config.GetOctopusClientType() == "octopus") {
        OctopusClient.Reset(new TOctopusClient(*Config.GetOctopusConfig()));
    } else if (Config.GetOctopusClientType() == "fake") {
        OctopusClient.Reset(new TFakeOctopusClient({NRobotPhoneCall::ECallStatus::OK}));
    } else {
        ERROR_LOG << "no valid octopus client type specified" << Endl;
    }

    if (Config.GetDocumentPhotosConfig()) {
        DocumentPhotosManager = MakeHolder<TDocumentPhotosManager>(*Config.GetDocumentPhotosConfig(), Database);
    }

    {
        TagSearches[NEntityTagsManager::EEntityType::Car] = MakeHolder<TTagsSearch>(GetTagsManager().GetDeviceTags(), GetTagsManager());
        TagSearches[NEntityTagsManager::EEntityType::User] = MakeHolder<TTagsSearch>(GetTagsManager().GetUserTags(), GetTagsManager());
        TagSearches[NEntityTagsManager::EEntityType::Trace] = MakeHolder<TTagsSearch>(GetTagsManager().GetTraceTags(), GetTagsManager());
    }
}

TDBTag TDriveAPI::GetUserDictionaryTag(const TString& userId, const TString& tagNameSetting, NDrive::TEntitySession& session) const {
    CHECK_WITH_LOG(TagsManager);
    CHECK_WITH_LOG(SettingsDB);
    TDBTag settingsTag;
    TString userSettingsTagName = SettingsDB->GetValueDef(tagNameSetting, tagNameSetting);

    TVector<TDBTag> userTags;
    {
        if (!TagsManager->GetUserTags().RestoreEntityTags(userId, { userSettingsTagName }, userTags, session)) {
            return settingsTag;
        }
    }

    for (auto&& tag: userTags) {
        if (tag.Is<TUserDictionaryTag>() && tag->GetName() == userSettingsTagName) {
            return tag;
        }
    }

    auto tag = TagsManager->GetTagsMeta().CreateTag(userSettingsTagName);
    if (!tag) {
        return settingsTag;
    }

    settingsTag.SetObjectId(userId);
    settingsTag.SetData(tag);
    return settingsTag;
}

TDBTag TDriveAPI::GetUserSettings(const TString& userId, NDrive::TEntitySession& session) const {
    return GetUserDictionaryTag(userId, NDrive::UserSettingsTagNameSetting, session);
}

TDBTag TDriveAPI::GetUserAppFlags(const TString& userId, NDrive::TEntitySession& session) const {
    return GetUserDictionaryTag(userId, NDrive::UserFlagsTagNameSetting, session);
}

bool TDriveAPI::CheckCrossloginFlag(const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    Y_UNUSED(server);
    if (!SettingsDB) {
        return false;
    }

    ui32 percent = 100;
    TString mode = ToString(EHeadAppMode::Disable);

    if (!SettingsDB->GetValue("launch_head_app", mode) || mode.empty()) {
        return false;
    }

    auto parts = SplitString(mode, "-");
    EHeadAppMode headAppMode = EHeadAppMode::Disable;
    if (!TryFromString(parts[0], headAppMode)) {
        return false;
    }

    if (parts.size() > 1) {
        if (!TryFromString(parts[1], percent) || percent > 100) {
            return false;
        }
    }

    if (percent < 100) {
        ui64 userHash = FnvHash<ui64>(permissions.GetUserId());
        if (userHash % 100 > percent) {
            return false;
        }
    }

    bool launchAppFlag = false;

    if (headAppMode == EHeadAppMode::Yandex) {
        launchAppFlag = permissions.GetUserFeatures().GetIsYandexUser();
    } else if (headAppMode == EHeadAppMode::Enable) {
        launchAppFlag = true;
    }

    TUserAppSettings userAppSettings(permissions.GetUserId());
    TExpectedSettingString crossLogin = userAppSettings.GetCrossLogin(session);
    if (!crossLogin) {
        ERROR_LOG << "CheckCrossloginFlag: can't get user settings " << crossLogin.GetError() << Endl;
        return false;
    }
    launchAppFlag = launchAppFlag && IsFalse(crossLogin.GetRef());

    return launchAppFlag;
}

bool TDriveAPI::ConfirmUserMail(const TString& key, const TString& user, const TString& userAuthHeader, const TString& clientIp, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    if (!server.GetAccountEmailBinder()) {
        session.SetErrorInfo("ConfirmUserMail", "absent email binder", EDriveSessionResult::InternalError);
        return false;
    }

    TVector<TDBTag> userTags;
    if (!TagsManager->GetUserTags().RestoreEntityTags(user, { TPassportTrackTag::TypeName }, userTags, session)) {
        return false;
    }

    const TPassportTrackTag* trackTag;
    if (userTags.empty() || !(trackTag = userTags.begin()->GetTagAs<TPassportTrackTag>())) {
        session.SetErrorInfo("ConfirmUserMail", "incorrect track", EDriveSessionResult::InternalError);
        return false;
    }

    TString error;
    if (!server.GetAccountEmailBinder()->ConfirmCode(key, trackTag->GetTrackId(), userAuthHeader, clientIp, error)) {
        session.SetErrorInfo("ConfirmUserMail", "passport error", EDriveSessionResult::InternalError);
        if (error) {
            session.SetLocalizedMessageKey("verification.mail.messages." + error);
        } else {
            session.SetLocalizedMessageKey("verification.mail.messages.internal_error");
        }
        return false;
    }
    return true;
}

bool TDriveAPI::BindUserMail(const TString& email, const TString& user, const TString& userAuthHeader, const TString& clientIp, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    if (!server.GetAccountEmailBinder()) {
        session.SetErrorInfo("BindUserMail", "passport error", EDriveSessionResult::InternalError);
        return false;
    }

    TString trackId;
    TString error;
    TString message = "verification.mail.messages.internal_error";
    if (!server.GetAccountEmailBinder()->BindSubmit(email, trackId, userAuthHeader, clientIp, error)) {
        session.SetErrorInfo("BindUserMail", "passport error", EDriveSessionResult::IncorrectUserReply);
        if (error) {
            message = "verification.mail.messages." + error;
        }
        session.SetLocalizedTitleKey(message);
        return false;
    }

    auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(TPassportTrackTag::TypeName, "Для регистрации " + email);
    auto passportTag = dynamic_cast<TPassportTrackTag*>(tag.Get());
    if (passportTag) {
        passportTag->SetTrackId(trackId);
    }

    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, user, user, &server, session)) {
        return false;
    }

    TDBTag settingsTag = server.GetDriveAPI()->GetUserSettings(user, session);
    TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
    if (!settingsData) {
        TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_incorrect_settings_configuration", 1);
        return false;
    }
    settingsData->SetField("sending_verification_mail_time", ::ToString(ModelingNow().Seconds()));
    settingsData->SetField("verified_mail", ::ToString(false));

    if (!!settingsTag.GetTagId()) {
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(settingsTag, user, session)) {
            TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_internal_error", 1);
            return false;
        }
    } else {
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(settingsTag.GetData(), user, user, &server, session)) {
            TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_internal_error", 1);
            return false;
        }
    }
    return true;
}

bool TDriveAPI::CheckReferralProgramParticipationImpl(TString& code, const TUserPermissions& permissions, NDrive::TEntitySession& session) const {
    TString& customCode = code;
    return CheckReferralProgramParticipationImpl(code, customCode, permissions, session);
}

bool TDriveAPI::CheckReferralProgramParticipationImpl(TString& code, TString& customCode, const TUserPermissions& permissions, NDrive::TEntitySession& session) const {
    if (!SettingsDB) {
        return false;
    }

    TString disableReferralProgramTag = "disable_referral_program";
    SettingsDB->GetValueStr("referral_program.disable_referral_program_tag", disableReferralProgramTag);

    TVector<TDBTag> userTags;
    {
        if (!TagsManager->GetUserTags().RestoreEntityTags(permissions.GetUserId(), { disableReferralProgramTag }, userTags, session)) {
            return false;
        }
    }

    for (auto&& tag : userTags) {
        if (tag->GetName() == disableReferralProgramTag) {
            return false;
        }
    }

    TDBTag settingsTag = GetUserSettings(permissions.GetUserId(), session);
    if (!settingsTag.HasData()) {
        // Because we don't know actual user decision
        return false;
    }

    TString referralCodeField = "referral_code";
    SettingsDB->GetValueStr("referral_program.user_settings_field", referralCodeField);

    auto settingsField = settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(referralCodeField);
    if (settingsField.Defined()) {
        code = settingsField.GetRef();
    }

    TString customReferralCodeField = "custom_referral_code";
    SettingsDB->GetValueStr("referral_program.user_settings_custom_code_field", customReferralCodeField);

    settingsField = settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(customReferralCodeField);
    if (settingsField.Defined() && *settingsField) {
        customCode = settingsField.GetRef();
    }

    return true;
}

bool TDriveAPI::CheckReferralProgramParticipation(TString& code, const TUserPermissions& permissions, NDrive::TEntitySession& session) const {
    TString& customCode = code;
    return CheckReferralProgramParticipation(code, customCode, permissions, session);
}

bool TDriveAPI::CheckReferralProgramParticipation(TString& code, TString& customCode, const TUserPermissions& permissions, NDrive::TEntitySession& session) const {
    if (!SettingsDB) {
        return false;
    }

    TString enableReferralProgramFlag = "enable_referral";
    SettingsDB->GetValueStr("referral_program.enable_referral_program_flag", enableReferralProgramFlag);
    if (!permissions.GetFlagsReport()[enableReferralProgramFlag].GetBooleanRobust()) {
        return false;
    }

    return CheckReferralProgramParticipationImpl(code, customCode, permissions, session);
}

bool TDriveAPI::InvalidateFirstRiding(const TString& userId, const TFullCompiledRiding& compiledRiding, const NDrive::IServer* server) const {
    auto session = NDrive::TEntitySession(Database->CreateTransaction(false));
    auto fetchResult = GetUsersData()->FetchInfo(userId, session);
    if (!fetchResult) {
        return false;
    }
    if (fetchResult.size() == 0) {
        return false;
    }
    auto user = fetchResult.begin()->second;
    user.SetFirstRiding(false);
    if (!GetUsersData()->UpdateUser(user, userId, session) || !session.Commit()) {
        return false;
    }
    if (!server || !server->GetUserDevicesManager()) {
        TUnistatSignalsCache::SignalAdd("frontend", "register-success-first-trip-fail-internal", 1);
    } else {
        Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::FirstTripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::Success), userId, compiledRiding.GetFinishInstant()));
        Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::SuccessFirstTrip, userId, compiledRiding.GetFinishInstant())); // deprecated
        auto offerPtr = compiledRiding.GetOffer().Get();
        if (dynamic_cast<const TFixPointOffer*>(offerPtr)) {
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::FirstTripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessFix), userId, compiledRiding.GetFinishInstant()));
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::SuccessFixFirstTrip, userId, compiledRiding.GetFinishInstant())); // deprecated
        } else if (dynamic_cast<const TPackOffer*>(offerPtr)) {
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::FirstTripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessPack), userId, compiledRiding.GetFinishInstant()));
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::SuccessPackFirstTrip, userId, compiledRiding.GetFinishInstant())); // deprecated
        } else if (dynamic_cast<const TStandartOffer*>(offerPtr)) {
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::FirstTripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessStandart), userId, compiledRiding.GetFinishInstant()));
            Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::SuccessStandartFirstTrip, userId, compiledRiding.GetFinishInstant())); // deprecated
        }

        auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(offerPtr->GetBehaviourConstructorId());
        if (!!action && !!action->GetAs<IOfferBuilderAction>()) {
            for (const auto& tag : (*action)->GetGrouppingTags()) {
                Y_UNUSED(server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::FirstTripTag, tag, userId, compiledRiding.GetFinishInstant()));
            }
        }
    }
    return true;
}

IOffer::TPtr TDriveAPI::GetCurrentCarOffer(const TString& objectId) const {
    auto sb = GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!sb) {
        return nullptr;
    }
    auto objectSession = sb->GetLastObjectSession(objectId);
    const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(objectSession.Get());
    if (bSession && !bSession->GetClosed() && !!bSession->GetCurrentOffer()) {
        return bSession->GetCurrentOffer();
    }
    return nullptr;
}

TDriveAPI::EDelegationAbility TDriveAPI::GetP2PAbility(const TDriveUserData& user, const TString& objectId, NDrive::TEntitySession& session, bool withOffer) const {
    if (user.GetStatus() == NDrive::UserStatusBlocked) {
        return EDelegationAbility::TargetUserBlocked;
    }
    if (user.GetStatus() != NDrive::UserStatusActive) {
        return EDelegationAbility::TargetUserOnboarding;
    }
    auto candidatePermissions = GetUserPermissions(user.GetUserId(), TUserPermissionsFeatures());
    if (!candidatePermissions) {
        return EDelegationAbility::UnknownError;
    }

    auto taggedCar = GetTagsManager().GetDeviceTags().RestoreObject(objectId, session);
    if (!taggedCar.Defined()) {
        return EDelegationAbility::UnknownError;
    }

    TVector<TDBTag> tags;
    for (auto&& tag : taggedCar->GetTags()) {
        if (tag->GetTagPriority(0) > 0 && tag.GetTagAs<IJsonSerializableTag>()) {
            auto tagCopy = tag;
            tagCopy.SetData(tag.GetTagAs<IJsonSerializableTag>()->Clone(GetTagsManager()));
            tagCopy->SetTagPriority(0);
            tags.emplace_back(std::move(tagCopy));
        } else if (tag->GetName().StartsWith("old_state_")) {
            auto tagImpl = tag.GetTagAs<TChargableTag>();
            auto tagCopy = tag;
            tagCopy.SetData(tagImpl->Clone(GetTagsManager()));
            tagCopy->SetName("old_state_reservation");
            tagCopy->SetPerformer("");
            tags.emplace_back(std::move(tagCopy));
        } else {
            tags.emplace_back(std::move(tag));
        }
    }
    auto targetVisibility = candidatePermissions->GetVisibility(TTaggedObject(objectId, tags, taggedCar->GetTimestamp()), NEntityTagsManager::EEntityType::Car);
    if (targetVisibility == TUserPermissions::EVisibility::NoVisible) {
        return EDelegationAbility::NoCarAccess;
    }

    auto currentOffer = GetCurrentCarOffer(objectId);
    if (!currentOffer) {
        return EDelegationAbility::UnknownError;
    }
    if (!currentOffer->IsTransferable() && currentOffer->GetTypeName() != TLongTermOffer::GetTypeNameStatic()) {
        return EDelegationAbility::OfferNotTransferable;
    }

    if (withOffer) {
        auto offerAction = currentOffer->GetBehaviourConstructorId();
        if (!candidatePermissions->HasAction(offerAction)) {
            return EDelegationAbility::NoTariffAccess;
        }
    }

    TDBTag settingsTag = GetUserSettings(user.GetUserId(), session);
    if (settingsTag.HasData() && SettingsDB) {
        auto disableP2PField = SettingsDB->GetValueDef<TString>("user_settings.f_disable_p2p_delegation", "disable_p2p_delegation");
        auto settingsField = settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(disableP2PField);
        bool isAllowed = !settingsField.Defined() || IsFalse(settingsField.GetRef());
        if (!isAllowed) {
            return EDelegationAbility::DisabledByUser;
        }
    }

    return EDelegationAbility::Possible;
}

NThreading::TFuture<TVector<TPaymentMethod>> TDriveAPI::GetUserPaymentMethods(const TUserPermissions& permissions, const NDrive::IServer& server, bool useCache) const {
    const auto& settings = server.GetSettings();
    NThreading::TFuture<TVector<TPaymentMethod>> methods;
    if (permissions.GetSetting<bool>(settings, "use_lpm_client").GetOrElse(false) && server.GetLPMClient()) {
        TString uid = permissions.GetPaymethodsUid(settings);
        if (!uid) {
            WARNING_LOG << "Trying to get cards for user(" << permissions.GetUserId() << ") with empty uid" << Endl;
            return NThreading::MakeFuture(TVector<TPaymentMethod>());
        }
        TString cacheId = useCache ? permissions.GetSetting<TString>(settings, "lpm_client_cache_id", "legacy") : "";
        methods = server.GetLPMClient()->GetPaymentMethods(uid, cacheId);
    } else if (!HasBillingManager()) {
        return NThreading::TExceptionFuture<yexception>() << "no billing manager";
    } else {
        auto cards = GetBillingManager().GetUserCards(permissions.GetUserId(), useCache);
        if (!cards) {
            return NThreading::TExceptionFuture<yexception>() << "cannot fetch cards";
        }
        methods = NThreading::MakeFuture(*cards);
    }
    auto ignoredSystems = MakeSet(SplitString(permissions.GetSetting<TString>(settings, "billing.client_ignored_systems", ""), ","));
    return methods.Apply([ignoredSystems = std::move(ignoredSystems)](const NThreading::TFuture<TVector<TPaymentMethod>>& f) -> NThreading::TFuture<TVector<TPaymentMethod>> {
        if (!ignoredSystems || !f.HasValue()) {
            return f;
        }
        auto result = f.GetValue();
        EraseIf(result, [&ignoredSystems](const auto& method){ return ignoredSystems.contains(method.GetPSystem()); });
        return NThreading::MakeFuture(result);
    });
}

TMaybe<TVector<TPaymentMethod>> TDriveAPI::GetUserPaymentMethodsSync(const TUserPermissions& permissions, const NDrive::IServer& server, bool useCache) const {
    TMaybe<TVector<TPaymentMethod>> result;
    auto resultF = GetUserPaymentMethods(permissions, server, useCache);
    resultF.Wait();
    if (resultF.HasValue()) {
        result = resultF.GetValue();
    }
    return result;
}


void TDriveAPI::UpdateUserPaymentMethods(const TUserPermissions& permissions, const NDrive::IServer& server) const {
    const auto& settings = server.GetSettings();
    if (permissions.GetSetting<bool>(settings, "use_lpm_client").GetOrElse(false) && server.GetLPMClient()) {
        if (TString uid = permissions.GetPaymethodsUid(settings)) {
            server.GetLPMClient()->UpdateValue(uid, permissions.GetSetting<TString>(settings, "lpm_client_cache_id", "legacy"));
        }
    } else if (HasBillingManager()) {
        if (GetBillingManager().GetTrustCache()) {
            GetBillingManager().GetTrustCache()->UpdateValue(permissions.GetUserId());
        }
    }
}

bool TDriveAPI::HasBillingManager() const {
    bool enabled = NDrive::HasServer() && NDrive::GetServer().GetSettings().GetValue<bool>("billing.enabled").GetOrElse(true);
    return BillingManager && enabled;
}

template<NSQL::TTransactionTraits traits>
NDrive::TEntitySession TDriveAPI::BuildYdbTx(const TString& objectName, const NDrive::IServer* server, TDuration statementTimeout, TDuration lockTimeout) const {
    if (!server || objectName == TYdbTransaction::NoYdb) {
        return NDrive::TEntitySession();
    }
    bool useYDB = server->GetSettings().GetValue<bool>("ydb." + objectName + ".use_ydb").GetOrElse(false);
    if (auto ydb = server->GetYDB(); useYDB && ydb) {
        return ydb->template BuildTx<traits>(lockTimeout, statementTimeout);
    }
    return NDrive::TEntitySession();
}

template NDrive::TEntitySession TDriveAPI::BuildYdbTx<NSQL::ReadOnly>(const TString& objectName, const NDrive::IServer* server, TDuration statementTimeout, TDuration lockTimeout) const;
template NDrive::TEntitySession TDriveAPI::BuildYdbTx<NSQL::ReadOnly | NSQL::Deferred>(const TString& objectName, const NDrive::IServer* server, TDuration statementTimeout, TDuration lockTimeout) const;
