#pragma once

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/areas/location.h>
#include <drive/backend/roles/permissions.h>

class IOctopusClient;
class IPrivateDataStorage;
class ITagsHistoryContext;
class ITakeoutClient;
class TBillingCompilation;
class TBillingManager;
class TCarAttachmentAssignmentDB;
class TCarGenericAttachmentDB;
class TCompiledRiding;
class TConsistencyStorage;
class TDatasyncQueueClient;
class TDocumentPhotosManager;
class TDriveAPIConfig;
class TDriveCarInfo;
class TDriveUserData;
class TFullCompiledRiding;
class THeadAccountManager;
class TImagesStorage;
class TInsuranceHistoryManager;
class TInsuranceNotification;
class TInternalPointContext;
class TIntroViewInfo;
class TLanding;
class TLandingAcceptance;
class TLoginsManager;
class TMajorClient;
class TMinimalRidingHistoryManager;
class TNamedFiltersStorage;
class TPassportClient;
class TRTLineAPIConfig;
class TRolesConfig;
class TRolesManager;
class TSamsaraClient;
class TShortSessionsManager;
class TSignalsConfigurationsStorage;
class TSolomonClient;
class TStaffClient;
class TTagsSearch;
class TTakeoutRequestDB;
class TYandexDiskClient;
struct TDeviceLocationOptions;
class TPaymentsData;
class TMaintenanceDB;
class TZoneStorage;
class IOffer;

namespace NDrive::NAutocode {
    class TAutocodeClient;
}

namespace NDrive {
    class TElementFinesClient;
}

namespace NDrive::NFine {
    class TFinesManager;
}

namespace NDrive {
    class TIncidentsManager;
}

namespace NDrive::NRenins {
    class TReninsClaimClient;
}

namespace NDrive {
    class TGeocoder;
    class TIngosClient;
    class TReninsClient;

    using TSearchEntities = ui32;
    static TSearchEntities AllSearchEntities = Max<TSearchEntities>();
    enum class ESearchEntity : TSearchEntities {
        Car = 1 << 0,
        User = 1 << 1
    };
}

class TDriveAPI final
    : public TDatabaseSessionConstructor
    , public TNonCopyable
    , public IStartStopProcess
    , public IDriveDatabase
{
private:
    const TDriveAPIConfig& Config;

    TAtomicSharedPtr<IDriveTagsManager> TagsManager;
    THolder<TSessionManager> SessionManager;
    THolder<TAdditionalServiceSessionManager> AdditionalServiceSessionManager;
    THolder<TDedicatedFleetSessionManager> DedicatedFleetSessionManager;

    THolder<TModelsDB> Models;
    THolder<TUsersDB> Users;
    THolder<TCarsDB> Cars;
    THolder<TCarNumbersDB> CarNumbers;
    THolder<TDBImei> CarsByImei;
    THolder<TDBVin> CarsByVin;

    THolder<TLandingsDB> LandingsDB;
    THolder<TLandingAcceptanceDB> UserLandings;
    const ISettings* SettingsDB;
    THolder<TAreasDB> AreasDB;
    THolder<TStateFiltersDB> StateFiltersDB;
    THolder<TNamedFiltersStorage> NamedFiltersDB;

    THolder<TSignalsConfigurationsStorage> SignalsConfigurationsDB;

    THolder<TMinimalRidingHistoryManager> MinimalCompiledRides;

    THolder<TLoginsManager> LoginsManager;
    THolder<TRolesManager> RolesManager;
    THolder<TBillingManager> BillingManager;
    THolder<NDrive::NRenins::TReninsClaimClient> ReninsClaimClient;
    THolder<NDrive::TIncidentsManager> IncidentsManager;
    THolder<NDrive::TReninsClient> ReninsClient;
    THolder<NDrive::TIngosClient> IngosClient;
    THolder<TPassportClient> PassportClient;

    THolder<THeadAccountManager> HeadAccountManager;
    THolder<TShortSessionsManager> ShortSessionsManager;
    THolder<TS3Client> MDSClient;
    THolder<NDrive::NAutocode::TAutocodeClient> AutocodeClient;
    THolder<TMaintenanceDB> MaintenanceDB;
    THolder<TMajorClient> MajorClient;
    THolder<NDrive::TElementFinesClient> ElementFinesClient;
    THolder<NDrive::NFine::TFinesManager> FinesManager;
    THolder<TYandexDiskClient> YandexDiskClient;
    THolder<TStartrekClient> StartrekClient;
    THolder<TSolomonClient> SolomonClient;
    THolder<TSolomonClient> DeprecatedSolomonClient;
    THolder<TSamsaraClient> SamsaraClient;
    THolder<TStaffClient> StaffClient;
    THolder<TDatasyncClient> DatasyncClient;
    THolder<IPrivateDataStorage> PrivateDataClient;
    THolder<TDocumentPhotosManager> DocumentPhotosManager;
    THolder<TTakeoutRequestDB> TakeoutRequests;
    THolder<ITakeoutClient> TakeoutClient;
    THolder<IOctopusClient> OctopusClient;
    THolder<NDrive::TGeocoder> GeocoderClient;
    THolder<TImagesStorage> ImagesDB;
    THolder<TDatasyncQueueClient> DatasyncQueueClient;

    THolder<TCarGenericAttachmentDB> CarGenericAttachments;
    THolder<TCarAttachmentAssignmentDB> CarAttachmentAssignments;

    THolder<TInsuranceHistoryManager> InsuranceHistoryManager;
    TMap<NEntityTagsManager::EEntityType, THolder<TTagsSearch>> TagSearches;

    THolder<TConsistencyStorage> ConsistencyDB;

    THolder<TZoneStorage> ZoneDB;

protected:
    bool DoStop() override;
    bool DoStart() override;

public:
    virtual TString GetDatabaseName() const;

    const ITagsHistoryContext& GetTagsHistoryContext() const {
        return TagsManager->GetContext();
    }

    TExpected<TPaymentsData, TString> AddClosedBillingInfo(const TBillingCompilation& compilation, const TString& userId, const NDrive::IServer& server, bool strict) const;
    TExpected<bool, TCodedException> CloseSession(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, bool strict = true) const;

    bool GetCurrentUserSessions(const TString& userId, TVector<TAtomicSharedPtr<const ISession>>& result, TInstant reqActuality, const TSet<TString>& states = TSet<TString>()) const;
    bool GetCurrentSessionsObjects(const TString& userId, TVector<TString>& result, const TInstant reqActuality, const TSet<TString>& states = TSet<TString>()) const;
    bool GetUserSession(const TString& userId, TAtomicSharedPtr<const ISession>& result, const TString& sessionId, TInstant reqActuality) const;

    const IOctopusClient* GetOctopusClient() const {
        return OctopusClient.Get();
    }

    const TImagesStorage& GetImagesDB() const {
        Y_ENSURE_BT(ImagesDB);
        return *ImagesDB;
    }

    const TTakeoutRequestDB& GetTakeoutRequests() const {
        return *TakeoutRequests;
    }

    ITakeoutClient& GetTakeoutClient() const {
        return *TakeoutClient;
    }

    bool HasDocumentPhotosManager() const {
        return !!DocumentPhotosManager;
    }

    TDocumentPhotosManager& GetDocumentPhotosManager() const {
        Y_ENSURE_BT(DocumentPhotosManager);
        return *DocumentPhotosManager;
    }

    bool HasDatasyncQueueClient() const {
        return !!DatasyncQueueClient;
    }

    const TDatasyncQueueClient* GetDatasyncQueueClient() const override {
        return DatasyncQueueClient.Get();
    }

    TCarGenericAttachmentDB& GetCarGenericAttachments() const {
        return *CarGenericAttachments;
    }

    const TCarAttachmentAssignmentDB& GetCarAttachmentAssignments() const override {
        return *CarAttachmentAssignments;
    }

    const TMinimalRidingHistoryManager& GetMinimalCompiledRides() const {
        Y_ENSURE_BT(MinimalCompiledRides);
        return *MinimalCompiledRides;
    }

    const TDriveAPIConfig& GetConfig() const {
        return Config;
    }

    TDriveAPI(const TDriveAPIConfig& config, const TMap<TString, TDatabasePtr>& databases, const ISettings* settings, const NDrive::IServer& server, const TAtomicSharedPtr<IDriveTagsManager> tagsManager);
    ~TDriveAPI();

    TMaybe<TIntroViewInfo> GetUserLandingAlertViewsInfo(const TString& userId, const TString& landingAlertId, NDrive::TEntitySession& session) const;

    bool IncrementUserLandingAlertViewsInfo(const TString& userId, const TString& landingAlertId, const TDuration intervalIgnore, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

    bool DetectObjectTagsOperator(const IEntityTagsManager*& tagsManager, const TSet<TString>& tagIds, NDrive::TEntitySession& session) const;
    bool DetectObjectTagsOperator(const IEntityTagsManager*& tagsManager, const TSet<TString>& tagIds, TVector<TDBTag>& tags, NDrive::TEntitySession& session) const;

    NDrive::TAreaIds GetAreaIds(const TGeoCoord& c) const;
    NDrive::TLocationTags GetTagsInPoint(const TGeoCoord& c, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;

    bool IncrementCashbackSessions(const TUserPermissions& permissions, NDrive::TEntitySession* session, const NDrive::IServer& server) const;
    bool IncrementCashbackSessions(const TUserPermissions& permissions, NDrive::TEntitySession& session, const NDrive::IServer& server) const;

    TVector<TInsuranceNotification> GetActualInsuranceNotifications() const;
    const TInsuranceHistoryManager& GetInsuranceHistoryManager() const;
    bool InsuranceNotification(const TString& sessionId, const TInstant start, const TInstant finish, const TString& userId, const TString& objectId, const TString& historyUser, NDrive::TEntitySession& session) const;
    bool CommitInsuranceNotification(ui64 notificationId, NDrive::TEntitySession& session) const;

    bool CleanLandings(const TString& userId, const TVector<TString>& landingIds, NDrive::TEntitySession* session) const;
    bool AcceptLandings(const TUserPermissions& permissions, const TVector<TLandingAcceptance>& ids, const TInstant reqActuality, NDrive::TEntitySession& session) const;
    bool GetLandingsForUser(const NDrive::IServer* server, const TUserPermissions& permissions, const TGeoCoord* coord, TVector<TLanding>& result, NDrive::TEntitySession& session, TVector<TDBTag>& skippedTags, bool oldStyle = true) const;

    TMap<TString, TGeoCoord> GetObjectFuturePositions() const override;

    const IEntityTagsManager& GetEntityTagsManager(NEntityTagsManager::EEntityType entity) const;

    const TAreasDB& GetAreaManager() const override {
        return *AreasDB;
    }
    const TCarsDB& GetCarManager() const override {
        return *Cars;
    }
    const TUsersDB& GetUserManager() const override {
        return *Users;
    }
    const TRolesManager& GetUserPermissionManager() const override {
        return *RolesManager;
    }
    TUserPermissions::TPtr GetUserPermissions(
        const TString& userId,
        const TUserPermissionsFeatures& userFeatures = {},
        TInstant reqActuality = TInstant::Zero(),
        IReplyContext::TPtr context = nullptr,
        bool addCustomActions = true,
        bool useCache = false,
        bool forceFetchPermissions = false
    ) const override;

    const THeadAccountManager& GetHeadAccountManager() const {
        return *HeadAccountManager;
    }

    const TShortSessionsManager& GetShortSessionsManager() const {
        return *ShortSessionsManager;
    }

    const TImagesStorage& GetImageStorage() const override {
        return GetImagesDB();
    }

    const TLandingsDB& GetLandingManager() const override {
        return *LandingsDB;
    }

    const TSessionManager& GetSessionManager() const override {
        return *SessionManager;
    }

    const TAdditionalServiceSessionManager& GetAdditionalServiceSessionManager() const override {
        return *AdditionalServiceSessionManager;
    }

    const TDedicatedFleetSessionManager& GetDedicatedFleetSessionManager() const override {
        return *DedicatedFleetSessionManager;
    }

    const TMinimalRidingHistoryManager& GetCompiledSessionManager() const override {
        return GetMinimalCompiledRides();
    }

    const IDriveTagsManager& GetTagsManager() const override {
        return *TagsManager;
    }

    const IPrivateDataStorage& GetPrivateDataClient() const {
        return *PrivateDataClient;
    }

    const TBillingManager& GetBillingManager() const {
        Y_ENSURE_BT(BillingManager);
        return *BillingManager;
    }

    bool HasBillingManager() const;

    const TS3Client& GetMDSClient() const {
        Y_ENSURE_BT(MDSClient);
        return *MDSClient;
    }

    bool HasMDSClient() const {
        return !!MDSClient;
    }

    const NDrive::NAutocode::TAutocodeClient& GetAutocodeClient() const {
        Y_ENSURE_BT(AutocodeClient);
        return *AutocodeClient;
    }

    bool HasAutocodeClient() const {
        return !!AutocodeClient;
    }

    const NDrive::TElementFinesClient& GetElementFinesClient() const {
        Y_ENSURE_BT(ElementFinesClient);
        return *ElementFinesClient;
    }

    bool HasElementFineClient() const {
        return !!ElementFinesClient;
    }

    const TMajorClient& GetMajorClient() const override {
        Y_ENSURE_BT(MajorClient);
        return *MajorClient;
    }

    const TMaintenanceDB& GetMaintenanceDB() const {
        Y_ENSURE_BT(MaintenanceDB);
        return *MaintenanceDB;
    }

    bool HasMajorClient() const {
        return !!MajorClient;
    }

    const TYandexDiskClient& GetYandexDiskClient() const {
        Y_ENSURE_BT(YandexDiskClient);
        return *YandexDiskClient;
    }

    bool HasYandexDiskClient() const {
        return !!YandexDiskClient;
    }

    const TStartrekClient& GetStartrekClient() const {
        Y_ENSURE_BT(StartrekClient);
        return *StartrekClient;
    }

    bool HasStartrekClient() const {
        return !!StartrekClient;
    }

    const TSolomonClient& GetSolomonClient() const {
        if (SettingsDB && SettingsDB->GetValueDef("new_solomon", false)) {
            Y_ENSURE_BT(SolomonClient);
            return *SolomonClient;
        }
        Y_ENSURE_BT(DeprecatedSolomonClient);
        return *DeprecatedSolomonClient;
    }

    bool HasSolomonClient() const {
        if (SettingsDB && SettingsDB->GetValueDef("new_solomon", false)) {
            return !!SolomonClient;
        }
        return !!DeprecatedSolomonClient;
    }

    const TSamsaraClient& GetSamsaraClient() const {
        Y_ENSURE_BT(SamsaraClient);
        return *SamsaraClient;
    }

    bool HasSamsaraClient() const {
        return !!SamsaraClient;
    }

    const TStaffClient& GetStaffClient() const {
        Y_ENSURE_BT(StaffClient);
        return *StaffClient;
    }

    bool HasStaffClient() const {
        return !!StaffClient;
    }

    const NDrive::NFine::TFinesManager& GetFinesManager() const {
        Y_ENSURE_BT(FinesManager);
        return *FinesManager;
    }

    bool HasFinesManager() const {
        return !!FinesManager;
    }

    const NDrive::NRenins::TReninsClaimClient& GetReninsClaimClient() const {
        Y_ENSURE_BT(ReninsClaimClient);
        return *ReninsClaimClient;
    }

    bool HasReninsClaimClient() const {
        return !!ReninsClaimClient;
    }

    const NDrive::TIncidentsManager& GetIncidentsManager() const {
        Y_ENSURE_BT(IncidentsManager);
        return *IncidentsManager;
    }

    bool HasIncidentsManager() const {
        return !!IncidentsManager;
    }

    const NDrive::TReninsClient* GetReninsClient() const {
        return ReninsClient.Get();
    }

    const NDrive::TIngosClient* GetIngosClient() const {
        return IngosClient.Get();
    }

    bool HasPassportClient() const {
        return !!PassportClient;
    }

    const TPassportClient& GetPassportClient() const {
        Y_ENSURE_BT(PassportClient);
        return *PassportClient;
    }

    bool HasGeocoderClient() const {
        return !!GeocoderClient;
    }

    const NDrive::TGeocoder& GetGeocoderClient() const {
        Y_ENSURE_BT(GeocoderClient);
        return *GeocoderClient;
    }

    TModelsDB* GetModelsData() const {
        return Models.Get();
    }

    virtual TModelsDB& GetModelsDB() const override;

    TTagsSearch* GetTagsSearch(const NEntityTagsManager::EEntityType entityType) const {
        auto it = TagSearches.find(entityType);
        if (it == TagSearches.end()) {
            return nullptr;
        }
        return it->second.Get();
    }

    virtual TUsersDB* GetUsersData() const override {
        return Users.Get();
    }

    virtual TLandingsDB* GetLandingsDB() const override {
        return LandingsDB.Get();
    }

    virtual TLandingAcceptanceDB* GetUserLandingData() const override {
        return UserLandings.Get();
    }

    virtual TStateFiltersDB* GetStateFiltersDB() const override {
        return StateFiltersDB.Get();
    }

    virtual TNamedFiltersStorage* GetNamedFiltersDB() const {
        return NamedFiltersDB.Get();
    }

    virtual TAreasDB* GetAreasDB() const {
        return AreasDB.Get();
    }

    virtual TSignalsConfigurationsStorage* GetSignalsConfigurationsDB() const {
        return SignalsConfigurationsDB.Get();
    }

    virtual TCarNumbersDB* GetCarNumbers() const override {
        return CarNumbers.Get();
    }

    virtual TDBImei* GetCarsImei() const override {
        return CarsByImei.Get();
    }

    virtual TDBVin* GetCarVins() const override {
        return CarsByVin.Get();
    }

    TCarsDB* GetCarsData() const {
        return Cars.Get();
    }

    const TLoginsManager* GetLoginsManager() const {
        return LoginsManager.Get();
    }

    const TRolesManager* GetRolesManager() const {
        return RolesManager.Get();
    }

    virtual TZoneStorage* GetZoneDB() const {
        Y_ENSURE_BT(ZoneDB);
        return ZoneDB.Get();
    }

    virtual TConsistencyStorage& GetConsistencyDB() const {
        Y_ENSURE_BT(ConsistencyDB);
        return *ConsistencyDB.Get();
    }

    TSet<TString> GetActions(ITag::TConstPtr tag) const;
    TSet<TString> GetActions(const TDBTag& tag) const;

    TVector<TString> GetIMEIs(TConstArrayRef<TString> carIds, const TMap<TString, TDriveCarInfo>& fetchResult) const;
    TVector<TString> GetIMEIs(TConstArrayRef<TString> carIds) const;
    TString GetIMEI(const TString& carId, const TMap<TString, TDriveCarInfo>& fetchResult) const;
    TString GetIMEI(const TString& carId) const;

    TVector<TString> GetCarIdByIMEIs(TConstArrayRef<TString> imeis) const;
    TVector<TString> GetCarIdByNumbers(TConstArrayRef<TString> numbers) const;
    TString GetCarIdByIMEI(const TString& imei) const;
    TString GetCarIdByNumber(const TString& number) const;

    TDBTag GetUserDictionaryTag(const TString& userId, const TString& tagNameSetting, NDrive::TEntitySession& session) const;
    TDBTag GetUserSettings(const TString& userId, NDrive::TEntitySession& session) const;
    TDBTag GetUserAppFlags(const TString& userId, NDrive::TEntitySession& session) const;
    bool CheckCrossloginFlag(const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    bool CheckReferralProgramParticipation(TString& code, const TUserPermissions& permissions, NDrive::TEntitySession& session) const;
    bool CheckReferralProgramParticipation(TString& code, TString& customCode, const TUserPermissions& permissions, NDrive::TEntitySession& session) const;
    bool CheckReferralProgramParticipationImpl(TString& code, const TUserPermissions& permissions, NDrive::TEntitySession& session) const;
    bool CheckReferralProgramParticipationImpl(TString& code, TString& customCode, const TUserPermissions& permissions, NDrive::TEntitySession& session) const;
    bool InvalidateFirstRiding(const TString& userId, const TFullCompiledRiding& compiledRiding, const NDrive::IServer* server) const;

    bool ConfirmUserMail(const TString& key, const TString& user, const TString& userAuthHeader, const TString& clientIp, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    bool BindUserMail(const TString& mail, const TString& user, const TString& userAuthHeader, const TString& clientIp, const NDrive::IServer& server, NDrive::TEntitySession& session) const;

    NThreading::TFuture<TVector<TPaymentMethod>> GetUserPaymentMethods(const TUserPermissions& permissions, const NDrive::IServer& server, bool useCache) const;
    TMaybe<TVector<TPaymentMethod>> GetUserPaymentMethodsSync(const TUserPermissions& permissions, const NDrive::IServer& server, bool useCache) const;
    void UpdateUserPaymentMethods(const TUserPermissions& permissions, const NDrive::IServer& server) const;

    enum EDelegationAbility {
        Possible /* "possible" */,
        SameUser /* "same_user" */,
        TargetUserBlocked /* "target_user_blocked" */,
        TargetUserOnboarding /* "target_user_onboarding" */,
        UserNotRegistered /* "user_not_registered" */,
        UnknownError /* "unknown_error" */,
        NoCarAccess /* "no_car_access" */,
        NoTariffAccess /* "no_tariff_access" */,
        DisabledByUser /* "disabled_by_user" */,
        OfferNotTransferable /* "offer_not_transferable" */,
    };

    TAtomicSharedPtr<IOffer> GetCurrentCarOffer(const TString& carId) const;
    EDelegationAbility GetP2PAbility(const TDriveUserData& user, const TString& objectId, NDrive::TEntitySession& session, bool withOffer = false) const;

    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildYdbTx(const TString& objectName, const NDrive::IServer* server, TDuration statementTimeout = TDuration::Zero(), TDuration lockTimeout = TDuration::Zero()) const;
};
