#pragma once

#include <drive/backend/database/transaction/tx.h>

#include <drive/backend/common/messages.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/storage/sql/query.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/types/cache_with_age.h>

#include <util/datetime/base.h>
#include <util/generic/iterator.h>
#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/random/random.h>
#include <util/system/guard.h>
#include <util/system/mutex.h>

template <bool OneToOne>
class TEntityResultsBuilder;

template <>
class TEntityResultsBuilder<true> {
public:
    template<class TEntity, class TOnReady>
    static void BuildResults(const TRecordsSet& records, TOnReady& onReady, const TString& /*colName*/) {
        for (auto&& i : records.GetRecords()) {
            TEntity entity;
            if (!entity.Parse(i)) {
                ERROR_LOG << "cannot parse object from record " << i.SerializeToJson() << Endl;
            } else {
                onReady(entity);
            }
        }
    }
};

template <>
class TEntityResultsBuilder<false> {
public:
    template <class TEntity, class TOnReady>
    static void BuildResults(const TRecordsSet& records, TOnReady& onReady, const TString& colName) {
        TMap<TString, TVector<NStorage::TTableRecord>> groupedRecords;
        for (auto&& i : records.GetRecords()) {
            TString id;
            Y_ENSURE_BT(i.TryGet(colName, id));
            groupedRecords[id].push_back(i);
        }
        for (auto&& i : groupedRecords) {
            TEntity entity;
            entity.Parse(i.first, i.second);
            onReady(entity);
        }
    }
};

template <class TEntity>
class TBaseEntityManager: public TDatabaseSessionConstructor {
public:
    using TEntities = TVector<TEntity>;
    using TOptionalEntity = TMaybe<TEntity>;
    using TOptionalEntities = TMaybe<TEntities>;
    using TQueryOptions = NSQL::TQueryOptions;

public:
    TBaseEntityManager(NStorage::IDatabase::TPtr db)
        : TDatabaseSessionConstructor(db)
    {
    }
    virtual ~TBaseEntityManager() = default;

    TBaseEntityManager(const TBaseEntityManager& other) = delete;
    TBaseEntityManager& operator=(const TBaseEntityManager& other) = delete;

    virtual TString GetTableName() const = 0;

    template <class TContext = TNull>
    TOptionalEntities Fetch(NStorage::ITransaction::TPtr transaction, const TQueryOptions& options, const TContext* context = nullptr) const {
        NStorage::TObjectRecordsSet<TEntity, TContext> records(context);
        auto query = options.PrintQuery(*Yensured(transaction), GetTableName());
        auto queryResult = transaction->Exec(query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return {};
        }
        return records.DetachObjects();
    }
    template <class TContext = TNull>
    TOptionalEntities Fetch(NDrive::TEntitySession& session, const TQueryOptions& options, const TContext* context = nullptr) const {
        auto transaction = session.GetTransaction();
        auto result = Fetch(transaction, options, context);
        if (!result) {
            session.SetErrorInfo("BaseEntityManager::Get", "transaction error", EDriveSessionResult::TransactionProblem);
        }
        return result;
    }
};

struct TNoFilter {
    template <class T>
    static T&& Filter(T&& container) {
        return std::forward<T>(container);
    }
};

template <typename TType>
struct TBaseFilter {
    template <class T>
    static TVector<TString> Filter(const T& container) {
        TVector<TString> result;
        for (auto&& key : container) {
            if (TType::FilterKey(key)) {
                result.push_back(key);
            } else {
                WARNING_LOG << "filtered out " << key << Endl;
            }
        }
        return result;
    }
};

struct TUuidFilter : public TBaseFilter<TUuidFilter> {
    static bool FilterKey(const TString& key);
};

struct TEmptyFilter : public TBaseFilter<TEmptyFilter> {
    static bool FilterKey(const TString& key);
};

template <class TEntity, bool OneToOne = true, class TKey = TString, class TKeyFilter = TNoFilter>
class TDatabaseEntityManager: public TBaseEntityManager<TEntity> {
private:
    using TBase = TBaseEntityManager<TEntity>;

public:
    using TOptionalEntity = typename TBase::TOptionalEntity;
    using TQueryOptions = typename TBase::TQueryOptions;

protected:
    using TBase::Database;

public:
    TDatabaseEntityManager(NStorage::IDatabase::TPtr db)
        : TBase(db)
    {
    }

    using TBase::Fetch;
    using TBase::GetTableName;

    virtual TString GetColumnName() const {
        return "id";
    }

    virtual TString GetFetchColumnName() const {
        return GetColumnName();
    }

    virtual TKey GetMainId(const TEntity& e) const = 0;

    virtual TString MakeQuery(TStringBuf condition = {}) const {
        auto result = TStringBuilder() << "SELECT * FROM " << GetTableName();
        if (condition) {
            result << " WHERE " << condition;
        }
        return result;
    }

public:
    class TFetchResult {
    private:
        TMap<TKey, TEntity> Result;
        TMaybe<TCodedException> Exception;

    public:
        explicit operator bool() const {
            return !Exception;
        }

        auto begin() const {
            return GetResult().begin();
        }
        auto begin() {
            return MutableResult().begin();
        }
        auto end() const {
            return GetResult().end();
        }
        auto end() {
            return MutableResult().end();
        }
        template <class T>
        auto find(const T& key) const {
            return GetResult().find(key);
        }
        template <class T>
        auto find(const T& key) {
            return MutableResult().find(key);
        }

        bool empty() const {
            return GetResult().empty();
        }
        size_t size() const {
            return GetResult().size();
        }

        void AddResult(const TKey& id, const TEntity& data) {
            MutableResult().emplace(id, data);
        }
        void AddResult(const TKey& id, TEntity&& data) {
            MutableResult().emplace(id, std::move(data));
        }

        const TMap<TKey, TEntity>& GetResult() const {
            EnsureValid();
            return Result;
        }

        TMap<TKey, TEntity>& MutableResult() {
            EnsureValid();
            return Result;
        }

        template <class T>
        const TEntity* GetResultPtr(T&& id) const {
            return GetResult().FindPtr(std::forward<T>(id));
        }
        template <class T>
        TMaybe<TEntity> ExtractResult(T&& id) {
            auto p = MutableResult().FindPtr(std::forward<T>(id));
            return p ? MakeMaybe(std::move(*p)) : Nothing();
        }

        const TCodedException* GetExceptionPtr() const {
            return Exception.Get();
        }
        const TCodedException& GetException() const {
            return *Yensured(Exception);
        }
        void SetException(TCodedException&& value) {
            Exception = std::move(value);
        }

    private:
        void EnsureValid() const noexcept(false) {
            if (Exception) {
                throw *std::move(Exception);
            }
        }
    };

public:
    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult FetchInfoByField(const TKeys& ids, TStringBuf fieldName, NStorage::ITransaction::TPtr transaction) const {
        if (ids.empty()) {
            return {};
        }
        auto condition = TStringBuilder() << fieldName << " IN (" << transaction->Quote(ids) << ")";
        auto query = MakeQuery(condition);
        return FetchWithCustomQuery(query, transaction);
    }

    TFetchResult FetchInfoByField(const TKey& id, TStringBuf fieldName, NStorage::ITransaction::TPtr transaction) const {
        return FetchInfoByField(NContainer::Scalar(id), fieldName, transaction);
    }

    template <class TArg>
    TFetchResult FetchInfoByField(TArg&& arg, TStringBuf fieldName, NDrive::TEntitySession& session) const {
        auto transaction = session.GetTransaction();
        auto result = FetchInfoByField(std::forward<TArg>(arg), fieldName, transaction);
        if (!result) {
            session.SetErrorInfo("FetchInfoByField", "query failed", EDriveSessionResult::TransactionProblem);
        }
        return result;
    }

    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult FetchInfo(const TKeys& ids, NStorage::ITransaction::TPtr transaction) const {
        const auto& filteredIds = TKeyFilter::Filter(ids);
        if (filteredIds.empty()) {
            return {};
        }
        auto query = MakeQuery(
            TStringBuilder() << GetColumnName() << " IN (" << Yensured(transaction)->Quote(filteredIds) << ")"
        );
        return FetchWithCustomQuery(query, transaction);
    }

    TFetchResult FetchInfo(const TKey& id, NStorage::ITransaction::TPtr transaction) const {
        return FetchInfo(NContainer::Scalar(id), transaction);
    }

    template <class TArg>
    TFetchResult FetchInfo(TArg&& arg, NDrive::TEntitySession& session) const {
        auto transaction = session.GetTransaction();
        auto result = FetchInfo(std::forward<TArg>(arg), transaction);
        if (!result) {
            session.SetErrorInfo("FetchInfo", "query failed", EDriveSessionResult::TransactionProblem);
        }
        return result;
    }

    template <class TArg>
    TFetchResult FetchInfo(TArg&& arg) const {
        auto tx = TBase::template BuildTx<NSQL::ReadOnly>();
        return FetchInfo(std::forward<TArg>(arg), tx);
    }

    TFetchResult FetchInfo(NStorage::ITransaction::TPtr transaction, const TQueryOptions& options = {}) const {
        auto query = options.PrintQuery(*Yensured(transaction), GetTableName());
        return FetchWithCustomQuery(query, transaction);
    }

    TFetchResult FetchInfo(NDrive::TEntitySession& session, const TQueryOptions& options = {}) const {
        auto transaction = session.GetTransaction();
        auto query = options.PrintQuery(*Yensured(transaction), GetTableName());
        return FetchWithCustomQuery(query, session);
    }

    TFetchResult FetchInfo(const TQueryOptions& options = {}) const {
        auto tx = TBase::template BuildTx<NSQL::ReadOnly>();
        return FetchInfo(tx, options);
    }

    TFetchResult FetchWithCustomQuery(const TString& query, NStorage::ITransaction::TPtr transaction) const {
        TRecordsSet records;
        TFetchResult result;
        auto queryResult = Yensured(transaction)->Exec(query, &records);
        if (queryResult && queryResult->IsSucceed()) {
            const auto onBuild = [this, &result](const TEntity& e) {
                result.AddResult(this->GetMainId(e), e);
            };
            TEntityResultsBuilder<OneToOne>::template BuildResults<TEntity>(records, onBuild, GetFetchColumnName());
        } else {
            auto e = TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                .AddInfo("query", query)
                .AddInfo("transaction", transaction->GetErrors().GetReport())
                .AddInfo("type", "FetchWithCustomQuery");
            result.SetException(std::move(e));
        }
        return result;
    }

    TFetchResult FetchWithCustomQuery(const TString& query, NDrive::TEntitySession& session) const {
        auto transaction = session.GetTransaction();
        auto result = FetchWithCustomQuery(query, transaction);
        if (!result) {
            session.SetErrorInfo("FetchWithCustomQuery", "query failed", EDriveSessionResult::TransactionProblem);
        }
        return result;
    }

    TFetchResult FetchWithCustomQuery(const TString& query) const {
        auto tx = TBase::template BuildTx<NSQL::ReadOnly>();
        return FetchWithCustomQuery(query, tx);
    }

    TOptionalEntity Fetch(const TKey& id, NDrive::TEntitySession& session) const {
        auto fetchResult = FetchInfo(id, session);
        if (!fetchResult) {
            return {};
        }
        auto p = fetchResult.MutableResult().find(id);
        if (p == fetchResult.MutableResult().end()) {
            session.SetErrorInfo("Fetch", TStringBuilder() << "cannot find id " << id << " in " << GetTableName());
            return {};
        }
        return std::move(p->second);
    }

    virtual void UpdateUniqueCondition(NStorage::TTableRecord& /*unique*/, const TEntity& /*info*/) const {
    }

    bool Upsert(const TEntity& entity, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* resultRecords = nullptr, bool* isUpdate = nullptr) const {
        NStorage::TTableRecord newEntityTableRecord = entity.SerializeToTableRecord();
        NStorage::TTableRecord unique;
        unique.Set(GetColumnName(), this->GetMainId(entity));
        UpdateUniqueCondition(unique, entity);
        NStorage::ITableAccessor::TPtr table = session->GetDatabase().GetTable(GetTableName());
        auto transaction = session.GetTransaction();
        auto result = table->Upsert(newEntityTableRecord, transaction, unique, isUpdate, resultRecords);
        if (!ParseQueryResult(result, session)) {
            return false;
        }
        auto affected = result->GetAffectedRows();
        if (affected != 1) {
            session.SetErrorInfo(GetTableName(), TStringBuilder() << "upserted " << affected << " rows", EDriveSessionResult::TransactionProblem);
            return false;
        }
        return true;
    }

    bool Insert(const TEntity& entity, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* resultRecords = nullptr) const {
        NStorage::TTableRecord newEntityTableRecord = entity.SerializeToTableRecord();
        NStorage::ITableAccessor::TPtr table = session->GetDatabase().GetTable(GetTableName());
        auto transaction = session.GetTransaction();
        auto result = table->AddRow(newEntityTableRecord, transaction, "", resultRecords);
        if (!ParseQueryResult(result, session)) {
            return false;
        }
        auto affected = result->GetAffectedRows();
        if (affected != 1) {
            session.SetErrorInfo(GetTableName(), TStringBuilder() << "inserted " << affected << " rows", EDriveSessionResult::TransactionProblem);
            return false;
        }
        return true;
    }

    bool AddIfNotExists(const TEntity& entity, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* resultRecords = nullptr, bool* isAdd = nullptr) const {
        NStorage::TTableRecord newEntityTableRecord = entity.SerializeToTableRecord();
        NStorage::TTableRecord unique;
        unique.Set(GetColumnName(), this->GetMainId(entity));
        UpdateUniqueCondition(unique, entity);
        NStorage::ITableAccessor::TPtr table = session->GetDatabase().GetTable(GetTableName());
        auto transaction = session.GetTransaction();
        auto result = table->AddIfNotExists(newEntityTableRecord, transaction, unique, resultRecords);
        if (!ParseQueryResult(result, session)) {
            return false;
        }
        if (isAdd) {
            *isAdd = result->GetAffectedRows();
        }
        return true;
    }

    bool Remove(const TKey& id, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* resultRecords = nullptr) const {
        return Remove(NContainer::Scalar(id), session, resultRecords);
    }

    template <class TKeys, class = typename TKeys::iterator>
    bool Remove(const TKeys& ids, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* resultRecords = nullptr) const {
        NStorage::ITableAccessor::TPtr table = session->GetDatabase().GetTable(GetTableName());
        NStorage::ITransaction::TPtr transaction = session.GetTransaction();
        TString condition = "true";
        if (ids.size()) {
            condition = GetColumnName() + " IN (" + transaction->Quote(ids) + ")";
        }
        NStorage::IQueryResult::TPtr result = table->RemoveRow(condition, transaction, resultRecords);
        if (!ParseQueryResult(result, session)) {
            return false;
        }
        return true;
    }
};

template <class TEntity, bool OneToOne = true, class TKey = TString>
using TDBEntities = TDatabaseEntityManager<TEntity, OneToOne, TKey>;

template <class TEntity, bool OneToOne = true, class TKey = TString, class TKeyFilter = TNoFilter>
class TCachedEntityManager
    : public TDatabaseEntityManager<TEntity, OneToOne, TKey, TKeyFilter>
    , public IAutoActualization
    , public IMessageProcessor
{
private:
    using TBase = TDatabaseEntityManager<TEntity, OneToOne, TKey, TKeyFilter>;

private:
    using TBase::GetDatabase;
    using TBase::GetTableName;
    using TBase::GetFetchColumnName;

    virtual bool Refresh() override {
        FetchInfo(Now(), nullptr);
        return true;
    }

protected:
    TDuration DatabaseWaiting = TDuration::Minutes(15);
    TDuration DatabasePingWaiting = TDuration::Seconds(1);
    mutable TCacheWithAge<TKey, TEntity> Cache;

protected:
    void DropCache() {
        TWriteGuard guard(FetchResultMutex);
        FetchAllInstant = TInstant::Zero();
        Cache.Drop();
    }

    virtual bool DoStart() override {
        TTimeGuardImpl<false, TLOG_INFO> tg("EntityDB construction for " + GetTableName());
        IAutoActualization::SetName("EntityDB:" + GetTableName());
        return IAutoActualization::DoStart();
    }

public:
    using TFetchResult = typename TBase::TFetchResult;
    using TRecordType = TEntity;

public:
    TCachedEntityManager(NStorage::IDatabase::TPtr db)
        : TBase(db)
        , IAutoActualization("FakeNameUntilStart")
    {
        RegisterGlobalMessageProcessor(this);
    }
    ~TCachedEntityManager() {
        UnregisterGlobalMessageProcessor(this);
    }

    using TBase::MakeQuery;

    virtual bool Process(IMessage* message) override {
        const NDrive::TCacheRefreshMessage* dropCache = dynamic_cast<const NDrive::TCacheRefreshMessage*>(message);
        if (dropCache) {
            if (dropCache->GetComponents().empty() || dropCache->GetComponents().contains(GetTableName())) {
                DropCache();
            }
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return "cache_" + ToString<const void*>(this);
    }

    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult GetCached(const TKeys& ids, const TInstant reqActuality = TInstant::Zero(), TVector<TKey>* missing = nullptr) const {
        TFetchResult result;
        auto onData = [&result](const TKey& id, const TEntity& info) {
            result.AddResult(id, info);
        };
        auto onMissing = [missing](const TKey& id) {
            if (missing) {
                missing->push_back(id);
            }
        };
        Cache.ProcessIds(ids, onData, onMissing, reqActuality);
        return result;
    }

    TFetchResult GetCached(const TKey& id, const TInstant reqActuality = TInstant::Zero(), TVector<TKey>* missing = nullptr) const {
        return GetCached(NContainer::Scalar(id), reqActuality, missing);
    }

    TFetchResult GetCached() const {
        TFetchResult result;
        FetchInfo(TInstant::Zero(), &result);
        return result;
    }

    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult GetCachedOrFetch(const TKeys& ids, NDrive::TEntitySession& session) const {
        TFetchResult result = GetCached(ids);
        if (!result || result.size() != ids.size()) {
            result = FetchInfo(ids, session);
        }
        return result;
    }

    TFetchResult GetCachedOrFetch(const TKey& id, NDrive::TEntitySession& session) const {
        return GetCachedOrFetch(NContainer::Scalar(id), session);
    }

    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult GetCachedOrFetch(const TKeys& ids) const {
        TFetchResult result = GetCached(ids);
        if (!result || result.size() != ids.size()) {
            result = FetchInfo(ids);
        }
        return result;
    }

    TFetchResult GetCachedOrFetch(const TKey& id) const {
        return GetCachedOrFetch(NContainer::Scalar(id));
    }

    using TBase::FetchInfo;

    TFetchResult FetchInfo(const TKey& id, const TInstant actuality, NStorage::ITransaction::TPtr transactionExt = nullptr) const {
        return FetchInfo(NContainer::Scalar(id), actuality, transactionExt);
    }

    template <class TKeys, class = typename TKeys::iterator>
    TFetchResult FetchInfo(const TKeys& ids, const TInstant reqActuality, NStorage::ITransaction::TPtr transactionExt = nullptr) const {
        TInstant actualityEffective = reqActuality;
        if (FetchAllInstant > actualityEffective) {
            actualityEffective = TInstant::Zero();
        }
        TTimeGuard tg("FETCH " + GetTableName());
        TVector<TKey> idsFetch;
        TFetchResult result = GetCached(ids, actualityEffective, &idsFetch);
        DEBUG_LOG << GetTableName() << "(" << actualityEffective.MilliSeconds() << "ms)" << " fetching. Re-fetch size: " << idsFetch.size() << "; Cached size: " << result.size() << Endl;
        if (idsFetch.size()) {
            auto transaction = !!transactionExt ? transactionExt : GetDatabase().CreateTransaction(true);
            auto fetchResult = FetchInfo(idsFetch, transaction);
            for (auto&& [key, value] : fetchResult) {
                result.AddResult(key, std::move(value));
            }
            for (auto&& [key, value] : result) {
                Cache.Refresh(key, value);
            }
            for (auto&& i : ids) {
                if (!result.GetResultPtr(i)) {
                    Cache.SetAbsent(i);
                }
            }
        }
        return result;
    }

    TFetchResult FetchInfo(const TInstant reqActuality) const {
        TFetchResult result;
        FetchInfo(reqActuality, &result);
        return result;
    }

private:
    void FetchInfo(const TInstant reqActuality, TFetchResult* resultReply) const {
        const TInstant start = Now();
        while (Now() - start < DatabaseWaiting) {
            if (FetchAllInstant > reqActuality) {
                TReadGuard rg(FetchResultMutex);
                if (resultReply) {
                    *resultReply = FetchAllResult;
                }
            } else {
                TWriteGuard wgRequest(RequestMutex);
                if (FetchAllInstant > reqActuality) {
                    TReadGuard rg(FetchResultMutex);
                    if (resultReply) {
                        *resultReply = FetchAllResult;
                    }
                    return;
                }

                TFetchResult result;
                const TInstant actualFetchInstantNext = ModelingNow();

                if (!FetchUpdatedInfoDelta(result)) {
                    Sleep(DatabasePingWaiting);
                    continue;
                }

                TWriteGuard rg(FetchResultMutex);
                FetchAllInstant = actualFetchInstantNext;
                FetchAllResult = std::move(result);
                if (resultReply) {
                    *resultReply = FetchAllResult;
                }
            }
            return;
        }
    }

    bool FetchUpdatedInfoDelta(TFetchResult& result) const {
        auto transaction = GetDatabase().CreateTransaction(true);
        TRecordsSet records;
        TString query = MakeQuery();
        NStorage::IQueryResult::TPtr qResult = transaction->Exec(query, &records);
        if (qResult && qResult->IsSucceed()) {
            const auto onBuild = [this, &result](const TEntity& e) {
                result.AddResult(this->GetMainId(e), e);
            };
            TEntityResultsBuilder<OneToOne>::template BuildResults<TEntity>(records, onBuild, GetFetchColumnName());
            for (auto&& i : result.GetResult()) {
                Cache.Refresh(i.first, i.second);
            }
            return true;
        } else {
            ERROR_LOG << transaction->GetErrors().GetReport() << Endl;
            return false;
        }
    }

private:
    mutable TInstant FetchAllInstant = TInstant::Zero();
    TRWMutex FetchResultMutex;
    TRWMutex RequestMutex;
    mutable TFetchResult FetchAllResult;
};
