#include "search_index.h"

TBasicSearchIndex::TSearchMatchedEntry::TSearchMatchedEntry(const TMapping& mapping, const TString& requiredPrefix, TTraits requiredFlags, TMapping::const_iterator keyIterator)
    : Mapping(mapping)
    , RequiredPrefix(requiredPrefix)
    , RequiredTraits(requiredFlags)
    , KeyIterator(keyIterator)
{
    if (KeyIterator != Mapping.end()) {
        RecordIterator = KeyIterator->second.begin();
    }
}

TString TBasicSearchIndex::TSearchMatchedEntry::GetNextId() {
    TString result;
    while (!result) {
        if (KeyIterator == Mapping.end()) {
            break;
        }
        if (!KeyIterator->first.StartsWith(RequiredPrefix)) {
            break;
        }
        Y_ASSERT(RecordIterator != KeyIterator->second.end());
        if (RecordIterator != KeyIterator->second.end()) {
            auto&& [id, traits] = *RecordIterator;
            if (traits & RequiredTraits) {
                result = id;
            }
            ++RecordIterator;
        }
        if (RecordIterator == KeyIterator->second.end()) {
            ++KeyIterator;
            if (KeyIterator != Mapping.end()) {
                RecordIterator = KeyIterator->second.begin();
            }
        }
    }
    return result;
}

void TBasicSearchIndex::AddEntity(const TString& id, const TVector<TBasicSearchIndex::TSearchEntityProp>& entityProps) {
    for (auto&& prop : entityProps) {
        Mapping[prop.GetValue()][id] |= prop.GetTraits();
    }
    if (!EntityProps.count(id)) {
        Ids.emplace_back(id);
    }
    EntityProps[id] = entityProps;
}

void TBasicSearchIndex::RemoveEntity(const TString& id, const TVector<TBasicSearchIndex::TSearchEntityProp>& entityProps) {
    for (auto&& prop : entityProps) {
        if (prop.IsUndying()) {
            continue;
        }
        auto mapping = Mapping.find(prop.GetValue());
        if (mapping == Mapping.end()) {
            ERROR_LOG << "trying to delete already deleted entity " << id << ": no mapping" << Endl;
            continue;
        }
        auto traitedId = mapping->second.find(id);
        if (traitedId == mapping->second.end()) {
            ERROR_LOG << "trying to delete already deleted entity " << id << ": no traited id" << Endl;
            continue;
        }
        traitedId->second &= (~prop.GetTraits());
        if (traitedId->second == 0) {
            mapping->second.erase(traitedId);
        }
        if (mapping->second.size() == 0) {
            Mapping.erase(mapping);
        }
    }
    EntityProps.erase(EntityProps.find(id));
}

bool TBasicSearchIndex::HasContinuation(const TString& prefix, const TVector<TBasicSearchIndex::TSearchEntityProp>& props) const {
    for (auto&& prop : props) {
        if (prop.GetValue().StartsWith(prefix)) {
            return true;
        }
    }
    return false;
}

TVector<TBasicSearchIndex::TSearchEntityProp> TBasicSearchIndex::GetPropsById(const TString& id) const {
    TReadGuard rg(RWMutex);
    auto p = EntityProps.find(id);
    if (p == EntityProps.end()) {
        return {};
    }
    return p->second;
}

TBasicSearchIndex::TSearchMatchedEntry TBasicSearchIndex::GetIdsMatchingPrefix(const TString& prefix, TTraits traits) const {
    TReadGuard rg(RWMutex);
    auto it = Mapping.lower_bound(prefix);
    if (it == Mapping.end() || !it->first.StartsWith(prefix)) {
        return { Mapping, prefix, traits, Mapping.end() };
    } else {
        return { Mapping, prefix, traits, it };
    }
}

void TBasicSearchIndex::Refresh(const TString& id, TVector<TBasicSearchIndex::TSearchEntityProp>& indexableProps) {
    TWriteGuard wg(RWMutex);

    size_t nextAvailablePosition = 0;
    for (size_t i = 0; i < indexableProps.size(); ++i) {
        try {
            indexableProps[nextAvailablePosition].SetValue(ToLowerUTF8(indexableProps[i].GetValue()));
        } catch (const std::exception& e) {
            ERROR_LOG << "Could not lowercase UTF8 string: " << indexableProps[i].GetValue() << "; " << FormatExc(e) << Endl;
            continue;
        }
        ++nextAvailablePosition;
    }
    indexableProps.resize(nextAvailablePosition);

    Remove(id);
    AddEntity(id, indexableProps);
}

void TBasicSearchIndex::Remove(const TString& id) {
    if (EntityProps.count(id)) {
        RemoveEntity(id, EntityProps[id]);
    }
}
