#pragma once

#include "search_request.h"

#include <drive/backend/database/transaction/tx.h>

#include <util/generic/iterator.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/system/guard.h>
#include <util/system/rwlock.h>
#include <rtline/util/algorithm/container.h>

class TBasicSearchIndex {
private:
    using TTraits = TSearchTraits;
    using TTraitedIds = TMap<TString, TTraits>;
    using TMapping = TMap<TString, TTraitedIds>;

    class TSearchMatchedEntry {
    private:
        const TMapping& Mapping;
        const TString RequiredPrefix;
        const TTraits RequiredTraits;

        TMapping::const_iterator KeyIterator;
        TTraitedIds::const_iterator RecordIterator;

    public:
        TSearchMatchedEntry(const TMapping& mapping, const TString& requiredPrefix, TTraits requiredFlags, TMapping::const_iterator keyIterator);

        TString GetNextId();
    };

public:
    class TSearchEntityProp {
        R_FIELD(TString, Value);
        R_FIELD(TTraits, Traits, AllSearchTraits);
        R_READONLY(bool, Undying, false);

    public:
        TSearchEntityProp() = default;
        TSearchEntityProp(const TString& value, const bool isUndying = false)
            : Value(value)
            , Undying(isUndying)
        {
        }
    };

private:
    TMapping Mapping;
    THashMap<TString, TVector<TSearchEntityProp>> EntityProps;
    TVector<TString> Ids;
    TRWMutex RWMutex;

private:
    void AddEntity(const TString& id, const TVector<TSearchEntityProp>& entityProps);
    void RemoveEntity(const TString& id, const TVector<TSearchEntityProp>& entityProps);
    bool HasContinuation(const TString& prefix, const TVector<TSearchEntityProp>& props) const;

    TVector<TSearchEntityProp> GetPropsById(const TString& id) const;
    TSearchMatchedEntry GetIdsMatchingPrefix(const TString& prefix, TTraits traits) const;

public:
    void Refresh(const TString& id, TVector<TSearchEntityProp>& indexableProps);
    void Remove(const TString& id);

    template<class TFunc>
    TVector<TString> GetMatchingIds(const TSearchRequest& searchRequest, TFunc&& entityFilter) const {
        TReadGuard rg(RWMutex);
        TSet<TString> takenIds;

        if ((searchRequest.RequiredMatches.empty() && searchRequest.OptionalMatches.empty()) || searchRequest.HasEmptyOptionalMatch) {
            TVector<TString> result;
            for (auto id : Ids) {
                if (entityFilter(id)) {
                    result.push_back(id);
                }
                if (result.size() == searchRequest.Limit) {
                    break;
                }
            }
            return result;
        }

        if (searchRequest.RequiredMatches.empty()) {
            for (auto token : searchRequest.OptionalMatches) {
                auto idIterator = GetIdsMatchingPrefix(token, searchRequest.Traits);
                while (auto id = idIterator.GetNextId()) {
                    if (entityFilter(id)) {
                        takenIds.emplace(id);
                    }
                    if (takenIds.size() == searchRequest.Limit) {
                        break;
                    }
                }
            }
        } else {
            auto idIterator = GetIdsMatchingPrefix(searchRequest.RequiredMatches[0], searchRequest.Traits);
            while (auto id = idIterator.GetNextId()) {
                bool isMatching = true;
                if (searchRequest.RequiredMatches.size() != 1 || !searchRequest.OptionalMatches.empty()) {
                    const TVector<TSearchEntityProp> props = GetPropsById(id);

                    for (size_t i = 1; i < searchRequest.RequiredMatches.size(); ++i) {
                        if (!HasContinuation(searchRequest.RequiredMatches[i], props)) {
                            isMatching = false;
                            break;
                        }
                    }
                    if (isMatching && searchRequest.OptionalMatches.size()) {
                        isMatching = false;
                        for (auto token : searchRequest.OptionalMatches) {
                            if (HasContinuation(token, props)) {
                                isMatching = true;
                                break;
                            }
                        }
                    }
                }
                if (isMatching) {
                    if (entityFilter(id)) {
                        takenIds.emplace(id);
                    }
                    if (takenIds.size() == searchRequest.Limit) {
                        break;
                    }
                }
            }
        }
        return MakeVector(takenIds);
    }
};
