#pragma once

#include <util/charset/utf8.h>
#include <util/generic/vector.h>

struct TMatchCondition {
    TString Token;
    bool IsRequired;

    TMatchCondition() = default;
    TMatchCondition(const TString& token, const bool isRequired)
        : Token(token)
        , IsRequired(isRequired)
    {
    }
};

using TSearchTraits = ui64;
constexpr TSearchTraits AllSearchTraits = Max<TSearchTraits>();

struct TSearchRequest {
    TVector<TString> RequiredMatches;
    TVector<TString> OptionalMatches;
    TVector<TString> OriginalOptionalMatches;
    TVector<TString> OriginalRequiredMatches;
    TSearchTraits Traits = AllSearchTraits;
    ui32 Limit = 20;
    bool HasEmptyOptionalMatch = false;

    void AddMatchingCondition(const TMatchCondition& matchCondition) {
        if (matchCondition.IsRequired) {
            if (!!matchCondition.Token) {
                OriginalRequiredMatches.push_back(matchCondition.Token);
                RequiredMatches.push_back(ToLowerUTF8(matchCondition.Token));
            }
        } else {
            OriginalOptionalMatches.push_back(matchCondition.Token);
            OptionalMatches.push_back(ToLowerUTF8(matchCondition.Token));
            if (!matchCondition.Token) {
                HasEmptyOptionalMatch = true;
            }
        }
    }

    bool IsOneTermSearch() const {
        return RequiredMatches.size() + OptionalMatches.size() == 1;
    }

    const TString& GetOriginalSearchTerm() const {
        if (!OriginalRequiredMatches.empty()) {
            return OriginalRequiredMatches.front();
        }
        return OriginalOptionalMatches.empty() ? Default<TString>() : OriginalOptionalMatches.front();
    }

    const TString& GetSearchTerm() const {
        if (!RequiredMatches.empty()) {
            return RequiredMatches.front();
        }
        return OptionalMatches.empty() ? Default<TString>() : OptionalMatches.front();
    }

    void SetTraits(TSearchTraits traits) {
        Traits = traits;
    }

    void SetLimit(const ui32 newLimit) {
        Limit = newLimit;
    }
};
