#pragma once

#include "common.h"
#include "config.h"
#include "manager.h"
#include "sequential.h"

#include <drive/backend/abstract/base.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/auto_actualization.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>
#include <util/random/random.h>
#include <util/system/mutex.h>
#include <util/system/rusage.h>
#include <util/system/rwlock.h>

template <class THistoryManager, class TEntity, class TIdType = TString>
class TDBCacheWithHistoryOwner
    : public IAutoActualization
    , public IMessageProcessor
    , public TDBEntitiesCacheImpl<TEntity, TIdType>
{
protected:
    using TBaseImpl = TDBEntitiesCacheImpl<TEntity, TIdType>;
    using TEventId = typename THistoryManager::TEventId;
    using TIdRef = typename TIdTypeSelector<TIdType>::TIdRef;

protected:
    using TBaseImpl::MakeObjectWriteGuard;
    using TBaseImpl::Objects;

    THolder<IHistoryContext> HistoryContext;
    THolder<THistoryManager> HistoryManager;

protected:
    virtual TDuration GetDefaultWaitingDuration() const {
        return TDuration::Seconds(1);
    }

private:
    mutable ui64 InsertedEventsCount = Max<ui64>();
    mutable TEventId LockedHistoryEventId = NDrive::IncorrectEventId;
    mutable TSet<TEventId> ReadyEvents;
    mutable TInstant LastRefreshInstant = TInstant::Zero();
    TMutex MutexRebuild;
    TAutoEvent EventWaitingHistory;

protected:
    NStorage::IDatabase::TPtr HistoryCacheDatabase;
    const THistoryConfig HistoryConfig;

protected:
    [[nodiscard]] bool RebuildCache(const TInstant reqActuality) const {
        const TSet<TString> deps = GetDependencesBefore();
        for (auto&& i : deps) {
            SendGlobalMessage<TBeforeCacheInvalidateNotification>(i, HistoryManager->GetCurrentInstant(), HistoryManager->GetTableName());
        }
        if (!HistoryManager->Update(reqActuality)) {
            return false;
        }
        INFO_LOG << "RESOURCE: Start snapshot construction for " << HistoryManager->GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
        bool result = false;
        {
            auto g = MakeObjectWriteGuard();
            const ui64 insertedEventsCount = HistoryManager->GetInsertedEventsCount();
            const ui64 lockedHistoryEventId = HistoryManager->GetLockedMaxEventId();
            const TInstant lastRefreshInstant = HistoryManager->GetCurrentInstant();

            Objects.clear();
            result = DoRebuildCacheUnsafe();

            if (result) {
                if (lockedHistoryEventId != HistoryManager->GetIncorrectEventId()) {
                    InsertedEventsCount = insertedEventsCount;
                    LockedHistoryEventId = lockedHistoryEventId;
                    LastRefreshInstant = lastRefreshInstant;
                } else {
                    InsertedEventsCount = 0;
                    LockedHistoryEventId = 0;
                    LastRefreshInstant = TInstant::Zero();
                }
            }
            INFO_LOG << "RESOURCE: Finish snapshot construction for " << HistoryManager->GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
        }
        {
            const TSet<TString> deps = GetDependencesAfter();
            for (auto&& i : deps) {
                INFO_LOG << "send TAfterCacheInvalidateNotification for " << i << Endl;
                SendGlobalMessage<TAfterCacheInvalidateNotification>(i, HistoryManager->GetCurrentInstant(), HistoryManager->GetTableName());
            }
        }
        return result;
    }

    virtual bool Process(IMessage* message) override {
        const TServerStartedMessage* activityChecker = dynamic_cast<const TServerStartedMessage*>(message);
        if (activityChecker) {
            CHECK_WITH_LOG(IsActive()) << HistoryManager->GetTableName() << " disabled" << Endl;
            return true;
        }

        const TCommonCacheInvalidateNotification* notifyHistoryInvalidate = dynamic_cast<const TCommonCacheInvalidateNotification*>(message);
        if (notifyHistoryInvalidate) {
            if (notifyHistoryInvalidate->GetTableName() == HistoryManager->GetTableName()) {
                bool refreshed = RefreshCache(notifyHistoryInvalidate->GetReqActuality());
                if (!refreshed) {
                    ERROR_LOG << "could not refresh cache invoked by CommonCacheInvalidateNotification message" << Endl;
                }
            }
            return true;
        }
        const TNotifyHistoryChanged* notifyHistoryChanged = dynamic_cast<const TNotifyHistoryChanged*>(message);
        if (notifyHistoryChanged) {
            if (notifyHistoryChanged->GetTableName() == HistoryManager->GetTableName()) {
                EventWaitingHistory.Signal();
            }
            return true;
        }
        const NDrive::TCacheRefreshMessage* dropCache = dynamic_cast<const NDrive::TCacheRefreshMessage*>(message);
        if (dropCache) {
            if (dropCache->GetComponents().empty() || dropCache->GetComponents().contains(HistoryManager->GetTableName())) {
                INFO_LOG << "Refresh Cache for " << HistoryManager->GetTableName() << Endl;
                bool rebuilt = RebuildCache(Now());
                if (rebuilt) {
                    INFO_LOG << "Refresh Cache for " << HistoryManager->GetTableName() << " OK" << Endl;
                } else {
                    ERROR_LOG << "Refresh Cache for " << HistoryManager->GetTableName() << " FAILURE" << Endl;
                }
            }
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return GetName();
    }

    virtual bool DoRebuildCacheUnsafe() const = 0;

    virtual TIdRef GetEventObjectId(const typename THistoryManager::THistoryEvent& ev) const = 0;
    virtual void AcceptHistoryEventUnsafe(const typename THistoryManager::THistoryEvent& ev) const = 0;

    virtual bool Refresh() override {
        if (EventWaitingHistory.WaitD(Now() + GetDefaultWaitingDuration())) {
            return RefreshCache(TInstant::Zero(), false);
        } else {
            auto lastRefreshInstant = HistoryManager->GetCurrentInstant();
            auto g = MakeObjectWriteGuard();
            LastRefreshInstant = std::max(LastRefreshInstant, lastRefreshInstant);
        }
        return true;
    }

public:
    ~TDBCacheWithHistoryOwner() {
        UnregisterGlobalMessageProcessor(this);
    }

    ui64 GetInsertedEventsCount() const {
        return InsertedEventsCount;
    }

    TEventId GetLockedHistoryEventId() const {
        return LockedHistoryEventId;
    }

    TInstant GetLastRefreshInstant() const {
        return LastRefreshInstant;
    }

    virtual TSet<TString> GetDependencesBefore() const {
        return {};
    }

    virtual TSet<TString> GetDependencesAfter() const {
        return {};
    }

    TDBCacheWithHistoryOwner(const TString& name, const IHistoryContext& context, const THistoryConfig& hConfig)
        : IAutoActualization(name + "-cache_refresh_actualization", hConfig.GetPingPeriod())
        , TBaseImpl(name)
        , HistoryManager(new THistoryManager(context, hConfig))
        , HistoryCacheDatabase(context.GetDatabase())
        , HistoryConfig(hConfig)
    {
        RegisterGlobalMessageProcessor(this);
    }

    TDBCacheWithHistoryOwner(const TString& name, THolder<IHistoryContext>&& context, const THistoryConfig& hConfig)
        : IAutoActualization(name + "-cache_refresh_actualization", hConfig.GetPingPeriod())
        , TBaseImpl(name)
        , HistoryContext(std::move(context))
        , HistoryManager(new THistoryManager(*HistoryContext, hConfig))
        , HistoryCacheDatabase(HistoryContext->GetDatabase())
        , HistoryConfig(hConfig)
    {
        RegisterGlobalMessageProcessor(this);
    }

    [[nodiscard]] bool RefreshCache(const TInstant reqActuality, const bool doActualizeHistory = true) const override {
        if (LockedHistoryEventId == NDrive::IncorrectEventId) {
            TGuard<TMutex> g(MutexRebuild);
            if (LockedHistoryEventId == NDrive::IncorrectEventId) {
                return RebuildCache(reqActuality);
            }
            return true;
        }
        if (doActualizeHistory) {
            if (LastRefreshInstant >= reqActuality) {
                if (reqActuality != TInstant::Zero()) {
                    DEBUG_LOG << "Skip re-fetch cache for " << HistoryManager->GetTableName() << "/" << LastRefreshInstant << ">=" << reqActuality << Endl;
                }
                return true;
            }
            if (!HistoryManager->Update(reqActuality)) {
                ERROR_LOG << "Cannot re-fetch cache for " << HistoryManager->GetTableName() << Endl;
                return false;
            }
        }
        const auto lockedHistoryEventId = HistoryManager->GetLockedMaxEventId();
        const ui64 insertedEventsCount = HistoryManager->GetInsertedEventsCount();
        const TInstant lastRefreshInstant = HistoryManager->GetCurrentInstant();
        if (insertedEventsCount == InsertedEventsCount && (lockedHistoryEventId == LockedHistoryEventId || lockedHistoryEventId == HistoryManager->GetIncorrectEventId())) {
            if (LastRefreshInstant < lastRefreshInstant) {
                auto g = MakeObjectWriteGuard();
                LastRefreshInstant = Max(LastRefreshInstant, lastRefreshInstant);
                DEBUG_LOG << "Cache re-fetching for " << HistoryManager->GetTableName() << " is useless but ts moved" << Endl;
            } else {
                DEBUG_LOG << "Cache re-fetching for " << HistoryManager->GetTableName() << " is useless" << Endl;
            }
            return true;
        }

        TVector<TAtomicSharedPtr<typename THistoryManager::THistoryEvent>> events;
        if (!HistoryManager->GetEventsSinceId(LockedHistoryEventId + 1, events, reqActuality)) {
            DEBUG_LOG << "Cannot re-fetch " << HistoryManager->GetTableName() << " cache" << Endl;
            return false;
        }
        if (events.size()) {
            const TSet<TString> deps = GetDependencesBefore();
            for (auto&& i : deps) {
                SendGlobalMessage<TBeforeCacheInvalidateNotification>(i, HistoryManager->GetCurrentInstant(), HistoryManager->GetTableName());
            }
        }
        {
            auto g = MakeObjectWriteGuard();
            if (doActualizeHistory && LastRefreshInstant >= reqActuality) {
                if (reqActuality != TInstant::Zero()) {
                    DEBUG_LOG << "Skip re-fetch cache for " << HistoryManager->GetTableName() << "/" << LastRefreshInstant << ">=" << reqActuality << Endl;
                }
                return true;
            }
            TTimeGuard tg("Objects from " + HistoryManager->GetTableName() + " cache re-fetching...");
            TSet<TIdRef> eventObjectIds;
            for (auto&& i : events) {
                TIdRef eventObjectId = GetEventObjectId(*i);
                if (ReadyEvents.contains(i->GetHistoryEventId()) && !eventObjectIds.contains(eventObjectId)) {
                    continue;
                }
                ReadyEvents.emplace(i->GetHistoryEventId());
                eventObjectIds.emplace(eventObjectId);
                AcceptHistoryEventUnsafe(*i);
            }
            if (lockedHistoryEventId != HistoryManager->GetIncorrectEventId() && LockedHistoryEventId < lockedHistoryEventId) {
                if (ReadyEvents.empty() || *ReadyEvents.rbegin() <= lockedHistoryEventId) {
                    ReadyEvents.clear();
                } else {
                    ReadyEvents.erase(ReadyEvents.begin(), ReadyEvents.upper_bound(lockedHistoryEventId));
                }
                LockedHistoryEventId = lockedHistoryEventId;
            }
            InsertedEventsCount = Max(InsertedEventsCount, insertedEventsCount);
            LastRefreshInstant = Max(LastRefreshInstant, lastRefreshInstant);
        }
        {
            const TSet<TString> deps = GetDependencesAfter();
            for (auto&& i : deps) {
                SendGlobalMessage<TAfterCacheInvalidateNotification>(i, HistoryManager->GetCurrentInstant(), HistoryManager->GetTableName());
            }
        }
        return true;
    }

    virtual bool DoStart() override {
        if (HistoryManager && !HistoryManager->Start()) {
            return false;
        }
        EventWaitingHistory.Signal();
        if (!IAutoActualization::DoStart()) {
            return false;
        }
        return true;
    }

    virtual bool DoStop() override {
        EventWaitingHistory.Signal();
        if (!IAutoActualization::DoStop()) {
            return false;
        }
        if (HistoryManager && !HistoryManager->Stop()) {
            return false;
        }
        return true;
    }

    TInstant GetLastUpdateTimestamp() const override {
        return LastRefreshInstant;
    }

    THistoryManager& GetHistoryManager() {
        return *HistoryManager;
    }
    const THistoryManager& GetHistoryManager() const {
        return *HistoryManager;
    }

    const THistoryConfig& GetHistoryConfig() const {
        return HistoryConfig;
    }
};

template <class TEntity, class TObjectId = TString>
class TAutoActualizingSnapshot
    : public IHistoryCallback<TObjectEvent<TEntity>, TObjectId>
    , public IMessageProcessor
{
private:
    using TBase = IHistoryCallback<TObjectEvent<TEntity>, TObjectId>;
    using THistoryReader = TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<TEntity>, TObjectId>>;

protected:
    using TBase::MakeObjectWriteGuard;
    using TBase::Objects;

protected:
    THistoryReader HistoryReader;

protected:
    virtual bool DoRebuildCacheUnsafe() const = 0;

public:
    TAutoActualizingSnapshot(const TString& name, THistoryReader historyReader)
        : TBase(name)
        , HistoryReader(historyReader)
        , Name_(name)
    {
        Y_ENSURE_BT(HistoryReader);
        HistoryReader->RegisterCallback(this);
        RegisterGlobalMessageProcessor(this);
    }

    virtual ~TAutoActualizingSnapshot() {
        UnregisterGlobalMessageProcessor(this);
        HistoryReader->UnregisterCallback(this);
    }

    TInstant GetLastUpdateTimestamp() const override {
        return HistoryReader->GetCurrentInstant();
    }

    [[nodiscard]] virtual bool RefreshCache(const TInstant reqActuality, const bool doActualizeHistory = true) const override {
        Y_UNUSED(doActualizeHistory);
        return HistoryReader->Update(reqActuality);
    }

    [[nodiscard]] bool RebuildCache() {
        auto wg = MakeObjectWriteGuard();
        return DoRebuildCacheUnsafe();
    }

    virtual bool Process(IMessage* message) override {
        const NDrive::TCacheRefreshMessage* dropCache = dynamic_cast<const NDrive::TCacheRefreshMessage*>(message);
        if (dropCache) {
            if (dropCache->GetComponents().empty() || dropCache->GetComponents().contains(HistoryReader->GetTableName())) {
                INFO_LOG << "Refresh Cache for " << HistoryReader->GetTableName() << Endl;
                bool updated = HistoryReader->Update(Now());
                if (updated) {
                    INFO_LOG << "Refresh Cache for " << HistoryReader->GetTableName() << " OK" << Endl;
                } else {
                    ERROR_LOG << "Refresh Cache for " << HistoryReader->GetTableName() << " failed" << Endl;
                }
            }
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return "callback_cache_history_" + Name_;
    }

private:
    const TString Name_;
};
