#pragma once

#include <rtline/library/storage/structured.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/ptr.h>

namespace NDrive {
    using TEventId = ui64;

    constexpr auto IncorrectEventId = std::numeric_limits<TEventId>::max();
}

class IHistoryContext {
public:
    virtual ~IHistoryContext() {
    }

    virtual TAtomicSharedPtr<NStorage::IDatabase> GetDatabase() const = 0;
};

class THistoryContext: public IHistoryContext {
private:
    TAtomicSharedPtr<NStorage::IDatabase> Database;

public:
    THistoryContext(TAtomicSharedPtr<NStorage::IDatabase> db)
        : Database(db)
    {
        CHECK_WITH_LOG(!!db);
    }

    virtual TAtomicSharedPtr<NStorage::IDatabase> GetDatabase() const override {
        return Database;
    }
};

template <class T>
class TIdTypeSelector {
public:
    using TId = T;
    using TIdRef = T;
};

template <>
class TIdTypeSelector<TString> {
public:
    using TId = TString;
    using TIdRef = TStringBuf;
};

template <class TEntity, class TIdType = TString>
class TDBEntitiesCacheImpl {
public:
    using TObjects = TMap<TIdType, TEntity>;

private:
    using TIdRef = typename TIdTypeSelector<TIdType>::TIdRef;

private:
    TRWMutex MutexCachedObjects;

protected:
    mutable TObjects Objects;

protected:
    auto MakeObjectReadGuard() const {
        auto guard = TReadGuard(MutexCachedObjects);
        auto lastUpdate = GetLastUpdateTimestamp();
        if (lastUpdate) {
            auto freshness = Now() - lastUpdate;
            ReadFreshnessSignal.Signal(freshness.MilliSeconds());
        }
        return guard;
    }
    auto MakeObjectWriteGuard() const {
        auto guard = TWriteGuard(MutexCachedObjects);
        return guard;
    }

private:
    template <bool Detach, class TAction, IterableContainer TObjectIds, class T>
    bool ForObjectsListImpl(T& objects, const TObjectIds* selectedObjectIds, const TAction& action) const {
        if constexpr (Detach) {
            const auto actionImpl = [&action](const TIdRef& /*id*/, TEntity&& entity) {
                action(std::move(entity));
            };
            return ForObjectsMapImpl<Detach>(objects, selectedObjectIds, actionImpl);
        } else {
            const auto actionImpl = [&action](const TIdRef& /*id*/, const TEntity& entity) {
                action(entity);
            };
            return ForObjectsMapImpl<Detach>(objects, selectedObjectIds, actionImpl);
        }
    }

    template <bool Detach, class TAction, IterableContainer TObjectIds, class T>
    bool ForObjectsMapImpl(T& objects, const TObjectIds* selectedObjectIds, const TAction& action) const {
        if (!!selectedObjectIds && selectedObjectIds->empty()) {
            return true;
        }
        if (!selectedObjectIds) {
            for (auto&& obj : objects) {
                action(obj.first, Detach ? std::move(obj.second) : obj.second);
            }
        } else {
            if (selectedObjectIds->size() < objects.size() / 20) {
                for (auto&& i : *selectedObjectIds) {
                    auto it = objects.find(i);
                    if (it != objects.end()) {
                        action(it->first, Detach ? std::move(it->second) : it->second);
                    }
                }
            } else {
                Y_ASSERT(std::is_sorted(selectedObjectIds->begin(), selectedObjectIds->end()));
                auto itId = selectedObjectIds->begin();
                auto itObj = objects.begin();
                while (itId != selectedObjectIds->end() && itObj != objects.end()) {
                    if (*itId < itObj->first) {
                        ++itId;
                    } else if (*itId > itObj->first) {
                        ++itObj;
                    } else {
                        action(itObj->first, Detach ? std::move(itObj->second) : itObj->second);
                        ++itId;
                        ++itObj;
                    }
                }
            }
        }
        return true;
    }

public:
    TDBEntitiesCacheImpl(TStringBuf name)
        : ReadFreshnessSignal({ TStringBuilder() << name << "-cache-read-freshness" }, NRTLineHistogramSignals::IntervalsTasks)
    {
    }
    virtual ~TDBEntitiesCacheImpl() = default;

    virtual TInstant GetLastUpdateTimestamp() const = 0;
    [[nodiscard]] virtual bool RefreshCache(TInstant reqActuality, bool doActualizeHistory = true) const = 0;

    TMaybe<TEntity> GetObject(const TIdType& id, const TInstant reqActuality = TInstant::Zero()) const {
        auto ids = NContainer::Scalar(id);
        TVector<TEntity> result;
        Y_ENSURE_BT(GetCustomObjectsFromCache(result, ids, reqActuality));
        return result.size() ? result.front() : TMaybe<TEntity>();
    }

    TVector<TEntity> GetCachedObjectsVector() const {
        TVector<TEntity> result;
        auto guard = MakeObjectReadGuard();
        for (auto&& i : Objects) {
            result.emplace_back(i.second);
        }
        return result;
    }

    template <class TContainer>
    [[nodiscard]] bool GetCustomObjectsMap(const TContainer& ids, TMap<TIdType, TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TIdType& key, const TEntity& entity) {
            result.emplace(key, entity);
        };
        return ForObjectsMap(action, reqActuality, &ids);
    }

    TMap<TIdType, TEntity> GetCachedObjectsMap() const {
        auto guard = MakeObjectReadGuard();
        return Objects;
    }

    template <class TContainer>
    TVector<TEntity> GetCachedObjectsVector(const TContainer& ids, TInstant actuality = TInstant::Zero()) const {
        Y_ENSURE_BT(RefreshCache(actuality));
        auto guard = MakeObjectReadGuard();
        TVector<TEntity> result;
        for (auto&& id : ids) {
            auto objectIt = Objects.find(id);
            if (objectIt == Objects.end()) {
                continue;
            }
            result.emplace_back(objectIt->second);
        }
        return result;
    }

    template <class TContainer>
    TMap<TIdType, TEntity> GetCachedObjectsMap(const TContainer& ids) const {
        auto guard = MakeObjectReadGuard();
        TMap<TIdType, TEntity> result;
        for (auto&& id : ids) {
            auto objectIt = Objects.find(id);
            if (objectIt == Objects.end()) {
                continue;
            }
            result.emplace(id, objectIt->second);
        }
        return result;
    }

    template <IterableContainer TObjectIds>
    [[nodiscard]] bool GetCustomObjectsFromCache(TVector<TEntity>& result, const TObjectIds& ids, TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [&result](const TIdType& /*key*/, const TEntity& entity) {
            result.emplace_back(entity);
        };
        return ForObjectsMap(action, reqActuality, &ids);
    }

    [[nodiscard]] bool GetAllObjectsFromCache(TVector<TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        if (!RefreshCache(reqActuality)) {
            return false;
        }
        auto guard = MakeObjectReadGuard();
        for (auto&& obj : Objects) {
            TEntity eFiltered(obj.second);
            result.emplace_back(std::move(eFiltered));
        }
        return true;
    }

    [[nodiscard]] bool GetAllObjectsFromCache(TMap<TIdType, TEntity>& result, TInstant reqActuality = TInstant::Zero()) const {
        if (!RefreshCache(reqActuality)) {
            return false;
        }
        auto guard = MakeObjectReadGuard();
        result = Objects;
        return true;
    }

    template <class TAction, IterableContainer TObjectIds = TSet<TIdType>>
    [[nodiscard]] bool ForObjectsList(const TAction& action, TInstant reqActuality = TInstant::Zero(), const TObjectIds* objectIds = nullptr) const {
        if (!RefreshCache(reqActuality)) {
            return false;
        }
        auto guard = MakeObjectReadGuard();
        return ForObjectsListImpl<false>(Objects, objectIds, action);
    }

    template <class TAction, IterableContainer TObjectIds = TSet<TIdType>>
    [[nodiscard]] bool ForObjectsMap(const TAction& action, TInstant reqActuality = TInstant::Zero(), const TObjectIds* objectIds = nullptr) const {
        if (!RefreshCache(reqActuality)) {
            return false;
        }
        auto guard = MakeObjectReadGuard();
        return ForObjectsMapImpl<false>(Objects, objectIds, action);
    }

private:
    TUnistatSignal<double> ReadFreshnessSignal;
};
