#pragma once

#include "cache.h"
#include "manager.h"
#include "propositions.h"

#include <rtline/util/types/accessor.h>

template <class TObjectContainer>
class TDBEntitiesHistoryManager: public TIndexedAbstractHistoryManager<TObjectContainer> {
public:
    using TIdRef = typename TIdTypeSelector<typename TObjectContainer::TId>::TIdRef;

private:
    using TBase = TIndexedAbstractHistoryManager<TObjectContainer>;

public:
    TDBEntitiesHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, TObjectContainer::GetHistoryTableName(), config)
    {
    }
};

class TDBEntitiesManagerConfig {
private:
    R_READONLY(THistoryConfig, HistoryConfig);
    R_READONLY(TString, DBName);

public:
    TDBEntitiesManagerConfig() = default;
    TDBEntitiesManagerConfig(const THistoryConfig& hConfig)
        : HistoryConfig(hConfig)
    {
    }

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

template <class TObjectContainer, class THistoryManager = TDBEntitiesHistoryManager<TObjectContainer>>
class TDBEntitiesCache: public TDBCacheWithHistoryOwner<THistoryManager, TObjectContainer, typename TObjectContainer::TId> {
private:
    using TBase = TDBCacheWithHistoryOwner<THistoryManager, TObjectContainer, typename TObjectContainer::TId>;
    using TIdRef = typename TBase::TIdRef;
    using TId = typename TObjectContainer::TId;

protected:
    using TBase::HistoryCacheDatabase;
    using TBase::HistoryManager;
    using TBase::MakeObjectReadGuard;
    using TBase::Objects;

protected:
    virtual TObjectContainer PrepareForUsage(const TObjectContainer& evHistory) const {
        return evHistory;
    }

    virtual bool DoRebuildCacheUnsafe() const override {
        NStorage::TObjectRecordsSet<TObjectContainer> objects;

        auto table = HistoryCacheDatabase->GetTable(TObjectContainer::GetTableName());
        {
            auto transaction = HistoryCacheDatabase->CreateTransaction(true);
            auto result = table->GetRows("", objects, transaction);
            if (!result->IsSucceed()) {
                ERROR_LOG << "Cannot refresh data for objects manager for " << TObjectContainer::GetTableName() << ", " << transaction->GetErrors().GetStringReport() << Endl;
                return false;
            }
        }
        for (auto&& i : objects) {
            Objects.emplace(i.GetInternalId(), PrepareForUsage(i));
        }

        return true;
    }

    virtual TIdRef GetEventObjectId(const TObjectEvent<TObjectContainer>& ev) const override {
        return ev.GetInternalId();
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TObjectContainer>& ev) const override {
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            DoAcceptHistoryEventBeforeRemoveUnsafe(ev);
            Objects.erase(ev.GetInternalId());
        } else {
            auto& object = (Objects[ev.GetInternalId()] = PrepareForUsage(ev));
            DoAcceptHistoryEventAfterChangeUnsafe(ev, object);
        }
    }

public:
    TDBEntitiesCache(const IHistoryContext& context, const THistoryConfig& historyConfig)
        : TBase(TObjectContainer::GetTableName(), context, historyConfig)
    {
    }

    TMaybe<TObjectContainer> GetCustomObject(const TId& objectId, const TInstant lastInstant = TInstant::Zero()) const {
        if (!TBase::RefreshCache(lastInstant)) {
            return Nothing();
        }
        auto rg = MakeObjectReadGuard();
        auto it = Objects.find(objectId);
        if (it == Objects.end()) {
            return Nothing();
        }
        return it->second;
    }

private:
    virtual void DoAcceptHistoryEventBeforeRemoveUnsafe(const TObjectEvent<TObjectContainer>& /*ev*/) const {
        return;
    }

    virtual void DoAcceptHistoryEventAfterChangeUnsafe(const TObjectEvent<TObjectContainer>& /*ev*/, TObjectContainer& /*object*/) const {
        return;
    }
};

template <class TObjectContainer, class TConditionConstructor, class THistoryManager, class Container = const TSet<typename TObjectContainer::TId>&>
bool RemoveObjects(const THistoryManager& historyManager, Container objects, const TString& userId, NDrive::TEntitySession& session) {
    if (objects.empty()) {
        return true;
    }
    auto transaction = session.GetTransaction();
    NStorage::TObjectRecordsSet<TObjectContainer> records;
    auto table = historyManager.GetDatabase().GetTable(TObjectContainer::GetTableName());
    const TString trCondition = TConditionConstructor::BuildCondition(objects, session);
    auto result = table->RemoveRow(trCondition, transaction, &records);
    if (!result || !result->IsSucceed() || records.size() != objects.size()) {
        session.SetErrorInfo("RemoveObject", "cannot remove objects from " + TObjectContainer::GetTableName(), session.GetResult());
        return false;
    }
    if (!historyManager.AddHistory(records, userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }
    return true;
}

template <class TObjectContainer, class TConditionConstructor, class THistoryManager>
bool AddObject(const THistoryManager& historyManager, const TRecordsSet& recordsRequest, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) {
    auto table = historyManager.GetDatabase().GetTable(TObjectContainer::GetTableName());
    NStorage::TObjectRecordsSet<TObjectContainer> recordsInt;
    NStorage::TObjectRecordsSet<TObjectContainer>* records = containerExt ? containerExt : &recordsInt;
    if (!table->AddRows(recordsRequest, session.GetTransaction(), "", records)) {
        session.SetErrorInfo("insert_object", session.GetTransaction()->GetErrors().GetStringReport(), EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!historyManager.AddHistory(records->GetObjects(), userId, EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

template <class TObjectContainer, class TConditionConstructor, class THistoryManager>
bool AddObjects(const THistoryManager& historyManager, TConstArrayRef<TObjectContainer> objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) {
    TRecordsSet recordsRequest;
    for (auto&& object : objects) {
        recordsRequest.AddRow(object.SerializeToTableRecord());
    }
    return AddObject<TObjectContainer, TConditionConstructor, THistoryManager>(historyManager, recordsRequest, userId, session, containerExt);
}

template <class TObjectContainer, class TConditionConstructor, class THistoryManager>
bool UpsertObject(const THistoryManager& historyManager, const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) {
    if (!object) {
        session.SetErrorInfo("incorrect user data", "upsert object into " + TObjectContainer::GetTableName(), EDriveSessionResult::DataCorrupted);
        return false;
    }
    NStorage::TTableRecord trUpdate = object.SerializeToTableRecord();
    NStorage::TTableRecord trCondition = TConditionConstructor::BuildCondition(object);
    auto table = historyManager.GetDatabase().GetTable(TObjectContainer::GetTableName());
    NStorage::TObjectRecordsSet<TObjectContainer> recordsInt;
    NStorage::TObjectRecordsSet<TObjectContainer>* records = containerExt ? containerExt : &recordsInt;

    bool isUpdate = false;
    if (!trCondition.Empty()) {
        if (!table->Upsert(trUpdate, session.GetTransaction(), trCondition, &isUpdate, records)) {
            session.SetErrorInfo("upsert_object", "Upsert failure", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!historyManager.AddHistory(records->GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
            return false;
        }
    } else {
        if (!table->AddRow(trUpdate, session.GetTransaction(), "", records)) {
            session.SetErrorInfo("insert_object", "AddRow failure", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!historyManager.AddHistory(records->GetObjects(), userId, EObjectHistoryAction::Add, session)) {
            return false;
        }
    }
    return true;
}

template <class TObjectContainer, class TConditionConstructor, class THistoryManager>
[[nodiscard]] bool UpsertWithRevision(const THistoryManager& historyManager, const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) {
        if (!object) {
            session.SetErrorInfo("incorrect user data", "upsert object into " + TObjectContainer::GetTableName(), EDriveSessionResult::DataCorrupted);
            return false;
        }

        NStorage::TTableRecord trUpdate = object.SerializeToTableRecord();
        NStorage::TTableRecord trCondition = TConditionConstructor::BuildCondition(object);
        auto table = historyManager.GetDatabase().GetTable(TObjectContainer::GetTableName());
        NStorage::TObjectRecordsSet<TObjectContainer> recordsInt;
        NStorage::TObjectRecordsSet<TObjectContainer>* records = containerExt ? containerExt : &recordsInt;

        bool isUpdate = false;
        switch (table->UpsertWithRevision(trUpdate, session.GetTransaction(), trCondition, object.OptionalRevision(), "revision", records)) {
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::IncorrectRevision:
                session.SetErrorInfo("upsert_object", "incorect_revision", EDriveSessionResult::InconsistencyUser);
                return false;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Failed:
                session.SetErrorInfo("upsert_object", "UpsertWithRevision failure", EDriveSessionResult::TransactionProblem);
                return false;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Updated:
                isUpdate = true;
            case NStorage::ITableAccessor::EUpdateWithRevisionResult::Inserted:
                break;
        }

        if (!historyManager.AddHistory(records->GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
            return false;
        }

        return true;
    }


class TDefaultConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "name IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("name", id);
        return trCondition;
    }

    template <class T>
    static NStorage::TTableRecord BuildCondition(const T& object) {
        return BuildCondition(object.GetName());
    }
};

template <class TObjectContainer, class TConditionConstructor = TDefaultConditionConstructor, class THistoryManager = TDBEntitiesHistoryManager<TObjectContainer>>
class TDBEntitiesManager
    : public TDBEntitiesCache<TObjectContainer, THistoryManager>
    , public virtual IDBEntitiesManager<TObjectContainer>
{
private:
    using TBase = TDBEntitiesCache<TObjectContainer, THistoryManager>;

protected:
    using TBase::HistoryCacheDatabase;
    using TBase::HistoryManager;
    using TBase::MakeObjectReadGuard;
    using TBase::Objects;

public:
    virtual TString BuildCondition(const TSet<typename TObjectContainer::TId>& ids, NDrive::TEntitySession& session) const final {
        return TConditionConstructor::BuildCondition(ids, session);
    }

    virtual NStorage::TTableRecord BuildCondition(const typename TObjectContainer::TId& id) const final {
        return TConditionConstructor::BuildCondition(id);
    }

    virtual NStorage::TTableRecord BuildCondition(const TObjectContainer& object) const final {
        return TConditionConstructor::BuildCondition(object);
    }

    [[nodiscard]] virtual bool GetObjects(TMap<typename TObjectContainer::TId, TObjectContainer>& objects, const TInstant reqActuality = TInstant::Zero()) const override {
        if (!TBase::RefreshCache(reqActuality)) {
            return false;
        }
        auto rg = MakeObjectReadGuard();
        objects = Objects;
        return true;
    }

    TSet<TString> GetObjectNames() const {
        TSet<TString> result;
        auto rg = MakeObjectReadGuard();
        for (auto&& i : Objects) {
            result.emplace(i.first);
        }
        return result;
    }

    TString GetTableName() const {
        return TObjectContainer::GetTableName();
    }

    TDBEntitiesManager(const IHistoryContext& context, const TDBEntitiesManagerConfig& config)
        : TBase(context, config.GetHistoryConfig())
    {
    }

    [[nodiscard]] virtual bool RemoveObject(const TSet<typename TObjectContainer::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const override {
        return RemoveObjects<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, ids, userId, session);
    }

    [[nodiscard]] virtual bool UpsertObject(const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session) const override {
        return UpsertWithRevision<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, object, userId, session);
    }

    [[nodiscard]] virtual bool ForceUpsertObject(const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const override {
        return UpsertObject<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, object, userId, session, containerExt);
    }

    [[nodiscard]] virtual bool AddObjects(const TVector<TObjectContainer>& objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const override {
        return AddObjects<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, objects, userId, session, containerExt);
    }
};

template <class TObjectContainer>
class IDBEntitiesManager {
public:
    virtual bool RemoveObject(const TSet<typename TObjectContainer::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const = 0;
    virtual bool UpsertObject(const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session) const = 0;
    virtual bool ForceUpsertObject(const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const = 0;
    virtual bool GetObjects(TMap<typename TObjectContainer::TId, TObjectContainer>& objects, const TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual bool AddObjects(const TVector<TObjectContainer>& objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const = 0;
};

template <class TObjectContainer>
class IDBEntitiesWithPropositionsManager: public virtual IDBEntitiesManager<TObjectContainer> {
public:
    virtual const IPropositionsManager<TObjectContainer>* GetPropositions() const = 0;
};

class TDBEntitiesManagerWithPropositionsConfig: public TDBEntitiesManagerConfig {
private:
    using TBase = TDBEntitiesManagerConfig;

private:
    R_READONLY(TPropositionsManagerConfig, PropositionsConfig);

public:
    TDBEntitiesManagerWithPropositionsConfig() = default;
    TDBEntitiesManagerWithPropositionsConfig(const THistoryConfig& hConfig, const TPropositionsManagerConfig& propositionsConfig)
        : TBase(hConfig)
        , PropositionsConfig(propositionsConfig)
    {
    }

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

template <class TObjectContainer, class TConditionConstructor = TDefaultConditionConstructor>
class TDBEntitiesManagerWithPropositions
    : public TDBEntitiesManager<TObjectContainer, TConditionConstructor>
    , public IDBEntitiesWithPropositionsManager<TObjectContainer>
{
private:
    using TBase = TDBEntitiesManager<TObjectContainer, TConditionConstructor>;

    class TPropositions: public TPropositionsManager<TObjectContainer> {
    public:
        TPropositions(const IHistoryContext& context, const TPropositionsManagerConfig& propositionsConfig)
            : TPropositionsManager<TObjectContainer>(context, TObjectContainer::GetPropositionsTableName(), propositionsConfig)
        {
        }
    };

private:
    TPropositions Propositions;

protected:
    virtual bool DoStart() override {
        return TBase::DoStart();
    }
    virtual bool DoStop() override {
        return TBase::DoStop();
    }

public:
    TDBEntitiesManagerWithPropositions(const IHistoryContext& context, const TDBEntitiesManagerWithPropositionsConfig& config)
        : TBase(context, config)
        , Propositions(context, config.GetPropositionsConfig())
    {
    }

    virtual const IPropositionsManager<TObjectContainer>* GetPropositions() const override {
        return &Propositions;
    }
};

template <class TObjectContainer, class TConditionConstructor = TDefaultConditionConstructor, class THistoryManager = TDatabaseHistoryManager<TObjectContainer>>
class TDatabaseEntitiesManager {
protected:
    THolder<THistoryManager> HistoryManager;

public:
    TDatabaseEntitiesManager(const IHistoryContext& context)
        : HistoryManager(new THistoryManager(context, TObjectContainer::GetHistoryTableName()))
    {
    }
    virtual ~TDatabaseEntitiesManager() = default;
    const THistoryManager& GetHistoryManager() const {
        return *HistoryManager;
    }

    [[nodiscard]] virtual bool RemoveObjects(const TSet<typename TObjectContainer::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const {
        return RemoveObjects<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, ids, userId, session);
    }

    [[nodiscard]] virtual bool AddObjects(TConstArrayRef<TObjectContainer> objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const {
        return AddObjects<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, objects, userId, session, containerExt);
    }

    [[nodiscard]] virtual bool UpsertObject(const TObjectContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TObjectContainer>* containerExt = nullptr) const {
        return UpsertObject<TObjectContainer, TConditionConstructor, THistoryManager>(*HistoryManager, object, userId, session, containerExt);
    }

    virtual TMaybe<TVector<TObjectContainer>> GetObjects(const TSet<typename TObjectContainer::TId>& ids, NDrive::TEntitySession& session) const {
        if (ids.empty()) {
            return TVector<TObjectContainer>();
        }
        return GetObjects(session, NSQL::TQueryOptions().AddCustomCondition(TConditionConstructor::BuildCondition(ids, session)));
    }

    virtual TMaybe<TVector<TObjectContainer>> GetObjects(NDrive::TEntitySession& session, const NSQL::TQueryOptions& options = {}) const {
        NStorage::TObjectRecordsSet<TObjectContainer> results;
        auto tx = session.GetTransaction();
        TString fields = TObjectContainer::TDecoder::GetFieldsForRequest();
        const TString query = options.PrintQuery(*tx, TObjectContainer::GetTableName(), NContainer::Scalar(fields));
        auto queryResult = tx->Exec(query, &results);
        if (!ParseQueryResult(queryResult, session)) {
            return {};
        }
        return results.DetachObjects();
    }

    template <class TKeys, class = typename TKeys::iterator>
    TMaybe<TVector<TObjectContainer>> GetObjectsByField(const TKeys& ids, const TString& fieldName, NDrive::TEntitySession& session) const {
        if (ids.empty()) {
            return TVector<TObjectContainer>();
        }
        return GetObjects(session, NSQL::TQueryOptions().SetGenericCondition(fieldName, ids));
    }

    TMaybe<TVector<TObjectContainer>> GetObjectsByField(const TString& id, const TString& fieldName, NDrive::TEntitySession& session) const {
        return GetObjectsByField(TSet<TString>{id}, fieldName, session);
    }
};
