#pragma once

#include "common.h"

#include <ydb/public/sdk/cpp/client/ydb_value/value.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/logger/global/global.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/timestamp.h>

#if defined(_asan_enabled_)
#   include <sanitizer/lsan_interface.h>
#endif

enum class EObjectHistoryAction {
    SetTagPerformer = 0 /* "set_performer" */,
    DropTagPerformer = 1 /* "drop_performer" */,
    AddSnapshot = 2 /* "add_snapshot" */,
    TagEvolve = 3 /* "evolve" */,
    ForceTagPerformer = 4 /* "force_performer" */,
    Add = 5 /* "add" */,
    Remove = 6 /* "remove" */,
    UpdateData = 7 /* "update_data" */,
    PutOff = 8 /* "put_off" */,

    Proposition = 9 /* "proposition" */,
    Confirmation = 10 /* "confirmation" */,
    Approved = 11 /* "approved" */,
    Rejected = 12 /* "rejected" */,

    Unknown = 100 /* "unknown" */,
};

template <class TObject>
class TObjectEventDecoder: public TObject::TDecoder {
private:
    using TBase = typename TObject::TDecoder;
    using TSelf = TObjectEventDecoder<TObject>;

private:
    R_FIELD(i32, HistoryAction, -1);
    R_FIELD(i32, HistoryUserId, -1);
    R_FIELD(i32, HistoryOriginatorId, -1);
    R_FIELD(i32, HistoryEventId, -1);
    R_FIELD(i32, HistoryComment, -1);
    R_FIELD(i32, HistoryTimestamp, -1);

public:
    TObjectEventDecoder() = default;

    static TString GetFieldsForRequest() {
        TString result = TObject::TDecoder::GetFieldsForRequest();
        if (result != "*") {
            result += ",history_action,history_user_id,history_originator_id,history_event_id,history_timestamp,history_comment";
        }
        return result;
    }

    TObjectEventDecoder(const TMap<TString, ui32>& decoderOriginal)
        : TObject::TDecoder(decoderOriginal)
    {
        HistoryAction = TBase::GetFieldDecodeIndexOrThrow("history_action", decoderOriginal);
        HistoryComment = TBase::GetFieldDecodeIndexOrThrow("history_comment", decoderOriginal);
        HistoryEventId = TBase::GetFieldDecodeIndexOrThrow("history_event_id", decoderOriginal);
        HistoryOriginatorId = TBase::GetFieldDecodeIndexOrThrow("history_originator_id", decoderOriginal);
        HistoryTimestamp = TBase::GetFieldDecodeIndexOrThrow("history_timestamp", decoderOriginal);
        HistoryUserId = TBase::GetFieldDecodeIndexOrThrow("history_user_id", decoderOriginal);
    }
};

template <class TObject>
class TObjectEvent: public TObject {
public:
    using TEventId = NDrive::TEventId;
    using TEntity = TObject;

public:
    static constexpr auto IncorrectEventId = NDrive::IncorrectEventId;

public:
    using TDecoder = TObjectEventDecoder<TObject>;
    using TPtr = TAtomicSharedPtr<TObjectEvent<TObject>>;

private:
    using TBase = TObject;
    using TSelf = TObjectEvent<TObject>;

public:
    R_FIELD(TEventId, HistoryEventId, IncorrectEventId);
    R_FIELD(TString, HistoryUserId);
    R_FIELD(TString, HistoryOriginatorId);
    R_FIELD(TString, HistoryComment);
    R_FIELD(TTimestamp, HistoryTimestamp, TInstant::Max());
    R_FIELD(EObjectHistoryAction, HistoryAction, EObjectHistoryAction::Unknown);

public:
    template <class TReportItemPolicy>
    NJson::TJsonValue BuildReportItemCustom() const {
        NJson::TJsonValue item;
        item.InsertValue("timestamp", HistoryTimestamp.Seconds());
        item.InsertValue("event_id", HistoryEventId);
        item.InsertValue("user_id", HistoryUserId);
        if (!!HistoryOriginatorId) {
            item.InsertValue("originator_id", HistoryOriginatorId);
        }
        item.InsertValue("action", ToString(HistoryAction));

        TReportItemPolicy::DoBuildReportItem(this, item);
        return item;
    }

    NJson::TJsonValue BuildReportItem() const {
        NJson::TJsonValue item;
        item.InsertValue("timestamp", HistoryTimestamp.Seconds());
        item.InsertValue("event_id", HistoryEventId);
        item.InsertValue("user_id", HistoryUserId);
        if (!!HistoryOriginatorId) {
            item.InsertValue("originator_id", HistoryOriginatorId);
        }
        item.InsertValue("action", ToString(HistoryAction));

        TObject::DoBuildReportItem(item);
        return item;
    }

    bool operator< (const TSelf& obj) const {
        if (HistoryTimestamp == obj.HistoryTimestamp) {
            return HistoryEventId < obj.HistoryEventId;
        } else {
            return HistoryTimestamp < obj.HistoryTimestamp;
        }
    }

    TObjectEvent() = default;
    TObjectEvent(const TObject& object, const EObjectHistoryAction action, TInstant instant, const TString& actorId, const TString& originatorId, const TString& comment)
        : TBase(object)
        , HistoryUserId(actorId)
        , HistoryOriginatorId(originatorId)
        , HistoryComment(comment)
        , HistoryTimestamp(instant)
        , HistoryAction(action)
    {
    }

    TInstant GetHistoryInstant() const {
        return HistoryTimestamp;
    }
    TSelf& SetHistoryInstant(TInstant value) {
        HistoryTimestamp = value;
        return *this;
    }

    NJson::TJsonValue SerializeToJson() const {
        return SerializeToTableRecord().SerializeToJson();
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result = TBase::SerializeToTableRecord();
        result
            .Set("history_action", ::ToString(HistoryAction))
            .Set("history_timestamp", HistoryTimestamp.Seconds())
            .Set("history_user_id", HistoryUserId);
        if (!!HistoryOriginatorId) {
            result.Set("history_originator_id", HistoryOriginatorId);
        }
        if (!!HistoryComment) {
            result.Set("history_comment", HistoryComment);
        }
        return result;
    }

    bool DeserializeWithDecoder(const TObjectEventDecoder<TObject>& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* context) {
        try {
            if (!TBase::DeserializeWithDecoder(decoder, values, context)) {
                return false;
            }
        } catch (const std::exception& e) {
            ERROR_LOG << FormatExc(e) << Endl;
            return false;
        }
#if defined(_asan_enabled_)
        __lsan::ScopedDisabler disableLeakCheck;
#endif
        READ_DECODER_VALUE(decoder, values, HistoryAction);
        READ_DECODER_VALUE(decoder, values, HistoryUserId);
        READ_DECODER_VALUE(decoder, values, HistoryOriginatorId);
        READ_DECODER_VALUE(decoder, values, HistoryEventId);
        READ_DECODER_VALUE(decoder, values, HistoryComment);
        READ_DECODER_VALUE_INSTANT(decoder, values, HistoryTimestamp);
        return true;
    }

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* context) {
        return TBaseDecoder::DeserializeFromTableRecord(*this, record, context);
    }

    static TMap<TString, NYdb::EPrimitiveType> GetYDBSchema() {
        using EPrimitiveType = NYdb::EPrimitiveType;
        TMap<TString, EPrimitiveType> schema = TObject::GetYDBSchema();
        schema.insert({"history_event_id",       EPrimitiveType::Int64});
        schema.insert({"history_user_id",        EPrimitiveType::String});
        schema.insert({"history_originator_id",  EPrimitiveType::String});
        schema.insert({"history_action",         EPrimitiveType::String});
        schema.insert({"history_timestamp",      EPrimitiveType::Int64});
        schema.insert({"history_comment",        EPrimitiveType::String});
        return schema;
    }
};

template <class TObject>
using TObjectEventId = typename TObjectEvent<TObject>::TEventId;

template <class TObject>
using TObjectEventPtr = typename TObjectEvent<TObject>::TPtr;

template <class TObject>
using TObjectEvents = TVector<TObjectEvent<TObject>>;

template <class TObject>
using TOptionalObjectEvent = TMaybe<TObjectEventPtr<TObject>>;

template <class TObject>
using TOptionalObjectEvents = TMaybe<TObjectEvents<TObject>>;

template <class TObject>
using TOptionalObjectEventId = TMaybe<TObjectEventId<TObject>>;
