#pragma once

#include "event.h"
#include "sequential.h"
#include "session_builder.h"

#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/container.h>

template <class TObject, class TBase = IBaseSequentialTableImpl>
class TDatabaseHistoryManager: public TBase {
public:
    enum class ESessionState {
        InProgress,
        Finished
    };

    using TEventId = typename TBase::TEventId;
    using TEvent = TObjectEvent<TObject>;
    using TEventPtr = TAtomicSharedPtr<TEvent>;
    using TEvents = TVector<TEvent>;
    using TEventPtrs = TVector<TEventPtr>;
    using TOptionalEvent = TMaybe<TEventPtr>;
    using TOptionalEvents = TMaybe<TEvents>;
    using TQueryOptions = typename TBase::TQueryOptions;

public:
    using TBase::TBase;
    using TBase::GetContext;
    using TBase::GetTableName;

    template <class TContainer>
    [[nodiscard]] bool AddHistory(const TContainer& objects, NDrive::TEntitySession& session, NStorage::IBaseRecordsSet* records = nullptr) const {
        if (objects.empty()) {
            return true;
        }
        TRecordsSet historyRecords;
        for (auto&& i : objects) {
            if (!i.GetHistoryUserId()) {
                session.SetErrorInfo("AddHistory", "Operator UserId empty", EDriveSessionResult::InconsistencySystem);
                return false;
            }
            {
                historyRecords.AddRow(i.SerializeToTableRecord());
            }
        }
        auto transaction = session.GetTransaction();
        if (historyRecords.size()) {
            auto historyTable = TBase::Database->GetTable(GetTableName());
            NStorage::IQueryResult::TPtr tResult = historyTable->AddRows(historyRecords, transaction, "", records);
            if (!tResult || !tResult->IsSucceed()) {
                session.SetErrorInfo("add_history", "history records AddRows failed", EDriveSessionResult::TransactionProblem);
                return false;
            }
        }
        TUnistatSignalsCache::SignalAdd("history_manager-" + GetTableName(), "add_records", objects.size());
        return true;
    }

    template <class TContainer>
    [[nodiscard]] bool AddHistory(const TContainer& objects, const TString& userId, const EObjectHistoryAction action, NDrive::TEntitySession& session, TRecordsSet* records = nullptr) const {
        TVector<TObjectEvent<TObject>> events;
        TInstant now = ModelingNow();
        for (auto&& i : objects) {
            TObjectEvent<TObject> objEvent(i, action, now, userId, session.GetOriginatorId(), session.GetComment());
            events.emplace_back(std::move(objEvent));
        }
        return AddHistory(events, session, records);
    }

    [[nodiscard]] bool AddHistory(const TObject& object, const TString& userId, const EObjectHistoryAction action, NDrive::TEntitySession& session, TRecordsSet* records = nullptr) const {
        return AddHistory(NContainer::Scalar(object), userId, action, session, records);
    }

    [[nodiscard]] bool AddHistory(const TObjectEvent<TObject>& objectEvent, const TString& userId, const EObjectHistoryAction action, NDrive::TEntitySession& session, TRecordsSet* records = nullptr) const {
        return AddHistory(NContainer::Scalar(objectEvent), userId, action, session, records);
    }

    template <class T>
    TOptionalObjectEvents<T> GetEvents(const TString& query, NDrive::TEntitySession& session) const {
        NStorage::TObjectRecordsSet<TObjectEvent<T>, IHistoryContext> events(&GetContext());
        auto transaction = session.GetTransaction();
        auto queryResult = transaction->Exec(query, &events);
        if (!ParseQueryResult(queryResult, session)) {
            return {};
        }
        return events.DetachObjects();
    }

    template <class T = TObject>
    TOptionalObjectEvents<T> GetEvents(TRange<TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& options = {}) const {
        auto fields = TObjectEvent<T>::TDecoder::GetFieldsForRequest();
        auto query = TBase::MakeEventsQuery(NContainer::Scalar(fields), std::move(idRange), std::move(timestampRange), session, options);
        auto result = GetEvents<T>(query, session);
        if (result && options.GetOrderBy().empty()) {
            Y_ASSERT(std::is_sorted(result->begin(), result->end(), [&options](const auto& left, const auto& right) {
                return (left.GetHistoryEventId() < right.GetHistoryEventId()) ^ options.GetDescending();
            }));
        }
        return result;
    }

    template <class T = TObject>
    TOptionalObjectEvents<T> GetEvents(TConstArrayRef<TEventId> ids, NDrive::TEntitySession& session) const {
        if (ids.empty()) {
            return TObjectEvents<T>();
        }
        auto fields = TObjectEvent<T>::TDecoder::GetFieldsForRequest();
        auto query = TBase::MakeEventsQuery(NContainer::Scalar(fields), ids, session);
        return GetEvents<T>(query, session);
    }

    template <class T = TObject>
    TOptionalObjectEvents<T> GetEvents(TEventId since, NDrive::TEntitySession& session, const TQueryOptions& options = {}) const {
        return GetEvents<T>(since, {}, session, options);
    }

    template <class T = TObject>
    TOptionalObjectEvent<T> GetEvent(TEventId id, NDrive::TEntitySession& session) const {
        auto optionalEvents = GetEvents<T>({id, id + 1}, {}, session);
        if (!optionalEvents) {
            return {};
        }
        if (optionalEvents->size() > 1) {
            session.SetErrorInfo("GetEvent", TStringBuilder() << "multiple events with id " << id << ": " << optionalEvents->size());
            return {};
        }
        if (optionalEvents->empty()) {
            return nullptr;
        }
        return MakeAtomicShared<TObjectEvent<T>>(std::move(optionalEvents->front()));
    }

protected:
    [[nodiscard]] bool GetEventsSinceIdDirect(TEventId id, TEventPtrs& results) const {
        auto session = TBase::BuildSession(false);
        auto optionalEvents = GetEvents(id, session);
        if (!optionalEvents) {
            ERROR_LOG << GetTableName() << ": cannot GetEventsSinceId " << id << ": " << session.GetStringReport() << Endl;
            return false;
        }
        for (auto&& ev : *optionalEvents) {
            results.push_back(MakeAtomicShared<TEvent>(std::move(ev)));
        }
        return true;
    }
};

template <class TObject, class TBase>
using TAbstractHistoryManagerImpl = TDatabaseHistoryManager<TObject, TBase>;

template <class T>
using TAbstractHistoryManager = TAbstractHistoryManagerImpl<T, TSequentialTableWithSessions<TObjectEvent<T>>>;

template <class T>
using TIndexedAbstractHistoryManager = TAbstractHistoryManagerImpl<T, TIndexedSequentialTableImpl<TObjectEvent<T>>>;
