#pragma once

#include "manager.h"

#include <drive/backend/database/history/event.h>
#include <drive/backend/common/messages.h>

#include <library/cpp/mediator/messenger.h>

#include <rtline/library/storage/sql/query.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/guid.h>
#include <util/generic/vector.h>
#include <util/generic/ylimits.h>
#include <util/system/types.h>

class IHistoryContext;

using TPropositionId = TString;

template <class T>
class TObjectProposition: public T {
private:
    using TBase = T;
    using TSelf = TObjectProposition<T>;

private:
    R_READONLY(TPropositionId, PropositionId);
    R_FIELD(ui32, ConfirmationsNeed, 1);
    R_FIELD(TString, Description);

public:
    NJson::TJsonValue BuildJsonReport() const {
        NJson::TJsonValue result = TBase::BuildJsonReport();
        result.InsertValue("proposition_id", PropositionId);
        result.InsertValue("confirmations_need", ConfirmationsNeed);
        result.InsertValue("proposition_description", Description);

        return result;
    }

    class TDecoder: public T::TDecoder {
        using TBase = typename T::TDecoder;
        R_FIELD(i32, PropositionId, -1);
        R_FIELD(i32, ConfirmationsNeed, -1);
        R_FIELD(i32, PropositionMeta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase)
            : TBase(decoderBase)
        {
            PropositionId = TBase::GetFieldDecodeIndex("proposition_id", decoderBase);
            ConfirmationsNeed = TBase::GetFieldDecodeIndex("confirmations_need", decoderBase);
            PropositionMeta = TBase::GetFieldDecodeIndex("proposition_meta", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
        if (!TBase::DeserializeWithDecoder(decoder, values, hContext)) {
            return false;
        }
        READ_DECODER_VALUE(decoder, values, PropositionId);
        READ_DECODER_VALUE(decoder, values, ConfirmationsNeed);

        NJson::TJsonValue jsonMeta;
        READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, PropositionMeta);
        const NJson::TJsonValue* jsonDescription;
        if (jsonMeta.GetValuePointer("description", &jsonDescription)) {
            Description = jsonDescription->GetStringRobust();
        }
        return true;
    }

    TObjectProposition() = default;
    TObjectProposition(const T& object, const ui32 confirmsCount)
        : TBase(object)
        , PropositionId(CreateGuidAsString())
        , ConfirmationsNeed(confirmsCount)
    {
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result = TBase::SerializeToTableRecord();
        result.Set("confirmations_need", ConfirmationsNeed);
        result.Set("proposition_id", PropositionId);

        NJson::TJsonValue jsonMeta = NJson::JSON_MAP;
        if (Description) {
            jsonMeta.InsertValue("description", Description);
        }
        result.Set("proposition_meta", jsonMeta.GetStringRobust());
        return result;
    }
};

class TConfirmationInfo {
    R_FIELD(TString, UserId);
    R_FIELD(TInstant, HistoryInstant);
    R_FIELD(TString, Comment);

public:
    TConfirmationInfo() = default;

    NJson::TJsonValue GetReport() const;
};

template <class T>
class TReportObjectProposition: public TObjectProposition<T> {
private:
    using TBase = TObjectProposition<T>;

private:
    R_FIELD(TString, PropositionAuthor);
    R_READONLY(TVector<TConfirmationInfo>, ConfirmationsInfo);
    R_FIELD(TInstant, StartedAt);

public:
    void FillUsers(TSet<TString>& userIds) const {
        for (auto&& c : ConfirmationsInfo) {
            userIds.emplace(c.GetUserId());
        }
        if (!!PropositionAuthor) {
            userIds.emplace(PropositionAuthor);
        }
    }

    void AddConfirmation(const TConfirmationInfo& info) {
        ConfirmationsInfo.emplace_back(info);
    }

    ui32 GetConfirmationsCount() const {
        return ConfirmationsInfo.size();
    }

    TReportObjectProposition() = default;
    TReportObjectProposition(const TBase& base)
        : TBase(base)
    {
    }

    NJson::TJsonValue BuildJsonReport() const {
        NJson::TJsonValue result = TBase::BuildJsonReport();
        result.InsertValue("confirmations_count", ConfirmationsInfo.size());
        result.InsertValue("proposition_author", PropositionAuthor);
        NJson::TJsonValue& confirmatorsJson = result.InsertValue("confirmators", NJson::JSON_ARRAY);
        NJson::TJsonValue& confirmationsJson = result.InsertValue("confirmations", NJson::JSON_ARRAY);
        for (auto&& i : ConfirmationsInfo) {
            confirmatorsJson.AppendValue(i.GetUserId());
            confirmationsJson.AppendValue(i.GetReport());
        }
        return result;
    }
};

enum class EPropositionAcceptance {
    ReadyForCommit,
    ConfirmWaiting,
    Rejected,
    Problems,
    UserConfirmAlready,
    PropositionRejectedAlready,
    PropositionApprovedAlready,
    SelfConfirmIsDenied
};

enum class EDoubleConfirmationPolicy {
    Accept,
    Ignore,
    Error,
};

enum class ESelfConfirmationPolicy {
    Accept,
    Ignore,
    Error,
};

class TPropositionsManagerConfig {
private:
    R_FIELD(ESelfConfirmationPolicy, DefaultSelfConfirmationPolicy, ESelfConfirmationPolicy::Error);
    R_FIELD(EDoubleConfirmationPolicy, DefaultDoubleConfirmationPolicy, EDoubleConfirmationPolicy::Error);
    R_FIELD(ui32, DefaultConfirmationsNeed, 1);
    R_FIELD(TDuration, DefaultPropositionLivetime, TDuration::Days(7));

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

template <class T>
class IPropositionsManager {
public:
    virtual ~IPropositionsManager() = default;
    virtual EPropositionAcceptance Propose(const TObjectProposition<T>& object, const TString& userId, NDrive::TEntitySession& session) const = 0;
    virtual EPropositionAcceptance Reject(const TString& propositionId, const TString& userId, NDrive::TEntitySession& session) const = 0;
    virtual EPropositionAcceptance Confirm(const TString& propositionId, const TString& userId, NDrive::TEntitySession& session) const = 0;

    virtual TMaybe<TMap<TString, TReportObjectProposition<T>>> Get(NDrive::TEntitySession& session) const {
        return Get(std::monostate(), session);
    }
    virtual TMaybe<TMap<TString, TReportObjectProposition<T>>> Get(const NSQL::TStringContainer& ids, NDrive::TEntitySession& session) const = 0;
};

template <class T>
class TPropositionsManager
    : private TDatabaseHistoryManager<TObjectProposition<T>>
    , public IPropositionsManager<T>
    , public IMessageProcessor
{
private:
    using TBase = TDatabaseHistoryManager<TObjectProposition<T>>;
    using TBaseInterface = IPropositionsManager<T>;
    using TSelf = TPropositionsManager<T>;

private:
    const TPropositionsManagerConfig Config;

    void CleanOld(const TString& userId, TMessagesCollector& report) const {
        TUnistatSignalsCache::SignalAdd("propositions_manager-" + TBase::GetTableName(), "clean_old", 1);
        TMap<TString, TReport> reports;
        {
            auto session = TBase::template BuildTx<NSQL::ReadOnly>();
            auto optionalReports = Get(session);
            if (!optionalReports) {
                report.AddMessage(TBase::GetTableName(), "Cannot get propositions: " + session.GetStringReport());
                return;
            }
            reports = *optionalReports;
        }

        ui32 errorsCount = 0;
        ui32 successCount = 0;
        for (auto&& i : reports) {
            if (GetDeadline(i.second, i.second.GetStartedAt()) < ModelingNow()) {
                auto session = TBase::template BuildTx<NSQL::Writable>();
                Reject(i.second.GetPropositionId(), userId, session);
                if (!session.Commit()) {
                    ++errorsCount;
                    ERROR_LOG << TBase::GetTableName() << "Reject proposition error" << ": " << i.first << session.GetStringReport();
                } else {
                    ++successCount;
                }
            }
        }
        if (errorsCount) {
            report.AddMessage(TBase::GetTableName(), "REJECT FAILED LINES COUNT = " + ::ToString(errorsCount));
        }
        if (successCount) {
            report.AddMessage(TBase::GetTableName(), "REJECT SUCCESS LINES COUNT = " + ::ToString(successCount));
        }
        if (!successCount && !errorsCount) {
            report.AddMessage(TBase::GetTableName(), "NO REJECTED LINES");
        }
    }

public:
    using TProposition = TObjectProposition<T>;
    using TReport = TReportObjectProposition<T>;

    TPropositionsManager(const IHistoryContext& context, const TString& tableName, const TPropositionsManagerConfig& config)
        : TBase(context, tableName)
        , Config(config)
    {
        RegisterGlobalMessageProcessor(this);
    }

    ~TPropositionsManager() {
        UnregisterGlobalMessageProcessor(this);
    }

    virtual bool Process(IMessage* message) override {
        TRegularDBServiceMessage* dbServiceMessage = dynamic_cast<TRegularDBServiceMessage*>(message);
        if (dbServiceMessage) {
            CleanOld(dbServiceMessage->GetUserId(), dbServiceMessage->MutableReport());
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return "mp_" + TBase::GetTableName();
    }

    using TBaseInterface::Get;

    const TPropositionsManagerConfig& GetConfig() const {
        return Config;
    }

    virtual TInstant GetDeadline(const T& /*obj*/, const TInstant start) const {
        return start + Config.GetDefaultPropositionLivetime();
    }

    virtual ESelfConfirmationPolicy GetSelfConfirmationPolicy(const T& /*obj*/) const {
        return Config.GetDefaultSelfConfirmationPolicy();
    }

    virtual EDoubleConfirmationPolicy GetDoubleConfirmationPolicy(const T& /*obj*/) const {
        return Config.GetDefaultDoubleConfirmationPolicy();
    }

    TMaybe<TMap<TString, TReport>> Get(const NSQL::TStringContainer& ids, NDrive::TEntitySession& session) const override {
        if (ids.Empty()) {
            return MakeMaybe<TMap<TString, TReport>>();
        }
        NSQL::TQueryOptions options;
        if (ids) {
            options.SetGenericCondition("proposition_id", ids.BuildCondition());
        }
        return Get(options, session);
    }

    EPropositionAcceptance Propose(const TProposition& object, const TString& userId, NDrive::TEntitySession& session) const override {
        if (object.GetConfirmationsNeed() == 0) {
            return EPropositionAcceptance::ReadyForCommit;
        }
        if (!TBase::AddHistory(object, userId, EObjectHistoryAction::Proposition, session)) {
            return EPropositionAcceptance::Problems;
        }
        return EPropositionAcceptance::ConfirmWaiting;
    }

    EPropositionAcceptance Reject(const TString& propositionId, const TString& userId, NDrive::TEntitySession& session) const override {
        auto optionalEvents = TBase::GetEvents(0, session, NSQL::TQueryOptions()
            .AddGenericCondition("proposition_id", propositionId)
        );
        if (!optionalEvents) {
            return EPropositionAcceptance::Problems;
        }
        auto& result = *optionalEvents;
        if (result.empty()) {
            session.SetErrorInfo("reject empty proposition", propositionId, EDriveSessionResult::IncorrectRequest);
            return EPropositionAcceptance::Problems;
        }
        for (auto&& i : result) {
            if (i.GetHistoryAction() == EObjectHistoryAction::Rejected) {
                return EPropositionAcceptance::Rejected;
            }
            if (i.GetHistoryAction() == EObjectHistoryAction::Approved) {
                session.SetErrorInfo("reject committed proposition", propositionId, EDriveSessionResult::IncorrectRequest);
                return EPropositionAcceptance::PropositionApprovedAlready;
            }
        }
        if (!TBase::AddHistory(result.front(), userId, EObjectHistoryAction::Rejected, session)) {
            return EPropositionAcceptance::Problems;
        }
        return EPropositionAcceptance::Rejected;
    }

    EPropositionAcceptance Confirm(const TString& propositionId, const TString& userId, NDrive::TEntitySession& session) const override {
        auto optionalEvents = TBase::GetEvents(0, session, NSQL::TQueryOptions()
            .AddGenericCondition("proposition_id", propositionId)
        );
        if (!optionalEvents) {
            return EPropositionAcceptance::Problems;
        }
        auto& result = *optionalEvents;
        if (result.empty()) {
            session.SetErrorInfo("confirm empty proposition", propositionId, EDriveSessionResult::IncorrectRequest);
            return EPropositionAcceptance::Problems;
        }

        const auto& firstEvent = result.front();
        if (firstEvent.GetHistoryUserId() == userId) {
            switch (GetSelfConfirmationPolicy(firstEvent)) {
            case ESelfConfirmationPolicy::Accept:
                break;
            case ESelfConfirmationPolicy::Ignore:
                return EPropositionAcceptance::ConfirmWaiting;
            case ESelfConfirmationPolicy::Error:
                session.SetErrorInfo("self confirm is denied", propositionId, EDriveSessionResult::IncorrectRequest);
                return EPropositionAcceptance::SelfConfirmIsDenied;
            }
        }

        if (!TBase::AddHistory(firstEvent, userId, EObjectHistoryAction::Confirmation, session)) {
            session.SetErrorInfo(session.GetStringReport(), propositionId, EDriveSessionResult::InternalError);
            return EPropositionAcceptance::Problems;
        }
        TVector<TString> userIds;
        TSet<TString> userIdsSet;
        for (auto&& i : result) {
            if (i.GetHistoryAction() == EObjectHistoryAction::Confirmation) {
                userIds.emplace_back(i.GetHistoryUserId());
                userIdsSet.emplace(i.GetHistoryUserId());
            }
            if (i.GetHistoryAction() == EObjectHistoryAction::Rejected) {
                session.SetErrorInfo("confirm rejected proposition", propositionId, EDriveSessionResult::IncorrectRequest);
                return EPropositionAcceptance::PropositionRejectedAlready;
            }
        }
        if (!userIdsSet.emplace(userId).second) {
            switch (GetDoubleConfirmationPolicy(firstEvent)) {
            case EDoubleConfirmationPolicy::Accept:
                userIds.emplace_back(userId);
                break;
            case EDoubleConfirmationPolicy::Ignore:
                break;
            case EDoubleConfirmationPolicy::Error:
                session.SetErrorInfo("confirm committed proposition", propositionId, EDriveSessionResult::IncorrectRequest);
                return EPropositionAcceptance::UserConfirmAlready;
            }
        } else {
            userIds.emplace_back(userId);
        }
        if (userIds.size() >= firstEvent.GetConfirmationsNeed()) {
            if (!TBase::AddHistory(firstEvent, userId, EObjectHistoryAction::Approved, session)) {
                return EPropositionAcceptance::Problems;
            }
            return EPropositionAcceptance::ReadyForCommit;
        } else {
            return EPropositionAcceptance::ConfirmWaiting;
        }
    }

protected:
    TMaybe<TMap<TString, TReport>> Get(const NSQL::TQueryOptions& baseOptions, NDrive::TEntitySession& session) const {
        TMap<TString, TReport> result;
        NSQL::TQueryOptions options = baseOptions;
        NSQL::TQueryOptions subqueryOptions = baseOptions;
        subqueryOptions.SetGenericCondition<TSet<TString>>("history_action", {ToString(EObjectHistoryAction::Rejected), ToString(EObjectHistoryAction::Approved)});
        auto subquery = subqueryOptions.PrintQuery(*session.GetTransaction(), TBase::GetTableName(), {"proposition_id"});
        options.AddCustomCondition("proposition_id NOT IN (" + subquery + ")");
        options.SetOrderBy({"history_event_id"});
        auto events = TBase::GetEvents(0, session, options);
        if (!events) {
            return Nothing();
        }
        TMap<TString, TVector<TObjectEvent<TProposition>>> propositionIdToEvents;
        for (auto event : *events) {
            propositionIdToEvents[event.GetPropositionId()].push_back(event);
        }
        for (auto&& i : propositionIdToEvents) {
            Y_ENSURE(!i.second.empty());
            auto newIt = result.emplace(i.first, i.second.back()).first;
            for (auto&& session : i.second) {
                if (session.GetHistoryAction() == EObjectHistoryAction::Confirmation) {
                    TConfirmationInfo confirmation;
                    confirmation.SetUserId(session.GetHistoryUserId()).SetComment(session.GetHistoryComment()).SetHistoryInstant(session.GetHistoryInstant());
                    newIt->second.AddConfirmation(confirmation);
                }
            }
            newIt->second.SetPropositionAuthor(i.second.front().GetHistoryUserId());
            newIt->second.SetStartedAt(i.second.front().GetHistoryInstant());
        }
        return MakeMaybe(result);
    }
};
