#include "sequential.h"

#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/logging/evlog.h>

#include <drive/library/cpp/threading/concurrent_hash.h>

#include <rtline/util/algorithm/container.h>

NUtil::TConcurrentHashMap<TString, const IBaseSequentialTableImpl*> SequentialTables;

IBaseSequentialTableImpl::IBaseSequentialTableImpl(const IHistoryContext& context, const TString& tableName)
    : TDatabaseSessionConstructor(context.GetDatabase())
    , Context(context)
    , TableName(tableName)
{
    bool inserted = SequentialTables.emplace(TableName, this).second;
    if (!inserted) {
        ERROR_LOG << "cannot register SequentialTable " << TableName << Endl;
    } else {
        INFO_LOG << "registered SequentialTable " << TableName << ": " << static_cast<const void*>(this) << Endl;
    }
}

IBaseSequentialTableImpl::~IBaseSequentialTableImpl() {
    auto current = SequentialTables.find(TableName);
    if (current && current->second == this) {
        SequentialTables.erase(std::move(current));
        INFO_LOG << "deregistered SequentialTable " << TableName << ": " << static_cast<const void*>(this) << Endl;
    } else {
        ERROR_LOG << "cannot deregister SequentialTable " << TableName << Endl;
    }
}

const IBaseSequentialTableImpl* IBaseSequentialTableImpl::Instance(const TString& tableName) {
    auto p = SequentialTables.find(tableName);
    if (!p) {
        return nullptr;
    }
    auto result = p->second;
    if (result) {
        Y_ENSURE(result->GetTableName() == tableName);
    }
    return result;
}

TMaybe<IBaseSequentialTableImpl::TOptionalEventId> IBaseSequentialTableImpl::GetHistoryEventId(const TString& query, NDrive::TEntitySession& session) const {
    auto transaction = session.GetTransaction();
    auto records = TRecordsSet();
    auto queryResult = transaction->Exec(query, &records);
    if (!ParseQueryResult(queryResult, session)) {
        return {};
    }
    return GetHistoryEventId(records, session);
}

TMaybe<IBaseSequentialTableImpl::TOptionalEventId> IBaseSequentialTableImpl::GetHistoryEventId(const TRecordsSet& records, NDrive::TEntitySession& session) const {
    if (records.size() != 1) {
        session.SetErrorInfo("GetHistoryEventId", TStringBuilder() << "incorrect records size: " << records.size(), EDriveSessionResult::TransactionProblem);
        return {};
    }
    const auto& historyEventIdString = records.GetRecords()[0].Get(GetSeqFieldName());
    if (!historyEventIdString) {
        return TOptionalEventId{};
    }
    IBaseSequentialTableImpl::TEventId result;
    if (!TryFromString(historyEventIdString, result)) {
        session.SetErrorInfo("GetHistoryEventId", TStringBuilder() << "cannot parse: " << historyEventIdString, EDriveSessionResult::TransactionProblem);
        return {};
    }
    return result;
}

IBaseSequentialTableImpl::TOptionalEventIds IBaseSequentialTableImpl::GetHistoryEventIds(TRange<TEventId> range, NDrive::TEntitySession& session, const TQueryOptions& options) const {
    auto transaction = session.GetTransaction();
    auto seqField = GetSeqFieldName();
    auto records = TRecordsSet();
    auto query = MakeEventsQuery(NContainer::Scalar(seqField), std::move(range), {}, session, options);
    auto queryResult = transaction->Exec(query, &records);
    if (!ParseQueryResult(queryResult, session)) {
        return {};
    }
    TEventIds result;
    for (auto&& record : records) {
        const auto& historyEventIdString = record.Get(GetSeqFieldName());
        TEventId id;
        if (!TryFromString(historyEventIdString, id)) {
            session.SetErrorInfo("GetHistoryEventIds", "cannot parse EventId from " + historyEventIdString, EDriveSessionResult::TransactionProblem);
            return {};
        }
        result.push_back(id);
    }
    if (options.GetDescending()) {
        Y_ASSERT(std::is_sorted(result.rbegin(), result.rend()));
    } else {
        Y_ASSERT(std::is_sorted(result.begin(), result.end()));
    }
    return result;
}

TString GetMaxHistoryEventQuery(const TString& tableName, const TString& seqFieldName, NDrive::TEntitySession& session, const NSQL::TQueryOptions& queryOptions = {}) {
    TString field = TStringBuilder() << "MAX(" << seqFieldName << ") as " << seqFieldName;
    return queryOptions.PrintQuery(
        *session,
        tableName,
        NContainer::Scalar(field)
    );
}

TMaybe<IBaseSequentialTableImpl::TOptionalEventId> IBaseSequentialTableImpl::GetMaxHistoryEventId(NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const {
    auto query = GetMaxHistoryEventQuery(GetTableName(), GetSeqFieldName(), session, queryOptions);
    return GetHistoryEventId(query, session);
}

IBaseSequentialTableImpl::TOptionalEventId IBaseSequentialTableImpl::GetMaxEventIdOrThrow(NDrive::TEntitySession& session) const {
    auto doubleOptionalEventId = GetMaxHistoryEventId(session);
    R_ENSURE(doubleOptionalEventId, {}, "cannot GetMaxHistoryEventId", session);
    auto optionalEventId = *doubleOptionalEventId;
    return optionalEventId;
}

NDrive::TEventId GetLockedMaxEventId(const TString& tableName, const TString& eventIdColumn, NDrive::TEntitySession&& tx) {
    TRecordsSet records;
    auto lockQuery = TStringBuilder() << "LOCK TABLE " << tableName << " IN SHARE MODE";
    {
        lockQuery << " NOWAIT";
    }
    auto selectQuery = GetMaxHistoryEventQuery(tableName, eventIdColumn, tx);
    auto queryResult = tx->MultiExec({
        {lockQuery},
        {selectQuery, records},
        {"COMMIT"}
    });
    R_ENSURE(ParseQueryResult(queryResult), {}, "cannot LockAndSelect from " + tableName, tx);

    NDrive::TEventId maxId = 0;
    R_ENSURE(
        !records.empty() && records.GetRecords().front().TryGet(eventIdColumn, maxId),
        HTTP_INTERNAL_SERVER_ERROR,
        "cannot parse EventId from " << records.GetRecords().front().SerializeToJson().GetStringRobust(),
        tx
    );
    return maxId;
}

IBaseSequentialTableImpl::TEventId IBaseSequentialTableImpl::GetLockedMaxEventId(TDuration timeout, TDuration pause) const {
    auto start = Now();
    auto finish = start + timeout;
    do {
        auto expectedEventId = TryGetLockedMaxEventId();
        if (expectedEventId) {
            return *expectedEventId;
        } else {
            WARNING_LOG << GetTableName() << ": cannot GetLockedMaxEventId: " << expectedEventId.GetError().GetReport().GetStringRobust() << Endl;
        }
        Sleep(pause);
    } while (Now() < finish);
    {
        auto current = LastLockedMaxHistoryEventId.load();
        auto threshold = start - MaxUncommittedTxDuration;
        auto tx = BuildTx<NSQL::Writable>();
        auto queryOptions = TQueryOptions()
            .SetGenericCondition(GetSeqFieldName(), MakeRange<TEventId>(current, Nothing()))
            .SetGenericCondition(GetTimestampFieldName(), MakeRange<ui64>(Nothing(), threshold.Seconds()));
        ;
        auto optionalMaxEventId = GetMaxHistoryEventId(tx, queryOptions);
        R_ENSURE(optionalMaxEventId, {}, "cannot GetMaxHistoryEventId", tx);
        auto maxEventId = *optionalMaxEventId;
        if (maxEventId) {
            UpdateLastLockedMaxHistoryEventId(*maxEventId);
        }
    }
    return LastLockedMaxHistoryEventId.load();
}

IBaseSequentialTableImpl::TExpectedEventId IBaseSequentialTableImpl::TryGetLockedMaxEventId() const {
    auto impl = [this] {
        auto tx = BuildTx<NSQL::Writable>();
        auto maxEventId = GetMaxEventIdOrThrow(tx);
        if (!maxEventId) {
            return static_cast<TEventId>(0);
        }
        auto previous = LastLockedMaxHistoryEventId.load();
        if (previous >= *maxEventId) {
            return previous;
        }
        auto eventId = ::GetLockedMaxEventId(GetTableName(), GetSeqFieldName(), std::move(tx));
        return UpdateLastLockedMaxHistoryEventId(eventId);
    };
    return WrapUnexpected<TCodedException>(impl);
}

IBaseSequentialTableImpl::TEventId IBaseSequentialTableImpl::UpdateLastLockedMaxHistoryEventId(TEventId id) const {
    TEventId previous = 0;
    TEventId current = 0;
    do {
        previous = LastLockedMaxHistoryEventId.load();
        current = std::max(previous, id);
    } while (!LastLockedMaxHistoryEventId.compare_exchange_weak(previous, current));
    return current;
}

TString IBaseSequentialTableImpl::MakeEventsQuery(TConstArrayRef<TString> fields, TRange<TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& options) const {
    auto transaction = session.GetTransaction();
    auto query = TStringBuilder() << "SELECT ";
    if (!fields.empty()) {
        query << JoinSeq(", ", fields);
    } else {
        query << '*';
    }
    query << " FROM " << GetTableName();
    if (options.GetSecondaryIndex() && transaction->UseSecondaryIndex()) {
        query << " VIEW " << options.GetSecondaryIndex();
    }
    query << " WHERE True";
    if (idRange) {
        query << " AND " << options.PrintCondition(GetSeqFieldName(), idRange, *transaction);
    }
    if (timestampRange.From) {
        query << " AND " << GetTimestampFieldName() << " >= " << transaction->Quote(timestampRange.From->Seconds());
    }
    if (timestampRange.To && timestampRange.To != TInstant::Max()) {
        query << " AND " << GetTimestampFieldName() << " < " << transaction->Quote(timestampRange.To->Seconds());
    }
    for (auto&& [field, condition] : options.GetGenericConditions()) {
        query << " AND (" << options.PrintCondition(field, condition, *transaction) << ")";
    }
    if (options.GetCustomCondition()) {
        query << " AND (" << options.GetCustomCondition() << ")";
    }
    if (const auto& orderBy = options.GetOrderBy(); !orderBy.empty()) {
        query << " ORDER BY";
        for (size_t i = 0; i < orderBy.size(); ++i) {
            if (i) {
                query << ',';
            }
            query << ' ' << orderBy[i];
            if (options.GetDescending()) {
                query << " DESC";
            }
        }
    } else {
        query << " ORDER BY " << GetSeqFieldName();
        if (options.IsDescending()) {
            query << " DESC";
        }
    }
    if (auto limit = options.GetLimit()) {
        query << " LIMIT " << limit;
    } else {
        query << " LIMIT ALL";
    }
    if (auto offset = options.GetOffset()) {
        query << " OFFSET " << offset;
    }
    return query;
}

TString IBaseSequentialTableImpl::MakeEventsQuery(TConstArrayRef<TString> fields, TConstArrayRef<TEventId> ids, NDrive::TEntitySession& session) const {
    auto transaction = session.GetTransaction();
    auto query = TStringBuilder() << "SELECT ";
    if (!fields.empty()) {
        query << JoinSeq(", ", fields);
    } else {
        query << '*';
    }
    query << " FROM " << GetTableName();
    query << " WHERE " << GetSeqFieldName() << " IN ("
          << transaction->Quote(ids)
          << ")";
    return query;
}

i64 IBaseSequentialTableImpl::GetHistoryEventIdByTimestamp(const TInstant instant, const TString& info) const {
    TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> gTime("history event by instant for " + info + " in " + GetTableName());
    if (!GetTimestampFieldName()) {
        return 0;
    }
    auto transaction = Database->CreateTransaction(true);

    TStringBuilder request;
    request << "SELECT " << GetSeqFieldName() << ", " << GetTimestampFieldName() << " FROM \n"
        << "(\n"
            << "SELECT * FROM (SELECT " << GetSeqFieldName() << ", " << GetTimestampFieldName() << " FROM " <<  GetTableName()
            << " WHERE " <<  GetTimestampFieldName() << " >= " << instant.Seconds() << " ORDER BY " << GetTimestampFieldName()
            << " LIMIT 1) as high"

            << "\n UNION ALL \n"

            << "SELECT * FROM (SELECT " << GetSeqFieldName() << ", " << GetTimestampFieldName() << " FROM " <<  GetTableName()
            << " WHERE " <<  GetTimestampFieldName() << " < " << instant.Seconds() << " ORDER BY " << GetTimestampFieldName() << " DESC"
            << " LIMIT 1) as low\n"
        << ") as limits\n"
        << " ORDER BY abs(" << GetTimestampFieldName() << " - " << instant.Seconds() << ") ASC"
        << " LIMIT 1";

    TRecordsSet records;
    auto reqResult = transaction->Exec(request, &records);
    CHECK_WITH_LOG(reqResult->IsSucceed()) << transaction->GetErrors().GetStringReport() << Endl;
    if (records.size() <  1) {
        return -1;
    }

    i64 eventId;
    CHECK_WITH_LOG(records.GetRecords().front().TryGetDefault<i64>(GetSeqFieldName(), eventId, -1));
    CHECK_WITH_LOG(eventId != -1) << Endl;
    return eventId;
}

void IBaseSequentialTableImpl::LogEvent(NJson::TJsonValue&& ev) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        ev["table"] = TableName;
        evlog->AddEvent(std::move(ev));
    }
}
