#pragma once

#include "common.h"
#include "config.h"

#include <drive/backend/database/transaction/tx.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/settings.h>
#include <drive/backend/common/messages.h>
#include <drive/backend/logging/events.h>

#include <kernel/daemon/messages.h>
#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/mediator/messenger.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/threading/future/future.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/storage/sql/query.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/types/expected.h>

#include <util/digest/fnv.h>
#include <util/generic/adaptor.h>
#include <util/generic/set.h>
#include <util/string/join.h>
#include <util/system/mutex.h>
#include <util/system/rusage.h>
#include <util/system/rwlock.h>
#include <util/thread/pool.h>

#include <atomic>

class TSignalFreshness: public TUnistatSignal<double> {
public:
    TSignalFreshness(const TString& name)
        : TUnistatSignal<double>({
            "history-freshness-" + name,
        }, EAggregationType::LastValue, "axxx")
    {
    }
};

class TSignalHistoryUpdateSuccess: public TUnistatSignal<double> {
public:
    TSignalHistoryUpdateSuccess(const TString& name)
        : TUnistatSignal<double>({
            "history-update_success-" + name,
        }, EAggregationType::Sum, "dmmm")
    {
    }
};

class TSignalHistoryUpdateFail: public TUnistatSignal<double> {
public:
    TSignalHistoryUpdateFail(const TString& name)
        : TUnistatSignal<double>({
            "history-update_fail-" + name,
        }, EAggregationType::Sum, "dmmm")
    {
    }
};

class TSignalHistoryRowsParsed: public TUnistatSignal<double> {
public:
    TSignalHistoryRowsParsed(const TString& name)
        : TUnistatSignal<double>({
            "history-rows_parsed-" + name,
        }, EAggregationType::LastValue, "axxx")
    {
    }
};

class TConfigurableCheckers {
    R_FIELD(bool, HistoryParsingAlertsActive, false);

public:
    bool AlertHistoryParsingOnFail(const bool isCacheBuilding) const {
        return isCacheBuilding || !HistoryParsingAlertsActive;
    }
};

class TCommonCacheInvalidateNotification: public NMessenger::IMessage {
    R_FIELD(TString, TableName);
    R_FIELD(TInstant, ReqActuality, TInstant::Zero());
    R_FIELD(TString, OriginatorTableName);

public:
    TCommonCacheInvalidateNotification() = default;
    TCommonCacheInvalidateNotification(const TString& tableName, const TInstant reqActuality, const TString& originatorTableName)
        : TableName(tableName)
        , ReqActuality(reqActuality)
        , OriginatorTableName(originatorTableName)
    {
    }
};

class TAfterCacheInvalidateNotification: public TCommonCacheInvalidateNotification {
public:
    using TCommonCacheInvalidateNotification::TCommonCacheInvalidateNotification;
};

class TBeforeCacheInvalidateNotification: public TCommonCacheInvalidateNotification {
public:
    using TCommonCacheInvalidateNotification::TCommonCacheInvalidateNotification;
};

class TNotifyHistoryChanged: public NMessenger::IMessage {
    R_FIELD(TString, TableName);
    R_FIELD(ui32, RecordsCount, 0);
    R_FIELD(TInstant, ReqActuality, TInstant::Zero());

public:
    TNotifyHistoryChanged() = default;
    TNotifyHistoryChanged(const TString& tableName, const ui32 recordsCount, const TInstant reqActuality)
        : TableName(tableName)
        , RecordsCount(recordsCount)
        , ReqActuality(reqActuality)
    {
    }
};

class IBaseSequentialTableImpl: public TDatabaseSessionConstructor {
public:
    using TEventId = NDrive::TEventId;
    using TEventIds = TVector<TEventId>;
    using TExpectedEventId = TExpected<TEventId, TCodedException>;
    using TOptionalEventId = TMaybe<TEventId>;
    using TOptionalEventIds = TMaybe<TEventIds>;

    using TQueryOptions = NSQL::TQueryOptions;

public:
    static const IBaseSequentialTableImpl* Instance(const TString& tableName);

protected:
    virtual TString GetTimestampFieldName() const {
        return "history_timestamp";
    }

    virtual TString GetSeqFieldName() const {
        return "history_event_id";
    }

    ui64 GetEventsCacheCheckDepth() const {
        return EventsCacheCheckDepth;
    }

    TString MakeEventsQuery(TConstArrayRef<TString> fields, TRange<TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& options = {}) const;
    TString MakeEventsQuery(TConstArrayRef<TString> fields, TConstArrayRef<TEventId> ids, NDrive::TEntitySession& session) const;

    i64 GetHistoryEventIdByTimestamp(const TInstant instant, const TString& info) const;

    void LogEvent(NJson::TJsonValue&& ev) const;

public:
    IBaseSequentialTableImpl(const IHistoryContext& context, const TString& tableName);
    virtual ~IBaseSequentialTableImpl();

    IBaseSequentialTableImpl(const IBaseSequentialTableImpl& other) = delete;
    IBaseSequentialTableImpl& operator=(const IBaseSequentialTableImpl& other) = delete;

    const IHistoryContext& GetContext() const {
        return Context;
    }

    const TString& GetTableName() const {
        return TableName;
    }

    TString GetEventIdFieldName() const {
        return GetSeqFieldName();
    }

    TOptionalEventIds GetHistoryEventIds(TRange<TEventId> range, NDrive::TEntitySession& session, const TQueryOptions& options = {}) const;
    TOptionalEventId GetMaxEventIdOrThrow(NDrive::TEntitySession& tx) const;

    virtual TEventId GetLockedMaxEventId(TDuration timeout = TDuration::MilliSeconds(500), TDuration pause = TDuration::MilliSeconds(50)) const;
    TExpectedEventId TryGetLockedMaxEventId() const;

private:
    TMaybe<TOptionalEventId> GetHistoryEventId(const TString& query, NDrive::TEntitySession& session) const;
    TMaybe<TOptionalEventId> GetHistoryEventId(const TRecordsSet& records, NDrive::TEntitySession& session) const;
    TMaybe<TOptionalEventId> GetMaxHistoryEventId(NDrive::TEntitySession& session, const TQueryOptions& queryOptions = {}) const;

    TEventId UpdateLastLockedMaxHistoryEventId(TEventId id) const;

private:
    const IHistoryContext& Context;
    const TDuration MaxUncommittedTxDuration = TDuration::Minutes(10);
    const TString TableName;

    mutable std::atomic<TEventId> LastLockedMaxHistoryEventId = 0;
    ui64 EventsCacheCheckDepth = 1000;
};

template <class TEvent>
class TBaseSequentialTableImpl
    : public IBaseSequentialTableImpl
    , public IMessageProcessor
    , public IAutoActualization
{
public:
    using THistoryEvent = TEvent;
    using TBaseEventId = typename IBaseSequentialTableImpl::TEventId;
    using TEventId = typename TEvent::TEventId;
    using TEventPtr = TAtomicSharedPtr<TEvent>;
    using TEvents = TVector<TEventPtr>;

    static_assert(sizeof(TBaseEventId) >= sizeof(TEventId));
    static_assert(sizeof(TEventId) >= sizeof(ui64));

protected:
    using TBase = IBaseSequentialTableImpl;
    using TSelf = TBaseSequentialTableImpl<TEvent>;

protected:
    mutable TMutex MutexRequest;
    mutable TRWMutex Mutex;
    mutable TInstant LastSuccessReadInstant = TInstant::Zero();
    mutable TInstant StartInstant = TInstant::Zero();

private:
    mutable TInstant CurrentInstant = TInstant::Zero();
    mutable TEventId CurrentMaxEventId = TEvent::IncorrectEventId;
    mutable TEventId LockedMaxEventId = TEvent::IncorrectEventId;
    mutable bool Invalid = false;

    TSignalHistoryRowsParsed SignalHistoryRowsParsed;
    TSignalFreshness SignalHistoryFreshness;
    TSignalHistoryUpdateSuccess SignalHistoryUpdateSuccess;
    TSignalHistoryUpdateFail SignalHistoryUpdateFail;
    TUnistatSignal<double> GetEventsInvoked;
    TUnistatSignal<double> GetEventsSinceIdInvoked;

protected:
    mutable std::deque<TEventPtr> Events;
    mutable std::atomic<ui64> InsertedEventsCount = 0;

protected:
    const THistoryConfig Config;
    mutable TSet<TEventId> ReadyEvents;
    mutable std::atomic<ui64> UpdatesCounter = 0;

protected:
    virtual void StartCacheConstruction() const {
    }

    virtual void FinishCacheConstruction() const {
        TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("FinishCacheConstruction for " + GetTableName() + "/" + ::ToString(Events.size()) + ")");
        INFO_LOG << "RESOURCE: FinishCacheConstruction start for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;

        InsertedEventsCount = Events.size();
        i64 eventIdPred = -1;
        for (auto&& i : Events) {
            CHECK_WITH_LOG(eventIdPred < (i64)i->GetHistoryEventId()) << eventIdPred << " / " << i->GetHistoryEventId() << Endl;
            eventIdPred = i->GetHistoryEventId();
        }

        if (Events.size()) {
            LockedMaxEventId = Events.back()->GetHistoryEventId();
            CurrentMaxEventId = Events.back()->GetHistoryEventId();
        }
        INFO_LOG << "RESOURCE: FinishCacheConstruction finish for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
    }

public:
    template <class T, class TV, class TFunc>
    static void InsertSorted(TV& v, T object, TFunc&& func) {
        v.emplace_back(object);
        auto itBack = v.rbegin();
        auto itBackPred = v.rbegin();
        ++itBackPred;
        const auto evId = func(object);
        for (; itBackPred != v.rend(); ++itBackPred, ++itBack) {
            if (func(*itBackPred) > evId) {
                std::swap(*itBackPred, *itBack);
            } else {
                Y_ENSURE_BT(func(*itBackPred) < func(object));
                break;
            }
        }
    }

    template <class T, class TV>
    static void InsertSorted(TV& v, T object) {
        InsertSorted(v, object, [](T ev) { return ev->GetHistoryEventId(); });
    }

    template <class T>
    static TVector<T> PopFrontEvent(std::deque<T>& v, TEventId historyEventId) {
        TVector<T> result;
        while (v.size() && v.front()->GetHistoryEventId() <= historyEventId) {
            result.emplace_back(v.front());
            v.pop_front();
        }
        return result;
    }

protected:
    virtual bool GetCallbackNeedsHistoryLoad() const {
        return false;
    }

    virtual void CleanCurrentBuffer(const bool /*locked*/) const {
        if (LockedMaxEventId != TEvent::IncorrectEventId) {
            auto depth = GetEventsCacheCheckDepth();
            auto limit = LockedMaxEventId > depth ? LockedMaxEventId - depth : 0;
            if (ReadyEvents.empty() || *ReadyEvents.rbegin() <= limit) {
                ReadyEvents.clear();
            } else {
                ReadyEvents.erase(ReadyEvents.begin(), ReadyEvents.upper_bound(limit));
            }
        }
    }

    virtual void DoAddEvent(const TEventPtr& /*event*/) const {
    }

    virtual bool ShouldCacheEvent(const TEvent& /*event*/) const {
        return Config.GetUseCache();
    }

    void AddEvent(const TEventPtr& currentEvent, TInstant& actualInstant, TInstant& modelingInstantNow, const bool locked) const {
        Y_ENSURE_BT(currentEvent);
        auto currentEventId = currentEvent->GetHistoryEventId();
        auto currentEventTimestamp = currentEvent->GetHistoryInstant();
        bool valid =
            LockedMaxEventId == TEvent::IncorrectEventId ||
            LockedMaxEventId < currentEventId ||
            false; // EventHoles were there
        if (!valid) {
            ALERT_LOG << Name() << ": invalid EventId " << currentEvent->GetHistoryEventId() << Endl;
            Invalid = true;
            throw TWithBackTrace<yexception>() << "invalid EventId " << currentEvent->GetHistoryEventId() << " for " << Name();
        }

        auto finishedTransactionTimestamp = ModelingNow() - Config.GetGuaranteeFinishedTransactionsDuration();
        if (ReadyEvents.contains(currentEvent->GetHistoryEventId())) {
            if (locked || currentEventTimestamp < finishedTransactionTimestamp) {
                LockedMaxEventId = std::max(LockedMaxEventId, currentEvent->GetHistoryEventId());
            }
            return;
        }
        ReadyEvents.emplace(currentEvent->GetHistoryEventId());

        if (ShouldCacheEvent(*currentEvent)) {
            InsertSorted(Events, currentEvent);
        }
        ++InsertedEventsCount;

        DoAddEvent(currentEvent);

        if (modelingInstantNow < currentEvent->GetHistoryInstant()) {
            modelingInstantNow = currentEvent->GetHistoryInstant();
        }
        if (actualInstant < currentEvent->GetHistoryInstant()) {
            actualInstant = currentEvent->GetHistoryInstant();
        }
        if (locked) {
            for (auto eventId = LockedMaxEventId + 1; eventId < currentEventId; ++eventId) {
                if (ReadyEvents.contains(eventId)) {
                    continue;
                }
            }
            LockedMaxEventId = currentEvent->GetHistoryEventId();
        } else if (LockedMaxEventId + 1 == currentEvent->GetHistoryEventId()) {
            LockedMaxEventId = currentEvent->GetHistoryEventId();
            DEBUG_LOG << Name() << ": advance non-locked to " << LockedMaxEventId << Endl;
        }
        if (CurrentMaxEventId == TEvent::IncorrectEventId || CurrentMaxEventId < currentEvent->GetHistoryEventId()) {
            CurrentMaxEventId = currentEvent->GetHistoryEventId();
        }
    }

protected:
    class TParsedObjects: public NStorage::TPackedRecordsSet {
        R_READONLY(TInstant, ActualInstant, TInstant::Zero());
        R_READONLY(TInstant, ModelingInstantNow, TInstant::Zero());
        R_FIELD(bool, Locked, true);
        R_READONLY(bool, ReadOnly, false);
        R_READONLY(bool, IgnoreParseErrors, false);

    private:
        using TDecoder = typename TEvent::TDecoder;

    private:
        const TSelf& Owner;
        TVector<TVector<TStringBuf>> RecordsPool;
        TVector<TEventPtr> ParsedRecordsPool;
        TVector<ui32> ParsedRecordsPoolCorrect;
        ui32 IterationIdx = 1;
        const ui32 MTParsingPoolSize = 200000;
        THolder<TThreadPool> Queue;
        i32 CurrentIdx = -1;
        i32 CurrentIdxGlobal = 0;
        TDecoder Decoder;
        const bool IsCacheBuilding = false;
        NRTLine::TAbstractLock::TPtr Lock;

        bool IsLockInitialized() const {
            return !!Lock && Lock->IsLocked();
        }

    public:
        const TDecoder& GetDecoder() const {
            return Decoder;
        }

        i32 GetCurrentIdxGlobal() const {
            return CurrentIdxGlobal;
        }

        TParsedObjects(const TSelf& owner, const TInstant actualInstant, const TInstant modelingInstantNow, bool readOnly, bool ignoreParseErrors)
            : ActualInstant(actualInstant)
            , ModelingInstantNow(modelingInstantNow)
            , ReadOnly(readOnly)
            , IgnoreParseErrors(ignoreParseErrors)
            , Owner(owner)
            , IsCacheBuilding(actualInstant == TInstant::Zero())
        {
            ParsedRecordsPoolCorrect.resize(MTParsingPoolSize * 1.5, 0);
            RecordsPool.resize(MTParsingPoolSize);
        }

        NDrive::TEntitySession BuildTx() const {
            if (ReadOnly) {
                return Owner.BuildTx<NSQL::ReadOnly>();
            } else {
                return Owner.BuildTx<NSQL::Writable>();
            }
        }

    public:
        class TParseActor: public IObjectInQueue {
        private:
            const TVector<TVector<TStringBuf>>& Records;
            const ui32 CurrentCount;
            TParsedObjects& ObjectsOwner;
            TVector<TEventPtr>& ParsedRecordsPool;
            TVector<ui32>& ParsedRecordsPoolCorrect;
            std::atomic<ui64>& ReadIndex;
            TReadGuard Guard;
            const ui32 IteratorIdx;
            const bool IsCacheBuilding = false;
            const bool IgnoreParseErrors = true;

        public:
            TParseActor(
                const TVector<TVector<TStringBuf>>& records,
                const ui32 currentCount,
                TParsedObjects& objectsOwner,
                TVector<TEventPtr>& parsedRecordsPool,
                TVector<ui32>& parsedRecordsPoolCorrect,
                std::atomic<ui64>& readIndex,
                TRWMutex& mutex,
                const ui32 iteratorIdx,
                const bool isCacheBuilding,
                const bool ignoreParseErrors
            )
                : Records(records)
                , CurrentCount(currentCount)
                , ObjectsOwner(objectsOwner)
                , ParsedRecordsPool(parsedRecordsPool)
                , ParsedRecordsPoolCorrect(parsedRecordsPoolCorrect)
                , ReadIndex(readIndex)
                , Guard(mutex)
                , IteratorIdx(iteratorIdx)
                , IsCacheBuilding(isCacheBuilding)
                , IgnoreParseErrors(ignoreParseErrors)
            {
                Y_ENSURE_BT(CurrentCount <= Records.size(), CurrentCount << '/' << Records.size());
            }

            virtual void Process(void* /*threadSpecificResource*/) override {
                while (true) {
                    ui64 currentIdx = ++ReadIndex - 1;
                    if (currentIdx >= CurrentCount) {
                        break;
                    }
                    TConstArrayRef<TStringBuf> arrLocal(Records[currentIdx]);
                    auto event = MakeAtomicShared<TEvent>();
                    if (!event->DeserializeWithDecoder(ObjectsOwner.Decoder, arrLocal, &ObjectsOwner.Owner.GetContext())) {
                        if (ObjectsOwner.Decoder.NeedVerboseParsingErrorLogging()) {
                            ERROR_LOG << "cannot parse from record: " + JoinSeq(", ", Records[currentIdx]) << ", table: " << ObjectsOwner.Owner.GetTableName() << Endl;
                        } else {
                            ERROR_LOG << "cannot parse from record: (...), table: " << ObjectsOwner.Owner.GetTableName() << Endl;
                        }
                        Y_ASSERT(Singleton<TConfigurableCheckers>()->AlertHistoryParsingOnFail(IgnoreParseErrors));
                        continue;
                    }
                    ParsedRecordsPool[currentIdx] = std::move(event);
                    ParsedRecordsPoolCorrect[currentIdx] = IteratorIdx;
                }
            }
        };

        void ParseMT(const ui32 currentCount) {
            TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("parsing " + ToString(currentCount) + " records (to " + ToString(currentCount + (IterationIdx - 1) * MTParsingPoolSize) + " for " + Owner.GetTableName() + ")");
            if (IterationIdx == 1) {
                Queue.Reset(new TThreadPool);
                Queue->Start(8);
            }
            ++IterationIdx;
            ParsedRecordsPool.clear();
            ParsedRecordsPool.resize(currentCount);
            std::atomic<ui64> counter = 0;
            {
                TRWMutex mutex;
                for (ui32 i = 0; i < 8; ++i) {
                    Queue->SafeAddAndOwn(THolder(new TParseActor(RecordsPool, currentCount, *this, ParsedRecordsPool, ParsedRecordsPoolCorrect, counter, mutex, IterationIdx, IsCacheBuilding, IgnoreParseErrors)));
                }

                TWriteGuard wg(mutex);
            }
            if (currentCount) {
                TWriteGuard wgOwner(Owner.Mutex);
                for (ui32 i = 0; i < currentCount; ++i) {
                    if (ParsedRecordsPoolCorrect[i] == IterationIdx) {
                        if (!IsCacheBuilding) {
                            Owner.AddEvent(ParsedRecordsPool[i], ActualInstant, ModelingInstantNow, Locked);
                        } else {
                            Owner.Events.emplace_back(std::move(ParsedRecordsPool[i]));
                        }
                    }
                }
            }
            if (!IsCacheBuilding) {
                Owner.SignalHistoryRowsParsed.Signal(currentCount);
            }
            CurrentIdx = -1;
        }

        void ParseST(const ui32 currentCount) {
            if (currentCount) {
                TWriteGuard wgOwner(Owner.Mutex);
                for (ui32 i = 0; i < currentCount; ++i) {
                    auto event = MakeAtomicShared<TEvent>();
                    if (!event->DeserializeWithDecoder(Decoder, RecordsPool[i], &Owner.GetContext())) {
                        if (Decoder.NeedVerboseParsingErrorLogging()) {
                            ERROR_LOG << "cannot parse from record: " + JoinSeq(", ", RecordsPool[i]) << ", table: " << Owner.GetTableName() << Endl;
                        } else {
                            ERROR_LOG << "cannot parse from record: (...), table: " << Owner.GetTableName() << Endl;
                        }
                        Y_ASSERT(Singleton<TConfigurableCheckers>()->AlertHistoryParsingOnFail(IsCacheBuilding));
                        continue;
                    }
                    if (!IsCacheBuilding) {
                        Owner.AddEvent(event, ActualInstant, ModelingInstantNow, Locked);
                    } else {
                        Owner.Events.push_back(std::move(event));
                    }
                }
            }
            if (!IsCacheBuilding) {
                Owner.SignalHistoryRowsParsed.Signal(currentCount);
            }
            CurrentIdx = -1;
        }

    public:
        virtual void AddRow(const TVector<TStringBuf>& values) override {
            ++CurrentIdx;
            ++CurrentIdxGlobal;
            const ui32 currentCount = GetCurrentRecordsCount();
            RecordsPool[currentCount - 1] = values;
            if (currentCount == MTParsingPoolSize) {
                ParseMT(currentCount);
            }
        }

        virtual void Initialize(const ui32 /*recordsCount*/, const TMap<TString, ui32>& remapColumns) override {
            Lock = nullptr;
            Decoder = TDecoder(remapColumns);
        }

        ui32 GetCurrentRecordsCount() const {
            if (CurrentIdx == -1) {
                return 0;
            }
            return CurrentIdx % (MTParsingPoolSize) + 1;
        }

        void Finish() {
            TTimeGuard tg("parsing last " + ToString(GetCurrentRecordsCount()) + " records (" + Owner.GetTableName() + ")");
            const ui32 currentCount = GetCurrentRecordsCount();
            if (currentCount > MTParsingPoolSize * 0.5) {
                ParseMT(currentCount);
            } else {
                ParseST(currentCount);
            }
            Owner.CleanCurrentBuffer(Locked);
            if (IterationIdx != 1) {
                Queue->Stop();
            }
        }
    };

protected:
    virtual void OnFinishCacheModification(const TParsedObjects& /*records*/) const {
    }

    bool FinishCacheModification(TParsedObjects& records) const {
        DEBUG_LOG << "FinishCacheModification for " << GetTableName() << " START..." << records.GetCurrentRecordsCount() << Endl;
        records.Finish();
        DEBUG_LOG << "FinishCacheModification for " << GetTableName() << " PARSING FINISHED..." << records.GetCurrentRecordsCount() << Endl;
        OnFinishCacheModification(records);
        if (records.GetCurrentIdxGlobal()) {
            SendGlobalMessage<TNotifyHistoryChanged>(GetTableName(), records.GetCurrentIdxGlobal(), records.GetActualInstant());
        }
        CurrentInstant = records.GetActualInstant();
        DEBUG_LOG << "FinishCacheModification for " << GetTableName() << " FINISHED" << Endl;
        return true;
    }

    virtual void FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& /*result*/, TEventId /*historyEventIdMaxWithFinishInstant*/) const {
    }

    TString BuildAdditionalEventsConditions(TEventId historyEventIdMaxWithFinishInstant) const {
        TMap<TString, TSet<TString>> result;
        FillAdditionalEventsConditions(result, historyEventIdMaxWithFinishInstant);
        TStringStream ss;
        TStringBuilder sbLog;
        for (auto&& i : result) {
            if (!ss.Empty()) {
                ss << " OR ";
            }
            ss << i.first << " IN ('" << JoinSeq("','", i.second) << "')";
            sbLog << "ADDITIONAL_EVENTS_INFO: " << i.first << ":" << i.second.size() << ";";
        }
        if (sbLog) {
            INFO_LOG << sbLog << Endl;
        }
        return ss.Str();
    }

    virtual void BuildCache() const {
        StartCacheConstruction();
        TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("CacheConstructionFor:" + GetTableName());
        auto eventLogTimeGuard = NDrive::TEventLog::Guard(GetTableName() + '-' + "BuildCache");
        INFO_LOG << "RESOURCE: Start cache construction for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
        auto guard = Guard(MutexRequest);
        auto finishInstant = ModelingNow() - TDuration::Minutes(5);
        StartInstant = Min(ModelingNow() - Config.GetDeep(), finishInstant);
        const TInstant deepStartInstant = Min(ModelingNow() - Config.GetMaxHistoryDeep(), StartInstant);

        const i64 historyEventIdMinDeep = GetHistoryEventIdByTimestamp(deepStartInstant, "deep event");
        const i64 historyEventIdMin = GetHistoryEventIdByTimestamp(StartInstant, "deepest event");
        i64 historyEventIdMax = GetHistoryEventIdByTimestamp(finishInstant, "fresh event");
        i64 historyEventIdMaxWithFinishInstant = historyEventIdMax;
        Y_ASSERT(historyEventIdMinDeep <= historyEventIdMin);
        Y_ASSERT(historyEventIdMin <= historyEventIdMax);

        while (historyEventIdMax > historyEventIdMin && (Events.empty() || Events.back()->GetHistoryInstant() > StartInstant)) {
            const i64 historyEventIdNext = Max(historyEventIdMin, historyEventIdMax - (i64)Config.GetChunkSize());
            TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("iteration for history loading from " + ::ToString(historyEventIdNext) + " to " + ToString(historyEventIdMax) + " for " + GetTableName());
            TParsedObjects records(*this, TInstant::Zero(), TInstant::Zero(), true, true);
            records.SetLocked(true);
            auto tx = records.BuildTx();
            auto result = tx->Exec("SELECT " + records.GetDecoder().GetFieldsForRequest() + " FROM " + GetTableName()
                + " WHERE " + GetSeqFieldName() + " > " + ToString(historyEventIdNext)
                + " AND " + GetSeqFieldName() + " <= " + ToString(historyEventIdMax)
                + " ORDER BY " + GetSeqFieldName() + " DESC", &records);
            CHECK_WITH_LOG(result->IsSucceed()) << tx.GetStringReport() << Endl;
            CHECK_WITH_LOG(FinishCacheModification(records));
            historyEventIdMax = historyEventIdNext;
        }

        while (historyEventIdMax >= 0 && (Events.empty() || Events.back()->GetHistoryInstant() > deepStartInstant)) {
            const TString additionalTags = BuildAdditionalEventsConditions(historyEventIdMaxWithFinishInstant);
            if (!additionalTags) {
                break;
            }
            i64 historyEventIdNext = historyEventIdMax - (i64)Config.GetChunkSize();
            TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("iteration for history deep loading from " + ::ToString(historyEventIdNext) + " to " + ToString(historyEventIdMax) + " + " + additionalTags + " for " + GetTableName());
            TParsedObjects records(*this, TInstant::Zero(), TInstant::Zero(), true, true);
            records.SetLocked(true);
            auto tx = records.BuildTx();
            auto result = tx->Exec("SELECT " + records.GetDecoder().GetFieldsForRequest() + " FROM " + GetTableName()
                + " WHERE " + GetSeqFieldName() + " > " + ToString(historyEventIdNext)
                + " AND " + GetSeqFieldName() + " <= " + ToString(historyEventIdMax)
                + " AND (" + additionalTags + ")"
                + " ORDER BY " + GetSeqFieldName() + " DESC", &records);
            CHECK_WITH_LOG(result->IsSucceed()) << tx.GetStringReport() << Endl;
            CHECK_WITH_LOG(FinishCacheModification(records));
            historyEventIdMax = historyEventIdNext;
        }

        std::reverse(Events.begin(), Events.end());
        FinishCacheConstruction();
        if (historyEventIdMaxWithFinishInstant != -1) {
            LockedMaxEventId = historyEventIdMaxWithFinishInstant;
        }
        CleanCache();
        INFO_LOG << "RESOURCE: Finish cache construction for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
    }

    void CleanCache() const {
        INFO_LOG << GetTableName() << ": before CleanCache: " << Events.size() << Endl;
        auto removed = std::remove_if(Events.begin(), Events.end(), [this](const TEventPtr& ev) {
            return !ev || !ShouldCacheEvent(*ev);
        });
        Events.erase(removed, Events.end());
        INFO_LOG << GetTableName() << ": after CleanCache: " << Events.size() << Endl;
    }

    virtual void GarbageCollectCache(TInstant /*deadline*/) {
    }

    bool RefreshCache(const TInstant reqActuality, const bool doFreezeExt, const bool ignoreParseErrors) const {
        bool doFreeze = doFreezeExt && (Now() - LastSuccessReadInstant) < Config.GetMaximalLockedIntervalDuration();
        DEBUG_LOG << "RefreshCache for " << GetTableName() << " START..." << Endl;
        TInstant actualInstant = Min(Now(), reqActuality);
        TInstant modelingActualInstant = ModelingNow();

        if (CurrentInstant > actualInstant && !doFreeze) {
            DEBUG_LOG << "Skip update inasmuch as " << CurrentInstant << " > " << actualInstant << Endl;
            return true;
        }

        if (Invalid) {
            ALERT_LOG << Name() << ": marked invalid" << Endl;
            throw TWithBackTrace<yexception>() << "invalid index " << Name();
        }

        LogEvent(NJson::TMapBuilder
            ("event", "RefreshSeqTableCache")
            ("actuality", NJson::ToJson(reqActuality))
            ("current", NJson::ToJson(CurrentInstant))
        );
        bool result = false;
        {
            result = RefreshCache1(actualInstant, modelingActualInstant, doFreeze, ignoreParseErrors);
        }
        LogEvent(NJson::TMapBuilder
            ("event", "RefreshSeqTableCacheResult")
            ("result", result)
        );
        DEBUG_LOG << "RefreshCache for " << GetTableName() << " FINISHED " << Events.size() << "/" << ReadyEvents.size() << Endl;
        return result;
    }

    bool RefreshCache1(TInstant actualInstant, TInstant modelingActualInstant, bool doFreeze, bool ignoreParseErrors) const {
        auto guard = Guard(MutexRequest);
        if (CurrentInstant > actualInstant && !doFreeze) {
            DEBUG_LOG << "Skip update inasmuch as " << CurrentInstant << " > " << actualInstant << Endl;
            LogEvent(NJson::TMapBuilder
                ("event", "SkipRefreshSeqTableCache1")
                ("actuality", NJson::ToJson(actualInstant))
                ("current", NJson::ToJson(CurrentInstant))
            );
            return true;
        }

        const bool readFull = LockedMaxEventId == TEvent::IncorrectEventId;
        THolder<TParsedObjects> records(new TParsedObjects(*this, actualInstant, modelingActualInstant, !doFreeze, ignoreParseErrors));
        records->SetLocked(doFreeze);

        auto fields = records->GetDecoder().GetFieldsForRequest();
        bool success = false;
        if (!success) {
            auto session = records->BuildTx();
            auto transaction = session.GetTransaction();
            NStorage::IQueryResult::TPtr result;

            TString selectRequest;
            if (readFull) {
                selectRequest = "SELECT " + fields + " FROM " + GetTableName() + " ORDER BY " + GetSeqFieldName();
            } else {
                selectRequest = "SELECT " + fields + " FROM " + GetTableName() + " WHERE " + GetSeqFieldName() + " > " + ToString(LockedMaxEventId) + " ORDER BY " + GetSeqFieldName();
            }
            if (doFreeze) {
                NDrive::TEventLog::TSourceGuard sourceGuard(Name());
                result = transaction->MultiExec({
                    {"LOCK TABLE " + GetTableName() + " IN SHARE MODE NOWAIT"},
                    {selectRequest, *records},
                    {"COMMIT"}
                });
                if (!result->IsSucceed()) {
                    records.Reset(new TParsedObjects(*this, actualInstant, modelingActualInstant, true, ignoreParseErrors));
                    records->SetLocked(false);
                    session = records->BuildTx();
                    transaction = session.GetTransaction();
                    WARNING_LOG << "REASK EVENTS FROM " << LockedMaxEventId << " !! / " << GetTableName() << Endl;
                    result = transaction->Exec(selectRequest, records.Get());
                } else if (!session.Commit()) {
                    WARNING_LOG << Name() << ": cannot Commit: " << session.GetStringReport() << Endl;
                }
            } else {
                result = transaction->Exec(selectRequest, records.Get());
            }
            success = ParseQueryResult(result, session);
            if (!success) {
                ERROR_LOG << Name() << ": could not RefreshCache: " << session.GetStringReport() << Endl;
                LogEvent(NJson::TMapBuilder
                    ("event", "RefreshSeqTableCache1Error")
                    ("error", session.GetReport())
                );
            }
        }
        if (!success) {
            return false;
        }
        LastSuccessReadInstant = Now();
        return FinishCacheModification(*records);
    }

    void SetLockedMaxEventId(TEventId value) const {
        TWriteGuard wg(Mutex);
        LockedMaxEventId = value;
    }

public:
    virtual bool IsNotUniqueTableSequential() const {
        return false;
    }

    virtual bool Refresh() override {
        bool isBuilding = ++UpdatesCounter == 1;
        if (isBuilding) {
            BuildCache();
        }
        auto actuality = Now();
        bool updateSucceed = RefreshCache(actuality, true, isBuilding);
        if (updateSucceed) {
            SignalHistoryUpdateSuccess.Signal(1);
        } else {
            SignalHistoryUpdateFail.Signal(1);
        }
        auto deadline = actuality - Config.GetDeep();
        GarbageCollectCache(deadline);
        return updateSucceed;
    }

    virtual bool MetricSignal() override {
        SignalHistoryFreshness.Signal((Now() - CurrentInstant).MilliSeconds());
        return true;
    }

    ui64 GetInsertedEventsCount() const {
        return InsertedEventsCount.load(std::memory_order_acquire);
    }

    TEventId GetCurrentMaxEventId() const {
        return CurrentMaxEventId;
    }

    TBaseEventId GetLockedMaxEventId(TDuration timeout = TDuration::MilliSeconds(500), TDuration pause = TDuration::MilliSeconds(50)) const override {
        Y_UNUSED(timeout);
        Y_UNUSED(pause);
        return LockedMaxEventId;
    }

    TInstant GetCurrentInstant() const {
        return CurrentInstant;
    }

    TEventId GetIncorrectEventId() const {
        return TEvent::IncorrectEventId;
    }

    const THistoryConfig& GetConfig() const {
        return Config;
    }

    [[nodiscard]] bool Update(const TInstant reqActuality) const {
        bool isBuilding = ++UpdatesCounter == 1;
        if (isBuilding) {
            BuildCache();
        }
        return RefreshCache(reqActuality, false, isBuilding);
    }

    virtual void FillServerInfo(NJson::TJsonValue& report) const {
        report["locked_max_event"] = LockedMaxEventId;
        report["current_max_event"] = CurrentMaxEventId;
        report["start_instant"] = StartInstant.Seconds();
        report["current_instant"] = CurrentInstant.Seconds();
        report["start_instant_hr"] = StartInstant.ToString();
        report["current_instant_hr"] = CurrentInstant.ToString();
        report["events_count"] = Events.size();
    }

    virtual bool Process(IMessage* message) override {
        TCollectServerInfo* messageSI = dynamic_cast<TCollectServerInfo*>(message);
        if (messageSI) {
            NJson::TJsonValue report;
            FillServerInfo(report);
            messageSI->Fields[Name()] = report;
            return true;
        }
        const NDrive::TCacheRefreshMessage* refreshCache = dynamic_cast<const NDrive::TCacheRefreshMessage*>(message);
        if (refreshCache && (refreshCache->GetComponents().empty() || refreshCache->GetComponents().contains(GetTableName()))) {
            bool updated = Update(Now());
            if (!updated) {
                ERROR_LOG << "cannot CacheRefresh in " << Name() << Endl;
            }
            return true;
        }
        TServerStartedMessage* serverStarted = dynamic_cast<TServerStartedMessage*>(message);
        if (serverStarted) {
            CHECK_WITH_LOG(serverStarted->MutableSequentialTableNames().emplace(GetTableName()).second || IsNotUniqueTableSequential()) << "table " << GetTableName() << " is used by more than one sequential";
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return "SeqTableFor:" + GetTableName();
    }

    TBaseSequentialTableImpl(const IHistoryContext& context, const TString& tableName, const THistoryConfig& config)
        : IBaseSequentialTableImpl(context, tableName)
        , IAutoActualization(tableName + "-auto-refresh", config.GetPingPeriod(), TDuration::Seconds(1))
        , SignalHistoryRowsParsed(GetTableName())
        , SignalHistoryFreshness(GetTableName())
        , SignalHistoryUpdateSuccess(GetTableName())
        , SignalHistoryUpdateFail(GetTableName())
        , GetEventsInvoked({ "history-" + tableName + "-GetEvents" }, false)
        , GetEventsSinceIdInvoked({ "history-" + tableName + "-GetEventsSinceId" }, false)
        , Config(config)
    {
        StartInstant = ModelingNow() - Config.GetDeep();
        CurrentInstant = StartInstant;
        RegisterGlobalMessageProcessor(this);
    }

    virtual ~TBaseSequentialTableImpl() {
        UnregisterGlobalMessageProcessor(this);
    }

    [[nodiscard]] bool GetEventsImpl(const std::deque<TEventPtr>& events, const TInstant startInstant, TEvents& results) const {
        GetEventsInvoked.Signal(1);
        results.clear();
        for (auto it = events.rbegin(); it != events.rend(); ++it) {
            if ((*it)->GetHistoryInstant() < startInstant)
                break;
            results.emplace_back(*it);
        }
        std::reverse(results.begin(), results.end());
        return true;
    }

    [[nodiscard]] bool GetEventsSinceIdImpl(const std::deque<TEventPtr>& events, TEventId id, TEvents& results) const {
        GetEventsSinceIdInvoked.Signal(1);
        Y_ASSERT(std::is_sorted(events.begin(), events.end(), [](const TAtomicSharedPtr<TEvent>& left, const TAtomicSharedPtr<TEvent>& right) {
            return left->GetHistoryEventId() < right->GetHistoryEventId();
        }));
        auto lower = std::lower_bound(events.begin(), events.end(), id, [](const TAtomicSharedPtr<TEvent>& ev, const TEventId id) {
            return ev->GetHistoryEventId() < id;
        });
        results.assign(lower, events.end());
        return true;
    }

    [[nodiscard]] virtual bool GetEventsSinceId(TEventId id, TEvents& results, const TInstant actuality) const {
        if (!Update(actuality)) {
            return false;
        }
        TReadGuard rg(Mutex);
        return GetEventsSinceIdImpl(Events, id, results);
    }

    [[nodiscard]] virtual bool GetEventsAll(const TInstant startInstant, TEvents& results, const TInstant reqActuality) const {
        if (!Update(reqActuality)) {
            return false;
        }
        TReadGuard rg(Mutex);
        return GetEventsImpl(Events, startInstant, results);
    }
};

template <class TEvent, class TObjectId, class TCachedObject = typename TEvent::TEntity>
class IHistoryCallback: public TDBEntitiesCacheImpl<TCachedObject, TObjectId> {
private:
    using TBase = TDBEntitiesCacheImpl<TCachedObject, TObjectId>;
    using TIdRef = typename TIdTypeSelector<TObjectId>::TIdRef;
    using TEventId = typename TEvent::TEventId;

private:
    TSet<TEventId> ReadyEvents;

protected:
    TEventId LockedMaxEventId = TEvent::IncorrectEventId;

protected:
    virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TEvent>& dbEvent, const bool isNewEvent) = 0;
    virtual TIdRef GetEventObjectId(const TEvent& ev) const = 0;

public:
    using TPtr = TAtomicSharedPtr<IHistoryCallback>;

public:
    using TBase::TBase;

    virtual bool Accept(const std::deque<TAtomicSharedPtr<TEvent>>& events) {
        TSet<TIdRef> eventObjectIds;
        auto wg = TBase::MakeObjectWriteGuard();
        for (auto&& i : events) {
            TIdRef eventObjectId = GetEventObjectId(*i);
            const bool isNewEvent = !ReadyEvents.contains(i->GetHistoryEventId());
            if (!isNewEvent && !eventObjectIds.contains(eventObjectId)) {
                continue;
            }
            ReadyEvents.emplace(i->GetHistoryEventId());
            eventObjectIds.emplace(eventObjectId);
            DoAcceptHistoryEventUnsafe(i, isNewEvent);
        }
        return true;
    }

    void SetLockedInfo(TEventId lockedMaxEventId) {
        if (lockedMaxEventId != TEvent::IncorrectEventId) {
            LockedMaxEventId = lockedMaxEventId;
            if (ReadyEvents.empty() || *ReadyEvents.rbegin() <= lockedMaxEventId) {
                ReadyEvents.clear();
            } else {
                ReadyEvents.erase(ReadyEvents.begin(), ReadyEvents.upper_bound(lockedMaxEventId));
            }
        }
    }
};

template <class TEvent, class TObjectId = TStringBuf, class TCachedObject = typename TEvent::TEntity>
class TCallbackSequentialTableImpl: public TBaseSequentialTableImpl<TEvent> {
private:
    using TBase = TBaseSequentialTableImpl<TEvent>;
    using IHistoryCallback = IHistoryCallback<TEvent, TObjectId, TCachedObject>;

private:
    TVector<IHistoryCallback*> Callbacks;

protected:
    using TBase::Config;
    using TBase::Events;
    using TBase::CleanCache;
    using TBase::FinishCacheConstruction;
    using TBase::GetCallbackNeedsHistoryLoad;
    using TBase::GetHistoryEventIdByTimestamp;
    using TBase::GetLockedMaxEventId;
    using TBase::ReadyEvents;
    using TBase::StartInstant;
    using TBase::SetLockedMaxEventId;

    virtual void BuildCache() const override {
        if (GetCallbackNeedsHistoryLoad()) {
            TBase::BuildCache();
            return;
        }
        StartInstant = ModelingNow() - TDuration::Minutes(5);
        const i64 historyEventIdMin = GetHistoryEventIdByTimestamp(StartInstant, "deepest event");
        FinishCacheConstruction();
        if (historyEventIdMin != -1) {
            SetLockedMaxEventId(historyEventIdMin);
        }
        CleanCache();
    }

    virtual void CleanCurrentBuffer(const bool locked) const override {
        auto lockedMaxEventId = GetLockedMaxEventId();
        for (auto&& c : Callbacks) {
            c->Accept(Events);
            c->SetLockedInfo(lockedMaxEventId);
        }
        TBase::CleanCurrentBuffer(locked);
        TBaseSequentialTableImpl<TEvent>::PopFrontEvent(Events, lockedMaxEventId);
    }

public:
    using TBase::TBase;

    void RegisterCallback(IHistoryCallback* callback) {
        Callbacks.emplace_back(callback);
    }

    void UnregisterCallback(IHistoryCallback* callback) {
        Callbacks.erase(std::remove_if(Callbacks.begin(), Callbacks.end(), [callback](const IHistoryCallback* item) {
            return callback == item;
        }), Callbacks.end());
    }
};

template <class TEvent>
class IEventsIndexWriter {
public:
    virtual void FillServerInfo(NJson::TJsonValue& report) const = 0;
    virtual void Rebuild(const std::deque<TAtomicSharedPtr<TEvent>>& events) = 0;
    virtual void InsertSorted(TAtomicSharedPtr<TEvent> ev) = 0;
    virtual void PopFrontEvent(TAtomicSharedPtr<TEvent> ev) = 0;
};

template <class TEvent>
class IIndexRefreshAgent {
public:
    virtual TString GetName() const = 0;
    virtual bool RefreshIndexData(const TInstant reqActuality) = 0;
    virtual bool RegisterIndex(IEventsIndexWriter<TEvent>* index) = 0;
    virtual bool UnregisterIndex(IEventsIndexWriter<TEvent>* index) = 0;
    virtual TReadGuard StartRead() = 0;
};

template <class T>
class TKeyIdAdapter {
};

template <>
class TKeyIdAdapter<TStringBuf> {
public:
    using TKeyId = TString;

public:
    static TKeyId MakeIndexKey(const TStringBuf id) {
        return TKeyId(id);
    }
};

template <>
class TKeyIdAdapter<ui32> {
public:
    using TKeyId = ui32;

public:
    static ui32 MakeIndexKey(const ui32 id) {
        return id;
    }
};

template <>
class TKeyIdAdapter<ui64> {
public:
    using TKeyId = ui64;

public:
    static ui32 MakeIndexKey(const ui64 id) {
        return id;
    }
};

template <class TEvent, class TKeyIdExt = TStringBuf>
class IEventsIndex: public IEventsIndexWriter<TEvent> {
public:
    using TKeyIdInt = typename TKeyIdAdapter<TKeyIdExt>::TKeyId;

private:
    using IIndexRefreshAgent = IIndexRefreshAgent<TEvent>;

protected:
    TMap<TKeyIdInt, std::deque<TAtomicSharedPtr<TEvent>>> Index;
    IIndexRefreshAgent* IndexRefreshAgent = nullptr;
    TRWMutex MutexIndex;
    TString Name = "undefined";

protected:
    virtual TMaybe<TKeyIdExt> ExtractKey(TAtomicSharedPtr<TEvent> ev) const = 0;

private:
    template <class T>
    static ui64 GetIdHash(const T id) {
        return id;
    }

    static ui64 GetIdHash(const TStringBuf id) {
        return FnvHash<ui64>(id.data(), id.size());
    }

    class TSortedEvent {
        R_READONLY(TKeyIdExt, Id, 0);
        R_READONLY(ui32, Idx, 0);
        ui64 Hash;
        TAtomicSharedPtr<TEvent> Object;

    public:
        TAtomicSharedPtr<TEvent> GetObject() const {
            return Object;
        }

        TSortedEvent(const TKeyIdExt& id, const ui32 idx, TAtomicSharedPtr<TEvent> object)
            : Id(id)
            , Idx(idx)
            , Object(object)
        {
            Hash = GetIdHash(Id);
        }

        bool operator<(const TSortedEvent& item) const {
            if (Hash < item.Hash) {
                return true;
            } else if (Hash > item.Hash) {
                return false;
            } else {
                return (Id < item.Id) || (Id == item.Id && Idx < item.Idx);
            }
        }
    };

    virtual void DoInsertSorted(std::deque<TAtomicSharedPtr<TEvent>>& v, TAtomicSharedPtr<TEvent> ev) {
        TBaseSequentialTableImpl<TEvent>::InsertSorted(v, ev);
    }

    virtual void DoPopFrontEvent(std::deque<TAtomicSharedPtr<TEvent>>& v, TAtomicSharedPtr<TEvent> ev) const {
        TBaseSequentialTableImpl<TEvent>::PopFrontEvent(v, ev->GetHistoryEventId());
    }

public:
    IEventsIndex(IIndexRefreshAgent* agent, const TString& name)
        : IndexRefreshAgent(agent)
        , Name(name)
        , GetCachedEventsInvoked({"history-" + agent->GetName() + "-index-" + name + "-GetCachedEvents"}, false)
        , GetEventsInvoked({"history-" + agent->GetName() + "-index-" + name + "-GetEvents"}, false)
    {
        IndexRefreshAgent->RegisterIndex(this);
    }

    virtual ~IEventsIndex() {
        IndexRefreshAgent->UnregisterIndex(this);
    }

    virtual void FillServerInfo(NJson::TJsonValue& report) const override {
        report["events_index_info_" + Name + "_keys_count"] = Index.size();
    }

    TMap<TKeyIdInt, std::deque<TAtomicSharedPtr<TEvent>>> GetCachedEvents() const {
        GetCachedEventsInvoked.Signal(1);
        TReadGuard gMutex = IndexRefreshAgent->StartRead();
        return Index;
    }

    virtual void PopFrontEvent(TAtomicSharedPtr<TEvent> ev) override {
        auto key = ExtractKey(ev);
        if (key) {
            auto it = Index.find(*key);
            if (it != Index.end()) {
                DoPopFrontEvent(it->second, ev);
            }
        }
    };

    virtual void InsertSorted(TAtomicSharedPtr<TEvent> ev) override {
        auto key = ExtractKey(ev);
        if (key) {
            auto it = Index.find(*key);
            if (it == Index.end()) {
                it = Index.emplace(TKeyIdAdapter<TKeyIdExt>::MakeIndexKey(*key), std::deque<TAtomicSharedPtr<TEvent>>()).first;
            }
            DoInsertSorted(it->second, ev);
        }
    }

    [[nodiscard]] bool GetEvents(const TKeyIdExt& keyId, const TInstant since, TVector<TAtomicSharedPtr<TEvent>>& results, const TInstant reqActuality) const {
        GetEventsInvoked.Signal(1);
        if (!Yensured(IndexRefreshAgent)->RefreshIndexData(reqActuality)) {
            return false;
        }
        results.clear();
        TReadGuard gMutex = IndexRefreshAgent->StartRead();
        auto itKey = Index.find(keyId);
        if (itKey == Index.end()) {
            return true;
        }
        for (auto it = itKey->second.rbegin(); it != itKey->second.rend(); ++it) {
            if ((*it)->GetHistoryInstant() < since)
                break;
            results.emplace_back(*it);
        }
        std::reverse(results.begin(), results.end());
        return true;
    }

    virtual void Rebuild(const std::deque<TAtomicSharedPtr<TEvent>>& events) override {
        TVector<TSortedEvent> evSorted;
        for (auto&& i : events) {
            TMaybe<TKeyIdExt> keyId = ExtractKey(i);
            if (keyId) {
                evSorted.emplace_back(*keyId, evSorted.size(), i);
            }
        }
        std::sort(evSorted.begin(), evSorted.end());

        TKeyIdExt currentId;
        bool isFirst = true;
        std::deque<TAtomicSharedPtr<TEvent>>* vEvents = nullptr;
        for (auto&& i : evSorted) {
            if (isFirst || currentId != i.GetId()) {
                vEvents = &Index.emplace(TKeyIdAdapter<TKeyIdExt>::MakeIndexKey(i.GetId()), std::deque<TAtomicSharedPtr<TEvent>>()).first->second;
                currentId = i.GetId();
                isFirst = false;
            }
            DoInsertSorted(*vEvents, i.GetObject());
        }
    }

    TString GetName() const {
        return Name;
    }

private:
    TUnistatSignal<double> GetCachedEventsInvoked;
    TUnistatSignal<double> GetEventsInvoked;
};

template <class TEvent>
class TIndexedSequentialTableImpl
    : public TBaseSequentialTableImpl<TEvent>
    , public IIndexRefreshAgent<TEvent>
{
private:
    using TBase = TBaseSequentialTableImpl<TEvent>;
    using IEventsIndexWriter = IEventsIndexWriter<TEvent>;

public:
    using TEventPtr = typename TBase::TEventPtr;
    using TEventsView = TVector<TEventPtr>;
    using TBase::GetTableName;
    using TBase::Mutex;
    using TBase::TBase;
    using TBase::Update;

private:
    mutable TVector<IEventsIndexWriter*> Indexes;

    virtual TString GetName() const override {
        return GetTableName();
    }

    virtual bool RegisterIndex(IEventsIndexWriter* index) override {
        TWriteGuard wg(Mutex);
        for (auto&& i : Indexes) {
            if (i == index) {
                return false;
            }
        }
        Indexes.emplace_back(index);
        return true;
    }

    virtual bool UnregisterIndex(IEventsIndexWriter* index) override {
        TWriteGuard wg(Mutex);
        const auto pred = [index](IEventsIndexWriter* i) {
            return i == index;
        };
        Indexes.erase(std::remove_if(Indexes.begin(), Indexes.end(), pred), Indexes.end());
        return true;
    }

protected:
    using TBase::Events;

    virtual TReadGuard StartRead() override {
        return TReadGuard(Mutex);
    }
    virtual bool RefreshIndexData(const TInstant reqActuality) override {
        return Update(reqActuality);
    }

    virtual void FillServerInfo(NJson::TJsonValue& report) const override {
        TBase::FillServerInfo(report);
        for (auto&& i : Indexes) {
            i->FillServerInfo(report);
        }
    }

    virtual void AddEventDeep(const TEventPtr& /*event*/) const {
    }

    virtual void DoAddEvent(const TEventPtr& currentEvent) const override {
        AddEventDeep(currentEvent);
        if (currentEvent && TBase::ShouldCacheEvent(*currentEvent)) {
            for (auto&& i : Indexes) {
                i->InsertSorted(currentEvent);
            }
        }
    }

    virtual void FinishCacheConstruction() const override {
        TTimeGuardImpl<false, ELogPriority::TLOG_NOTICE> tg("FinishCacheConstruction for " + GetTableName() + "/" + ::ToString(Events.size()) + ")");
        INFO_LOG << "RESOURCE: FinishCacheConstruction start for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
        TBase::FinishCacheConstruction();

        for (auto&& i : Events) {
            AddEventDeep(i);
        }

        TAutoPtr<IThreadPool> pool = CreateThreadPool(Indexes.size());
        TVector<NThreading::TFuture<void>> futures;
        for (auto&& index : Indexes) {
            futures.emplace_back(NThreading::Async([this, index]() {
                index->Rebuild(Events);
            }, *pool));
        }

        NThreading::WaitExceptionOrAll(futures).Wait();
        INFO_LOG << "RESOURCE: FinishCacheConstruction finish for " << GetTableName() << ": " << TRusage::GetCurrentRSS() / 1024 / 1024 << "Mb" << Endl;
    }
};
