#include "session.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/threading/eventlogger.h>

ISession::ISession(const ISession& other)
    : StartTS(other.StartTS)
    , LastTS(other.LastTS)
    , LastEventId(other.LastEventId)
    , InstanceId(other.InstanceId)
    , ObjectId(other.ObjectId)
    , UserId(other.UserId)
    , CurrentState(other.CurrentState)
    , Compiled(false)
    , Deprecated(other.Deprecated)
{
}

bool ISession::Compile() const {
    TGuard<TMutex> g(Mutex);
    if (Compiled && GetClosed()) {
        return true;
    }
    if (CurrentState == ECurrentState::Corrupted) {
        return false;
    }
    Compiled = DoCompile();
    return Compiled;
}

NJson::TJsonValue ISession::GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const {
    TGuard<TMutex> g(Mutex);
    NJson::TJsonValue result;
    if (!Compile()) {
        result.InsertValue("compiled", false);
    }
    if (customization && customization->GetNeedMeta()) {
        result.InsertValue("meta", GetReportMeta());
    }
    result.InsertValue("session", DoGetReport(locale, server, customization));
    return result;
}

NJson::TJsonValue ISession::GetReportImpl(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const {
    return DoGetReport(locale, server, customization);
}

NJson::TJsonValue ISession::GetReportMeta() const {
    NJson::TJsonValue result;
    if (GetSessionId()) {
        result.InsertValue("session_id", GetSessionId());
    }
    result.InsertValue("last_event_id", LastEventId);
    result.InsertValue("instance_id", InstanceId);
    result.InsertValue("object_id", ObjectId);
    result.InsertValue("user_id", UserId);
    result.InsertValue("start", StartTS.Seconds());
    if (GetClosed()) {
        result.InsertValue("finish", LastTS.Seconds());
    } else {
        result.InsertValue("current", LastTS.Seconds());
    }
    result.InsertValue("finished", GetClosed());
    result.InsertValue("corrupted", CurrentState == ECurrentState::Corrupted);
    return result;
}

void ISession::MarkCorrupted(const NJson::TJsonValue& info) {
    NDrive::TEventLog::Log("MarkCorrupted", info);
    SetProtectedCurrentState(ECurrentState::Corrupted);
}

void ISession::SignalRefresh(TInstant refreshTimestamp) {
    if (GetClosed() || refreshTimestamp <= LastTS) {
        return;
    }
    TGuard<TMutex> g(Mutex);
    if (!GetClosed() && refreshTimestamp > LastTS) {
        LastTS = refreshTimestamp;
        Compiled = false;
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NEventsSession::TTimeEvent& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetEventId();
    result["index"] = object.GetEventIndex();
    result["timestamp"] = NJson::ToJson(object.GetEventInstant());
    result["type"] = ToString(object.GetTimeEvent());
    return result;
}
