#pragma once

#include "common.h"

#include <drive/backend/abstract/base.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/logger/global/global.h>

#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/generic/cast.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/system/mutex.h>

class ITagsMeta;

namespace NDrive {
    class IObjectSnapshot;
    class IServer;
}

class TSessionCorruptedGlobalMessage: public IMessage {
    R_FIELD(bool, NeedAlert, true);
};

class ISessionReportCustomization {
public:
    R_FIELD(bool, NeedMeta, true);

public:
    using TPtr = TAtomicSharedPtr<ISessionReportCustomization>;

public:
    virtual ~ISessionReportCustomization() = default;
};

class ISession {
public:
    using TPtr = TAtomicSharedPtr<ISession>;
    using TConstPtr = TAtomicSharedPtr<const ISession>;

    enum class ECurrentState {
        Undefined,
        Started,
        Closed,
        Corrupted
    };

public:
    R_READONLY(TInstant, StartTS, TInstant::Zero());
    R_READONLY(TInstant, LastTS, TInstant::Zero());
    R_READONLY(NDrive::TEventId, LastEventId, 0);
    R_READONLY(TString, InstanceId);
    R_READONLY(TString, ObjectId);
    R_READONLY(TString, UserId);
    R_READONLY(ECurrentState, CurrentState, ECurrentState::Undefined);
    R_READONLY(bool, Compiled, false, mutable);
    R_FIELD(bool, Deprecated, false);

protected:
    TMutex Mutex;

protected:
    ISession(const ISession& other);

    virtual bool DoCompile() const = 0;
    virtual NJson::TJsonValue DoGetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const = 0;
    NJson::TJsonValue GetReportMeta() const;
    void MarkCorrupted(const NJson::TJsonValue& info);

public:
    ISession() = default;
    virtual ~ISession() = default;

    [[nodiscard]] bool Compile() const;

    bool GetClosed() const {
        return CurrentState == ECurrentState::Closed;
    }

    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const;
    NJson::TJsonValue GetReportImpl(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const;
    virtual const TString& GetSessionId() const = 0;

    void SignalRefresh(TInstant refreshTimestamp);
};

namespace NEventsSession {
    const i64 MaxEventId = Max<i64>();

    enum EEventCategory {
        Begin = 1,
        End = 1 << 1,
        Internal = 1 << 2,
        IgnoreExternal = 1 << 3,
        IgnoreInternal = 1 << 4,
        Switch = 1 << 5,
    };

    enum class EEvent {
        Tag,
        CurrentFinish
    };

    class TTimeEvent {
    public:
        R_FIELD(EEvent, TimeEvent, EEvent::Tag);
        R_FIELD(i64, EventId, -1);
        R_FIELD(TInstant, EventInstant, TInstant::Zero());
        R_FIELD(ui32, EventIndex, Max<ui32>());

    public:
        bool operator<(const TTimeEvent& item) const {
            return std::make_tuple(EventInstant, EventId) < std::make_tuple(item.EventInstant, item.EventId);
        }
        bool operator<(TInstant timestamp) const {
            return EventInstant < timestamp;
        }
    };

    inline bool operator<(TInstant timestamp, const TTimeEvent& ev) noexcept {
        return timestamp < ev.GetEventInstant();
    }
}

template <class TEvent>
class IEventsSession: public ISession {
public:
    using EEvent = NEventsSession::EEvent;
    using TTimeEvent = NEventsSession::TTimeEvent;

    class ICompilation {
    public:
        virtual ~ICompilation() = default;
        virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TEvent>>& events) = 0;
        virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const = 0;
        virtual const TString& GetSessionId() const = 0;
    };

    using TPtr = TAtomicSharedPtr<IEventsSession<TEvent>>;
    using TConstPtr = TAtomicSharedPtr<const IEventsSession<TEvent>>;
    using TUniquePtr = THolder<IEventsSession<TEvent>>;

protected:
    virtual THolder<ICompilation> BuildDefaultCompilation() const {
        return nullptr;
    }

    virtual TUniquePtr DoClone() const = 0;

    [[nodiscard]] bool DoCompile() const final {
        if (!DefaultCompilation) {
            DefaultCompilation = BuildDefaultCompilation();
        }
        Y_ENSURE_BT(DefaultCompilation);
        return FillCompilation(*DefaultCompilation);
    }

    virtual NJson::TJsonValue DoGetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const override final {
        Y_ENSURE_BT(DefaultCompilation);
        NJson::TJsonValue result;
        result.InsertValue("specials", DefaultCompilation->GetReport(locale, server, customization));
        return result;
    }

private:
    mutable THolder<ICompilation> DefaultCompilation;
    TVector<TAtomicSharedPtr<TEvent>> Events;
    TAtomicSharedPtr<TEvent> CurrentEvent;
    mutable TVector<TTimeEvent> Timeline;
    mutable ui32 PredTimelineSize = 0;

private:
    void PrepareTimeline() const {
        if (PredTimelineSize == Timeline.size()) {
            return;
        }
        std::sort(Timeline.begin(), Timeline.end());
        PredTimelineSize = Timeline.size();
    }

public:
    IEventsSession() = default;
    IEventsSession(const IEventsSession& other)
        : ISession(other)
        , Events(other.Events)
        , CurrentEvent(other.CurrentEvent)
        , Timeline(other.Timeline)
        , PredTimelineSize(other.PredTimelineSize)
    {
    }

    TUniquePtr Clone() const {
        TGuard<TMutex> g(Mutex);
        return DoClone();
    }

    TMaybe<TEvent> GetLastEvent() const {
        TGuard<TMutex> g(Mutex);
        if (CurrentEvent) {
            return *CurrentEvent;
        }
        return {};
    }

    TMaybe<TEvent> GetLastEventAt(TInstant timestamp) const {
        TGuard<TMutex> g(Mutex);
        PrepareTimeline();
        auto lk = LowerKey(Timeline, timestamp);
        if (lk == Timeline.end()) {
            return {};
        }
        if (lk->GetTimeEvent() != EEvent::Tag) {
            return {};
        }
        Y_ASSERT(lk->GetEventIndex() < Events.size());
        const auto& ev = Events[lk->GetEventIndex()];
        if (!ev) {
            return {};
        }
        return *ev;
    }

    TMaybe<TEvent> GetFirstEvent() const {
        TGuard<TMutex> g(Mutex);
        PrepareTimeline();
        if (Timeline.size() && Timeline.front().GetTimeEvent() == EEvent::Tag) {
            return *Events[Timeline.front().GetEventIndex()];
        }
        return {};
    }

    template <class T>
    TMaybe<T> GetCompilationAs() const {
        TGuard<TMutex> g(Mutex);
        if (!Compile()) {
            WARNING_LOG << "Cannot compile session: " << GetSessionId() << Endl;
            return {};
        }
        auto val = dynamic_cast<const T*>(DefaultCompilation.Get());
        if (val) {
            return *val;
        } else {
            return {};
        }
    }

    virtual const TString& GetSessionId() const override {
        TGuard<TMutex> g(Mutex);
        if (!DefaultCompilation) {
            if (!Compile()) {
                return Default<TString>();
            }
        }
        return DefaultCompilation->GetSessionId();
    }

    [[nodiscard]] bool FillCompilation(ICompilation& compilation) const {
        TGuard<TMutex> g(Mutex);
        PrepareTimeline();
        TVector<TTimeEvent> timeline = Timeline;
        if (Timeline.size()) {
            TInstant max = Timeline[0].GetEventInstant();
            for (ui32 i = 0; i + 1 < Timeline.size(); ++i) {
                Y_ASSERT(Timeline[i].GetEventId() != Timeline[i + 1].GetEventId());
                Y_ASSERT(Timeline[i].GetEventInstant() <= Timeline[i + 1].GetEventInstant());
                max = Max(max, Timeline[i + 1].GetEventInstant());
            }
            if (!GetClosed()) {
                TTimeEvent eventPoint;
                eventPoint.SetTimeEvent(EEvent::CurrentFinish).SetEventInstant(Max(max, GetLastTS())).SetEventId(NEventsSession::MaxEventId);
                if (timeline.empty() || timeline.back().GetTimeEvent() != EEvent::CurrentFinish) {
                    timeline.emplace_back(eventPoint);
                }
            }
        }
        return compilation.Fill(timeline, Events);
    }

    template <class T, class... TArgs>
    TMaybe<T> MakeCompilation(TArgs&&... args) const {
        auto result = MakeMaybe<T>(std::forward<TArgs>(args)...);
        if (!FillCompilation(*result)) {
            return {};
        }
        return result;
    }

    class TDefaultItemReportConstructor {
    public:
        static void DoBuildReportItem(const TEvent* ev, NJson::TJsonValue& item) {
            ev->DoBuildReportItem(item);
        }
    };

    bool HasEvents(const TInstant since = TInstant::Zero(), const TInstant until = TInstant::Max()) const {
        TGuard<TMutex> g(Mutex);
        if (!Compile()) {
            return false;
        }
        std::sort(Timeline.begin(), Timeline.end());
        for (auto&& i : Timeline) {
            if (i.GetEventInstant() >= since && i.GetEventInstant() < until) {
                return true;
            }
        }
        return false;
    }

    template <class TItemReportConstructor = TDefaultItemReportConstructor>
    NJson::TJsonValue GetEventsReport(const TInstant since = TInstant::Zero(), const TInstant until = TInstant::Max()) const {
        TGuard<TMutex> g(Mutex);
        NJson::TJsonValue result(NJson::JSON_MAP);
        result.InsertValue("timeline", GetTimelineReport<TItemReportConstructor>(since, until));
        result.InsertValue("meta", ISession::GetReportMeta());
        return result;
    }

    template <class TItemReportConstructor = TDefaultItemReportConstructor>
    NJson::TJsonValue GetTimelineReport(const TInstant since = TInstant::Zero(), const TInstant until = TInstant::Max()) const {
        TGuard<TMutex> g(Mutex);
        if (!Compile()) {
            NJson::TJsonValue result;
            result["compiled"] = false;
            return result;
        }

        std::sort(Timeline.begin(), Timeline.end());
        NJson::TJsonValue jsonTimeline(NJson::JSON_ARRAY);
        for (auto&& i : Timeline) {
            if (i.GetEventInstant() >= since && i.GetEventInstant() < until) {
                jsonTimeline.AppendValue(Events[i.GetEventIndex()]->template BuildReportItemCustom<TItemReportConstructor>());
            }
        }
        return jsonTimeline;
    }

    NJson::TJsonValue GetDebugInfo() const {
        TGuard<TMutex> g(Mutex);
        NJson::TJsonValue info;
        info["compiled"] = GetCompiled();
        info["current"] = NJson::ToJson(CurrentEvent);
        info["events"] = NJson::ToJson(Events);
        info["finish"] = NJson::ToJson(GetLastTS());
        info["instance_id"] = GetInstanceId();
        info["last_event_id"] = GetLastEventId();
        info["object_id"] = GetObjectId();
        info["previous_timeline_size"] = NJson::ToJson(PredTimelineSize);
        info["start"] = NJson::ToJson(GetStartTS());
        info["timeline"] = NJson::ToJson(Timeline);
        info["user_id"] = GetUserId();
        return info;
    }

    void MarkCorrupted() {
        TSessionCorruptedGlobalMessage message;
        SendGlobalMessage(message);

        TBackTrace backtrace;
        backtrace.Capture();
        NJson::TJsonValue info = GetDebugInfo();
        info["backtrace"] = NJson::ToJson(backtrace);

        if (message.GetNeedAlert()) {
            ALERT_LOG << "Session corrupted: " << info.GetStringRobust() << Endl;
        } else {
            WARNING_LOG << "Session corrupted: " << info.GetStringRobust() << Endl;
        }
        ISession::MarkCorrupted(info);
    }

    virtual bool TestEvent(const TEvent& histEvent) const = 0;

    void InitializeSession(const TEvent& histEvent, const NEventsSession::EEventCategory cat) {
        SetProtectedStartTS(histEvent.GetHistoryInstant());
        SetProtectedLastTS(histEvent.GetHistoryInstant());
        SetProtectedLastEventId(histEvent.GetHistoryEventId());
        SetProtectedObjectId(TEvent::GetObjectId(histEvent));
        SetProtectedUserId(histEvent.GetHistoryUserId());
        SetProtectedInstanceId(TEvent::GetInstanceId(histEvent));
        SetProtectedCurrentState(ISession::ECurrentState::Started);
        if (cat != NEventsSession::EEventCategory::Begin && cat != NEventsSession::EEventCategory::Switch) {
            MarkCorrupted();
        }
    }

    void InsertInternalEvent(TAtomicSharedPtr<TEvent> histEvent) {
        if (histEvent->GetHistoryInstant() < GetStartTS()) {
            SetProtectedStartTS(histEvent->GetHistoryInstant());
        }
        if (histEvent->GetHistoryEventId() > GetLastEventId()) {
            SetProtectedLastEventId(histEvent->GetHistoryEventId());
        }
        if (histEvent->GetHistoryInstant() > GetLastTS()) {
            SetProtectedLastTS(histEvent->GetHistoryInstant());
        }
        Events.emplace_back(histEvent);

        TTimeEvent eventPoint;
        eventPoint.SetEventIndex(Events.size() - 1).SetEventInstant(histEvent->GetHistoryInstant()).SetEventId(histEvent->GetHistoryEventId()).SetTimeEvent(EEvent::Tag);
        Timeline.emplace_back(eventPoint);
        if (!CurrentEvent || *CurrentEvent < *histEvent) {
            CurrentEvent = histEvent;
        }
        SetProtectedCompiled(false);
    }

    void CheckClosed() {
        if (GetCurrentState() != ISession::ECurrentState::Closed) {
            MarkCorrupted();
        }
    }

    void AddEvent(TAtomicSharedPtr<TEvent> histEvent, const NEventsSession::EEventCategory cat) {
        TGuard<TMutex> g(Mutex);
        if (GetCurrentState() == ECurrentState::Corrupted) {
            return;
        }
        Y_ASSERT(!CurrentEvent || (CurrentEvent->GetHistoryEventId() != histEvent->GetHistoryEventId()));
        if (GetCurrentState() == ISession::ECurrentState::Undefined || GetCurrentState() == ISession::ECurrentState::Started) {
            if (GetCurrentState() == ISession::ECurrentState::Undefined) {
                InitializeSession(*histEvent, cat);
            } else {
                if (cat == NEventsSession::EEventCategory::Internal) {

                } else if (cat == NEventsSession::EEventCategory::End) {
                    SetProtectedCurrentState(ISession::ECurrentState::Closed);
                } else {
                    MarkCorrupted();
                    return;
                }
            }
            if (!TestEvent(*histEvent)) {
                MarkCorrupted();
                return;
            }
            InsertInternalEvent(histEvent);
        } else if (GetCurrentState() != ISession::ECurrentState::Closed) {
            MarkCorrupted();
        }
    }
};

template <class TEvent>
class ISessionSelector {
public:
    using TEventId = typename TEvent::TEventId;
    using TPtr = TAtomicSharedPtr<ISessionSelector<TEvent>>;

public:
    virtual ~ISessionSelector() = default;

    virtual NEventsSession::EEventCategory Accept(const TEvent& e) const = 0;

    virtual void FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& /*result*/, const std::deque<TAtomicSharedPtr<TEvent>>& /*events*/, TEventId /*historyEventIdMaxWithFinishInstant*/) const {
    }
    virtual TAtomicSharedPtr<IEventsSession<TEvent>> BuildSession() const = 0;
    virtual TString GetName() const = 0;
};
