#pragma once

#include "sequential.h"
#include "session.h"

#include <util/generic/map.h>

bool CompareSessionsByActiveness(const ISession& left, const ISession& right);

template <class TEvent>
class TSessionsBuilder {
private:
    using TEventId = typename TEvent::TEventId;
    using TEventsSessionPtr = typename ::IEventsSession<TEvent>::TPtr;
    using ISessionSelector = ::ISessionSelector<TEvent>;

private:
    TRWMutex Mutex;
    typename ISessionSelector::TPtr Selector;
    TMap<TString, TEventsSessionPtr> Sessions;
    TMap<TString, TEventsSessionPtr> TagSessions;
    TMap<TString, std::deque<TEventsSessionPtr>> UserSessions;
    TMap<TString, std::deque<TEventsSessionPtr>> ObjectSessions;
    std::multimap<TInstant, TEventsSessionPtr> ClosedSessions;

public:
    using TPtr = TAtomicSharedPtr<TSessionsBuilder<TEvent>>;

public:
    static bool CompareSessionsByActiveness(const TEventsSessionPtr& left, const TEventsSessionPtr& right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return ::CompareSessionsByActiveness(*left, *right);
    }

public:
    void GarbageCollect(TInstant deadline) {
        TVector<std::pair<TInstant, TEventsSessionPtr>> victims;
        {
            TWriteGuard guard(Mutex);
            auto begin = ClosedSessions.begin();
            auto end = ClosedSessions.upper_bound(deadline);
            victims = {
                std::make_move_iterator(begin),
                std::make_move_iterator(end)
            };
            ClosedSessions.erase(begin, end);
        }
        for (auto&& [timestamp, session] : victims) {
            if (!session) {
                WARNING_LOG << "null session encountered" << Endl;
                continue;
            }
            const auto& instanceId = session->GetInstanceId();
            const auto& objectId = session->GetObjectId();
            const auto& sessionId = session->GetSessionId();
            const auto& userId = session->GetUserId();

            size_t instanceIdErased = 0;
            size_t objectIdErased = 0;
            size_t sessionIdErased = 0;
            size_t userIdErased = 0;
            auto erase = [&instanceId](std::deque<TEventsSessionPtr>& sessions) {
                size_t count = 0;
                for (auto i = sessions.begin(); i != sessions.end();) {
                    auto s = i->Get();
                    if (s && s->GetInstanceId() == instanceId) {
                        i = sessions.erase(i);
                        ++count;
                    } else {
                        ++i;
                    }
                }
                return count;
            };
            {
                TWriteGuard guard(Mutex);
                instanceIdErased = TagSessions.erase(instanceId);
                sessionIdErased = Sessions.erase(sessionId);
                objectIdErased = erase(ObjectSessions[objectId]);
                userIdErased = erase(UserSessions[userId]);
            }
            INFO_LOG << "SessionsBuilder " << Selector->GetName() << ": garbage collected " << sessionId
                << ' ' << instanceIdErased
                << ' ' << objectIdErased
                << ' ' << sessionIdErased
                << ' ' << userIdErased
                << Endl;
        }
    }

    void FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& result, const std::deque<TAtomicSharedPtr<TEvent>>& events, TEventId historyEventIdMaxWithFinishInstant) const {
        if (!!Selector) {
            Selector->FillAdditionalEventsConditions(result, events, historyEventIdMaxWithFinishInstant);
        }
    }

    TEventsSessionPtr GetLastObjectSession(const TString& objectId) const {
        TReadGuard rg(Mutex);
        auto it = ObjectSessions.find(objectId);
        if (it == ObjectSessions.end() || it->second.empty()) {
            return nullptr;
        }
        const auto& objectSessions = it->second;
        Y_ASSERT(!objectSessions.empty());
        Y_ASSERT(std::is_sorted(objectSessions.begin(), objectSessions.end(), CompareSessionsByActiveness));
        return objectSessions.size() > 0 ? objectSessions.back() : nullptr;
    }

    void SignalRefresh(const TInstant refreshInstant) {
        TReadGuard rg(Mutex);
        for (auto&& [objectId, objectSessions] : ObjectSessions) {
            for (auto&& session : objectSessions) {
                if (session && !session->GetClosed()) {
                    session->SignalRefresh(refreshInstant);
                }
            }
        }
    }

    TEventsSessionPtr GetSession(const TString& sessionId) const {
        TReadGuard rg(Mutex);
        auto it = Sessions.find(sessionId);
        if (it == Sessions.end()) {
            return nullptr;
        }
        return it->second;
    }

    TVector<TEventsSessionPtr> GetUserSessions(const TString& userId) const {
        TReadGuard rg(Mutex);
        auto it = UserSessions.find(userId);
        TVector<TEventsSessionPtr> result;
        if (it == UserSessions.end()) {
            return result;
        }
        result.reserve(it->second.size());
        result.insert(result.end(), it->second.begin(), it->second.end());
        return result;
    }

    TVector<TEventsSessionPtr> GetObjectSessions(const TString& objectId) const {
        TReadGuard rg(Mutex);
        auto it = ObjectSessions.find(objectId);
        TVector<TEventsSessionPtr> result;
        if (it == ObjectSessions.end()) {
            return result;
        }
        result.reserve(it->second.size());
        result.insert(result.end(), it->second.begin(), it->second.end());
        return result;
    }

    void FillResultSession(const TInstant since, const TInstant until, const TString& id, const std::deque<TEventsSessionPtr>& sessions, TMap<TString, TVector<TEventsSessionPtr>>& result) const {
        auto itResult = result.end();
        for (auto it = sessions.rbegin(); it != sessions.rend(); ++it) {
            if ((*it)->GetLastTS() < since) {
                break;
            }
            if ((*it)->GetStartTS() > until) {
                continue;
            }
            if (itResult == result.end()) {
                itResult = result.emplace(id, TVector<TEventsSessionPtr>()).first;
            }
            itResult->second.emplace_back(*it);
        }
    }

    TMap<TString, TVector<TEventsSessionPtr>> GetSessionsActualImpl(const TInstant since, const TInstant until, const TVector<TString>& ids, const TMap<TString, std::deque<TEventsSessionPtr>>& sessions) const {
        TMap<TString, TVector<TEventsSessionPtr>> result;
        for (auto&& id : ids) {
            result.emplace(id, TVector<TEventsSessionPtr>());
            auto itSessions = sessions.find(id);
            if (itSessions == sessions.end()) {
                continue;
            }

            FillResultSession(since, until, itSessions->first, itSessions->second, result);
        }
        return result;
    }

    TMap<TString, TVector<TEventsSessionPtr>> GetSessionsActualByObjects(const TInstant since, const TInstant until, const TVector<TString>& objectIds) const {
        TReadGuard rg(Mutex);
        return GetSessionsActualImpl(since, until, objectIds, ObjectSessions);
    }

    TVector<TEventsSessionPtr> GetSessionsActualSinceId(TEventId eventId) const {
        TReadGuard rg(Mutex);
        TVector<TEventsSessionPtr> result;
        for (auto&& session : Sessions) {
            if (session.second->GetLastEventId() > eventId) {
                result.emplace_back(session.second);
            }
        }
        return result;
    }

    TMap<TString, TVector<TEventsSessionPtr>> GetSessionsActual(const TInstant since, const TInstant until) const {
        TMap<TString, TVector<TEventsSessionPtr>> result;
        TReadGuard rg(Mutex);
        for (auto&& session : ObjectSessions) {
            for (auto it = session.second.rbegin(); it != session.second.rend(); ++it) {
                if ((*it)->GetLastTS() < since) {
                    break;
                }
                if ((*it)->GetStartTS() >= until) {
                    continue;
                }
                result[(*it)->GetUserId()].emplace_back(*it);
            }
        }
        return result;
    }

    TVector<TEventsSessionPtr> GetSessionsActual() const {
        TReadGuard rg(Mutex);
        TVector<TEventsSessionPtr> result;
        for (auto&& [objectId, objectSessions] : ObjectSessions) {
            for (auto&& session : objectSessions) {
                if (session && !session->GetClosed()) {
                    result.push_back(session);
                }
            }
        }
        return result;
    }

    std::multimap<TString, TEventsSessionPtr> GetSessionsActualByObjects() const {
        TReadGuard rg(Mutex);
        std::multimap<TString, TEventsSessionPtr> result;
        for (auto&& [objectId, objectSessions] : ObjectSessions) {
            for (auto&& session : objectSessions) {
                if (session && !session->GetClosed()) {
                    result.emplace(objectId, session);
                }
            }
        }
        return result;
    }

    std::multimap<TString, TEventsSessionPtr> GetSessionsActualByObjects(const TSet<TString>* objectIds) const {
        if (!objectIds) {
            return GetSessionsActualByObjects();
        }
        TReadGuard rg(Mutex);
        std::multimap<TString, TEventsSessionPtr> result;
        if (objectIds->size() < ObjectSessions.size() / 10) {
            for (auto&& i : *objectIds) {
                auto it = ObjectSessions.find(i);
                if (it == ObjectSessions.end()) {
                    continue;
                }
                for (auto&& session : it->second) {
                    if (session && !session->GetClosed()) {
                        result.emplace(i, session);
                    }
                }
            }
        } else {
            auto it = ObjectSessions.begin();
            for (auto&& i : *objectIds) {
                if (!Advance(it, ObjectSessions.end(), i)) {
                    continue;
                }
                for (auto&& session : it->second) {
                    if (session && !session->GetClosed()) {
                        result.emplace(i, session);
                    }
                }
            }
        }
        return result;
    }

    void AddEvent(TAtomicSharedPtr<TEvent> e) {
        if (!e) {
            return;
        }
        const auto& instanceId = TEvent::GetInstanceId(*e);
        if (!instanceId) {
            return;
        }
        const auto& objectId = TEvent::GetObjectId(*e);
        const NEventsSession::EEventCategory cat = Selector->Accept(*e);
        if (cat == NEventsSession::EEventCategory::IgnoreExternal || cat == NEventsSession::EEventCategory::IgnoreInternal) {
            return;
        }
        TWriteGuard wg(Mutex);
        auto it = TagSessions.find(instanceId);
        if (cat == NEventsSession::Switch && it != TagSessions.end() && !Yasserted(it->second)->GetClosed()) {
            it->second->AddEvent(e, NEventsSession::End);
        }
        bool stateChanged = false;
        TEventsSessionPtr session;
        if (cat == NEventsSession::Begin || cat == NEventsSession::Switch) {
            if (it != TagSessions.end()) {
                auto session = Yasserted(it->second.Get());
                session->CheckClosed();
            }
            session = Selector->BuildSession();
            Y_ENSURE(session);
            session->AddEvent(e, NEventsSession::Begin);
            Sessions[session->GetSessionId()] = session;
            TagSessions[instanceId] = session;
            UserSessions[e->GetHistoryUserId()].emplace_back(session);
            ObjectSessions[objectId].emplace_back(session);
            stateChanged = true;
        } else if (it != TagSessions.end()) {
            session = Yasserted(it->second);
            auto stateBefore = session->GetCurrentState();
            session->AddEvent(e, cat);
            auto stateAfter = session->GetCurrentState();
            stateChanged = stateAfter != stateBefore;
        }
        auto objectSessionsIterator = ObjectSessions.find(objectId);
        if (objectSessionsIterator != ObjectSessions.end()) {
            auto& objectSessions = objectSessionsIterator->second;
            Y_ASSERT(!objectSessions.empty());
            auto lastObjectSession = objectSessions.size() > 0 ? objectSessions.back() : nullptr;
            bool instanceIdChanged = lastObjectSession ? lastObjectSession->GetInstanceId() != instanceId : false;
            if (instanceIdChanged || stateChanged) {
                std::sort(objectSessions.begin(), objectSessions.end(), CompareSessionsByActiveness);
            }
        }
        if (session && session->GetClosed()) {
            auto finish = session->GetLastTS();
            ClosedSessions.emplace(finish, session);
        }
    }

    TSessionsBuilder(typename ISessionSelector::TPtr selector, const std::deque<TAtomicSharedPtr<TEvent>>& events)
        : Selector(selector)
    {
        for (auto&& e : events) {
            AddEvent(e);
        }
    }
};

template <class TEvent>
class TSequentialTableWithSessions: public TIndexedSequentialTableImpl<TEvent> {
private:
    using ISessionSelector = ::ISessionSelector<TEvent>;
    using TSessionsBuilder = ::TSessionsBuilder<TEvent>;
    using TBase = TIndexedSequentialTableImpl<TEvent>;
    using TEventPtr = typename TBase::TEventPtr;

public:
    using TEventId = typename TBase::TEventId;

private:
    TRWMutex MutexSessions;
    TMap<TString, typename TSessionsBuilder::TPtr> Sessions;

protected:
    void OnFinishCacheModification(const typename TBase::TParsedObjects& records) const override {
        TReadGuard rgSessions(MutexSessions);
        for (auto&& sessionBuilder : Sessions) {
            sessionBuilder.second->SignalRefresh(records.GetModelingInstantNow());
        }
    }

    void AddEventDeep(const TEventPtr& eventObj) const final {
        TReadGuard rgSessions(MutexSessions);
        for (auto&& sessionsBuilder : Sessions) {
            sessionsBuilder.second->AddEvent(eventObj);
        }
    }

    void FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& result, TEventId historyEventIdMaxWithFinishInstant) const override {
        for (auto&& i : Sessions) {
            i.second->FillAdditionalEventsConditions(result, TBase::Events, historyEventIdMaxWithFinishInstant);
        }
    }

    void GarbageCollectCache(TInstant deadline) override {
        TReadGuard rgSessions(MutexSessions);
        for (auto&& [id, builder] : Sessions) {
            builder->GarbageCollect(deadline);
        }
    }

public:
    using TBase::TBase;

    typename TSessionsBuilder::TPtr GetSessionsBuilderSafe(TStringBuf name, const TInstant actuality = TInstant::Zero()) const {
        if (!TBase::Update(actuality)) {
            return nullptr;
        }
        TReadGuard wg(MutexSessions);
        auto it = Sessions.find(name);
        if (it == Sessions.end()) {
            return nullptr;
        }
        return it->second;
    }

    typename TSessionsBuilder::TPtr GetSessionsBuilder(TStringBuf name, const TInstant actuality = TInstant::Zero()) const {
        if (!TBase::Update(actuality)) {
            WARNING_LOG << "Cannot refresh sessions builder data for " << name << Endl;
        }
        TReadGuard wg(MutexSessions);
        auto it = Sessions.find(name);
        if (it == Sessions.end()) {
            return nullptr;
        }
        return it->second;
    }

    void RegisterSessionBuilder(typename ISessionSelector::TPtr sessionSelector) {
        TWriteGuard wg(MutexSessions);
        Sessions.emplace(sessionSelector->GetName(), new TSessionsBuilder(sessionSelector, TBase::Events));
    }
};
