#include "assert.h"

#include "tx.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/localization.h>

#include <drive/library/cpp/searchserver/http_status_config.h>

void FillError(TMaybe<int> code, TMaybe<EDriveSessionResult> result, TSourceLocation location, TStringBuf function, TStringBuf condition, NDrive::TInfoEntitySession& session) {
    auto source = function ? TString{TStringBuf(function).Before('(').RAfter(' ')} : "NDrive::Throw";
    auto info = TStringBuilder() << "assertion " << condition << " failed";
    session.AddErrorMessage("source_location", ToString(location));
    if (code) {
        session.SetCode(code);
    }
    if (session.GetResult() == EDriveSessionResult::Success) {
        if (!result) {
            result = (code && IsUserError(*code)) ? EDriveSessionResult::IncorrectRequest : EDriveSessionResult::InternalError;
        }
        session.SetErrorInfo(source, info, *result);
    } else {
        session.AddErrorMessage(source, info);
    }
}

void ThrowError(TStringBuf error, NDrive::TInfoEntitySession& session) {
    const auto& httpStatusManager = NDrive::HasServer() ? NDrive::GetServer().GetHttpStatusManagerConfig() : Default<THttpStatusManagerConfig>();
    const auto localization = NDrive::HasServer() ? NDrive::GetServer().GetLocalization() : nullptr;
    session.DoExceptionOnFail(httpStatusManager, localization, error);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error, TMaybe<EDriveSessionResult> result, NDrive::TInfoEntitySession& session) {
    FillError(code, result, location, function, condition, session);
    ThrowError(error, session);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error, TMaybe<EDriveSessionResult> result) {
    NDrive::TInfoEntitySession session;
    NDrive::Throw(location, function, condition, code, error, result, session);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error, NDrive::TError&& err, NDrive::TInfoEntitySession& session) {
    FillError(code, {}, location, function, condition, session);
    session.SetError(std::move(err));
    ThrowError(error, session);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error, NDrive::TError&& err) {
    NDrive::TInfoEntitySession session;
    NDrive::Throw(location, function, condition, code, error, std::move(err), session);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error, NDrive::TInfoEntitySession& session) {
    NDrive::Throw(location, function, condition, code, error, Nothing(), session);
}

void NDrive::Throw(TSourceLocation location, TStringBuf function, TStringBuf condition, TMaybe<int> code, TStringBuf error) {
    NDrive::TInfoEntitySession session;
    NDrive::Throw(location, function, condition, code, error, session);
}
