#include "tx.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>

#include <drive/library/cpp/searchserver/http_status_config.h>

#include <rtline/util/types/exception.h>

#include <util/system/hostname.h>

static bool TransactionNamesEnabled = false;

NDrive::TInfoEntitySession::~TInfoEntitySession() {
    auto result = GetResult();
    if (AlertOnDirty && result != EDriveSessionResult::Success && !std::uncaught_exceptions()) {
        ALERT_LOG << "Destroying dirty Tx: " << GetStringReport() << Endl;
        Y_ASSERT(result == EDriveSessionResult::Success);
    }
}

NDrive::TEntitySession::TEntitySession(TDeferredTransaction&& deferred, TAtomicSharedPtr<ISessionContext> context)
    : DeferredTransaction(std::move(deferred))
    , Context(context)
{
    if (auto eventLogger = NDrive::GetThreadEventLogger(); eventLogger && DeferredTransaction) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "create_session")
            ("deferred", true)
        );
    }
}

NDrive::TEntitySession::TEntitySession(NStorage::ITransaction::TPtr transaction, TAtomicSharedPtr<ISessionContext> context)
    : Transaction(transaction)
    , Context(context)
{
    if (auto eventLogger = NDrive::GetThreadEventLogger(); eventLogger && Transaction) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "create_session")
            ("writable", Transaction->IsWritable())
            ("db_location", Transaction->GetDBLocation())
        );
    }
}

NDrive::TEntitySession::~TEntitySession() {
    if (auto eventLogger = NDrive::GetThreadEventLogger(); eventLogger && Transaction) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "destroy_session")
            ("affected_rows", Transaction->GetAffectedRows())
            ("committed", Transaction->GetStatus() == NStorage::ETransactionStatus::Commited)
            ("duration", NJson::ToJson(Transaction->GetDuration()))
            ("messages", GetMessages().GetReport())
            ("queries", NJson::ToJson(Transaction->GetQueries()))
            ("status", ToString(Transaction->GetStatus()))
            ("db_location", Transaction->GetDBLocation())
        );
    }
    bool failedTransaction = Transaction && Transaction->IsWritable() &&
        Transaction->GetStatus() != NStorage::ETransactionStatus::Commited;
    bool heavyTransaction = Transaction && Transaction->GetAffectedRows() > 100000;
    bool longTransaction = Transaction && Transaction->GetDuration() > TDuration::Seconds(1);
    if (longTransaction || heavyTransaction || failedTransaction) {
        NDrive::TEventLog::Log("SessionDestroy", NJson::TMapBuilder
            ("created", Transaction->GetStartInstant().Seconds())
            ("finished", Transaction->GetFinishInstant().Seconds())
            ("duration",  NJson::ToJson(Transaction->GetDuration()))
            ("affected_rows", Transaction->GetAffectedRows())
            ("committed", Transaction->GetStatus() == NStorage::ETransactionStatus::Commited)
            ("messages", GetMessages().GetReport())
            ("queries", NJson::ToJson(Transaction->GetQueries()))
            ("status", ToString(Transaction->GetStatus()))
            ("writable", Transaction->IsWritable())
            ("db_location", Transaction->GetDBLocation())
        );
    }
    if (Transaction && Transaction->IsFailed()) {
        MergeErrorMessages(Transaction->GetErrors(), "transaction");
        if (GetResult() == EDriveSessionResult::Success && Transaction->GetStatus() == NStorage::ETransactionStatus::Failed) {
            SetResult(EDriveSessionResult::TransactionProblem);
        }
        if (GetResult() == EDriveSessionResult::Success && Transaction->GetStatus() == NStorage::ETransactionStatus::SerializationFailed) {
            SetResult(EDriveSessionResult::SerializationProblem);
        }
        if (GetResult() == EDriveSessionResult::TransactionProblem && Transaction->GetStatus() == NStorage::ETransactionStatus::SerializationFailed) {
            SetResult(EDriveSessionResult::SerializationProblem);
        }
        if (GetResult() == EDriveSessionResult::ForeignKeyViolation && Transaction->GetStatus() == NStorage::ETransactionStatus::ForeignKeyViolation) {
            SetResult(EDriveSessionResult::ForeignKeyViolation);
        }
    }
}

TMessagesCollector NDrive::TEntitySession::GetMessages() const {
    TMessagesCollector result = TBase::GetMessages();
    if (Transaction) {
        result.MergeMessages(Transaction->GetErrors(), "transaction");
    }
    return result;
}

EDriveSessionResult NDrive::TEntitySession::GetResult() const {
    if (HasResult() && GetResultRef() == EDriveSessionResult::TransactionProblem && Transaction && Transaction->GetStatus() == NStorage::ETransactionStatus::SerializationFailed) {
        return EDriveSessionResult::SerializationProblem;
    }
    if (HasResult()) {
        return GetResultRef();
    }
    if (Transaction && Transaction->GetStatus() == NStorage::ETransactionStatus::Failed) {
        return EDriveSessionResult::TransactionProblem;
    }
    if (Transaction && Transaction->GetStatus() == NStorage::ETransactionStatus::SerializationFailed) {
        return EDriveSessionResult::SerializationProblem;
    }
    if (Transaction && Transaction->GetStatus() == NStorage::ETransactionStatus::ForeignKeyViolation) {
        return EDriveSessionResult::ForeignKeyViolation;
    }
    return EDriveSessionResult::Success;
}

NStorage::ITransaction::TPtr NDrive::TEntitySession::GetTransaction() {
    if (!Transaction && DeferredTransaction) {
        Transaction = DeferredTransaction();
        auto eventLogger = NDrive::GetThreadEventLogger();
        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "create_transaction")
                ("writable", Transaction ? NJson::ToJson(Transaction->IsWritable()) : NJson::JSON_NULL)
                ("db_location", Transaction ? NJson::ToJson(Transaction->GetDBLocation()) : NJson::JSON_NULL)
            );
        }
    }
    return Yensured(Transaction);
}

bool NDrive::TEntitySession::HasTransaction() const {
    return Transaction != nullptr;
}

bool NDrive::TEntitySession::Commit() {
    if (GetResult() != EDriveSessionResult::Success) {
        AddErrorMessage("Commit", "Tx is dirty");
        return false;
    }

    if (!Transaction) {
        return true;
    }

    if (!Transaction->Commit()) {
        SetErrorInfo("SessionCommit", "COMMIT failed", EDriveSessionResult::TransactionProblem);
        NDrive::TEventLog::Log("SessionCommitFailure", NJson::TMapBuilder
            ("created", Transaction->GetStartInstant().Seconds())
            ("errors", Transaction->GetErrors().GetReport())
            ("db_location", Transaction->GetDBLocation())
        );
        return false;
    }

    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        eventLogger->AddEvent("commit_session");
    }

    CommitPromise.SetValue();
    if (Context) {
        Context->OnAfterCommit();
    }
    return true;
}

bool NDrive::TEntitySession::Rollback() {
    if (!Transaction) {
        return true;
    }

    if (!Transaction->Rollback()) {
        SetErrorInfo("SessionRollback", "ROLLBACK failed", EDriveSessionResult::TransactionProblem);
        NDrive::TEventLog::Log("SessionRollbackFailure", NJson::TMapBuilder
            ("created", Transaction->GetStartInstant().Seconds())
            ("errors", Transaction->GetErrors().GetReport())
            ("db_location", Transaction->GetDBLocation())
        );
        return false;
    }

    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        eventLogger->AddEvent("rollback_session");
    }

    return true;
}

TMaybe<bool> NDrive::TEntitySession::TryLock(TStringBuf id) {
    if (!GetTransaction()) {
        return {};
    }
    auto result = Transaction->TryLock(id);
    if (!result) {
        SetErrorInfo("TryLock", TStringBuilder() << "error while locking " << id, EDriveSessionResult::ResourceLocked);
        return {};
    }
    return result;
}

TMaybe<bool> NDrive::TEntitySession::TrySharedLock(TStringBuf id) {
    if (!Transaction) {
        return {};
    }
    auto result = Transaction->TrySharedLock(id);
    if (!result) {
        SetErrorInfo("TrySharedLock", TStringBuilder() << "error while locking " << id, EDriveSessionResult::ResourceLocked);
        return {};
    }
    return result;
}

void NDrive::TInfoEntitySession::Check() noexcept(false) {
    const auto& statuses = NDrive::HasServer() ? NDrive::GetServer().GetHttpStatusManagerConfig() : Default<THttpStatusManagerConfig>();
    DoExceptionOnFail(statuses);
}

void NDrive::TInfoEntitySession::DoExceptionOnFail(const THttpStatusManagerConfig& config, const ILocalization* localization, TStringBuf message) const {
    auto code = Code.GetOrElse(HTTP_OK);
    auto locale = Locale.GetOrElse(DefaultLocale);
    if (!localization && NDrive::HasServer()) {
        localization = NDrive::GetServer().GetLocalization();
    }
    TString localizedMessage;
    TString localizedTitle;
    if (!localizedMessage && localization && Error) {
        localizedMessage = localization->GetLocalString(locale, "error." + Error.Value + ".message", localizedMessage);
    }
    if (!localizedTitle && localization && Error) {
        localizedTitle = localization->GetLocalString(locale, "error." + Error.Value + ".title", localizedTitle);
    }
    if (!localizedMessage && localization && LocalizedMessageKey) {
        localizedMessage = localization->GetLocalString(locale, LocalizedMessageKey);
    }
    if (!localizedTitle && localization && LocalizedTitleKey) {
        localizedTitle = localization->GetLocalString(locale, LocalizedTitleKey);
    }
    auto result = GetResult();
    if (locale == LegacyLocale) {
        if (!localizedMessage && !DecodeEnumMessage(result, localizedMessage)) {
            return;
        }
    } else {
        if (!localizedMessage && localization) {
            localizedMessage = localization->GetLocalString(locale, TStringBuilder() << "error." << result << ".message");
        }
        if (!localizedTitle && localization) {
            localizedTitle = localization->GetLocalString(locale, TStringBuilder() << "error." << result << ".title", localizedTitle);
        }
    }
    if (!Code && !DecodeEnumCode(result, config, code)) {
        return;
    }
    NJson::TJsonValue landings = NJson::JSON_NULL;
    if (Landing.IsDefined()) {
        landings.AppendValue(Landing);
    }
    throw TCodedException(code)
        .AddInfo("session_info", GetMessages().GetReport())
        .AddPublicInfo("error_details", ErrorDetails)
        .AddPublicInfo("landings", landings)
        .SetErrorCode(Error ? Error.Value : ToString(result))
        .SetLocalizedMessage(localizedMessage)
        .SetLocalizedTitle(localizedTitle)
        << message;
}

EDriveSessionResult NDrive::TInfoEntitySession::GetResult() const {
    return Result.GetOrElse(EDriveSessionResult::Success);
}

EDriveSessionResult NDrive::TInfoEntitySession::DetachResult() {
    EDriveSessionResult result = GetResult();
    ClearErrors();
    return result;
}

TMessagesCollector NDrive::TInfoEntitySession::GetMessages() const {
    return ErrorsMessenger;
}

void NDrive::TInfoEntitySession::AddEvent(NJson::TJsonValue&& ev) {
    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        eventLogger->AddEvent(std::move(ev));
    }
}

bool NDrive::TInfoEntitySession::EventsEnabled() const {
    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        return eventLogger->ShouldRecordEventLog();
    } else {
        return false;
    }
}

void NDrive::TInfoEntitySession::ClearErrors() {
    ErrorsInfoCounter = 0;
    ErrorsMessenger.ClearMessages();
    Result = EDriveSessionResult::Success;
}

void NDrive::TInfoEntitySession::SetErrorLanding(const TString& source, const NJson::TJsonValue& landing, EDriveSessionResult result) {
    ASSERT_WITH_LOG(++ErrorsInfoCounter == 1) << source << "/" << landing << ": " << GetStringReport() << Endl;
    ErrorsMessenger.AddMessage(source, landing.GetStringRobust());
    SetLanding(landing);
    if (!Result || Result == EDriveSessionResult::Success) {
        Result = result;
    }
}

void NDrive::TInfoEntitySession::SetErrorInfo(const TString& source, const TMessagesCollector& collector, EDriveSessionResult result) {
    ASSERT_WITH_LOG(++ErrorsInfoCounter == 1) << collector.GetStringReport() << ": " << GetStringReport() << Endl;
    ErrorsMessenger.MergeMessages(collector, source);
    if (!Result || Result == EDriveSessionResult::Success) {
        Result = result;
    }
}

void NDrive::TInfoEntitySession::SetErrorInfo(const TString& source, const TMessagesCollector& collector, TError&& error, EDriveSessionResult result) {
    Error = std::move(error);
    SetErrorInfo(source, collector, result);
}

void NDrive::TInfoEntitySession::SetErrorInfo(const TString& source, const TString& info, EDriveSessionResult result) {
    ASSERT_WITH_LOG(++ErrorsInfoCounter == 1) << source << "/" << info << ": " << GetStringReport() << Endl;
    ErrorsMessenger.AddMessage(source, info);
    if (!Result || Result == EDriveSessionResult::Success) {
        Result = result;
    }
}

void NDrive::TInfoEntitySession::SetErrorInfo(const TString& source, const TString& info, TError&& error, EDriveSessionResult result) {
    Error = std::move(error);
    SetErrorInfo(source, info, result);
}

bool NDrive::DecodeEnumMessage(const EDriveSessionResult value, TString& localizedMessage) {
    localizedMessage = "";
    switch (value) {
        case EDriveSessionResult::Success:
            return false;
        case EDriveSessionResult::InternalError:
        case EDriveSessionResult::InconsistencySystem:
        case EDriveSessionResult::SerializationProblem:
        case EDriveSessionResult::TransactionProblem:
        case EDriveSessionResult::DataCorrupted:
        case EDriveSessionResult::OfferCannotRead:
            localizedMessage = NDrive::TLocalization::ServerProblems();
            break;
        case EDriveSessionResult::OfferNotFound:
            localizedMessage = NDrive::TLocalization::OfferProblems();
            break;
        case EDriveSessionResult::InconsistencyOffer:
            localizedMessage = NDrive::TLocalization::OfferInconsistency();
            break;
        case EDriveSessionResult::InconsistencyUser:
            localizedMessage = NDrive::TLocalization::InconsistencyUser();
            break;
        case EDriveSessionResult::OfferExpired:
            localizedMessage = NDrive::TLocalization::OfferExpired();
            break;
        case EDriveSessionResult::ResourceLocked:
            localizedMessage = NDrive::TLocalization::CarInUsing();
            break;
        case EDriveSessionResult::IncorrectRequest:
            localizedMessage = NDrive::TLocalization::IncorrectRequest();
            break;
        case EDriveSessionResult::NoUserPermissions:
            localizedMessage = NDrive::TLocalization::NoPermissions();
            break;
        case EDriveSessionResult::PaymentRequired:
        case EDriveSessionResult::DepositFails:
            localizedMessage = NDrive::TLocalization::PaymentRequired();
            break;
        case EDriveSessionResult::RequiredDepositIsNotHeld:
            localizedMessage = NDrive::TLocalization::RequiredDepositIsNotHeld();
            break;
        case EDriveSessionResult::IncorrectCarPosition:
            localizedMessage = NDrive::TLocalization::IncorrectCarPosition();
            break;
        case EDriveSessionResult::IncorrectCarStatus:
            localizedMessage = NDrive::TLocalization::IncorrectCarStatus();
            break;
        case EDriveSessionResult::IncorrectCarSignal:
            localizedMessage = NDrive::TLocalization::IncorrectCarSignal();
            break;
        case EDriveSessionResult::ResourceHasMajorProblems:
            localizedMessage = NDrive::TLocalization::ResourceHasMajorProblems();
            break;
        case EDriveSessionResult::IncorrectTagObjectType:
            localizedMessage = NDrive::TLocalization::IncorrectTagObjectType();
            break;
        case EDriveSessionResult::LockedResourcesLimitEnriched:
            localizedMessage = NDrive::TLocalization::LockedResourcesLimitEnriched();
            break;
        case EDriveSessionResult::UserHaveRentedCar:
            localizedMessage = NDrive::TLocalization::UserHaveRentedCar();
            break;
        case EDriveSessionResult::IncorrectCarTags:
            localizedMessage = NDrive::TLocalization::IncorrectCarTags();
            break;
        case EDriveSessionResult::CarIsBusy:
            localizedMessage = NDrive::TLocalization::CarIsBusy();
            break;
        case EDriveSessionResult::NoPermissionsForPerform:
            localizedMessage = NDrive::TLocalization::NoPermissionsForPerform();
            break;
        case EDriveSessionResult::IncorrectFilter:
            localizedMessage = NDrive::TLocalization::IncorrectFilter();
            break;
        case EDriveSessionResult::ScannerExpired:
            localizedMessage = NDrive::TLocalization::ScannerExpired();
            break;
        case EDriveSessionResult::IncorrectHeadId:
            localizedMessage = NDrive::TLocalization::IncorrectHeadId();
            break;
        case EDriveSessionResult::CarCooldown:
            localizedMessage = NDrive::TLocalization::CarCooldown();
            break;
        case EDriveSessionResult::DuplicatePromoCode:
            localizedMessage = NDrive::TLocalization::DuplicatePromoCode();
            break;
        case EDriveSessionResult::RequestForUser:
            localizedMessage = NDrive::TLocalization::RequestForUser();
            break;
        case EDriveSessionResult::IncorrectUserReply:
            localizedMessage = NDrive::TLocalization::IncorrectUserReply();
            break;
        case EDriveSessionResult::UserShouldBeBlocked:
            localizedMessage = NDrive::TLocalization::UserShouldBeBlocked();
            break;
        case EDriveSessionResult::Unauthenticated:
            localizedMessage = NDrive::TLocalization::BlackboxProblems();
            break;
        case EDriveSessionResult::ApplicationOutdated:
            localizedMessage = NDrive::TLocalization::TooOldApplication();
            break;
        case EDriveSessionResult::ForeignKeyViolation:
            localizedMessage = NDrive::TLocalization::ResourceCantBeDeleted();
            break;
    }
    return true;
}

bool NDrive::DecodeEnumCode(const EDriveSessionResult value, const THttpStatusManagerConfig& config, int& code) {
    code = (int)HTTP_OK;
    switch (value) {
        case EDriveSessionResult::Success:
            return false;
        case EDriveSessionResult::InternalError:
        case EDriveSessionResult::InconsistencySystem:
        case EDriveSessionResult::TransactionProblem:
        case EDriveSessionResult::DataCorrupted:
        case EDriveSessionResult::OfferCannotRead:
            code = config.UnknownErrorStatus;
            break;
        case EDriveSessionResult::OfferNotFound:
        case EDriveSessionResult::IncorrectUserReply:
        case EDriveSessionResult::RequestForUser:
        case EDriveSessionResult::InconsistencyOffer:
        case EDriveSessionResult::InconsistencyUser:
        case EDriveSessionResult::OfferExpired:
        case EDriveSessionResult::ResourceLocked:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::IncorrectRequest:
            code = config.SyntaxErrorStatus;
            break;
        case EDriveSessionResult::NoUserPermissions:
        case EDriveSessionResult::UserShouldBeBlocked:
            code = config.PermissionDeniedStatus;
            break;
        case EDriveSessionResult::PaymentRequired:
        case EDriveSessionResult::DepositFails:
        case EDriveSessionResult::RequiredDepositIsNotHeld:
            code = config.PaymentRequiredState;
            break;
        case EDriveSessionResult::IncorrectCarPosition:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::IncorrectCarStatus:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::IncorrectCarSignal:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::ResourceHasMajorProblems:
            code = config.ConflictRequest;
            break;
        case EDriveSessionResult::IncorrectTagObjectType:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::LockedResourcesLimitEnriched:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::UserHaveRentedCar:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::IncorrectCarTags:
            code = config.UnknownErrorStatus;
            break;
        case EDriveSessionResult::CarIsBusy:
            code = config.ConflictRequest;
            break;
        case EDriveSessionResult::NoPermissionsForPerform:
            code = config.PermissionDeniedStatus;
            break;
        case EDriveSessionResult::IncorrectFilter:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::ScannerExpired:
            code = config.EmptyRequestStatus;
            break;
        case EDriveSessionResult::IncorrectHeadId:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::CarCooldown:
            code = config.TooManyRequestsStatus;
            break;
        case EDriveSessionResult::DuplicatePromoCode:
            code = config.UserErrorState;
            break;
        case EDriveSessionResult::SerializationProblem:
            code = HTTP_LOOP_DETECTED;
            break;
        case EDriveSessionResult::Unauthenticated:
            code = HTTP_UNAUTHORIZED;
            break;
        case EDriveSessionResult::ApplicationOutdated:
            code = HTTP_UPGRADE_REQUIRED;
            break;
        case EDriveSessionResult::ForeignKeyViolation:
            code = config.ResourceIsBusy;
            break;
    }
    return true;
}

void TDatabaseSessionConstructor::EnableTransactionNames() {
    TransactionNamesEnabled = true;
}

void TDatabaseSessionConstructor::DisableTransactionNames() {
    TransactionNamesEnabled = false;
}

NStorage::ITransaction::TPtr BuildTxImpl(const NStorage::IDatabase* database, bool readonly, bool repeatableRead, TDuration lockTimeout, TDuration statementTimeout) {
    NStorage::TCreateTransactionOptions options;
    options.ReadOnly = readonly;
    options.RepeatableRead = repeatableRead;
    options.LockTimeout = lockTimeout;
    options.StatementTimeout = statementTimeout;
    if (TransactionNamesEnabled) {
        auto source = NDrive::TEventLog::GetSource();
        if (source) {
            source.append('@');
            source.append(HostName());
            source.resize(std::min<size_t>(source.size(), 64));
            options.Name = std::move(source);
        }
    }
    return Yensured(database)->CreateTransaction(options);
}

NDrive::TEntitySession TDatabaseSessionConstructor::BuildTxImpl(
    NSQL::TTransactionTraits traits,
    TDuration lockTimeout = TDuration::Zero(),
    TDuration statementTimeout = TDuration::Zero()
) const {
    bool deferred = traits & NSQL::Deferred;
    bool readonly = traits & NSQL::ReadOnly;
    bool repeatableRead = traits & NSQL::RepeatableRead;
    if (deferred) {
        return NDrive::TEntitySession([
            database = Database,
            readonly,
            repeatableRead,
            lockTimeout,
            statementTimeout
        ] {
            return ::BuildTxImpl(database.Get(), readonly, repeatableRead, lockTimeout, statementTimeout);
        }, nullptr);
    } else {
        return ::BuildTxImpl(Database.Get(), readonly, repeatableRead, lockTimeout, statementTimeout);
    }
}

NDrive::TEntitySession TDatabaseSessionConstructor::BuildSession(bool readonly, bool repeatableRead, TDuration lockTimeout, TDuration statementTimeout) const {
    return ::BuildTxImpl(Database.Get(), readonly, repeatableRead, lockTimeout, statementTimeout);
}

bool ParseQueryResult(NStorage::IQueryResult::TPtr queryResult, NDrive::TEntitySession* session) {
    if (!queryResult) {
        if (session) {
            session->SetErrorInfo("ParseQueryResult", "null QueryResult", EDriveSessionResult::TransactionProblem);
        }
        return false;
    }
    if (!queryResult->IsSucceed()) {
        if (session) {
            session->SetErrorInfo("ParseQueryResult", "unsuccessful QueryResult", EDriveSessionResult::TransactionProblem);
        }
        return false;
    }
    return true;
}

bool ParseQueryResult(NStorage::IQueryResult::TPtr queryResult, NDrive::TEntitySession& session) {
    return ParseQueryResult(queryResult, &session);
}
