#pragma once

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_value.h>

#include <rtline/library/storage/sql/transaction.h>

#include <util/generic/cast.h>

class ILocalization;
struct THttpStatusManagerConfig;
enum class ELocalization;

enum class EDriveSessionResult {
    Success = 0 /* "success" */,
    ApplicationOutdated /* "application_outdated" */,
    InconsistencyUser /* "inconsistency_user" */,
    InconsistencySystem /* "inconsistency_system" */,
    SerializationProblem /* "serialization_problem" */,
    TransactionProblem /* "transaction_problem" */,
    InternalError /* "internal_error" */,
    ResourceLocked /* "resource_locked" */,
    NoUserPermissions /* "no_user_permissions" */,
    DataCorrupted /* "data_corrupted" */,
    IncorrectRequest /* "incorrect_request" */,

    InconsistencyOffer /* "inconsistency_offer" */,
    OfferNotFound /* "offer_not_found" */,
    OfferCannotRead /* "offer_cannot_read" */,
    OfferExpired /* "offer_expired" */,

    PaymentRequired /* "payment_required" */,
    IncorrectCarPosition /* "incorrect_car_position" */,
    IncorrectCarSignal /* "incorrect_car_signal" */,
    IncorrectCarStatus /* "incorrect_car_status" */,
    CarCooldown /* "command_cooldown" */,
    ResourceHasMajorProblems /* "resource_major_problems" */,
    IncorrectTagObjectType /* "incorrect_tag_object_type" */,
    LockedResourcesLimitEnriched /* "locked_resources_limit_enriched" */,

    UserHaveRentedCar /* "user_have_rented_car" */,
    CarIsBusy /* "car_is_busy" */,
    IncorrectCarTags /* "incorrect_car_tags" */,

    NoPermissionsForPerform /* "no_permissions_for_perform" */,
    IncorrectFilter /* "incorrect_filter" */,
    ScannerExpired /* "scanner_expired" */,
    IncorrectHeadId /* "incorrect_head_id" */,

    DuplicatePromoCode /* "promo_code_already_been_used" */,
    DepositFails /* "deposit_fails" */,
    RequiredDepositIsNotHeld /* required_deposit_is_not_held */,

    RequestForUser /* "request_for_user" */,
    IncorrectUserReply /* "incorrect_user_reply" */,
    UserShouldBeBlocked /* "user_should_be_blocked" */,
    Unauthenticated /* "unauthenticated" */,

    ForeignKeyViolation /* "foreign_key_violation" */,
};

namespace NDrive {
    struct TError {
    public:
        TString Value;

    public:
        TError() = default;

        explicit operator bool() const {
            return !Value.empty();
        }
    };
    template <class T>
    TError MakeError(T&& value) {
        TError result;
        result.Value = std::forward<T>(value);
        return result;
    }

    class TSessionRequestContext {
    private:
        R_FIELD(TString, UserChoice);
    };

    class ISessionContext {
    public:
        virtual ~ISessionContext() {}
        virtual void OnAfterCommit() = 0;
    };

    class TInfoEntitySession {
    private:
        R_FIELD(TString, OriginatorId);
        R_FIELD(TString, Comment);
        R_FIELD(NJson::TJsonValue, ErrorDetails);
        R_FIELD(NJson::TJsonValue, Landing, {});
        R_FIELD(TString, LocalizedMessageKey);
        R_FIELD(TString, LocalizedTitleKey);
        R_FIELD(TError, Error);
        R_FIELD(bool, AlertOnDirty, false);
        R_OPTIONAL(int, Code);
        R_OPTIONAL(ELocalization, Locale);
        R_OPTIONAL(EDriveSessionResult, Result);

    private:
        TSessionRequestContext RequestContext;
        TMessagesCollector ErrorsMessenger;
        ui64 ErrorsInfoCounter = 0;

    public:
        TInfoEntitySession() = default;
        virtual ~TInfoEntitySession();

        TInfoEntitySession(TInfoEntitySession&& other) = default;
        TInfoEntitySession(const TInfoEntitySession& other) = delete;

        TInfoEntitySession& operator=(TInfoEntitySession&& other) = default;
        TInfoEntitySession& operator=(const TInfoEntitySession& other) = delete;

        const TSessionRequestContext& GetRequestContext() const {
            return RequestContext;
        }
        TSessionRequestContext& MutableRequestContext() {
            return RequestContext;
        }

        virtual EDriveSessionResult GetResult() const;
        EDriveSessionResult DetachResult();
        void DoExceptionOnFail(const THttpStatusManagerConfig& config, const ILocalization* localization = nullptr, TStringBuf message = {}) const;

        virtual TMessagesCollector GetMessages() const;
        NJson::TJsonValue GetReport() const {
            return GetMessages().GetReport();
        }
        TString GetStringReport() const {
            return GetMessages().GetStringReport();
        }

        void AddEvent(NJson::TJsonValue&& ev);
        bool EventsEnabled() const;

        void Check() noexcept(false);
        void ClearErrors();
        void SetErrorInfo(const TString& source, const TMessagesCollector& collector, EDriveSessionResult result = EDriveSessionResult::InternalError);
        void SetErrorInfo(const TString& source, const TMessagesCollector& collector, TError&& error, EDriveSessionResult result = EDriveSessionResult::InternalError);
        void SetErrorInfo(const TString& source, const TString& info, EDriveSessionResult result = EDriveSessionResult::InternalError);
        void SetErrorInfo(const TString& source, const TString& info, TError&& error, EDriveSessionResult result = EDriveSessionResult::InternalError);
        void SetErrorLanding(const TString& source, const NJson::TJsonValue& landing, EDriveSessionResult result = EDriveSessionResult::RequestForUser);

        void AddErrorMessage(const TString& source, const TString& info) {
            ErrorsMessenger.AddMessage<TLOG_DEBUG>(source, info);
        }
        void MergeErrorMessages(const TMessagesCollector& collector, const TString& comment) {
            ErrorsMessenger.MergeMessages(collector, comment);
        }
        TMessagesCollector& MutableMessages() {
            return ErrorsMessenger;
        }
    };

    class TEntitySession: public TInfoEntitySession {
    private:
        using TBase = TInfoEntitySession;

    public:
        using TDeferredTransaction = std::function<NStorage::ITransaction::TPtr()>;

    private:
        TDeferredTransaction DeferredTransaction;
        NStorage::ITransaction::TPtr Transaction;
        NThreading::TLazyPromise CommitPromise;
        TAtomicSharedPtr<ISessionContext> Context;

    public:
        TEntitySession() = default;
        TEntitySession(TDeferredTransaction&& deferred, TAtomicSharedPtr<ISessionContext> context);
        TEntitySession(NStorage::ITransaction::TPtr transaction, TAtomicSharedPtr<ISessionContext> context = nullptr);
        ~TEntitySession();

        TEntitySession(TEntitySession&& other) = default;
        TEntitySession(const TEntitySession& other) = delete;

        TEntitySession& operator=(TEntitySession&& other) = default;
        TEntitySession& operator=(const TEntitySession& other) = delete;

        explicit operator bool() const {
            return Transaction || DeferredTransaction;
        }

        NStorage::ITransactionBase* operator->() {
            return GetTransaction().Get();
        }
        NStorage::ITransactionBase& operator*() {
            return *GetTransaction();
        }

        virtual TMessagesCollector GetMessages() const override;
        virtual EDriveSessionResult GetResult() const override;

        NStorage::ITransaction::TPtr GetTransaction();
        bool HasTransaction() const;

        [[nodiscard]] bool Commit();
        [[nodiscard]] bool Rollback();

        [[nodiscard]] TMaybe<bool> TryLock(TStringBuf id);
        [[nodiscard]] TMaybe<bool> TrySharedLock(TStringBuf id);

        NThreading::TFuture<void> Committed() {
            return CommitPromise.GetFuture();
        }

        template <class T>
        auto GetContextAs() const {
            return std::dynamic_pointer_cast<T>(Context);
        }
    };

    bool DecodeEnumMessage(const EDriveSessionResult value, TString& localizedMessage);
    bool DecodeEnumCode(const EDriveSessionResult value, const THttpStatusManagerConfig& config, int& code);
}

template <class D>
class THolder<NDrive::TEntitySession, D> {
};

template <class C, class D>
class TSharedPtr<NDrive::TEntitySession, C, D> {
};

NStorage::ITransaction::TPtr BuildTxImpl(const NStorage::IDatabase* database, bool readonly, bool repeatableRead, TDuration lockTimeout, TDuration statementTimeout);

class TDatabaseSessionConstructor {
private:
    NDrive::TEntitySession BuildTxImpl(NSQL::TTransactionTraits traits, TDuration lockTimeout, TDuration statementTimeout) const;

protected:
    NStorage::IDatabase::TPtr Database;

public:
    static void EnableTransactionNames();
    static void DisableTransactionNames();

public:
    TDatabaseSessionConstructor(NStorage::IDatabase::TPtr database)
        : Database(database)
    {
    }

    const NStorage::IDatabase& GetDatabase() const {
        Y_ENSURE_BT(Database);
        return *Database;
    }
    NStorage::IDatabase::TPtr GetDatabasePtr() const {
        return Database;
    }

    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx(
        TDuration lockTimeout = TDuration::Zero(),
        TDuration statementTimeout = TDuration::Zero()
    ) const {
        static_assert(bool(traits & NSQL::Writable) ^ bool(traits & NSQL::ReadOnly), "transaction should be either Writable or ReadOnly");
        return BuildTxImpl(traits, lockTimeout, statementTimeout);
    }
    NDrive::TEntitySession BuildSession(
        bool readonly = false,
        bool repeatableRead = false,
        TDuration lockTimeout = TDuration::Zero(),
        TDuration statementTimeout = TDuration::Zero()
    ) const;
};

bool ParseQueryResult(NStorage::IQueryResult::TPtr queryResult, NDrive::TEntitySession* session = nullptr);
bool ParseQueryResult(NStorage::IQueryResult::TPtr queryResult, NDrive::TEntitySession& session);
