#include "config.h"

#include <library/cpp/mediator/global_notifications/system_status.h>

namespace NDevicesSnapshotManager {
    namespace {
        template<typename TKey, typename TValue>
        TMap<TKey, TValue> InitMap(const TYandexConfig::Section& section, const TString& emptyKeyAlias) {
            auto children = section.GetAllChildren();
            TMap<TKey, TValue> result;
            for (const auto& [sectionName, sectionPtr] : children) {
                TKey key{};
                if (sectionName != emptyKeyAlias) {
                    key = FromString<TKey>(sectionName);
                }
                TValue value;
                value.Init(*sectionPtr);
                result[key] = std::move(value);
            }
            return result;
        }

        template<typename TKey, typename TValue>
        void DumpMap(const TMap<TKey, TValue>& map, const TKey& emptyKeyAlias, IOutputStream& os) {
            for (const auto& [key, value] : map) {
                auto dumpedKey = key;
                if (!dumpedKey) {
                    dumpedKey = emptyKeyAlias;
                }
                os << "<" << ToString(dumpedKey) << ">" << Endl;
                value.ToString(os);
                os << "</" << ToString(dumpedKey) << ">" << Endl;
            }
        }
    }

    void TFetchOptions::Init(const TYandexConfig::Section& section) {
        UpdatePeriod = section.GetDirectives().Value("UpdatePeriod", UpdatePeriod);
        EnableSecondStage = section.GetDirectives().Value("EnableSecondStage", EnableSecondStage);
    }

    void TFetchOptions::ToString(IOutputStream& os) const {
        os << "UpdatePeriod: " << UpdatePeriod << Endl;
        os << "EnableSecondStage: " << EnableSecondStage << Endl;
    }

    void TOptions::Init(const TYandexConfig::Section& section) {
        auto children = section.GetAllChildren();
        {
            auto it = children.find("HeartbeatOptions");
            if (it != children.end()) {
                HeartbeatOptions = InitMap<TString, THeartbeatsOptions>(*it->second, "default");
            }
        }
        {
            auto it = children.find("LocationOptions");
            if (it != children.end()) {
                LocationOptions = InitMap<TString, TLocationOptions>(*it->second, "raw");
            }
        }
        {
            auto it = children.find("SensorOptions");
            if (it != children.end()) {
                SensorOptions = InitMap<NDrive::TSensorId, TSensorOptions>(*it->second, "");
            }
        }
        FutureFilter = section.GetDirectives().Value("FutureFilter", FutureFilter);
        SensorTimeout = section.GetDirectives().Value("SensorTimeout", SensorTimeout);
        UpdatePeriod = section.GetDirectives().Value("UpdatePeriod", UpdatePeriod);
        HashUpdatePeriod = section.GetDirectives().Value("HashUpdatePeriod", HashUpdatePeriod);
        EnableFirstStageFetching = section.GetDirectives().Value("EnableFirstStageFetching", EnableFirstStageFetching);
        EnableSecondStageFetching = section.GetDirectives().Value("EnableSecondStageFetching", EnableSecondStageFetching);
        AssertCorrectConfig(EnableFirstStageFetching || EnableSecondStageFetching, "at least one stage should be enabled");
    }

    void TOptions::ToString(IOutputStream& os) const {
        os << "<HeartbeatOptions>" << Endl;
        DumpMap<TString>(HeartbeatOptions, "default", os);
        os << "</HeartbeatOptions>" << Endl;

        os << "<LocationOptions>" << Endl;
        DumpMap<TString>(LocationOptions, "raw", os);
        os << "</LocationOptions>" << Endl;

        os << "<SensorOptions>" << Endl;
        DumpMap<NDrive::TSensorId>(SensorOptions, NDrive::TSensorId{}, os);
        os << "</SensorOptions>" << Endl;

        os << "FutureFilter: " << FutureFilter << Endl;
        os << "SensorTimeout: " << SensorTimeout << Endl;
        os << "UpdatePeriod: " << UpdatePeriod << Endl;
        os << "HashUpdatePeriod: " << HashUpdatePeriod << Endl;
        os << "EnableFirstStageFetching: " << EnableFirstStageFetching << Endl;
        os << "EnableSecondStageFetching: " << EnableSecondStageFetching << Endl;
    }
}
