#include "fetchers.h"

#include <drive/telematics/api/sensor/interface.h>

#include <rtline/util/algorithm/iterator.h>

NThreading::TFuture<NDrive::ISensorApi::TSensors> TSensorsFetcher::FetchFirst(const TFetchedKey& id, TDuration timeout) {
    NDrive::ISensorApi::TSensorsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    return Server.GetSensorApi()->GetSensor(id, queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> TSensorsFetcher::FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) {
    NDrive::ISensorApi::TSensorsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetMaxKeysPerQuery(maxKeysPerQuery);
    queryOptions.MutableFetchOptions().SetQuorum(Quorum);
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    return Server.GetSensorApi()->GetSensor(imeis, id, queryOptions);
}

void TSensorsFetcher::Prepare() {
    TBase::DataByKey.clear();
    for (auto&& i : NDevicesSnapshotManager::AllSensors) {
        TBase::DataByKey.emplace(i, nullptr);
    }
}

void TSensorsFetcher::FinishFetching(const TVector<TString>& imeis, const TDuration timeout) {
    Y_ASSERT(std::is_sorted(imeis.begin(), imeis.end()));
    for (auto&&[id, asyncSensors] : MultiAsync) {
        if (asyncSensors.Wait(FetchStart + timeout) && asyncSensors.HasValue()) {
            const auto& values = asyncSensors.GetValue();
            auto i = Sensors.begin();
            auto j = values.begin();
            for (auto&& imei : imeis) {
                if (!Advance(i, Sensors.end(), imei)) {
                    i = Sensors.emplace(imei, NDrive::ISensorApi::TMultiSensor()).first;
                }
                NDrive::ISensorApi::TMultiSensor& multisensor = i->second;
                if (Advance(j, values.end(), imei)) {
                    const NDrive::ISensorApi::TMultiSensor& mss = j->second;
                    multisensor.insert(multisensor.end(), mss.begin(), mss.end());
                }
            }
            LastUpdates.at(id) = FetchStart;
        } else {
            ERROR_LOG << "an exception occurred during " << id << " sensor request: " << NThreading::GetExceptionMessage(asyncSensors) << Endl;
        }
    }
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> THeartbeatsFetcher::FetchFirst(const TFetchedKey& id, TDuration timeout) {
    NDrive::ISensorApi::THeartbeatsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    queryOptions.SetName(id);
    return Server.GetSensorApi()->GetHeartbeats(queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> THeartbeatsFetcher::FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) {
    NDrive::ISensorApi::THeartbeatsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetMaxKeysPerQuery(maxKeysPerQuery);
    queryOptions.MutableFetchOptions().SetQuorum(Quorum);
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    queryOptions.SetName(id);
    return Server.GetSensorApi()->GetHeartbeats(imeis, queryOptions);
}

void THeartbeatsFetcher::Prepare() {
    TBase::DataByKey.clear();
    TBase::DataByKey.emplace(NDrive::DefaultHeartbeatName, &Heartbeats);
    TBase::DataByKey.emplace(NDrive::ConfiguratorHeartbeatName, &ConfiguratorHeartbeats);
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> TLocationsFetcher::FetchFirst(const TFetchedKey& id, TDuration timeout) {
    NDrive::ISensorApi::TLocationsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    queryOptions.SetName(id);
    return Server.GetSensorApi()->GetLocations(queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> TLocationsFetcher::FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) {
    NDrive::ISensorApi::TLocationsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetMaxKeysPerQuery(maxKeysPerQuery);
    queryOptions.MutableFetchOptions().SetQuorum(Quorum);
    queryOptions.MutableFetchOptions().SetTimeout(timeout);
    queryOptions.SetName(id);
    return Server.GetSensorApi()->GetLocations(imeis, queryOptions);
}


void TLocationsFetcher::Prepare() {
    TBase::DataByKey.clear();
    TBase::DataByKey.emplace(NDrive::RawLocationName, &RawLocations);
    TBase::DataByKey.emplace(NDrive::LBSLocationName, &LBSLocations);
    TBase::DataByKey.emplace(NDrive::LinkedLocationName, &LinkedLocations);
    TBase::DataByKey.emplace(NDrive::HeadLocationName, &HeadLocations);
    TBase::DataByKey.emplace(NDrive::BeaconsLocationName, &BeaconsLocations);
    TBase::DataByKey.emplace(NDrive::GeocodedLocationName, &GeocodedLocations);
}
