#pragma once

#include <drive/backend/device_snapshot/config/config.h>

#include <drive/backend/abstract/frontend.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/server/location/location.h>

#include <random>

template <class TFetcherConfig, class TFetchedData, class TDataKey, class TFetchedKeyExt>
class IFetcher {
protected:
    using TConfig = TFetcherConfig;
    using TData = TFetchedData;
    using TKey = TDataKey;
    using TFetchedKey = TFetchedKeyExt;
    using TDataSet = TMap<TKey, TData>;
    using TFetchedDataSet = TMap<TDataKey, TData>;

protected:
    const NDrive::IServer& Server;
    const TString FetcherName;
    TMap<TFetchedKey, TDataSet*> DataByKey;
    TMap<TFetchedKey, TFetcherConfig>& FetcherConfigs;
    TMap<TFetchedKey, TInstant>& LastUpdates;
    TMap<TFetchedKey, NThreading::TFuture<TFetchedDataSet>> MultiAsync;
    const TInstant FetchStart = Now();

    ui32 MaxSecondStageInvocations = 5;
    ui32 SecondStageInvocations = 0;
    float Quorum = 0.5;

protected:
    virtual NThreading::TFuture<TFetchedDataSet> FetchFirst(const TFetchedKey& id, TDuration timeout) = 0;
    virtual NThreading::TFuture<TFetchedDataSet> FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) = 0;


public:
    IFetcher(const NDrive::IServer& server, const TString& fetcherName, TMap<TFetchedKey, TFetcherConfig>& fetcherConfigs, TMap<TFetchedKey, TInstant>& lastUpdates)
        : Server(server)
        , FetcherName(fetcherName)
        , FetcherConfigs(fetcherConfigs)
        , LastUpdates(lastUpdates)
    {
    }

    void PrefetchFirst(TDuration timeout) {
        for (auto&&[name, p] : DataByKey) {
            const TInstant updated = LastUpdates[name];
            auto i = FetcherConfigs.find(name);
            const TFetcherConfig& options = (i != FetcherConfigs.end()) ? i->second : Default<TFetcherConfig>();
            if (updated + options.UpdatePeriod < FetchStart) {
                INFO_LOG << "first stage fetching of " << FetcherName << ": " << name << Endl;
                MultiAsync.emplace(name, FetchFirst(name, timeout));
            }
        }
    }

    void PrefetchSecond(const TVector<TString>& imeis, const TDuration timeout, TMaybe<ui32> maxKeysPerQuery = {}) {
        auto data = MakeVector<std::pair<TFetchedKey, TDataSet*>>(DataByKey);
        {
            std::mt19937_64 g(Now().MicroSeconds());
            std::shuffle(data.begin(), data.end(), g);
        }
        for (auto&&[name, p] : data) {
            auto i = FetcherConfigs.find(name);
            const TFetcherConfig& options = (i != FetcherConfigs.end()) ? i->second : Default<TFetcherConfig>();
            bool shouldLaunch = false;
            auto it = MultiAsync.find(name);
            if (it == MultiAsync.end()) {
                const TInstant updated = LastUpdates[name];
                shouldLaunch = (updated + options.UpdatePeriod < FetchStart);
            } else {
                auto& asyncValue = it->second;
                shouldLaunch = !asyncValue.Initialized() || !asyncValue.Wait(FetchStart + timeout) || !asyncValue.HasValue();
            }
            if (shouldLaunch) {
                if (it != MultiAsync.end()) {
                    auto& asyncValue = it->second;
                    WARNING_LOG << "an exception occurred during " + FetcherName + " request: " << NThreading::GetExceptionMessage(asyncValue) << Endl;
                }
                if (options.EnableSecondStage && SecondStageInvocations < MaxSecondStageInvocations) {
                    INFO_LOG << "second stage fetching of " << FetcherName << ": " << name << Endl;
                    MultiAsync[name] = FetchSecond(imeis, name, timeout, maxKeysPerQuery);
                    SecondStageInvocations += 1;
                }
            }
        }
    }

    virtual void FinishFetching(const TVector<TString>& imeis, const TDuration timeout) = 0;
};

template <class TFetcherConfig, class TFetchedData, class TDataKey, class TFetchedKeyExt>
class ISimpleReadFetcher: public IFetcher<TFetcherConfig, TFetchedData, TDataKey, TFetchedKeyExt> {
private:
    using TBase = IFetcher<TFetcherConfig, TFetchedData, TDataKey, TFetchedKeyExt>;
protected:
    using TBase::MultiAsync;
    using TBase::DataByKey;
    using TBase::FetchStart;
    using TBase::FetcherName;
    using TBase::LastUpdates;
    using TFetchedKey = TFetchedKeyExt;

public:
    using TBase::TBase;

    virtual void FinishFetching(const TVector<TString>& /*imeis*/, const TDuration timeout) override {
        for (auto&&[name, asyncValue] : MultiAsync) {
            if (asyncValue.Wait(FetchStart + timeout) && asyncValue.HasValue()) {
                auto p = DataByKey.find(name);
                CHECK_WITH_LOG(p != DataByKey.end());
                CHECK_WITH_LOG(p->second);
                *(p->second) = asyncValue.ExtractValue();
                LastUpdates.at(name) = FetchStart;
            } else {
                ERROR_LOG << "an exception occurred during " << name << " " << FetcherName << " request: " << NThreading::GetExceptionMessage(asyncValue) << Endl;
            }
        }
    }
};

class THeartbeatsFetcher: public ISimpleReadFetcher<NDevicesSnapshotManager::THeartbeatsOptions, NDrive::THeartbeat, TString, TString> {
private:
    using TBase = ISimpleReadFetcher<NDevicesSnapshotManager::THeartbeatsOptions, NDrive::THeartbeat, TString, TString>;

private:
    R_READONLY(TDataSet, Heartbeats);
    R_READONLY(TDataSet, ConfiguratorHeartbeats);

protected:
    virtual NThreading::TFuture<TFetchedDataSet> FetchFirst(const TFetchedKey& id, TDuration timeout) override;
    virtual NThreading::TFuture<TFetchedDataSet> FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) override;

public:
    using TBase::TBase;

    void Prepare();
};

class TLocationsFetcher: public ISimpleReadFetcher<NDevicesSnapshotManager::TLocationOptions, NDrive::TLocation, TString, TString> {
private:
    using TBase = ISimpleReadFetcher<NDevicesSnapshotManager::TLocationOptions, NDrive::TLocation, TString, TString>;

private:
    R_READONLY(TDataSet, RawLocations);
    R_READONLY(TDataSet, LinkedLocations);
    R_READONLY(TDataSet, LBSLocations);
    R_READONLY(TDataSet, HeadLocations);
    R_READONLY(TDataSet, BeaconsLocations);
    R_READONLY(TDataSet, GeocodedLocations);

protected:
    virtual NThreading::TFuture<TFetchedDataSet> FetchFirst(const TFetchedKey& id, TDuration timeout) override;
    virtual NThreading::TFuture<TFetchedDataSet> FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) override;

public:
    using TBase::TBase;

    void Prepare();
};

class TSensorsFetcher: public IFetcher<NDevicesSnapshotManager::TSensorOptions, NDrive::TMultiSensor, TString, NDrive::TSensorId> {
private:
    using TBase = IFetcher<NDevicesSnapshotManager::TSensorOptions, NDrive::TMultiSensor, TString, NDrive::TSensorId>;

private:
    R_FIELD(TDataSet, Sensors);

protected:
    virtual NThreading::TFuture<TFetchedDataSet> FetchFirst(const TFetchedKey& id, TDuration timeout) override;
    virtual NThreading::TFuture<TFetchedDataSet> FetchSecond(const TVector<TString>& imeis, const TFetchedKey& id, TDuration timeout, TMaybe<ui32> maxKeysPerQuery) override;

public:
    using TBase::TBase;

    void Prepare();

    virtual void FinishFetching(const TVector<TString>& imeis, const TDuration timeout) override;
};

