#include "manager.h"

#include "fetchers.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/hardware.h>

#include <drive/telematics/api/sensor/interface.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/iterator.h>
#include <rtline/util/types/string_pool.h>

#include <util/random/random.h>

namespace {
    TDevicesSnapshotManager::THashPtr EmptyHash() {
        return new TDevicesSnapshotManager::THash(TRect<TGeoCoord>(-180, -90, 180, 90));
    }

    template <std::contiguous_iterator Iterator>
    TVector<TArrayRef<typename std::iterator_traits<Iterator>::value_type>>
    SplitContainer(Iterator begin, Iterator end, ui32 chunkSize) {
        using ValueType = typename std::iterator_traits<Iterator>::value_type;
        TVector<TArrayRef<ValueType>> chunks;
        chunks.reserve((end - begin + chunkSize - 1) / chunkSize);
        for (auto it = begin; it < end; it += chunkSize) {
            chunks.emplace_back(std::to_address(it), Min(chunkSize, static_cast<ui32>(end - it)));
        }
        return chunks;
    }

    NThreading::TFuture<NSignalq::TV1EventsRetrieveResponse::TSerialNumberToDataMap>
    GetSignalqEventsBySerialNumbers(const ITaxiSignalqDrivematicsApiClient *client, TVector<TString>&& serialNumbers, ui32 maxChunkSize) {
        INFO_LOG << "Fetching SignalQ events, maxChunkSize = " << maxChunkSize;

        auto serialNumbersChunks = SplitContainer(serialNumbers.begin(), serialNumbers.end(), maxChunkSize);
        NThreading::TFutures<NSignalq::TV1EventsRetrieveResponse> eventsRetrieveResponseFutures;

        eventsRetrieveResponseFutures.reserve(serialNumbersChunks.size());
        for (auto& chunk : serialNumbersChunks) {
            NDrive::NSignalq::TV1EventsRetrieveRequest retrieveRequestParams;
            retrieveRequestParams.SetSerialNumbers(TVector<TString>(std::make_move_iterator(chunk.begin()), std::make_move_iterator(chunk.end())));
            auto signalqEventsResponse = client->GetSignalqEvents(retrieveRequestParams);
            eventsRetrieveResponseFutures.push_back(std::move(signalqEventsResponse));
        }

        auto waiter = NThreading::WaitAll(eventsRetrieveResponseFutures);
        return waiter.Apply([eventsRetrieveResponseFutures = std::move(eventsRetrieveResponseFutures)](const NThreading::TFuture<void>&) mutable {
            NSignalq::TV1EventsRetrieveResponse::TSerialNumberToDataMap result;
            INFO_LOG << "Fetched " << eventsRetrieveResponseFutures.size() << " event, building map...";
            for (auto&& futureMap : eventsRetrieveResponseFutures) {
                auto map = futureMap.ExtractValue().BuildSerialNumberToDataMap();
                result.insert(std::make_move_iterator(map.begin()), std::make_move_iterator(map.end()));
            }
            return result;
        });
    }
}

TDevicesSnapshot TDevicesSnapshot::Clone() const {
    TDevicesSnapshot result;
    *result.SnapshotsById = *SnapshotsById;
    *result.SnapshotsByIMEI = *SnapshotsByIMEI;
    *result.SignalqOnlySnapshotsById = *SignalqOnlySnapshotsById;
    *result.AreaCarsCount = *AreaCarsCount;
    *result.FutureAreaCarsCount = *FutureAreaCarsCount;
    return result;
}

const TRTDeviceSnapshot* TDevicesSnapshotManager::TDevicesSnapshotSelection::TConstSelectionIterator::SkipTo(const TString& id) {
    if (Advance(ItA, FinishA, id)) {
        return &ItA->second;
    }
    return nullptr;
}

TDeviceLocationOptions TDevicesSnapshotManager::GetLocationOptions() const {
    TDeviceLocationOptions result;
    const auto& settings = Server.GetSettings();
    TString bannedTags;
    if (settings.GetValue("linked_location_banned_tags", bannedTags) && bannedTags) {
        StringSplitter(bannedTags).Split(',').SkipEmpty().Collect(&result.BannedTags);
    } else {
        DEBUG_LOG << "linked_location_banned_tags is not set" << Endl;
    }
    if (!settings.GetValue("linked_location_merge_threshold", result.MergeThreshold)) {
        DEBUG_LOG << "linked_location_merge_threshold is not set" << Endl;
    }
    return result;
}

TDevicesSnapshotManager::THashPtr TDevicesSnapshotManager::GetGeoHash() const {
    const ui64 idx = GetCurrentIdx();
    TReadGuard rg(GeoHashLock[idx]);
    return GeoHash[idx];
}

TDevicesSnapshotManager::TDevicesSnapshotSelection TDevicesSnapshotManager::GetSnapshotsVector(const TSet<TString>& carIds) const {
    TConstDevicesSnapshot cds;
    {
        const ui64 idx = GetCurrentIdx();
        TReadGuard rg(CurrentSnapshotsLock[idx]);
        cds = CurrentSnapshots[idx];
    }
    return TDevicesSnapshotSelection(std::move(cds), carIds);
}

TDevicesSnapshot TDevicesSnapshotManager::GetSnapshots(const TSet<TString>& carIds) const {
    TDevicesSnapshot result;
    const auto inserter = [&result](const TString& id, const TRTDeviceSnapshot& ds) {
        result.ById().emplace(id, ds);
    };
    const ui64 idx = GetCurrentIdx();
    TReadGuard rg(CurrentSnapshotsLock[idx]);
    CurrentSnapshots[idx].ApplyForSnapshots(carIds, inserter);
    return result;
}

TConstDevicesSnapshot TDevicesSnapshotManager::GetSnapshots() const {
    const ui64 idx = GetCurrentIdx();
    TReadGuard rg(CurrentSnapshotsLock[idx]);
    return CurrentSnapshots[idx];
}

TRTDeviceSnapshot TDevicesSnapshotManager::GetSnapshot(const TString& carId) const {
    const ui64 idx = GetCurrentIdx();
    TReadGuard rg(CurrentSnapshotsLock[idx]);
    auto it = CurrentSnapshots[idx].find(carId);
    if (it != CurrentSnapshots[idx].end()) {
        return it->second;
    }
    return TRTDeviceSnapshot();
}

TAtomicSharedPtr<TRTDeviceSnapshot> TDevicesSnapshotManager::GetSnapshotPtr(const TString& carId) const {
    const ui64 idx = GetCurrentIdx();
    TReadGuard rg(CurrentSnapshotsLock[idx]);
    auto it = CurrentSnapshots[idx].find(carId);
    if (it != CurrentSnapshots[idx].end()) {
        return MakeAtomicShared<TRTDeviceSnapshot>(it->second);
    }
    return nullptr;
}

TAtomicSharedPtr<TRTDeviceSnapshot> TDevicesSnapshotManager::FetchSnapshot(const TString& carId, TInstant deadline) const {
    auto result = GetSnapshotPtr(carId);
    if (!result) {
        result = MakeAtomicShared<TRTDeviceSnapshot>();
    }
    auto object = Server.GetDriveDatabase().GetCarManager().GetObject(carId);
    auto imei = object ? object->GetIMEI() : TString{};
    if (!imei) {
        return result;
    } else {
        result->SetImei(imei);
    }

    auto client = Yensured(Server.GetSensorApi());
    auto asyncLocation = client->GetLocation(imei);
    auto asyncSensors = client->GetSensors(imei);
    asyncLocation.Wait(deadline);
    asyncSensors.Wait(deadline);
    auto futureDeadline = Now() + Options.FutureFilter;
    if (auto location = asyncLocation.HasValue() ? asyncLocation.GetValue().Get() : nullptr) {
        result->UpdateRawLocation(*location, futureDeadline);
    }
    if (asyncSensors.HasValue()) {
        auto sensors = asyncSensors.GetValue();
        result->UpdateSensors(sensors, futureDeadline);
    }
    NDrive::TEventLog::Log("FetchDeviceSnapshot", NJson::TMapBuilder
        ("car_id", carId)
        ("async_location", NJson::ToJson(asyncLocation))
        ("async_sensors", NJson::ToJson(asyncSensors))
        ("sensors", NJson::ToJson(result->GetSensors()))
        ("location", NJson::ToJson(result->GetLocation()))
    );
    return result;
}

void TDevicesSnapshotManager::DropCache() {
    for (ui32 i = 0; i < BucketsNumber; ++i) {
        {
            TWriteGuard guard(CurrentSnapshotsLock[i]);
            CurrentSnapshots[i] = TDevicesSnapshot();
        }
        {
            TWriteGuard guard(GeoHashLock[i]);
            GeoHash[i] = EmptyHash();
        }
    }
}

bool TDevicesSnapshotManager::Refresh() {
    INFO_LOG << GetName() << " start" << Endl;
    auto gCars = Server.GetDriveDatabase().GetCarManager().GetCachedObjectsVector();

    TMap<TString, TString> carIdToSerialSignalq;
    TVector<TString> signalqSerialNumbers;
    auto mapToAttachments = Server.GetDriveDatabase().GetCarAttachmentAssignments().GetAssignmentsOfType(TVector<TString> {}, EDocumentAttachmentType::CarSignalDevice, TInstant::Zero());
    for (auto it : mapToAttachments) {
        auto carSignalDeviceImpl = dynamic_cast<const TCarSignalDevice*>(it.second.Get());
        if (carSignalDeviceImpl) {
            auto serialNumber = carSignalDeviceImpl->GetSerialNumber();
            carIdToSerialSignalq[it.first] = serialNumber;
            signalqSerialNumbers.push_back(std::move(serialNumber));
        }
    }

    NThreading::TFuture<NDrive::NSignalq::TV1StatusesRetrieveResponse::TSerialNumberToDataMap> signalqSerialsToStatusFuture = NThreading::MakeFuture(NDrive::NSignalq::TV1StatusesRetrieveResponse::TSerialNumberToDataMap());
    if (!signalqSerialNumbers.empty()) {
        auto client = Server.GetTaxiSignalqDrivematicsApiClient();
        if (!client) {
            ERROR_LOG << "Cant fetch signalq statuses: no client" << Endl;
        } else {
            NDrive::NSignalq::TV1StatusesRetrieveRequest retriveRequestParams;
            retriveRequestParams.SetSerialNumbers(signalqSerialNumbers);
            const auto start = Now();
            INFO_LOG << "Trying to fetch " << signalqSerialNumbers.size() << " SignalqStatuses " << Endl;
            auto signalqStatusesResponse = client->GetSignalqStatuses(retriveRequestParams);
            signalqSerialsToStatusFuture = signalqStatusesResponse.Apply([start](const NThreading::TFuture<NDrive::NSignalq::TV1StatusesRetrieveResponse>& response) {
                if (response.HasValue()) {
                    const auto finish = Now();
                    const auto duration = finish - start;
                    INFO_LOG << "Fetched SignalqStatuses, duration: " << duration << Endl;
                    const auto itemsSize = response.GetValue().GetItems().size();
                    INFO_LOG << "Fetched SignalqStatuses, amount: " << itemsSize << Endl;
                    return response.GetValue().BuildSerialNumberToDataMap();
                } else {
                    ythrow yexception() << "Cant fetch signalq statuses: no response, exp: " << NThreading::GetExceptionMessage(response) << Endl;
                }
            });
        }
    }

    NThreading::TFuture signalqSerialsToEventsFuture = NThreading::MakeFuture(NDrive::NSignalq::TV1EventsRetrieveResponse::TSerialNumberToDataMap());
    if (!signalqSerialNumbers.empty()) {
        auto client = Server.GetTaxiSignalqDrivematicsApiClient();
        if (!client) {
            ERROR_LOG << "Cant fetch signalq events: no client" << Endl;
        } else {
            auto optionalChunkSize = Server.GetSettings().GetValue<ui32>("signalq.signalq_events.events_retrieve_chunk_size");
            const auto chunkSize = optionalChunkSize && *optionalChunkSize != 0 ? *optionalChunkSize : 1000;
            INFO_LOG << "Trying to fetch " << signalqSerialNumbers.size() << " SignalqEvents " << Endl;
            const auto start = Now();
            auto signalqEventsResponse = GetSignalqEventsBySerialNumbers(client, std::move(signalqSerialNumbers), chunkSize);
            signalqSerialsToEventsFuture = signalqEventsResponse.Apply([start](const NThreading::TFuture<NDrive::NSignalq::TV1EventsRetrieveResponse::TSerialNumberToDataMap>& response){
                if (response.HasValue()) {
                    const auto finish = Now();
                    const auto duration = finish - start;
                    INFO_LOG << "Fetched SignalqEvents, duration: " << duration << Endl;
                    const auto itemsSize = response.GetValue().size();
                    INFO_LOG << "Fetched SignalqEvents, amount: " << itemsSize << Endl;
                    return response.GetValue();
                } else {
                    ythrow yexception() << "Cant fetch signalq events: no response, exp: " << NThreading::GetExceptionMessage(response) << Endl;
                }
            });
        }
    }

    TMap<TString, TString> imeiToCarId;
    TVector<TString> imeis;
    TVector<TString> carIdsOnlySignalq;
    for (auto&& i : gCars) {
        if (!!i.GetIMEI()) {
            imeiToCarId.emplace(i.GetIMEI(), i.GetId());
            imeis.emplace_back(i.GetIMEI());
        } else if (auto it = carIdToSerialSignalq.find(i.GetId()); it != carIdToSerialSignalq.end()) {
            carIdsOnlySignalq.emplace_back(i.GetId());
        }
    }
    std::sort(imeis.begin(), imeis.end());
    std::sort(carIdsOnlySignalq.begin(), carIdsOnlySignalq.end());

    TDevicesSnapshot snapshots;
    {
        const ui64 idx = GetCurrentIdx();
        TReadGuard rg(CurrentSnapshotsLock[idx]);
        snapshots = CurrentSnapshots[idx].Clone();
    }

    INFO_LOG << "updating Areas from: " << State.AreasLastUpdate << "/" << State.InsertedAreasEventsCount << Endl;
    const auto& areaManager = Server.GetDriveDatabase().GetAreaManager();
    const bool areasUpdated = State.InsertedAreasEventsCount != areaManager.GetInsertedEventsCount();
    if (areasUpdated) {
        State.AreasLastUpdate = std::max(State.AreasLastUpdate, areaManager.GetLastRefreshInstant());
        State.InsertedAreasEventsCount = areaManager.GetInsertedEventsCount();
    }
    INFO_LOG << "updated Areas to: " << State.AreasLastUpdate << "/" << State.InsertedAreasEventsCount << "/" << areaManager.GetInsertedEventsCount() << Endl;

    THeartbeatsFetcher heartbeats(Server, "heartbeats", Options.HeartbeatOptions, State.HeartbeatLastUpdate);
    TLocationsFetcher locations(Server, "locations", Options.LocationOptions, State.LocationLastUpdate);
    TSensorsFetcher fSensors(Server, "sensors", Options.SensorOptions, State.SensorLastUpdate);

    auto areaTagsPrecision = Server.GetSettings().GetValueDef<double>("snapshots.area_tags_precision", 20);

    TInstant now = Now();
    heartbeats.Prepare();
    locations.Prepare();
    fSensors.Prepare();
    TDuration fetchTimeout = TDuration::Zero();
    if (Options.EnableFirstStageFetching) {
        fetchTimeout += Options.SensorTimeout;
        heartbeats.PrefetchFirst(Options.SensorTimeout);
        locations.PrefetchFirst(Options.SensorTimeout);
        fSensors.PrefetchFirst(Options.SensorTimeout);
    }
    if (Options.EnableSecondStageFetching) {
        fetchTimeout += Options.SensorTimeout;
        auto maxKeysPerQuery = Server.GetSettings().GetValue<ui32>("snapshots.max_keys_per_query");
        heartbeats.PrefetchSecond(imeis, Options.SensorTimeout, maxKeysPerQuery);
        locations.PrefetchSecond(imeis, Options.SensorTimeout, maxKeysPerQuery);
        fSensors.PrefetchSecond(imeis, Options.SensorTimeout, maxKeysPerQuery);
    }

    heartbeats.FinishFetching(imeis, fetchTimeout);
    locations.FinishFetching(imeis, fetchTimeout);
    fSensors.FinishFetching(imeis, fetchTimeout);

    TInstant futureDeadline = now + Options.FutureFilter;
    {
        auto itSensors = fSensors.MutableSensors().begin();
        auto itLocation = locations.GetRawLocations().begin();
        auto itLinkedLocation = locations.GetLinkedLocations().begin();
        auto itLBSLocation = locations.GetLBSLocations().begin();
        auto itHeadLocation = locations.GetHeadLocations().begin();
        auto itBeaconsLocation = locations.GetBeaconsLocations().begin();
        auto itGeocodedLocation = locations.GetGeocodedLocations().begin();
        auto itIMEI = imeiToCarId.begin();
        auto itSnapshot = snapshots.ByIMEI().begin();

        TMap<TString, ui32>& areaCarsCount = snapshots.MutableAreaCarsCount();
        areaCarsCount.clear();
        auto itHeartbeats = heartbeats.GetHeartbeats().begin();
        auto itConfiguratorHeartbeats = heartbeats.GetConfiguratorHeartbeats().begin();

        auto& signalqOnlySnapshots = snapshots.SignalqOnlyById();
        NDrive::NSignalq::TV1StatusesRetrieveResponse::TSerialNumberToDataMap signalqSerialsToStatusesData;
        signalqSerialsToStatusFuture.Wait();
        if (signalqSerialsToStatusFuture.HasValue()) {
            signalqSerialsToStatusesData = signalqSerialsToStatusFuture.ExtractValue();
        } else {
            ERROR_LOG << "Failed to fetch signalq statuses" << NThreading::GetExceptionMessage(signalqSerialsToStatusFuture) << Endl;
        }

        NDrive::NSignalq::TV1EventsRetrieveResponse::TSerialNumberToDataMap signalqSerialsToEventsData;
        signalqSerialsToEventsFuture.Wait();
        if (signalqSerialsToEventsFuture.HasValue()) {
            signalqSerialsToEventsData = signalqSerialsToEventsFuture.ExtractValue();
        } else {
            ERROR_LOG << "Failed to fetch signalq events" << NThreading::GetExceptionMessage(signalqSerialsToEventsFuture) << Endl;
        }

        TVector<TString> signalqWithNewTelematicsCarIds;
        for (; itIMEI != imeiToCarId.end(); ++itIMEI) {
            const TString& imei = itIMEI->first;
            if (!Advance(itSnapshot, snapshots.ByIMEI().end(), imei)) {
                // We do not want to lose signalq snapshot data if snapshot obtained imei
                if (auto itExistingSignalqSnapshot = signalqOnlySnapshots.find(itIMEI->second);
                        itExistingSignalqSnapshot != signalqOnlySnapshots.end()) {
                    itSnapshot = snapshots.ByIMEI().emplace(itIMEI->first, itExistingSignalqSnapshot->second).first;
                    signalqWithNewTelematicsCarIds.push_back(itIMEI->second);
                } else {
                    itSnapshot = snapshots.ByIMEI().emplace(itIMEI->first, TRTDeviceSnapshot()).first;
                    itSnapshot->second.SetImei(imei);
                }
            }
            auto& snapshot = itSnapshot->second;
            Y_ASSERT(snapshot.GetImei() == imei);

            if (auto it = carIdToSerialSignalq.find(itIMEI->second); it != carIdToSerialSignalq.end()) {
                if (auto statusIt = signalqSerialsToStatusesData.find(it->second); statusIt != signalqSerialsToStatusesData.end()) {
                    snapshot.UpdateSignalqStatusAndLocation(statusIt->second, futureDeadline, false);
                }
                if (auto eventIt = signalqSerialsToEventsData.find(it->second); eventIt != signalqSerialsToEventsData.end()) {
                    snapshot.UpdateLastSignalqEvent(eventIt->second);
                }
            }
            TSet<TString> ids;
            TSet<TString> idsGuarantee;
            TVector<TDBTag> hardTags;
            TSet<TString> tagsInternal;
            TSet<TString> tagsGuarantee;
            TSet<TString> tagsPotentially;
            TSet<TString> clusterIds;
            const auto actor = [&](const TFullAreaInfo& info, const TInternalPointContext::EInternalPointType pointType) {
                if (pointType == TInternalPointContext::EInternalPointType::InternalGuarantee || pointType == TInternalPointContext::EInternalPointType::InternalPotentially) {
                    ids.insert(info.GetArea().GetIdentifier());
                    hardTags.insert(hardTags.end(), info.GetTags().begin(), info.GetTags().end());
                }
                switch (pointType) {
                    case TInternalPointContext::EInternalPointType::InternalGuarantee:
                        idsGuarantee.emplace(info.GetId());
                        tagsGuarantee.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
                        tagsGuarantee.insert(info.GetHardTags().begin(), info.GetHardTags().end());
                    case TInternalPointContext::EInternalPointType::InternalPotentially:
                        if (info.GetArea().IsCluster()) {
                            clusterIds.emplace(info.GetArea().GetIdentifier());
                        }
                        tagsInternal.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
                        tagsInternal.insert(info.GetHardTags().begin(), info.GetHardTags().end());
                    case TInternalPointContext::EInternalPointType::ExternalPotentially:
                        tagsPotentially.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
                        tagsPotentially.insert(info.GetHardTags().begin(), info.GetHardTags().end());
                    case TInternalPointContext::EInternalPointType::ExternalGuarantee:
                        break;
                };
                return true;
            };

            const auto updateLocationWithTags = [&](auto& locationsIterator, const auto locationsEnd, const auto& locationGetter, const auto& locationUpdater) {
                bool locationUpdated = false;
                if (Advance(locationsIterator, locationsEnd, imei)) {
                    const auto newLocation = locationsIterator->second;
                    locationUpdated = locationUpdater(snapshot, newLocation, futureDeadline);
                }

                if (const TMaybe<NDrive::TLocation> location = locationGetter(snapshot); location && (areasUpdated || locationUpdated)) {
                    if (!areaManager.ProcessTagsInPointWithInternalTypes(location->GetCoord(), actor, {}, TInstant::Zero(), areaTagsPrecision)) {
                        ERROR_LOG << "could not process area tags in " << location->ToJson().GetStringRobust() << Endl;
                        return false;
                    } else {
                        return true;
                    }
                }
                return false;
            };

            const bool needSetRawLocationTags = updateLocationWithTags(itLocation, locations.GetRawLocations().end(),
                                    [](const auto& snapshot) {
                                        return snapshot.GetRawLocation();
                                    },
                                    [](auto& snapshot, const auto& location, auto futureDeadline) {
                                        return snapshot.UpdateRawLocation(location, futureDeadline);
                                    });

            const bool needSetBeaconsLocationTags = updateLocationWithTags(itBeaconsLocation, locations.GetBeaconsLocations().end(),
                                    [](const auto& snapshot) {
                                        return snapshot.GetBeaconsLocation();
                                    },
                                    [](auto& snapshot, const auto& location, auto futureDeadline) {
                                        return snapshot.UpdateBeaconsLocation(location, futureDeadline);
                                    });

            if (needSetRawLocationTags || needSetBeaconsLocationTags) {
                snapshot.SetLocationTags(std::move(tagsInternal));
                snapshot.SetLocationTagsGuarantee(std::move(tagsGuarantee));
                snapshot.SetLocationTagsPotentially(std::move(tagsPotentially));
                snapshot.SetAreaIds(std::move(ids));
                snapshot.SetAreaIdsGuarantee(std::move(idsGuarantee));
                snapshot.SetHardTags(std::move(hardTags));
                snapshot.SetClusterIds(std::move(clusterIds));
            }

            for (auto&& areaId : snapshot.GetAreaIdsGuarantee()) {
                ++areaCarsCount[areaId];
            }
            if (Advance(itLinkedLocation, locations.GetLinkedLocations().end(), imei)) {
                snapshot.UpdateLinkedLocation(itLinkedLocation->second, futureDeadline);
            }
            if (Advance(itLBSLocation, locations.GetLBSLocations().end(), imei)) {
                const auto& location = itLBSLocation->second;
                bool locationUpdated = snapshot.UpdateLBSLocation(location, futureDeadline);
                if (locationUpdated || areasUpdated) {
                    snapshot.SetLBSLocationTags(areaManager.GetTagsInPoint(location.GetCoord()));
                }
            }
            if (Advance(itHeadLocation, locations.GetHeadLocations().end(), imei)) {
                snapshot.UpdateHeadLocation(itHeadLocation->second, futureDeadline);
            }
            if (Advance(itGeocodedLocation, locations.GetGeocodedLocations().end(), imei)) {
                snapshot.UpdateGeocodedLocation(itGeocodedLocation->second, futureDeadline);
            }
            if (Advance(itSensors, fSensors.MutableSensors().end(), imei)) {
                snapshot.UpdateSensors(itSensors->second, futureDeadline);
            }
            if (Advance(itHeartbeats, heartbeats.GetHeartbeats().end(), imei)) {
                snapshot.UpdateHeartbeats(itHeartbeats->second, futureDeadline);
            }
            if (Advance(itConfiguratorHeartbeats, heartbeats.GetConfiguratorHeartbeats().end(), imei)) {
                snapshot.UpdateConfiguratorHeartbeat(itConfiguratorHeartbeats->second, futureDeadline);
            }
        }

        auto itSignalqSnapshot = signalqOnlySnapshots.begin();
        for (const auto& carId : carIdsOnlySignalq) {
            if (!Advance(itSignalqSnapshot, signalqOnlySnapshots.end(), carId)) {
                itSignalqSnapshot = signalqOnlySnapshots.emplace(carId, TRTDeviceSnapshot()).first;
            }
            auto& snapshot = itSignalqSnapshot->second;
            if (auto it = carIdToSerialSignalq.find(carId); it != carIdToSerialSignalq.end()) {
                if (auto statusIt = signalqSerialsToStatusesData.find(it->second); statusIt != signalqSerialsToStatusesData.end()) {
                    snapshot.UpdateSignalqStatusAndLocation(statusIt->second, futureDeadline, true);
                }
                if (auto eventIt = signalqSerialsToEventsData.find(it->second); eventIt != signalqSerialsToEventsData.end()) {
                    snapshot.UpdateLastSignalqEvent(eventIt->second);
                }
            }
        }

        {
            snapshots.BuildById(imeiToCarId, carIdsOnlySignalq, Server);
            snapshots.RemoveDuplicatingSignalqOnlySnapshotsById(signalqWithNewTelematicsCarIds);

            const ui64 idx = GetCurrentIdx(1);
            {
                TWriteGuard rg(CurrentSnapshotsLock[idx]);
                LastRefreshInstant = Now();
                CurrentSnapshots[idx] = snapshots;
            }

            if (State.HashLastUpdate + Options.HashUpdatePeriod < now) {
                INFO_LOG << "start building geo index" << Endl;
                TMap<TString, TString> models;
                for (auto&& i : gCars) {
                    models[i.GetId()] = i.GetModel();
                }

                auto locationOptions = GetLocationOptions();
                THashPtr geoHash = EmptyHash();
                NDrive::TLocation location;
                for (auto&& [id, snapshot] : snapshots.ById()) {
                    auto location = snapshot.GetLocation(TDuration::Max(), locationOptions);
                    if (location) {
                        geoHash->AddObject(THashedDevice(
                            id,
                            models[id],
                            std::move(*location)
                        ));
                    }
                }
                INFO_LOG << "finish building geo index:" << geoHash->Size() << Endl;
                State.HashLastUpdate = now;
                {
                    TWriteGuard rg(GeoHashLock[idx]);
                    GeoHash[idx] = geoHash;
                }
            }
            SetCurrentIdx(idx);
        }
    }
    INFO_LOG << GetName() << " finish" << Endl;
    return true;
}

bool TDevicesSnapshotManager::MetricSignal() {
    if (LastRefreshInstant) {
        TUnistatSignalsCache::SignalLastX("devices_snapshot_manager", "freshness", (Now() - *LastRefreshInstant).MilliSeconds());
    }
    return true;
}

TDevicesSnapshotManager::TDevicesSnapshotManager(const NDrive::IServer& server, const TOptions& options)
    : IAutoActualization("devices_snapshot_manager")
    , Server(server)
    , Options(options)
{
    for (ui32 i = 0; i < BucketsNumber; ++i) {
        GeoHash[i] = EmptyHash();
    }
    State.AreasLastUpdate = TInstant::Zero();
    State.HashLastUpdate = TInstant::Zero();
    for (auto&& location : NDevicesSnapshotManager::AllLocations) {
        State.LocationLastUpdate[location] = TInstant::Zero();
    }
    for (auto&& sensor : NDevicesSnapshotManager::AllSensors) {
        State.SensorLastUpdate[sensor] = TInstant::Zero();
    }
    SetPeriod(Options.UpdatePeriod);
    SetMetricPeriod(TDuration::Seconds(1));
    RegisterGlobalMessageProcessor(this);
}

TDevicesSnapshotManager::~TDevicesSnapshotManager() {
    UnregisterGlobalMessageProcessor(this);
}

TString TDevicesSnapshotManager::Name() const {
    return GetName();
}

bool TDevicesSnapshotManager::Process(IMessage* message_) {
    if (auto message = message_->As<NDrive::TCacheRefreshMessage>()) {
        if (message->GetComponents().contains(GetName())) {
            DropCache();
            Refresh();
        }
        return true;
    }
    return false;
}

bool TDevicesSnapshotManager::UpdatedTo(TInstant threshold /*= TInstant::Zero()*/) const {
    if (State.AreasLastUpdate <= threshold) {
        WARNING_LOG << "AreasLastUpdate " << State.AreasLastUpdate << " <= " << threshold << Endl;
        return false;
    }
    for (auto&&[location, timestamp] : State.LocationLastUpdate) {
        if (timestamp <= threshold) {
            WARNING_LOG << "Location " << location << " last updated at " << timestamp << " <= " << threshold << Endl;
            return false;
        }
    }
    for (auto&&[sensor, timestamp] : State.SensorLastUpdate) {
        if (timestamp <= threshold) {
            WARNING_LOG << "Sensor " << sensor.GetName() << " last updated at " << timestamp << " <= " << threshold << Endl;
            return false;
        }
    }
    return true;
}

bool TDevicesSnapshotManager::Wait(TInstant deadline /*= TInstant::Max()*/, TInstant threshold /*= TInstant::Zero()*/) const {
    bool result = false;
    while (true) {
        result = UpdatedTo(threshold);
        if (result) {
            break;
        }
        auto now = Now();
        if (now > deadline) {
            break;
        }
        auto pause = std::min(GetPeriod(), deadline - now);
        Sleep(pause);
    }
    return false;
}

void TDevicesSnapshot::BuildById(
    const TMap<TString, TString>& imeiToCarId,
    const TVector<TString>& carIdsOnlySignalq,
    const NDrive::IServer& server
) {
    SnapshotsById->clear();

    auto itId = imeiToCarId.begin();
    for (auto&& itSnapshot : *SnapshotsByIMEI) {
        if (Advance(itId, imeiToCarId.end(), itSnapshot.first)) {
            SnapshotsById->emplace(itId->second, itSnapshot.second);
        }
    }

    auto itCarIdsOnlySignalq = carIdsOnlySignalq.begin();
    for (auto&& itSignalqSnapshot : *SignalqOnlySnapshotsById) {
        if (AdvanceSimple(itCarIdsOnlySignalq, carIdsOnlySignalq.end(), itSignalqSnapshot.first)) {
            SnapshotsById->emplace(*itCarIdsOnlySignalq, itSignalqSnapshot.second);
        }
    }

    TMap<TString, TGeoCoord> futureLocations = server.GetDriveDatabase().GetObjectFuturePositions();
    auto itFutureLocation = futureLocations.begin();
    FutureAreaCarsCount->clear();

    for (auto&& itSnapshot : *SnapshotsById) {
        if (Advance(itFutureLocation, futureLocations.end(), itSnapshot.first)) {
            if (itSnapshot.second.UpdateFutureLocation(itFutureLocation->second)) {
                itSnapshot.second.SetFutureAreaIds(server.GetDriveDatabase().GetAreaManager().GetAreaIdsInPoint(itFutureLocation->second, TInstant::Zero()));
            }
        } else {
            itSnapshot.second.DropFutureLocation();
        }
        for (auto&& areaId : itSnapshot.second.GetFutureAreaIds()) {
            ++(*FutureAreaCarsCount)[areaId];
        }
    }
}

void TDevicesSnapshot::RemoveDuplicatingSignalqOnlySnapshotsById(const TVector<TString>& signalqWithNewTelematicsCarIds) {
    for (const auto& carId : signalqWithNewTelematicsCarIds) {
        if (auto it = SignalqOnlySnapshotsById->find(carId); it != SignalqOnlySnapshotsById->end()) {
            SignalqOnlySnapshotsById->erase(it);
        }
    }
}

TSet<TString> TConstDevicesSnapshot::GetNearestIds(const TGeoCoord& c, const ui32 count, const bool withIncorrectLocation, const bool withClusters, const TSet<TString>* preparedIds) const {
    TSet<TString> result;
    TVector<std::pair<double, TString>> carsWithDistance;
    carsWithDistance.reserve(SnapshotsById->size());
    const TSet<TString>& preparedIdsLocal = preparedIds ? *preparedIds : Default<TSet<TString>>();
    auto itIds = preparedIdsLocal.begin();
    for (auto&& i : *SnapshotsById) {
        if (preparedIds && !AdvanceSimple(itIds, preparedIdsLocal.end(), i.first)) {
            continue;
        }
        if (withClusters && i.second.GetClusterIds().size()) {
            result.emplace(i.first);
        } else {
            TMaybe<NDrive::TLocation> location = i.second.GetRawLocation();
            if (!!location) {
                carsWithDistance.emplace_back(std::make_pair(c.GetLengthTo(location->GetCoord()), i.first));
            } else if (withIncorrectLocation) {
                result.emplace(i.first);
            }
        }
    }
    std::sort(carsWithDistance.begin(), carsWithDistance.end());
    if (count < carsWithDistance.size()) {
        carsWithDistance.resize(count);
    }
    for (auto&& i : carsWithDistance) {
        result.emplace(std::move(i.second));
    }
    return result;
}
